/*=========================================================================

  Library:   CTK

  Copyright (c) Kitware Inc.

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

      http://www.commontk.org/LICENSE

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.

=========================================================================*/

// Qt includes
#include <QDebug>
#include <QDir>
#include <QTextStream>
#include <QTime>

// CTK includes
#include "ctkCoreTestingMacros.h"
#include "ctkScopedCurrentDir.h"
#include "ctkUtils.h"

// STD includes
#include <cstdlib>
#include <iostream>

namespace
{
//-----------------------------------------------------------------------------
int createFile(int line, const QDir& dir, const QString& relativePath, const QString& fileName)
{
  QDir newDir(dir);
  newDir.mkpath(relativePath);
  newDir.cd(relativePath);
  QString filePath = QFileInfo(newDir, fileName).filePath();
  QFile file(filePath);
  file.open(QIODevice::Text | QIODevice::WriteOnly);
  QTextStream out(&file);
  out << "Generated by ctkUtilsTest4" << endl;
  file.close();

  if (!QFile::exists(filePath))
    {
    std::cerr << "Line " << line << " - Failed to create file" << qPrintable(filePath) << std::endl;
    return EXIT_FAILURE;
    }

  return EXIT_SUCCESS;
}

} // end of anonymous namespace


//-----------------------------------------------------------------------------
int ctkUtilsTest4(int argc, char * argv [] )
{
  Q_UNUSED(argc);
  Q_UNUSED(argv);

  // --------------------------------------------------------------------------
  // Test removeDirRecursively(const QString & dirName);
  // --------------------------------------------------------------------------
  
  QDir tmp = QDir::temp();
  QString temporaryDirName =
      QString("ctkUtilsTest4.%1").arg(QTime::currentTime().toString("hhmmsszzz"));

  // Attempt to delete nonexistent relative directory
  QString nonexistentRelativeDirPath = temporaryDirName;
  CHECK_BOOL(QDir(nonexistentRelativeDirPath).exists(), false);
  if (ctk::removeDirRecursively(nonexistentRelativeDirPath))
    {
    std::cerr << "Line " << __LINE__ << " - Problem with ctk::removeDirRecursively() !"
              << " - It should fail to delete nonexistent directory: " <<
              qPrintable(nonexistentRelativeDirPath)<< std::endl;
    return EXIT_FAILURE;
    }

  // Attempt to delete nonexistent absolute directory
  QString nonexistentAbsoluteDirPath = QFileInfo(tmp, temporaryDirName).absoluteFilePath();
  CHECK_BOOL(QDir(nonexistentAbsoluteDirPath).exists(), false);
  if (ctk::removeDirRecursively(nonexistentAbsoluteDirPath))
    {
    std::cerr << "Line " << __LINE__ << " - Problem with ctk::removeDirRecursively() !"
              << " - It should fail to delete nonexistent directory: " <<
              qPrintable(nonexistentAbsoluteDirPath)<< std::endl;
    return EXIT_FAILURE;
    }

  // Absolute path
  {
    // Create a directory structure
    CHECK_BOOL(tmp.mkdir(temporaryDirName), true);
    QDir caseTmp = QDir(tmp);
    CHECK_BOOL(caseTmp.cd(temporaryDirName), true);
    CHECK_EXIT_SUCCESS(createFile(__LINE__, caseTmp, "foo", "a.txt"));
    CHECK_EXIT_SUCCESS(createFile(__LINE__, caseTmp, "foo/bar", "b.txt"));
    CHECK_EXIT_SUCCESS(createFile(__LINE__, caseTmp, "foo/zoo", "c.txt"));

    if (!ctk::removeDirRecursively(caseTmp.absolutePath()))
      {
      std::cerr << "Line " << __LINE__ << " - Problem with ctk::removeDirRecursively()"
                << " - Failed to delete directory:" << qPrintable(caseTmp.absolutePath()) << std::endl;
      return EXIT_FAILURE;
      }
    CHECK_BOOL(caseTmp.exists(), false);
  }

  // Relative path
  {
    // Create a directory structure
    CHECK_BOOL(tmp.mkdir(temporaryDirName), true);
    QDir caseTmp = QDir(tmp);
    CHECK_BOOL(caseTmp.cd(temporaryDirName), true);
    CHECK_EXIT_SUCCESS(createFile(__LINE__, caseTmp, "foo", "a.txt"));
    CHECK_EXIT_SUCCESS(createFile(__LINE__, caseTmp, "foo/bar", "b.txt"));
    CHECK_EXIT_SUCCESS(createFile(__LINE__, caseTmp, "foo/zoo", "c.txt"));

    ctkScopedCurrentDir currentDir(tmp.path());

    if (!ctk::removeDirRecursively(temporaryDirName))
      {
      std::cerr << "Line " << __LINE__ << " - Problem with ctk::removeDirRecursively()"
                << " - Failed to delete directory:" << qPrintable(temporaryDirName)
                << " - Current directory:" << qPrintable(QDir::currentPath())
                << std::endl;
      return EXIT_FAILURE;
      }
    CHECK_BOOL(caseTmp.exists(), false);
  }

  return EXIT_SUCCESS;
}
