/*=========================================================================

  Library:   CTK

  Copyright (c) Kitware Inc.

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

      http://www.commontk.org/LICENSE

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.

=========================================================================*/

// Qt includes
#include <QDebug>
#include <QDir>
#include <QTextStream>
#include <QTime>

// CTK includes
#include "ctkUtils.h"

// STD includes
#include <cstdlib>
#include <iostream>

namespace
{
//-----------------------------------------------------------------------------
bool createFile(int line, const QDir& dir, const QString& relativePath, const QString& fileName)
{
  QDir newDir(dir);
  newDir.mkpath(relativePath);
  newDir.cd(relativePath);
  QString filePath = QFileInfo(newDir, fileName).filePath();
  QFile file(filePath);
  file.open(QIODevice::Text | QIODevice::WriteOnly);
  QTextStream out(&file);
  out << "Generated by ctkUtilsCopyDirRecursivelyTest1" << endl;
  file.close();

  if (!QFile::exists(filePath))
    {
    std::cerr << "Line " << line << " - Failed to create file" << qPrintable(filePath) << std::endl;
    return false;
    }

  return true;
}

} // end of anonymous namespace


//-----------------------------------------------------------------------------
int ctkUtilsCopyDirRecursivelyTest1(int argc, char * argv [] )
{
  Q_UNUSED(argc);
  Q_UNUSED(argv);

  // --------------------------------------------------------------------------
  // Test copyDirRecursively(const QString &srcPath, const QString &dstPath);
  // --------------------------------------------------------------------------

  QDir tmp = QDir::temp();
  QString temporaryDirName =
      QString("ctkUtilsCopyDirRecursivelyTest1.%1").arg(QTime::currentTime().toString("hhmmsszzz"));

  {
  // Attempt to copy nonexistent relative directory
  QString nonexistentRelativeDirPath = temporaryDirName;
  if (ctk::copyDirRecursively(nonexistentRelativeDirPath, tmp.absolutePath()))
    {
    std::cerr << "Line " << __LINE__ << " - Problem with ctk::copyDirRecursively() !"
              << " - It should fail to copy nonexistent directory: " <<
              qPrintable(nonexistentRelativeDirPath)<< std::endl;
    return EXIT_FAILURE;
    }
  }

  {
  // Attempt to copy nonexistent absolute directory
  QString nonexistentAbsoluteDirPath = tmp.absolutePath() + "/" + temporaryDirName;
  if (ctk::copyDirRecursively(nonexistentAbsoluteDirPath, tmp.absolutePath()))
    {
    std::cerr << "Line " << __LINE__ << " - Problem with ctk::copyDirRecursively() !"
              << " - It should fail to delete nonexistent directory: " <<
              qPrintable(nonexistentAbsoluteDirPath)<< std::endl;
    return EXIT_FAILURE;
    }
  }

  // Create a directory structure
  tmp.mkdir(temporaryDirName);
  tmp.cd(temporaryDirName);
  if (!createFile(__LINE__, tmp, "foo", "a.txt"))
    {
    return EXIT_FAILURE;
    }
  if (!createFile(__LINE__, tmp, "foo/bar", "b.txt"))
    {
    return EXIT_FAILURE;
    }
  if (!createFile(__LINE__, tmp, "foo/zoo", "c.txt"))
    {
    return EXIT_FAILURE;
    }

  {
  QString srcPath(tmp.path() + "/foo");
  QString destPath(tmp.path() + "/dest");
  if (!ctk::copyDirRecursively(srcPath, destPath))
    {
    std::cerr << "Line " << __LINE__ << " - Problem with ctk::copyDirRecursively()"
              << " - Failed to copy directory: " << qPrintable(srcPath)
              << "into" << qPrintable(destPath) << std::endl;
    return EXIT_FAILURE;
    }
  }

  {
  // Atempt to copy a directory into itself
  QString srcPath(tmp.path());
  QString destPath(tmp.path() + "/dest");
  if (ctk::copyDirRecursively(srcPath, destPath))
    {
    std::cerr << "Line " << __LINE__ << " - Problem with ctk::copyDirRecursively()"
              << " - Copy of directory: " << qPrintable(srcPath)
              << "into itself " << qPrintable(destPath)
              << " should fail !" << std::endl;
    return EXIT_FAILURE;
    }
  }

  return EXIT_SUCCESS;
}
