/*=========================================================================

  Library:   CTK

  Copyright (c) Kitware Inc.

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

      http://www.commontk.org/LICENSE

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.

=========================================================================*/

// Qt includes
#include <QDebug>
#include <QDir>
#include <QTextStream>
#include <QTime>

// CTK includes
#include "ctkUtils.h"

// STD includes
#include <cstdlib>
#include <iostream>

namespace
{
//-----------------------------------------------------------------------------
bool createFile(int line, const QDir& dir, const QString& relativePath, const QString& fileName)
{
  QDir newDir(dir);
  newDir.mkpath(relativePath);
  newDir.cd(relativePath);
  QString filePath = QFileInfo(newDir, fileName).filePath();
  QFile file(filePath);
  file.open(QIODevice::Text | QIODevice::WriteOnly);
  QTextStream out(&file);
  out << "Generated by ctkUtilsTest4" << endl;
  file.close();

  if (!QFile::exists(filePath))
    {
    std::cerr << "Line " << line << " - Failed to create file" << qPrintable(filePath) << std::endl;
    return false;
    }

  return true;
}

} // end of anonymous namespace


//-----------------------------------------------------------------------------
int ctkUtilsTest4(int argc, char * argv [] )
{
  Q_UNUSED(argc);
  Q_UNUSED(argv);

  // --------------------------------------------------------------------------
  // Test removeDirRecursively(const QString & dirName);
  // --------------------------------------------------------------------------
  
  QDir tmp = QDir::temp();
  QString temporaryDirName =
      QString("ctkUtilsTest4.%1").arg(QTime::currentTime().toString("hhmmsszzz"));

  // Attempt to delete nonexistent relative directory
  QString nonexistentRelativeDirPath = temporaryDirName;
  if (ctk::removeDirRecursively(nonexistentRelativeDirPath))
    {
    std::cerr << "Line " << __LINE__ << " - Problem with ctk::removeDirRecursively() !"
              << " - It should fail to delete nonexistent directory: " <<
              qPrintable(nonexistentRelativeDirPath)<< std::endl;
    return EXIT_FAILURE;
    }

  // Attempt to delete nonexistent absolute directory
  QString nonexistentAbsoluteDirPath = QFileInfo(tmp, temporaryDirName).dir().absolutePath();
  if (ctk::removeDirRecursively(nonexistentAbsoluteDirPath))
    {
    std::cerr << "Line " << __LINE__ << " - Problem with ctk::removeDirRecursively() !"
              << " - It should fail to delete nonexistent directory: " <<
              qPrintable(nonexistentAbsoluteDirPath)<< std::endl;
    return EXIT_FAILURE;
    }

  // Create a directory structure
  tmp.mkdir(temporaryDirName);
  tmp.cd(temporaryDirName);
  if (!createFile(__LINE__, tmp, "foo", "a.txt"))
    {
    return EXIT_FAILURE;
    }
  if (!createFile(__LINE__, tmp, "foo/bar", "b.txt"))
    {
    return EXIT_FAILURE;
    }
  if (!createFile(__LINE__, tmp, "foo/zoo", "c.txt"))
    {
    return EXIT_FAILURE;
    }

  if (!ctk::removeDirRecursively(tmp.absolutePath()))
    {
    std::cerr << "Line " << __LINE__ << " - Problem with ctk::removeDirRecursively()"
              << " - Failed to delete directory:" << qPrintable(tmp.absolutePath()) << std::endl;
    return EXIT_FAILURE;
    }

  return EXIT_SUCCESS;
}
