/*============================================================================

The Medical Imaging Interaction Toolkit (MITK)

Copyright (c) German Cancer Research Center (DKFZ)
All rights reserved.

Use of this source code is governed by a 3-clause BSD license that can be
found in the LICENSE file.

============================================================================*/

#ifndef mitkLabel_h
#define mitkLabel_h

#include "MitkMultilabelExports.h"
#include <mitkColorProperty.h>
#include <mitkPropertyList.h>
#include <mitkPoint.h>
#include <mitkVector.h>
#include <mitkPropertyKeyPath.h>
#include <mitkDICOMCodeSequence.h>
#include <mitkDICOMCodeSequenceWithModifiers.h>

#include <optional>
#include <map>

namespace mitk
{
  /**
   * @brief Constants for property key path elements of properties of the label.
   */
  namespace LabelPropertyConstants
  {
    /**
     * @brief Gets the property key path element for anatomic region.
     * @return The string "anatomic_region"
     */
    MITKMULTILABEL_EXPORT const std::string& GetAnatomicRegionPropertyBaseName();

    /**
     * @brief Gets the property key path element for primary anatomic structure.
     * @return The string "primary_anatomic_structure"
     */
    MITKMULTILABEL_EXPORT const std::string& GetPrimaryAnatomicStructurePropertyBaseName();

    /**
     * @brief Gets the property key path element for segmented property category.
     * @return The string "segmented_property_category"
     */
    MITKMULTILABEL_EXPORT const std::string& GetSegmentedPropertyCategoryPropertyBaseName();

    /**
     * @brief Gets the property key path element for segmented property type.
     * @return The string "segmented_property_type"
     */
    MITKMULTILABEL_EXPORT const std::string& GetSegmentedPropertyTypePropertyBaseName();

    /**
     * @brief Gets the property key path element for modifier.
     * @return The string "modifier"
     */
    MITKMULTILABEL_EXPORT const std::string& GetModifierPropertySubName();

    /**
     * @brief Gets the property key path element for code value.
     * @return The string "value"
     */
    MITKMULTILABEL_EXPORT const std::string& GetValuePropertySubName();

    /**
     * @brief Gets the property key path element for code scheme.
     * @return The string "scheme"
     */
    MITKMULTILABEL_EXPORT const std::string& GetSchemePropertySubName();

    /**
     * @brief Gets the property key path element for code meaning.
     * @return The string "meaning"
     */
    MITKMULTILABEL_EXPORT const std::string& GetMeaningPropertySubName();
  }

  //##
  //##Documentation
  //## @brief A data structure describing a label.
  //## @ingroup Data
  //##
  class MITKMULTILABEL_EXPORT Label : public PropertyList
  {
  public:
    mitkClassMacro(Label, mitk::PropertyList);

    typedef unsigned short PixelType;

    itkNewMacro(Self);
    mitkNewMacro2Param(Self, PixelType, const std::string&);

    /// The maximum value a label can get: Since the value is of type unsigned short MAX_LABEL_VALUE = 65535
    static const PixelType MAX_LABEL_VALUE;

    //** Value indicating pixels that are not labeled at all.*/
    static constexpr PixelType UNLABELED_VALUE = 0;

    void SetLocked(bool locked);
    bool GetLocked() const;

    void SetVisible(bool visible);
    bool GetVisible() const;

    void SetOpacity(float opacity);
    float GetOpacity() const;

    void SetName(const std::string &name);
    std::string GetName() const;

    void SetTrackingID(const std::string& trackingID);
    std::string GetTrackingID() const;

    void SetTrackingUID(const std::string& trackingUID);
    std::string GetTrackingUID() const;

    void SetDescription(const std::string& description);
    std::string GetDescription() const;

    mitk::Point3D GetCenterOfMassIndex() const;

    mitk::Point3D GetCenterOfMassCoordinates() const;
    void ResetCenterOfMass();
    //returns the mtime of the last center of mass property update.
    itk::ModifiedTimeType GetCenterOfMassMTime() const;
    void UpdateCenterOfMass(const mitk::Point3D& index, const mitk::Point3D& coordinates);

    void SetColor(const mitk::Color &);
    const mitk::Color &GetColor() const;

    void SetValue(PixelType pixelValue);
    PixelType GetValue() const;

    enum class AlgorithmType
    {
      /// undefined type
      Undefined = 0,
      /// user generated labels with simple tooling (e.g. Add tool)
      MANUAL,
      /// calculated label by user with algorithmic assistance (e.g. nnInteractive)
      SEMIAUTOMATIC,
      /// fully automatically generated by an algorithm (e.g. nnUnet)
      AUTOMATIC
    };

    void SetAlgorithmType(AlgorithmType algoType);
    void SetAlgorithmTypeStr(const std::string& algoType);
    AlgorithmType GetAlgorithmType() const;
    std::string GetAlgorithmTypeStr() const;
    void SetAlgorithmName(const std::string& algoName);
    std::string GetAlgorithmName() const;

    /** @brief Helper function to add the usage of multiple tools correctly to the label.
     Mixture of types always lead to semiautomatic. If the algorithm name is empty, the name
     will directly set. If it is not empty and algoName is not already a sub string of the
     current algorithm name, the new algoName will be appended (separated by "|").
     */
    void AddToolUse(AlgorithmType algoType, const std::string& algoName);

    /**
         * @brief Sets an anatomic region code at the specified index.
         * @param code The DICOMCodeSequenceWithModifiers representing the anatomic region
         * @param index The index at which to store the code (default: 0)
         */
    void SetAnatomicRegion(const DICOMCodeSequenceWithModifiers& code, std::size_t index = 0);

    /**
     * @brief Gets an anatomic region code at the specified index.
     * @param index The index of the code to retrieve (default: 0)
     * @return The DICOMCodeSequenceWithModifiers at the specified index, or an empty code if not found
     */
    DICOMCodeSequenceWithModifiers GetAnatomicRegion(std::size_t index = 0) const;

    /**
     * @brief Gets the number of anatomic region codes stored.
     * @return The count of anatomic region codes
     */
    std::size_t GetAnatomicRegionCount() const;

    /**
     * @brief Removes an anatomic region code at the specified index.
     * @param index The index of the code to remove
     */
    void RemoveAnatomicRegion(std::size_t index);

    /**
     * @brief Sets a primary anatomic structure code at the specified index.
     * @param code The DICOMCodeSequenceWithModifiers representing the primary anatomic structure
     * @param index The index at which to store the code (default: 0)
     */
    void SetPrimaryAnatomicStructure(const DICOMCodeSequenceWithModifiers& code, std::size_t index = 0);

    /**
     * @brief Gets a primary anatomic structure code at the specified index.
     * @param index The index of the code to retrieve (default: 0)
     * @return The DICOMCodeSequenceWithModifiers at the specified index, or an empty code if not found
     */
    DICOMCodeSequenceWithModifiers GetPrimaryAnatomicStructure(std::size_t index = 0) const;

    /**
     * @brief Gets the number of primary anatomic structure codes stored.
     * @return The count of primary anatomic structure codes
     */
    std::size_t GetPrimaryAnatomicStructureCount() const;

    /**
     * @brief Removes a primary anatomic structure code at the specified index.
     * @param index The index of the code to remove
     */
    void RemovePrimaryAnatomicStructure(std::size_t index);

    /**
     * @brief Sets the segmented property category code.
     * @param code The DICOMCodeSequence representing the segmented property category
     */
    void SetSegmentedPropertyCategory(const DICOMCodeSequence& code);

    /**
     * @brief Gets the segmented property category code.
     * @return The DICOMCodeSequence representing the segmented property category
     */
    std::optional<DICOMCodeSequence> GetSegmentedPropertyCategory() const;

    /**
     * @brief Sets the segmented property type code with optional modifiers.
     * @param code The DICOMCodeSequenceWithModifiers representing the segmented property type
     */
    void SetSegmentedPropertyType(const DICOMCodeSequenceWithModifiers& code);

    /**
     * @brief Gets the segmented property type code with modifiers.
     * @return The DICOMCodeSequenceWithModifiers representing the segmented property type
     */
    std::optional<DICOMCodeSequenceWithModifiers> GetSegmentedPropertyType() const;

    void SetProperty(const std::string &propertyKey, BaseProperty *property, const std::string &contextName = "", bool fallBackOnDefaultContext = false) override;
    BaseProperty::ConstPointer GetConstProperty(const std::string& propertyKey, const std::string& contextName = "", bool fallBackOnDefaultContext = true) const override;
    BaseProperty* GetNonConstProperty(const std::string& propertyKey, const std::string& contextName = "", bool fallBackOnDefaultContext = true) override;

    using itk::Object::Modified;
    void Modified() { Superclass::Modified(); }

    /** @brief Function updates the property values of a label provided as templates.
    If a property exists in the destination label only the value will be updated (therefor observers and pointers
    stay valid). If the property does not exist, it will be cloned and added. No properties will be removed.
    @param templateLabel Label instance that provides the new values for properties
    @param updateLabelValue Indicate if also the label value should be updated.
    */
    void Update(const Label* templateLabel, bool updateLabelValue = false);

    Label();
    Label(PixelType value, const std::string& name);
    ~Label() override;

  protected:
    void PrintSelf(std::ostream &os, itk::Indent indent) const override;
    using PropertyList::GetProperty;

    Label(const Label &other);

    void SetCenterOfMassIndex(const mitk::Point3D& center);
    void SetCenterOfMassCoordinates(const mitk::Point3D& center);

  private:
    PixelType m_Value;

    itk::LightObject::Pointer InternalClone() const override;

    /**
     * @brief Helper function to set a DICOMCode as properties.
     * @param basePath The base property key path
     * @param code The DICOMCode to store
     * @param withModifiers If true, also store modifiers (code must be DICOMCodeSequenceWithModifiers)
     */
    void SetDICOMCodeSequenceAsProperties(const PropertyKeyPath& basePath,
      const DICOMCodeSequence& code,
      bool withModifiers);

    /**
     * @brief Helper function to get a DICOMCodeSequence from properties.
     * @param basePath The base property key path
     * @return The DICOMCodeSequence retrieved from properties
     */
    std::optional<DICOMCodeSequence> GetDICOMCodeSequenceFromProperties(const PropertyKeyPath& basePath) const;

    /**
     * @brief Helper function to get a DICOMCodeSequenceWithModifiers from properties.
     * @param basePath The base property key path
     * @return The DICOMCodeSequenceWithModifiers retrieved from properties including modifiers
     */
    std::optional<DICOMCodeSequenceWithModifiers> GetDICOMCodeSequenceWithModifiersFromProperties(const PropertyKeyPath& basePath) const;

    /**
     * @brief Helper function to remove DICOMCodeSequence properties.
     * @param basePath The base property key path
     */
    void RemoveDICOMCodeSequenceProperties(const PropertyKeyPath& basePath);
  };

  using LabelVector = std::vector<Label::Pointer>;
  using ConstLabelVector = std::vector<Label::ConstPointer>;

  /**
  * @brief Equal A function comparing two labels for being equal in data
  *
  * @ingroup MITKTestingAPI
  *
  * Following aspects are tested for equality:
  *  - Lebel equality via Equal-PropetyList
  *
  * @param rightHandSide An image to be compared
  * @param leftHandSide An image to be compared
  * @param eps Tolarence for comparison. You can use mitk::eps in most cases.
  * @param verbose Flag indicating if the user wants detailed console output or not.
  * @return true, if all subsequent comparisons are true, false otherwise
  */
  MITKMULTILABEL_EXPORT bool Equal(const mitk::Label &leftHandSide,
                                   const mitk::Label &rightHandSide,
                                   ScalarType eps,
                                   bool verbose);

} // namespace mitk

#endif
