/*============================================================================

The Medical Imaging Interaction Toolkit (MITK)

Copyright (c) German Cancer Research Center (DKFZ)
All rights reserved.

Use of this source code is governed by a 3-clause BSD license that can be
found in the LICENSE file.

============================================================================*/

#ifndef mitkOperation_h
#define mitkOperation_h

#include <MitkCoreExports.h>
#include <itkEventObject.h>

#include <mitkCommon.h>

namespace mitk
{
  typedef int OperationType;

  //##Documentation
  //## @brief Base class of all Operation-classes
  //##
  //## @ingroup Undo
  class MITKCORE_EXPORT Operation
  {
  public:
    mitkClassMacroNoParent(Operation)

    /** Function indicates if the operation is currently still valid and
     could be conducted. Default implementation returns always true.*/
    virtual bool IsValid() const;

    virtual ~Operation() = default;
    OperationType GetOperationType();

  protected:
    Operation(OperationType operationType);
    Operation(const Operation&) = default;
    Operation(Operation&&) = default;
    Operation& operator=(const Operation&) = default;
    Operation& operator=(Operation&&) = default;

    OperationType m_OperationType;
  };

  class MITKCORE_EXPORT OperationEndEvent : public itk::EndEvent
  {
  public:
    typedef OperationEndEvent Self;
    typedef itk::EndEvent Superclass;
    OperationEndEvent(Operation *operation = nullptr) : m_Operation(operation) {}
    ~OperationEndEvent() override {}
    const char *GetEventName() const override { return "OperationEndEvent"; }
    bool CheckEvent(const ::itk::EventObject *e) const override { return dynamic_cast<const Self *>(e); }
    ::itk::EventObject *MakeObject() const override { return new Self(m_Operation); }
    Operation *GetOperation() const { return m_Operation; }
  private:
    Operation *m_Operation;
    OperationEndEvent(const Self &);
    void operator=(const Self &);
  };

} // namespace mitk
#endif
