/*============================================================================

The Medical Imaging Interaction Toolkit (MITK)

Copyright (c) German Cancer Research Center (DKFZ)
All rights reserved.

Use of this source code is governed by a 3-clause BSD license that can be
found in the LICENSE file.

============================================================================*/

// Testing
#include "mitkTestFixture.h"
#include "mitkTestingMacros.h"

// std includes
#include <string>

// MITK includes
#include "mitkCustomTagParser.h"
#include "mitkCESTPropertyHelper.h"
#include <mitkTemporoSpatialStringProperty.h>

//itksys
#include <itksys/SystemTools.hxx>
#include <itkFileTools.h>

// microservice includes
#include "usGetModuleContext.h"
#include "usModule.h"
#include "usModuleContext.h"

// VTK includes
#include <vtkDebugLeaks.h>

class mitkCustomTagParserTestSuite : public mitk::TestFixture
{
  CPPUNIT_TEST_SUITE(mitkCustomTagParserTestSuite);

  // Test the dicom property parsing
  MITK_TEST(ValidPropertyParsedToPropertyList_Success);
  MITK_TEST(ValidPropertyMissingParametersParsedToEmptyPropertiesPropertyList_Success);
  MITK_TEST(ValidPropertyRevisionVeryLow_UseDefault_Success);
  MITK_TEST(ValidPropertyNoExactRevisionMatchUseInternal_Success);
  MITK_TEST(ValidPropertyNoExactRevisionMatchUseExternal_Success);
  MITK_TEST(ValidPropertyWindowsLineEndings_Success);
  MITK_TEST(InvalidPropertyInvalidRevision_Failure);
  MITK_TEST(InvalidPropertyNoCESTSequence_Failure);
  MITK_TEST(InvalidPropertyGarbageInDelimiters_Failure);
  MITK_TEST(ValidPropertyAlternatingOffset_Success);
  MITK_TEST(ValidPropertySimpleOffset_Success);
  MITK_TEST(ValidPropertyListOffset_Success);
  MITK_TEST(ExtractRevision);
  CPPUNIT_TEST_SUITE_END();

private:
  std::string m_PathToModule;
  std::string m_ValidCESTCustomTag;
  std::string m_ValidCESTCustomTagAllParametersMissing;
  std::string m_ValidCESTCustomTagUnsupportedRevisionTooLow; // rev 12
  std::string m_ValidCESTCustomTagUnsupportedRevisionNoExactJSONUseInternal; // rev 1417
  std::string m_ValidCESTCustomTagUnsupportedRevisionNoExactJSONUseExternal; // rev 120
  std::string m_InvalidCESTCustomTagRevisionNoNumber; //rev abc
  std::string m_NonCESTCustomTag; // no CEST_Rev substring
  std::string m_GarbageWithinDelimiters; // ASCII part of custom tag is just garbage
  std::string m_ValidCESTCustomTagWindowsLineEndings; // have windows line endings encoded
  std::string m_ValidCESTCustomTagAlternatingOffset;
  std::string m_ValidCESTCustomTagSingleOffset;
  std::string m_ValidCESTCustomTagListOffset;

public:
  void setUp() override
  {
    std::string moduleLocation = us::GetModuleContext()->GetModule()->GetLocation();
    std::string libraryName;
    itksys::SystemTools::SplitProgramPath(moduleLocation, m_PathToModule, libraryName);

    m_ValidCESTCustomTag                                          = "20\\20\\20\\20\\20\\20\\20\\20\\3C\\43\\6F\\6E\\6E\\65\\63\\74\\69\\6F\\6E\\2E\\22\\22\\63\\31\\22\\22\\3E\\20\\20\\7B\\20\\22\\22\\49\\6D\\61\\67\\65\\52\\65\\61\\64\\79\\22\\22\\20\\22\\22\\22\\22\\20\\22\\22\\43\\6F\\6D\\70\\75\\74\\65\\49\\6D\\61\\67\\65\\22\\22\\20\\20\\7D\\0A\\20\\20\\20\\20\\20\\20\\7D\\0A\\20\\20\\20\\20\\7D\\0A\\20\\20\\7D\\0A\\7D\\0A\\23\\23\\23\\20\\41\\53\\43\\43\\4F\\4E\\56\\20\\42\\45\\47\\49\\4E\\20\\23\\23\\23\\0A\\75\\6C\\56\\65\\72\\73\\69\\6F\\6E\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\30\\78\\31\\34\\62\\34\\34\\62\\36\\0A\\74\\53\\65\\71\\75\\65\\6E\\63\\65\\46\\69\\6C\\65\\4E\\61\\6D\\65\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\22\\22\\25\\43\\75\\73\\74\\6F\\6D\\65\\72\\53\\65\\71\\25\\5C\\58\\58\\58\\58\\58\\5F\\43\\45\\53\\54\\5F\\52\\65\\76\\31\\34\\31\\36\\22\\22\\0A\\74\\50\\72\\6F\\74\\6F\\63\\6F\\6C\\4E\\61\\6D\\65\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\22\\22\\57\\41\\53\\41\\42\\49\\2B\\41\\46\\38\\2D\\4D\\49\\54\\4B\\2D\\54\\45\\53\\54\\2B\\41\\46\\38\\2D\\37\\54\\2B\\41\\46\\38\\2D\\33\\73\\6C\\69\\63\\65\\73\\22\\22\\0A\\73\\50\\72\\6F\\74\\43\\6F\\6E\\73\\69\\73\\74\\65\\6E\\63\\79\\49\\6E\\66\\6F\\2E\\74\\53\\79\\73\\74\\65\\6D\\54\\79\\70\\65\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\22\\22\\30\\39\\35\\22\\22\\0A\\73\\50\\72\\6F\\74\\43\\6F\\6E\\73\\69\\73\\74\\65\\6E\\63\\79\\49\\6E\\66\\6F\\2E\\66\\6C\\4E\\6F\\6D\\69\\6E\\61\\6C\\42\\30\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\36\\2E\\39\\38\\0A\\73\\54\\58\\53\\50\\45\\43\\2E\\61\\73\\4E\\75\\63\\6C\\65\\75\\73\\49\\6E\\66\\6F\\5B\\30\\5D\\2E\\6C\\46\\72\\65\\71\\75\\65\\6E\\63\\79\\20\\20\\20\\20\\20\\20\\3D\\20\\32\\39\\37\\31\\35\\34\\37\\35\\36\\0A\\73\\54\\58\\53\\50\\45\\43\\2E\\61\\73\\4E\\75\\63\\6C\\65\\75\\73\\49\\6E\\66\\6F\\5B\\30\\5D\\2E\\66\\6C\\52\\65\\66\\65\\72\\65\\6E\\63\\65\\41\\6D\\70\\6C\\69\\74\\75\\64\\65\\20\\3D\\20\\31\\36\\31\\2E\\34\\33\\35\\0A\\61\\6C\\54\\52\\5B\\30\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\32\\33\\30\\30\\30\\0A\\61\\6C\\54\\45\\5B\\30\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\33\\36\\34\\30\\0A\\6C\\41\\76\\65\\72\\61\\67\\65\\73\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\0A\\6C\\52\\65\\70\\65\\74\\69\\74\\69\\6F\\6E\\73\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\33\\31\\0A\\61\\64\\46\\6C\\69\\70\\41\\6E\\67\\6C\\65\\44\\65\\67\\72\\65\\65\\5B\\30\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\30\\0A\\6C\\54\\6F\\74\\61\\6C\\53\\63\\61\\6E\\54\\69\\6D\\65\\53\\65\\63\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\39\\37\\0A\\73\\45\\46\\49\\53\\50\\45\\43\\2E\\62\\45\\46\\49\\44\\61\\74\\61\\56\\61\\6C\\69\\64\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\31\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\32\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\33\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\34\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\35\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\32\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\36\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\35\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\37\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\33\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\38\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\39\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\31\\30\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\33\\32\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\31\\31\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\31\\32\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\31\\33\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\35\\30\\30\\30\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\31\\34\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\35\\36\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\31\\35\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\33\\30\\30\\30\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\31\\36\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\39\\30\\30\\30\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\31\\37\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\30\\30\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\31\\38\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\35\\30\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\31\\39\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\35\\30\\30\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\32\\30\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\32\\32\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\64\\46\\72\\65\\65\\5B\\31\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\32\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\64\\46\\72\\65\\65\\5B\\32\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\33\\2E\\37\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\64\\46\\72\\65\\65\\5B\\33\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\36\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\64\\46\\72\\65\\65\\5B\\34\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\32\\30\\30\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\64\\46\\72\\65\\65\\5B\\35\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\38\\30\\30\\30\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\64\\46\\72\\65\\65\\5B\\36\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\34\\30\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\64\\46\\72\\65\\65\\5B\\37\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\30\\2E\\31\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\64\\46\\72\\65\\65\\5B\\38\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\35\\31\\32\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\64\\46\\72\\65\\65\\5B\\39\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\35\\30\\30\\30\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\64\\46\\72\\65\\65\\5B\\31\\30\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\33\\2E\\37\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\64\\46\\72\\65\\65\\5B\\31\\31\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\0A\\23\\23\\23\\20\\41\\53\\43\\43\\4F\\4E\\56\\20\\45\\4E\\44\\20\\23\\23\\23\\22\\20\\0A\\20\\20\\20\\20\\7D\\0A\\20\\20\\7D\\0A\\7D\\0A\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\CD\\A0\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\CD\\A0\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\CD\\A0\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\CD\\A0\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\CD\\A0\\20\\20\\20\\20\\20\\20\\47\\72\\61\\64\\69\\65\\6E\\74\\4D\\6F\\64\\65\\20\\72\\6D\\61\\6E\\63\\65\\43\\61\\63\\68\\65\\2E\\69\\6E\\6C\\69\\6E\\65\\5F\\70\\6F\\73\\64\\69\\73\\70\\5F\\63\\61\\6E\\5F\\73\\65\\74\\22\\22\\20\\3C\\44\\6C\\6C\\3E\\20\\22\\22\\4D\\72\\4D\\75\\6C\\74\\01\\20\\20\\20\\53\\48\\20\\20\\16\\20\\20\\20\\06\\20\\20\\20\\4D\\20\\20\\20\\05\\20\\20\\20\\05\\20\\20\\20\\4D\\20\\20\\20\\05\\20\\20\\20\\46\\61\\73\\74\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\CD\\A0\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\CD\\A0\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\CD\\A0\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\CD\\A0\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\CD\\A0\\20\\20\\20\\20\\20\\20\\46\\6C\\6F\\77\\43\\6F\\6D\\70\\65\\6E\\73\\61\\74\\69\\6F\\6E\\20\\72\\65\\53\\6F\\75\\6E\\64\\22\\22\\20\\22\\22\\50\\72\\6F\\70\\65\\72\\74\\69\\65\\73\\2E\\53\\6F\\75\\6E\\64\\2E\\50\\6F\\73\\74\\53\\6F\\75\\6E\\64\\22\\22";
    m_ValidCESTCustomTagAllParametersMissing                      = "20\\20\\20\\20\\20\\20\\20\\20\\3C\\43\\6F\\6E\\6E\\65\\63\\74\\69\\6F\\6E\\2E\\22\\22\\63\\31\\22\\22\\3E\\20\\20\\7B\\20\\22\\22\\49\\6D\\61\\67\\65\\52\\65\\61\\64\\79\\22\\22\\20\\22\\22\\22\\22\\20\\22\\22\\43\\6F\\6D\\70\\75\\74\\65\\49\\6D\\61\\67\\65\\22\\22\\20\\20\\7D\\0A\\20\\20\\20\\20\\20\\20\\7D\\0A\\20\\20\\20\\20\\7D\\0A\\20\\20\\7D\\0A\\7D\\0A\\23\\23\\23\\20\\41\\53\\43\\43\\4F\\4E\\56\\20\\42\\45\\47\\49\\4E\\20\\23\\23\\23\\0A\\75\\6C\\56\\65\\72\\73\\69\\6F\\6E\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\30\\78\\31\\34\\62\\34\\34\\62\\36\\0A\\74\\53\\65\\71\\75\\65\\6E\\63\\65\\46\\69\\6C\\65\\4E\\61\\6D\\65\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\22\\22\\25\\43\\75\\73\\74\\6F\\6D\\65\\72\\53\\65\\71\\25\\5C\\58\\58\\58\\58\\58\\5F\\43\\45\\53\\54\\5F\\52\\65\\76\\31\\34\\31\\36\\22\\22\\0A\\23\\23\\23\\20\\41\\53\\43\\43\\4F\\4E\\56\\20\\45\\4E\\44\\20\\23\\23\\23\\22\\20\\0A\\20\\20\\20\\20\\7D\\0A\\20\\20\\7D\\0A\\7D\\0A\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\CD\\A0\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\CD\\A0\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\CD\\A0\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\CD\\A0\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\CD\\A0\\20\\20\\20\\20\\20\\20\\47\\72\\61\\64\\69\\65\\6E\\74\\4D\\6F\\64\\65\\20\\72\\6D\\61\\6E\\63\\65\\43\\61\\63\\68\\65\\2E\\69\\6E\\6C\\69\\6E\\65\\5F\\70\\6F\\73\\64\\69\\73\\70\\5F\\63\\61\\6E\\5F\\73\\65\\74\\22\\22\\20\\3C\\44\\6C\\6C\\3E\\20\\22\\22\\4D\\72\\4D\\75\\6C\\74\\01\\20\\20\\20\\53\\48\\20\\20\\16\\20\\20\\20\\06\\20\\20\\20\\4D\\20\\20\\20\\05\\20\\20\\20\\05\\20\\20\\20\\4D\\20\\20\\20\\05\\20\\20\\20\\46\\61\\73\\74\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\CD\\A0\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\CD\\A0\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\CD\\A0\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\CD\\A0\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\CD\\A0\\20\\20\\20\\20\\20\\20\\46\\6C\\6F\\77\\43\\6F\\6D\\70\\65\\6E\\73\\61\\74\\69\\6F\\6E\\20\\72\\65\\53\\6F\\75\\6E\\64\\22\\22\\20\\22\\22\\50\\72\\6F\\70\\65\\72\\74\\69\\65\\73\\2E\\53\\6F\\75\\6E\\64\\2E\\50\\6F\\73\\74\\53\\6F\\75\\6E\\64\\22\\22";
    m_ValidCESTCustomTagUnsupportedRevisionTooLow                 = "20\\20\\20\\20\\20\\20\\20\\20\\3C\\43\\6F\\6E\\6E\\65\\63\\74\\69\\6F\\6E\\2E\\22\\22\\63\\31\\22\\22\\3E\\20\\20\\7B\\20\\22\\22\\49\\6D\\61\\67\\65\\52\\65\\61\\64\\79\\22\\22\\20\\22\\22\\22\\22\\20\\22\\22\\43\\6F\\6D\\70\\75\\74\\65\\49\\6D\\61\\67\\65\\22\\22\\20\\20\\7D\\0A\\20\\20\\20\\20\\20\\20\\7D\\0A\\20\\20\\20\\20\\7D\\0A\\20\\20\\7D\\0A\\7D\\0A\\23\\23\\23\\20\\41\\53\\43\\43\\4F\\4E\\56\\20\\42\\45\\47\\49\\4E\\20\\23\\23\\23\\0A\\75\\6C\\56\\65\\72\\73\\69\\6F\\6E\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\30\\78\\31\\34\\62\\34\\34\\62\\36\\0A\\74\\53\\65\\71\\75\\65\\6E\\63\\65\\46\\69\\6C\\65\\4E\\61\\6D\\65\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\22\\22\\25\\43\\75\\73\\74\\6F\\6D\\65\\72\\53\\65\\71\\25\\5C\\58\\58\\58\\58\\58\\5F\\43\\45\\53\\54\\5F\\52\\65\\76\\31\\32\\22\\22\\0A\\74\\50\\72\\6F\\74\\6F\\63\\6F\\6C\\4E\\61\\6D\\65\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\22\\22\\57\\41\\53\\41\\42\\49\\2B\\41\\46\\38\\2D\\4D\\49\\54\\4B\\2D\\54\\45\\53\\54\\2B\\41\\46\\38\\2D\\37\\54\\2B\\41\\46\\38\\2D\\33\\73\\6C\\69\\63\\65\\73\\22\\22\\0A\\73\\50\\72\\6F\\74\\43\\6F\\6E\\73\\69\\73\\74\\65\\6E\\63\\79\\49\\6E\\66\\6F\\2E\\74\\53\\79\\73\\74\\65\\6D\\54\\79\\70\\65\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\22\\22\\30\\39\\35\\22\\22\\0A\\73\\50\\72\\6F\\74\\43\\6F\\6E\\73\\69\\73\\74\\65\\6E\\63\\79\\49\\6E\\66\\6F\\2E\\66\\6C\\4E\\6F\\6D\\69\\6E\\61\\6C\\42\\30\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\36\\2E\\39\\38\\0A\\73\\54\\58\\53\\50\\45\\43\\2E\\61\\73\\4E\\75\\63\\6C\\65\\75\\73\\49\\6E\\66\\6F\\5B\\30\\5D\\2E\\6C\\46\\72\\65\\71\\75\\65\\6E\\63\\79\\20\\20\\20\\20\\20\\20\\3D\\20\\32\\39\\37\\31\\35\\34\\37\\35\\36\\0A\\73\\54\\58\\53\\50\\45\\43\\2E\\61\\73\\4E\\75\\63\\6C\\65\\75\\73\\49\\6E\\66\\6F\\5B\\30\\5D\\2E\\66\\6C\\52\\65\\66\\65\\72\\65\\6E\\63\\65\\41\\6D\\70\\6C\\69\\74\\75\\64\\65\\20\\3D\\20\\31\\36\\31\\2E\\34\\33\\35\\0A\\61\\6C\\54\\52\\5B\\30\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\32\\33\\30\\30\\30\\0A\\61\\6C\\54\\45\\5B\\30\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\33\\36\\34\\30\\0A\\6C\\41\\76\\65\\72\\61\\67\\65\\73\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\0A\\6C\\52\\65\\70\\65\\74\\69\\74\\69\\6F\\6E\\73\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\33\\31\\0A\\61\\64\\46\\6C\\69\\70\\41\\6E\\67\\6C\\65\\44\\65\\67\\72\\65\\65\\5B\\30\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\30\\0A\\6C\\54\\6F\\74\\61\\6C\\53\\63\\61\\6E\\54\\69\\6D\\65\\53\\65\\63\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\39\\37\\0A\\73\\45\\46\\49\\53\\50\\45\\43\\2E\\62\\45\\46\\49\\44\\61\\74\\61\\56\\61\\6C\\69\\64\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\31\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\32\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\33\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\34\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\35\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\32\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\36\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\35\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\37\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\33\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\38\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\39\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\31\\30\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\33\\32\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\31\\31\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\31\\32\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\31\\33\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\35\\30\\30\\30\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\31\\34\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\35\\36\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\31\\35\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\33\\30\\30\\30\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\31\\36\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\39\\30\\30\\30\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\31\\37\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\30\\30\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\31\\38\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\35\\30\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\31\\39\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\35\\30\\30\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\32\\30\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\32\\32\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\64\\46\\72\\65\\65\\5B\\31\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\32\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\64\\46\\72\\65\\65\\5B\\32\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\33\\2E\\37\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\64\\46\\72\\65\\65\\5B\\33\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\36\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\64\\46\\72\\65\\65\\5B\\34\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\32\\30\\30\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\64\\46\\72\\65\\65\\5B\\35\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\38\\30\\30\\30\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\64\\46\\72\\65\\65\\5B\\36\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\34\\30\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\64\\46\\72\\65\\65\\5B\\37\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\30\\2E\\31\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\64\\46\\72\\65\\65\\5B\\38\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\35\\31\\32\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\64\\46\\72\\65\\65\\5B\\39\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\35\\30\\30\\30\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\64\\46\\72\\65\\65\\5B\\31\\30\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\33\\2E\\37\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\64\\46\\72\\65\\65\\5B\\31\\31\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\0A\\23\\23\\23\\20\\41\\53\\43\\43\\4F\\4E\\56\\20\\45\\4E\\44\\20\\23\\23\\23\\22\\20\\0A\\20\\20\\20\\20\\7D\\0A\\20\\20\\7D\\0A\\7D\\0A\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\CD\\A0\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\CD\\A0\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\CD\\A0\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\CD\\A0\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\CD\\A0\\20\\20\\20\\20\\20\\20\\47\\72\\61\\64\\69\\65\\6E\\74\\4D\\6F\\64\\65\\20\\72\\6D\\61\\6E\\63\\65\\43\\61\\63\\68\\65\\2E\\69\\6E\\6C\\69\\6E\\65\\5F\\70\\6F\\73\\64\\69\\73\\70\\5F\\63\\61\\6E\\5F\\73\\65\\74\\22\\22\\20\\3C\\44\\6C\\6C\\3E\\20\\22\\22\\4D\\72\\4D\\75\\6C\\74\\01\\20\\20\\20\\53\\48\\20\\20\\16\\20\\20\\20\\06\\20\\20\\20\\4D\\20\\20\\20\\05\\20\\20\\20\\05\\20\\20\\20\\4D\\20\\20\\20\\05\\20\\20\\20\\46\\61\\73\\74\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\CD\\A0\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\CD\\A0\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\CD\\A0\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\CD\\A0\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\CD\\A0\\20\\20\\20\\20\\20\\20\\46\\6C\\6F\\77\\43\\6F\\6D\\70\\65\\6E\\73\\61\\74\\69\\6F\\6E\\20\\72\\65\\53\\6F\\75\\6E\\64\\22\\22\\20\\22\\22\\50\\72\\6F\\70\\65\\72\\74\\69\\65\\73\\2E\\53\\6F\\75\\6E\\64\\2E\\50\\6F\\73\\74\\53\\6F\\75\\6E\\64\\22\\22";
    m_ValidCESTCustomTagUnsupportedRevisionNoExactJSONUseInternal = "20\\20\\20\\20\\20\\20\\20\\20\\3C\\43\\6F\\6E\\6E\\65\\63\\74\\69\\6F\\6E\\2E\\22\\22\\63\\31\\22\\22\\3E\\20\\20\\7B\\20\\22\\22\\49\\6D\\61\\67\\65\\52\\65\\61\\64\\79\\22\\22\\20\\22\\22\\22\\22\\20\\22\\22\\43\\6F\\6D\\70\\75\\74\\65\\49\\6D\\61\\67\\65\\22\\22\\20\\20\\7D\\0A\\20\\20\\20\\20\\20\\20\\7D\\0A\\20\\20\\20\\20\\7D\\0A\\20\\20\\7D\\0A\\7D\\0A\\23\\23\\23\\20\\41\\53\\43\\43\\4F\\4E\\56\\20\\42\\45\\47\\49\\4E\\20\\23\\23\\23\\0A\\75\\6C\\56\\65\\72\\73\\69\\6F\\6E\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\30\\78\\31\\34\\62\\34\\34\\62\\36\\0A\\74\\53\\65\\71\\75\\65\\6E\\63\\65\\46\\69\\6C\\65\\4E\\61\\6D\\65\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\22\\22\\25\\43\\75\\73\\74\\6F\\6D\\65\\72\\53\\65\\71\\25\\5C\\58\\58\\58\\58\\58\\5F\\43\\45\\53\\54\\5F\\52\\65\\76\\31\\34\\31\\37\\22\\22\\0A\\74\\50\\72\\6F\\74\\6F\\63\\6F\\6C\\4E\\61\\6D\\65\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\22\\22\\57\\41\\53\\41\\42\\49\\2B\\41\\46\\38\\2D\\4D\\49\\54\\4B\\2D\\54\\45\\53\\54\\2B\\41\\46\\38\\2D\\37\\54\\2B\\41\\46\\38\\2D\\33\\73\\6C\\69\\63\\65\\73\\22\\22\\0A\\73\\50\\72\\6F\\74\\43\\6F\\6E\\73\\69\\73\\74\\65\\6E\\63\\79\\49\\6E\\66\\6F\\2E\\74\\53\\79\\73\\74\\65\\6D\\54\\79\\70\\65\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\22\\22\\30\\39\\35\\22\\22\\0A\\73\\50\\72\\6F\\74\\43\\6F\\6E\\73\\69\\73\\74\\65\\6E\\63\\79\\49\\6E\\66\\6F\\2E\\66\\6C\\4E\\6F\\6D\\69\\6E\\61\\6C\\42\\30\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\36\\2E\\39\\38\\0A\\73\\54\\58\\53\\50\\45\\43\\2E\\61\\73\\4E\\75\\63\\6C\\65\\75\\73\\49\\6E\\66\\6F\\5B\\30\\5D\\2E\\6C\\46\\72\\65\\71\\75\\65\\6E\\63\\79\\20\\20\\20\\20\\20\\20\\3D\\20\\32\\39\\37\\31\\35\\34\\37\\35\\36\\0A\\73\\54\\58\\53\\50\\45\\43\\2E\\61\\73\\4E\\75\\63\\6C\\65\\75\\73\\49\\6E\\66\\6F\\5B\\30\\5D\\2E\\66\\6C\\52\\65\\66\\65\\72\\65\\6E\\63\\65\\41\\6D\\70\\6C\\69\\74\\75\\64\\65\\20\\3D\\20\\31\\36\\31\\2E\\34\\33\\35\\0A\\61\\6C\\54\\52\\5B\\30\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\32\\33\\30\\30\\30\\0A\\61\\6C\\54\\45\\5B\\30\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\33\\36\\34\\30\\0A\\6C\\41\\76\\65\\72\\61\\67\\65\\73\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\0A\\6C\\52\\65\\70\\65\\74\\69\\74\\69\\6F\\6E\\73\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\33\\31\\0A\\61\\64\\46\\6C\\69\\70\\41\\6E\\67\\6C\\65\\44\\65\\67\\72\\65\\65\\5B\\30\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\30\\0A\\6C\\54\\6F\\74\\61\\6C\\53\\63\\61\\6E\\54\\69\\6D\\65\\53\\65\\63\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\39\\37\\0A\\73\\45\\46\\49\\53\\50\\45\\43\\2E\\62\\45\\46\\49\\44\\61\\74\\61\\56\\61\\6C\\69\\64\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\31\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\32\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\33\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\34\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\35\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\32\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\36\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\35\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\37\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\33\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\38\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\39\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\31\\30\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\33\\32\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\31\\31\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\31\\32\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\31\\33\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\35\\30\\30\\30\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\31\\34\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\35\\36\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\31\\35\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\33\\30\\30\\30\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\31\\36\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\39\\30\\30\\30\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\31\\37\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\30\\30\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\31\\38\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\35\\30\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\31\\39\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\35\\30\\30\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\32\\30\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\32\\32\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\64\\46\\72\\65\\65\\5B\\31\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\32\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\64\\46\\72\\65\\65\\5B\\32\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\33\\2E\\37\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\64\\46\\72\\65\\65\\5B\\33\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\36\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\64\\46\\72\\65\\65\\5B\\34\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\32\\30\\30\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\64\\46\\72\\65\\65\\5B\\35\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\38\\30\\30\\30\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\64\\46\\72\\65\\65\\5B\\36\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\34\\30\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\64\\46\\72\\65\\65\\5B\\37\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\30\\2E\\31\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\64\\46\\72\\65\\65\\5B\\38\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\35\\31\\32\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\64\\46\\72\\65\\65\\5B\\39\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\35\\30\\30\\30\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\64\\46\\72\\65\\65\\5B\\31\\30\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\33\\2E\\37\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\64\\46\\72\\65\\65\\5B\\31\\31\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\0A\\23\\23\\23\\20\\41\\53\\43\\43\\4F\\4E\\56\\20\\45\\4E\\44\\20\\23\\23\\23\\22\\20\\0A\\20\\20\\20\\20\\7D\\0A\\20\\20\\7D\\0A\\7D\\0A\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\CD\\A0\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\CD\\A0\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\CD\\A0\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\CD\\A0\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\CD\\A0\\20\\20\\20\\20\\20\\20\\47\\72\\61\\64\\69\\65\\6E\\74\\4D\\6F\\64\\65\\20\\72\\6D\\61\\6E\\63\\65\\43\\61\\63\\68\\65\\2E\\69\\6E\\6C\\69\\6E\\65\\5F\\70\\6F\\73\\64\\69\\73\\70\\5F\\63\\61\\6E\\5F\\73\\65\\74\\22\\22\\20\\3C\\44\\6C\\6C\\3E\\20\\22\\22\\4D\\72\\4D\\75\\6C\\74\\01\\20\\20\\20\\53\\48\\20\\20\\16\\20\\20\\20\\06\\20\\20\\20\\4D\\20\\20\\20\\05\\20\\20\\20\\05\\20\\20\\20\\4D\\20\\20\\20\\05\\20\\20\\20\\46\\61\\73\\74\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\CD\\A0\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\CD\\A0\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\CD\\A0\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\CD\\A0\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\CD\\A0\\20\\20\\20\\20\\20\\20\\46\\6C\\6F\\77\\43\\6F\\6D\\70\\65\\6E\\73\\61\\74\\69\\6F\\6E\\20\\72\\65\\53\\6F\\75\\6E\\64\\22\\22\\20\\22\\22\\50\\72\\6F\\70\\65\\72\\74\\69\\65\\73\\2E\\53\\6F\\75\\6E\\64\\2E\\50\\6F\\73\\74\\53\\6F\\75\\6E\\64\\22\\22";
    m_ValidCESTCustomTagUnsupportedRevisionNoExactJSONUseExternal = "20\\20\\20\\20\\20\\20\\20\\20\\3C\\43\\6F\\6E\\6E\\65\\63\\74\\69\\6F\\6E\\2E\\22\\22\\63\\31\\22\\22\\3E\\20\\20\\7B\\20\\22\\22\\49\\6D\\61\\67\\65\\52\\65\\61\\64\\79\\22\\22\\20\\22\\22\\22\\22\\20\\22\\22\\43\\6F\\6D\\70\\75\\74\\65\\49\\6D\\61\\67\\65\\22\\22\\20\\20\\7D\\0A\\20\\20\\20\\20\\20\\20\\7D\\0A\\20\\20\\20\\20\\7D\\0A\\20\\20\\7D\\0A\\7D\\0A\\23\\23\\23\\20\\41\\53\\43\\43\\4F\\4E\\56\\20\\42\\45\\47\\49\\4E\\20\\23\\23\\23\\0A\\75\\6C\\56\\65\\72\\73\\69\\6F\\6E\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\30\\78\\31\\34\\62\\34\\34\\62\\36\\0A\\74\\53\\65\\71\\75\\65\\6E\\63\\65\\46\\69\\6C\\65\\4E\\61\\6D\\65\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\22\\22\\25\\43\\75\\73\\74\\6F\\6D\\65\\72\\53\\65\\71\\25\\5C\\58\\58\\58\\58\\58\\5F\\43\\45\\53\\54\\5F\\52\\65\\76\\31\\32\\30\\22\\22\\0A\\74\\50\\72\\6F\\74\\6F\\63\\6F\\6C\\4E\\61\\6D\\65\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\22\\22\\57\\41\\53\\41\\42\\49\\2B\\41\\46\\38\\2D\\4D\\49\\54\\4B\\2D\\54\\45\\53\\54\\2B\\41\\46\\38\\2D\\37\\54\\2B\\41\\46\\38\\2D\\33\\73\\6C\\69\\63\\65\\73\\22\\22\\0A\\73\\50\\72\\6F\\74\\43\\6F\\6E\\73\\69\\73\\74\\65\\6E\\63\\79\\49\\6E\\66\\6F\\2E\\74\\53\\79\\73\\74\\65\\6D\\54\\79\\70\\65\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\22\\22\\30\\39\\35\\22\\22\\0A\\73\\50\\72\\6F\\74\\43\\6F\\6E\\73\\69\\73\\74\\65\\6E\\63\\79\\49\\6E\\66\\6F\\2E\\66\\6C\\4E\\6F\\6D\\69\\6E\\61\\6C\\42\\30\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\36\\2E\\39\\38\\0A\\73\\54\\58\\53\\50\\45\\43\\2E\\61\\73\\4E\\75\\63\\6C\\65\\75\\73\\49\\6E\\66\\6F\\5B\\30\\5D\\2E\\6C\\46\\72\\65\\71\\75\\65\\6E\\63\\79\\20\\20\\20\\20\\20\\20\\3D\\20\\32\\39\\37\\31\\35\\34\\37\\35\\36\\0A\\73\\54\\58\\53\\50\\45\\43\\2E\\61\\73\\4E\\75\\63\\6C\\65\\75\\73\\49\\6E\\66\\6F\\5B\\30\\5D\\2E\\66\\6C\\52\\65\\66\\65\\72\\65\\6E\\63\\65\\41\\6D\\70\\6C\\69\\74\\75\\64\\65\\20\\3D\\20\\31\\36\\31\\2E\\34\\33\\35\\0A\\61\\6C\\54\\52\\5B\\30\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\32\\33\\30\\30\\30\\0A\\61\\6C\\54\\45\\5B\\30\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\33\\36\\34\\30\\0A\\6C\\41\\76\\65\\72\\61\\67\\65\\73\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\0A\\6C\\52\\65\\70\\65\\74\\69\\74\\69\\6F\\6E\\73\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\33\\31\\0A\\61\\64\\46\\6C\\69\\70\\41\\6E\\67\\6C\\65\\44\\65\\67\\72\\65\\65\\5B\\30\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\30\\0A\\6C\\54\\6F\\74\\61\\6C\\53\\63\\61\\6E\\54\\69\\6D\\65\\53\\65\\63\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\39\\37\\0A\\73\\45\\46\\49\\53\\50\\45\\43\\2E\\62\\45\\46\\49\\44\\61\\74\\61\\56\\61\\6C\\69\\64\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\31\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\32\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\33\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\34\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\35\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\32\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\36\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\35\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\37\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\33\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\38\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\39\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\31\\30\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\33\\32\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\31\\31\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\31\\32\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\31\\33\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\35\\30\\30\\30\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\31\\34\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\35\\36\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\31\\35\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\33\\30\\30\\30\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\31\\36\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\39\\30\\30\\30\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\31\\37\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\30\\30\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\31\\38\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\35\\30\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\31\\39\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\35\\30\\30\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\32\\30\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\32\\32\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\64\\46\\72\\65\\65\\5B\\31\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\32\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\64\\46\\72\\65\\65\\5B\\32\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\33\\2E\\37\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\64\\46\\72\\65\\65\\5B\\33\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\36\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\64\\46\\72\\65\\65\\5B\\34\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\32\\30\\30\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\64\\46\\72\\65\\65\\5B\\35\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\38\\30\\30\\30\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\64\\46\\72\\65\\65\\5B\\36\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\34\\30\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\64\\46\\72\\65\\65\\5B\\37\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\30\\2E\\31\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\64\\46\\72\\65\\65\\5B\\38\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\35\\31\\32\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\64\\46\\72\\65\\65\\5B\\39\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\35\\30\\30\\30\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\64\\46\\72\\65\\65\\5B\\31\\30\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\33\\2E\\37\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\64\\46\\72\\65\\65\\5B\\31\\31\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\0A\\23\\23\\23\\20\\41\\53\\43\\43\\4F\\4E\\56\\20\\45\\4E\\44\\20\\23\\23\\23\\22\\20\\0A\\20\\20\\20\\20\\7D\\0A\\20\\20\\7D\\0A\\7D\\0A\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\CD\\A0\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\CD\\A0\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\CD\\A0\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\CD\\A0\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\CD\\A0\\20\\20\\20\\20\\20\\20\\47\\72\\61\\64\\69\\65\\6E\\74\\4D\\6F\\64\\65\\20\\72\\6D\\61\\6E\\63\\65\\43\\61\\63\\68\\65\\2E\\69\\6E\\6C\\69\\6E\\65\\5F\\70\\6F\\73\\64\\69\\73\\70\\5F\\63\\61\\6E\\5F\\73\\65\\74\\22\\22\\20\\3C\\44\\6C\\6C\\3E\\20\\22\\22\\4D\\72\\4D\\75\\6C\\74\\01\\20\\20\\20\\53\\48\\20\\20\\16\\20\\20\\20\\06\\20\\20\\20\\4D\\20\\20\\20\\05\\20\\20\\20\\05\\20\\20\\20\\4D\\20\\20\\20\\05\\20\\20\\20\\46\\61\\73\\74\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\CD\\A0\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\CD\\A0\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\CD\\A0\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\CD\\A0\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\CD\\A0\\20\\20\\20\\20\\20\\20\\46\\6C\\6F\\77\\43\\6F\\6D\\70\\65\\6E\\73\\61\\74\\69\\6F\\6E\\20\\72\\65\\53\\6F\\75\\6E\\64\\22\\22\\20\\22\\22\\50\\72\\6F\\70\\65\\72\\74\\69\\65\\73\\2E\\53\\6F\\75\\6E\\64\\2E\\50\\6F\\73\\74\\53\\6F\\75\\6E\\64\\22\\22";
    m_InvalidCESTCustomTagRevisionNoNumber                        = "20\\20\\20\\20\\20\\20\\20\\20\\3C\\43\\6F\\6E\\6E\\65\\63\\74\\69\\6F\\6E\\2E\\22\\22\\63\\31\\22\\22\\3E\\20\\20\\7B\\20\\22\\22\\49\\6D\\61\\67\\65\\52\\65\\61\\64\\79\\22\\22\\20\\22\\22\\22\\22\\20\\22\\22\\43\\6F\\6D\\70\\75\\74\\65\\49\\6D\\61\\67\\65\\22\\22\\20\\20\\7D\\0A\\20\\20\\20\\20\\20\\20\\7D\\0A\\20\\20\\20\\20\\7D\\0A\\20\\20\\7D\\0A\\7D\\0A\\23\\23\\23\\20\\41\\53\\43\\43\\4F\\4E\\56\\20\\42\\45\\47\\49\\4E\\20\\23\\23\\23\\0A\\75\\6C\\56\\65\\72\\73\\69\\6F\\6E\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\30\\78\\31\\34\\62\\34\\34\\62\\36\\0A\\74\\53\\65\\71\\75\\65\\6E\\63\\65\\46\\69\\6C\\65\\4E\\61\\6D\\65\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\22\\22\\25\\43\\75\\73\\74\\6F\\6D\\65\\72\\53\\65\\71\\25\\5C\\58\\58\\58\\58\\58\\5F\\43\\45\\53\\54\\5F\\52\\65\\76\\61\\62\\63\\22\\22\\0A\\74\\50\\72\\6F\\74\\6F\\63\\6F\\6C\\4E\\61\\6D\\65\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\22\\22\\57\\41\\53\\41\\42\\49\\2B\\41\\46\\38\\2D\\4D\\49\\54\\4B\\2D\\54\\45\\53\\54\\2B\\41\\46\\38\\2D\\37\\54\\2B\\41\\46\\38\\2D\\33\\73\\6C\\69\\63\\65\\73\\22\\22\\0A\\73\\50\\72\\6F\\74\\43\\6F\\6E\\73\\69\\73\\74\\65\\6E\\63\\79\\49\\6E\\66\\6F\\2E\\74\\53\\79\\73\\74\\65\\6D\\54\\79\\70\\65\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\22\\22\\30\\39\\35\\22\\22\\0A\\73\\50\\72\\6F\\74\\43\\6F\\6E\\73\\69\\73\\74\\65\\6E\\63\\79\\49\\6E\\66\\6F\\2E\\66\\6C\\4E\\6F\\6D\\69\\6E\\61\\6C\\42\\30\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\36\\2E\\39\\38\\0A\\73\\54\\58\\53\\50\\45\\43\\2E\\61\\73\\4E\\75\\63\\6C\\65\\75\\73\\49\\6E\\66\\6F\\5B\\30\\5D\\2E\\6C\\46\\72\\65\\71\\75\\65\\6E\\63\\79\\20\\20\\20\\20\\20\\20\\3D\\20\\32\\39\\37\\31\\35\\34\\37\\35\\36\\0A\\73\\54\\58\\53\\50\\45\\43\\2E\\61\\73\\4E\\75\\63\\6C\\65\\75\\73\\49\\6E\\66\\6F\\5B\\30\\5D\\2E\\66\\6C\\52\\65\\66\\65\\72\\65\\6E\\63\\65\\41\\6D\\70\\6C\\69\\74\\75\\64\\65\\20\\3D\\20\\31\\36\\31\\2E\\34\\33\\35\\0A\\61\\6C\\54\\52\\5B\\30\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\32\\33\\30\\30\\30\\0A\\61\\6C\\54\\45\\5B\\30\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\33\\36\\34\\30\\0A\\6C\\41\\76\\65\\72\\61\\67\\65\\73\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\0A\\6C\\52\\65\\70\\65\\74\\69\\74\\69\\6F\\6E\\73\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\33\\31\\0A\\61\\64\\46\\6C\\69\\70\\41\\6E\\67\\6C\\65\\44\\65\\67\\72\\65\\65\\5B\\30\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\30\\0A\\6C\\54\\6F\\74\\61\\6C\\53\\63\\61\\6E\\54\\69\\6D\\65\\53\\65\\63\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\39\\37\\0A\\73\\45\\46\\49\\53\\50\\45\\43\\2E\\62\\45\\46\\49\\44\\61\\74\\61\\56\\61\\6C\\69\\64\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\31\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\32\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\33\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\34\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\35\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\32\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\36\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\35\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\37\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\33\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\38\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\39\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\31\\30\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\33\\32\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\31\\31\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\31\\32\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\31\\33\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\35\\30\\30\\30\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\31\\34\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\35\\36\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\31\\35\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\33\\30\\30\\30\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\31\\36\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\39\\30\\30\\30\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\31\\37\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\30\\30\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\31\\38\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\35\\30\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\31\\39\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\35\\30\\30\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\32\\30\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\32\\32\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\64\\46\\72\\65\\65\\5B\\31\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\32\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\64\\46\\72\\65\\65\\5B\\32\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\33\\2E\\37\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\64\\46\\72\\65\\65\\5B\\33\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\36\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\64\\46\\72\\65\\65\\5B\\34\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\32\\30\\30\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\64\\46\\72\\65\\65\\5B\\35\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\38\\30\\30\\30\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\64\\46\\72\\65\\65\\5B\\36\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\34\\30\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\64\\46\\72\\65\\65\\5B\\37\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\30\\2E\\31\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\64\\46\\72\\65\\65\\5B\\38\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\35\\31\\32\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\64\\46\\72\\65\\65\\5B\\39\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\35\\30\\30\\30\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\64\\46\\72\\65\\65\\5B\\31\\30\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\33\\2E\\37\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\64\\46\\72\\65\\65\\5B\\31\\31\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\0A\\23\\23\\23\\20\\41\\53\\43\\43\\4F\\4E\\56\\20\\45\\4E\\44\\20\\23\\23\\23\\22\\20\\0A\\20\\20\\20\\20\\7D\\0A\\20\\20\\7D\\0A\\7D\\0A\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\CD\\A0\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\CD\\A0\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\CD\\A0\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\CD\\A0\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\CD\\A0\\20\\20\\20\\20\\20\\20\\47\\72\\61\\64\\69\\65\\6E\\74\\4D\\6F\\64\\65\\20\\72\\6D\\61\\6E\\63\\65\\43\\61\\63\\68\\65\\2E\\69\\6E\\6C\\69\\6E\\65\\5F\\70\\6F\\73\\64\\69\\73\\70\\5F\\63\\61\\6E\\5F\\73\\65\\74\\22\\22\\20\\3C\\44\\6C\\6C\\3E\\20\\22\\22\\4D\\72\\4D\\75\\6C\\74\\01\\20\\20\\20\\53\\48\\20\\20\\16\\20\\20\\20\\06\\20\\20\\20\\4D\\20\\20\\20\\05\\20\\20\\20\\05\\20\\20\\20\\4D\\20\\20\\20\\05\\20\\20\\20\\46\\61\\73\\74\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\CD\\A0\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\CD\\A0\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\CD\\A0\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\CD\\A0\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\CD\\A0\\20\\20\\20\\20\\20\\20\\46\\6C\\6F\\77\\43\\6F\\6D\\70\\65\\6E\\73\\61\\74\\69\\6F\\6E\\20\\72\\65\\53\\6F\\75\\6E\\64\\22\\22\\20\\22\\22\\50\\72\\6F\\70\\65\\72\\74\\69\\65\\73\\2E\\53\\6F\\75\\6E\\64\\2E\\50\\6F\\73\\74\\53\\6F\\75\\6E\\64\\22\\22";
    m_NonCESTCustomTag                                            = "20\\20\\20\\20\\20\\20\\20\\20\\3C\\43\\6F\\6E\\6E\\65\\63\\74\\69\\6F\\6E\\2E\\22\\22\\63\\31\\22\\22\\3E\\20\\20\\7B\\20\\22\\22\\49\\6D\\61\\67\\65\\52\\65\\61\\64\\79\\22\\22\\20\\22\\22\\22\\22\\20\\22\\22\\43\\6F\\6D\\70\\75\\74\\65\\49\\6D\\61\\67\\65\\22\\22\\20\\20\\7D\\0A\\20\\20\\20\\20\\20\\20\\7D\\0A\\20\\20\\20\\20\\7D\\0A\\20\\20\\7D\\0A\\7D\\0A\\23\\23\\23\\20\\41\\53\\43\\43\\4F\\4E\\56\\20\\42\\45\\47\\49\\4E\\20\\23\\23\\23\\0A\\75\\6C\\56\\65\\72\\73\\69\\6F\\6E\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\30\\78\\31\\34\\62\\34\\34\\62\\36\\0A\\74\\53\\65\\71\\75\\65\\6E\\63\\65\\46\\69\\6C\\65\\4E\\61\\6D\\65\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\22\\22\\25\\43\\75\\73\\74\\6F\\6D\\65\\72\\53\\65\\71\\25\\5C\\58\\58\\58\\58\\58\\5F\\59\\59\\59\\59\\5F\\5A\\5A\\5A\\5A\\5A\\5A\\5A\\22\\22\\0A\\74\\50\\72\\6F\\74\\6F\\63\\6F\\6C\\4E\\61\\6D\\65\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\22\\22\\57\\41\\53\\41\\42\\49\\2B\\41\\46\\38\\2D\\4D\\49\\54\\4B\\2D\\54\\45\\53\\54\\2B\\41\\46\\38\\2D\\37\\54\\2B\\41\\46\\38\\2D\\33\\73\\6C\\69\\63\\65\\73\\22\\22\\0A\\73\\50\\72\\6F\\74\\43\\6F\\6E\\73\\69\\73\\74\\65\\6E\\63\\79\\49\\6E\\66\\6F\\2E\\74\\53\\79\\73\\74\\65\\6D\\54\\79\\70\\65\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\22\\22\\30\\39\\35\\22\\22\\0A\\73\\50\\72\\6F\\74\\43\\6F\\6E\\73\\69\\73\\74\\65\\6E\\63\\79\\49\\6E\\66\\6F\\2E\\66\\6C\\4E\\6F\\6D\\69\\6E\\61\\6C\\42\\30\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\36\\2E\\39\\38\\0A\\73\\54\\58\\53\\50\\45\\43\\2E\\61\\73\\4E\\75\\63\\6C\\65\\75\\73\\49\\6E\\66\\6F\\5B\\30\\5D\\2E\\6C\\46\\72\\65\\71\\75\\65\\6E\\63\\79\\20\\20\\20\\20\\20\\20\\3D\\20\\32\\39\\37\\31\\35\\34\\37\\35\\36\\0A\\73\\54\\58\\53\\50\\45\\43\\2E\\61\\73\\4E\\75\\63\\6C\\65\\75\\73\\49\\6E\\66\\6F\\5B\\30\\5D\\2E\\66\\6C\\52\\65\\66\\65\\72\\65\\6E\\63\\65\\41\\6D\\70\\6C\\69\\74\\75\\64\\65\\20\\3D\\20\\31\\36\\31\\2E\\34\\33\\35\\0A\\61\\6C\\54\\52\\5B\\30\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\32\\33\\30\\30\\30\\0A\\61\\6C\\54\\45\\5B\\30\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\33\\36\\34\\30\\0A\\6C\\41\\76\\65\\72\\61\\67\\65\\73\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\0A\\6C\\52\\65\\70\\65\\74\\69\\74\\69\\6F\\6E\\73\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\33\\31\\0A\\61\\64\\46\\6C\\69\\70\\41\\6E\\67\\6C\\65\\44\\65\\67\\72\\65\\65\\5B\\30\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\30\\0A\\6C\\54\\6F\\74\\61\\6C\\53\\63\\61\\6E\\54\\69\\6D\\65\\53\\65\\63\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\39\\37\\0A\\73\\45\\46\\49\\53\\50\\45\\43\\2E\\62\\45\\46\\49\\44\\61\\74\\61\\56\\61\\6C\\69\\64\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\31\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\32\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\33\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\34\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\35\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\32\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\36\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\35\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\37\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\33\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\38\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\39\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\31\\30\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\33\\32\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\31\\31\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\31\\32\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\31\\33\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\35\\30\\30\\30\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\31\\34\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\35\\36\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\31\\35\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\33\\30\\30\\30\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\31\\36\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\39\\30\\30\\30\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\31\\37\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\30\\30\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\31\\38\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\35\\30\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\31\\39\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\35\\30\\30\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\32\\30\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\32\\32\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\64\\46\\72\\65\\65\\5B\\31\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\32\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\64\\46\\72\\65\\65\\5B\\32\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\33\\2E\\37\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\64\\46\\72\\65\\65\\5B\\33\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\36\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\64\\46\\72\\65\\65\\5B\\34\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\32\\30\\30\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\64\\46\\72\\65\\65\\5B\\35\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\38\\30\\30\\30\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\64\\46\\72\\65\\65\\5B\\36\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\34\\30\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\64\\46\\72\\65\\65\\5B\\37\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\30\\2E\\31\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\64\\46\\72\\65\\65\\5B\\38\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\35\\31\\32\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\64\\46\\72\\65\\65\\5B\\39\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\35\\30\\30\\30\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\64\\46\\72\\65\\65\\5B\\31\\30\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\33\\2E\\37\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\64\\46\\72\\65\\65\\5B\\31\\31\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\0A\\23\\23\\23\\20\\41\\53\\43\\43\\4F\\4E\\56\\20\\45\\4E\\44\\20\\23\\23\\23\\22\\20\\0A\\20\\20\\20\\20\\7D\\0A\\20\\20\\7D\\0A\\7D\\0A\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\CD\\A0\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\CD\\A0\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\CD\\A0\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\CD\\A0\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\CD\\A0\\20\\20\\20\\20\\20\\20\\47\\72\\61\\64\\69\\65\\6E\\74\\4D\\6F\\64\\65\\20\\72\\6D\\61\\6E\\63\\65\\43\\61\\63\\68\\65\\2E\\69\\6E\\6C\\69\\6E\\65\\5F\\70\\6F\\73\\64\\69\\73\\70\\5F\\63\\61\\6E\\5F\\73\\65\\74\\22\\22\\20\\3C\\44\\6C\\6C\\3E\\20\\22\\22\\4D\\72\\4D\\75\\6C\\74\\01\\20\\20\\20\\53\\48\\20\\20\\16\\20\\20\\20\\06\\20\\20\\20\\4D\\20\\20\\20\\05\\20\\20\\20\\05\\20\\20\\20\\4D\\20\\20\\20\\05\\20\\20\\20\\46\\61\\73\\74\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\CD\\A0\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\CD\\A0\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\CD\\A0\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\CD\\A0\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\CD\\A0\\20\\20\\20\\20\\20\\20\\46\\6C\\6F\\77\\43\\6F\\6D\\70\\65\\6E\\73\\61\\74\\69\\6F\\6E\\20\\72\\65\\53\\6F\\75\\6E\\64\\22\\22\\20\\22\\22\\50\\72\\6F\\70\\65\\72\\74\\69\\65\\73\\2E\\53\\6F\\75\\6E\\64\\2E\\50\\6F\\73\\74\\53\\6F\\75\\6E\\64\\22\\22";
    m_GarbageWithinDelimiters                                     = "20\\20\\20\\20\\20\\20\\20\\20\\3C\\43\\6F\\6E\\6E\\65\\63\\74\\69\\6F\\6E\\2E\\22\\22\\63\\31\\22\\22\\3E\\20\\20\\7B\\20\\22\\22\\49\\6D\\61\\67\\65\\52\\65\\61\\64\\79\\22\\22\\20\\22\\22\\22\\22\\20\\22\\22\\43\\6F\\6D\\70\\75\\74\\65\\49\\6D\\61\\67\\65\\22\\22\\20\\20\\7D\\0A\\20\\20\\20\\20\\20\\20\\7D\\0A\\20\\20\\20\\20\\7D\\0A\\20\\20\\7D\\0A\\7D\\0A\\23\\23\\23\\20\\41\\53\\43\\43\\4F\\4E\\56\\20\\42\\45\\47\\49\\4E\\20\\23\\23\\23\\0A\\32\\33\\39\\37\\34\\30\\32\\62\\76\\30\\77\\6E\\65\\72\\62\\66\\6B\\3C\\73\\79\\64\\68\\66\\6A\\6F\\5A\\49\\47\\28\\3D\\2F\\51\\22\\26\\54\\24\\29\\28\\2F\\26\\54\\C2\\A7\\51\\3D\\28\\25\\2F\\36\\7A\\C3\\9F\\61\\62\\72\\6E\\65\\62\\74\\6C\\61\\6B\\73\\64\\20\\76\\6E\\66\\6C\\61\\69\\72\\62\\74\\7A\\38\\33\\34\\35\\74\\31\\38\\33\\30\\37\\34\\7A\\62\\74\\28\\2F\\54\\26\\3D\\26\\28\\51\\50\\57\\C2\\A7\\45\\25\\3D\\51\\57\\C2\\A7\\62\\70\\39\\38\\65\\72\\74\\76\\68\\6E\\38\\39\\33\\34\\36\\7A\\31\\76\\C3\\9F\\30\\35\\39\\34\\75\\31\\6E\\30\\20\\74\\76\\6C\\61\\62\\72\\7A\\70\\39\\30\\65\\77\\35\\37\\32\\33\\38\\34\\36\\30\\39\\74\\7A\\C3\\9F\\39\\42\\20\\4E\\50\\29\\28\\20\\4E\\26\\2F\\3D\\2F\\24\\22\\26\\20\\50\\29\\4E\\72\\65\\74\\62\\68\\61\\70\\74\\6E\\76\\61\\70\\39\\72\\38\\74\\7A\\76\\42\\20\\26\\60\\3F\\29\\C2\\A7\\3D\\57\\26\\2F\\28\\29\\3F\\57\\26\\2F\\42\\20\\56\\24\\45\\57\\28\\26\\3F\\20\\4E\\62\\38\\34\\77\\7A\\61\\77\\C3\\9F\\20\\38\\34\\62\\6D\\C3\\9F\\61\\39\\65\\6D\\63\\36\\7A\\76\\77\\75\\69\\35\\34\\62\\36\\C3\\9F\\61\\38\\39\\6E\\36\\7A\\34\\35\\76\\6D\\38\\20\\6E\\75\\20\\20\\20\\20\\38\\33\\7A\\6E\\36\\76\\30\\33\\7A\\35\\36\\76\\C3\\9F\\5E\\5E\\39\\20\\5E\\62\\74\\7A\\C3\\9F\\34\\39\\38\\62\\7A\\6E\\20\\71\\38\\34\\65\\36\\7A\\76\\37\\38\\39\\C3\\9F\\C3\\9F\\20\\28\\2F\\20\\24\\3F\\57\\20\\26\\2F\\5A\\3F\\4E\\57\\24\\42\\20\\4D\\48\\4A\\55\\20\\28\\24\\2F\\20\\56\\4E\\5A\\3F\\57\\29\\28\\24\\29\\42\\26\\24\\0A\\23\\23\\23\\20\\41\\53\\43\\43\\4F\\4E\\56\\20\\45\\4E\\44\\20\\23\\23\\23\\22\\20\\0A\\20\\20\\20\\20\\7D\\0A\\20\\20\\7D\\0A\\7D\\0A\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\CD\\A0\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\CD\\A0\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\CD\\A0\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\CD\\A0\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\CD\\A0\\20\\20\\20\\20\\20\\20\\47\\72\\61\\64\\69\\65\\6E\\74\\4D\\6F\\64\\65\\20\\72\\6D\\61\\6E\\63\\65\\43\\61\\63\\68\\65\\2E\\69\\6E\\6C\\69\\6E\\65\\5F\\70\\6F\\73\\64\\69\\73\\70\\5F\\63\\61\\6E\\5F\\73\\65\\74\\22\\22\\20\\3C\\44\\6C\\6C\\3E\\20\\22\\22\\4D\\72\\4D\\75\\6C\\74\\01\\20\\20\\20\\53\\48\\20\\20\\16\\20\\20\\20\\06\\20\\20\\20\\4D\\20\\20\\20\\05\\20\\20\\20\\05\\20\\20\\20\\4D\\20\\20\\20\\05\\20\\20\\20\\46\\61\\73\\74\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\CD\\A0\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\CD\\A0\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\CD\\A0\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\CD\\A0\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\CD\\A0\\20\\20\\20\\20\\20\\20\\46\\6C\\6F\\77\\43\\6F\\6D\\70\\65\\6E\\73\\61\\74\\69\\6F\\6E\\20\\72\\65\\53\\6F\\75\\6E\\64\\22\\22\\20\\22\\22\\50\\72\\6F\\70\\65\\72\\74\\69\\65\\73\\2E\\53\\6F\\75\\6E\\64\\2E\\50\\6F\\73\\74\\53\\6F\\75\\6E\\64\\22\\22";
    m_ValidCESTCustomTagWindowsLineEndings                        = "20\\20\\20\\20\\20\\20\\20\\20\\3C\\43\\6F\\6E\\6E\\65\\63\\74\\69\\6F\\6E\\2E\\22\\22\\63\\31\\22\\22\\3E\\20\\20\\7B\\20\\22\\22\\49\\6D\\61\\67\\65\\52\\65\\61\\64\\79\\22\\22\\20\\22\\22\\22\\22\\20\\22\\22\\43\\6F\\6D\\70\\75\\74\\65\\49\\6D\\61\\67\\65\\22\\22\\20\\20\\7D\\0D\\0A\\20\\20\\20\\20\\20\\20\\7D\\0D\\0A\\20\\20\\20\\20\\7D\\0D\\0A\\20\\20\\7D\\0D\\0A\\7D\\0D\\0A\\23\\23\\23\\20\\41\\53\\43\\43\\4F\\4E\\56\\20\\42\\45\\47\\49\\4E\\20\\23\\23\\23\\0D\\0A\\75\\6C\\56\\65\\72\\73\\69\\6F\\6E\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\30\\78\\31\\34\\62\\34\\34\\62\\36\\0D\\0A\\74\\53\\65\\71\\75\\65\\6E\\63\\65\\46\\69\\6C\\65\\4E\\61\\6D\\65\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\22\\22\\25\\43\\75\\73\\74\\6F\\6D\\65\\72\\53\\65\\71\\25\\5C\\58\\58\\58\\58\\58\\5F\\43\\45\\53\\54\\5F\\52\\65\\76\\31\\34\\31\\36\\22\\22\\0D\\0A\\74\\50\\72\\6F\\74\\6F\\63\\6F\\6C\\4E\\61\\6D\\65\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\22\\22\\57\\41\\53\\41\\42\\49\\2B\\41\\46\\38\\2D\\4D\\49\\54\\4B\\2D\\54\\45\\53\\54\\2B\\41\\46\\38\\2D\\37\\54\\2B\\41\\46\\38\\2D\\33\\73\\6C\\69\\63\\65\\73\\22\\22\\0D\\0A\\73\\50\\72\\6F\\74\\43\\6F\\6E\\73\\69\\73\\74\\65\\6E\\63\\79\\49\\6E\\66\\6F\\2E\\74\\53\\79\\73\\74\\65\\6D\\54\\79\\70\\65\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\22\\22\\30\\39\\35\\22\\22\\0D\\0A\\73\\50\\72\\6F\\74\\43\\6F\\6E\\73\\69\\73\\74\\65\\6E\\63\\79\\49\\6E\\66\\6F\\2E\\66\\6C\\4E\\6F\\6D\\69\\6E\\61\\6C\\42\\30\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\36\\2E\\39\\38\\0D\\0A\\73\\54\\58\\53\\50\\45\\43\\2E\\61\\73\\4E\\75\\63\\6C\\65\\75\\73\\49\\6E\\66\\6F\\5B\\30\\5D\\2E\\6C\\46\\72\\65\\71\\75\\65\\6E\\63\\79\\20\\20\\20\\20\\20\\20\\3D\\20\\32\\39\\37\\31\\35\\34\\37\\35\\36\\0D\\0A\\73\\54\\58\\53\\50\\45\\43\\2E\\61\\73\\4E\\75\\63\\6C\\65\\75\\73\\49\\6E\\66\\6F\\5B\\30\\5D\\2E\\66\\6C\\52\\65\\66\\65\\72\\65\\6E\\63\\65\\41\\6D\\70\\6C\\69\\74\\75\\64\\65\\20\\3D\\20\\31\\36\\31\\2E\\34\\33\\35\\0D\\0A\\61\\6C\\54\\52\\5B\\30\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\32\\33\\30\\30\\30\\0D\\0A\\61\\6C\\54\\45\\5B\\30\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\33\\36\\34\\30\\0D\\0A\\6C\\41\\76\\65\\72\\61\\67\\65\\73\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\0D\\0A\\6C\\52\\65\\70\\65\\74\\69\\74\\69\\6F\\6E\\73\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\33\\31\\0D\\0A\\61\\64\\46\\6C\\69\\70\\41\\6E\\67\\6C\\65\\44\\65\\67\\72\\65\\65\\5B\\30\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\30\\0D\\0A\\6C\\54\\6F\\74\\61\\6C\\53\\63\\61\\6E\\54\\69\\6D\\65\\53\\65\\63\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\39\\37\\0D\\0A\\73\\45\\46\\49\\53\\50\\45\\43\\2E\\62\\45\\46\\49\\44\\61\\74\\61\\56\\61\\6C\\69\\64\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\0D\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\31\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\0D\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\32\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\0D\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\33\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\0D\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\34\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\0D\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\35\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\32\\0D\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\36\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\35\\0D\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\37\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\33\\0D\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\38\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\0D\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\39\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\0D\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\31\\30\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\33\\32\\0D\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\31\\31\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\0D\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\31\\32\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\0D\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\31\\33\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\35\\30\\30\\30\\0D\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\31\\34\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\35\\36\\0D\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\31\\35\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\33\\30\\30\\30\\0D\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\31\\36\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\39\\30\\30\\30\\0D\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\31\\37\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\30\\30\\0D\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\31\\38\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\35\\30\\0D\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\31\\39\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\35\\30\\30\\0D\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\32\\30\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\0D\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\32\\32\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\0D\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\64\\46\\72\\65\\65\\5B\\31\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\32\\0D\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\64\\46\\72\\65\\65\\5B\\32\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\33\\2E\\37\\0D\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\64\\46\\72\\65\\65\\5B\\33\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\36\\0D\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\64\\46\\72\\65\\65\\5B\\34\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\32\\30\\30\\0D\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\64\\46\\72\\65\\65\\5B\\35\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\38\\30\\30\\30\\0D\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\64\\46\\72\\65\\65\\5B\\36\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\34\\30\\0D\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\64\\46\\72\\65\\65\\5B\\37\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\30\\2E\\31\\0D\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\64\\46\\72\\65\\65\\5B\\38\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\35\\31\\32\\0D\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\64\\46\\72\\65\\65\\5B\\39\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\35\\30\\30\\30\\0D\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\64\\46\\72\\65\\65\\5B\\31\\30\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\33\\2E\\37\\0D\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\64\\46\\72\\65\\65\\5B\\31\\31\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\31\\0D\\0A\\23\\23\\23\\20\\41\\53\\43\\43\\4F\\4E\\56\\20\\45\\4E\\44\\20\\23\\23\\23\\22\\20\\0D\\0A\\20\\20\\20\\20\\7D\\0D\\0A\\20\\20\\7D\\0D\\0A\\7D\\0D\\0A\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\CD\\A0\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\CD\\A0\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\CD\\A0\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\CD\\A0\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\CD\\A0\\20\\20\\20\\20\\20\\20\\47\\72\\61\\64\\69\\65\\6E\\74\\4D\\6F\\64\\65\\20\\72\\6D\\61\\6E\\63\\65\\43\\61\\63\\68\\65\\2E\\69\\6E\\6C\\69\\6E\\65\\5F\\70\\6F\\73\\64\\69\\73\\70\\5F\\63\\61\\6E\\5F\\73\\65\\74\\22\\22\\20\\3C\\44\\6C\\6C\\3E\\20\\22\\22\\4D\\72\\4D\\75\\6C\\74\\01\\20\\20\\20\\53\\48\\20\\20\\16\\20\\20\\20\\06\\20\\20\\20\\4D\\20\\20\\20\\05\\20\\20\\20\\05\\20\\20\\20\\4D\\20\\20\\20\\05\\20\\20\\20\\46\\61\\73\\74\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\CD\\A0\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\CD\\A0\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\CD\\A0\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\CD\\A0\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\CD\\A0\\20\\20\\20\\20\\20\\20\\46\\6C\\6F\\77\\43\\6F\\6D\\70\\65\\6E\\73\\61\\74\\69\\6F\\6E\\20\\72\\65\\53\\6F\\75\\6E\\64\\22\\22\\20\\22\\22\\50\\72\\6F\\70\\65\\72\\74\\69\\65\\73\\2E\\53\\6F\\75\\6E\\64\\2E\\50\\6F\\73\\74\\53\\6F\\75\\6E\\64\\22\\22";
    m_ValidCESTCustomTagAlternatingOffset                         = "36\\23\\23\\23\\20\\41\\53\\43\\43\\4F\\4E\\56\\20\\42\\45\\47\\49\\4E\\20\\23\\23\\23\\0A\\75\\6C\\56\\65\\72\\73\\69\\6F\\6E\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\30\\78\\31\\34\\62\\34\\34\\62\\36\\0A\\74\\53\\65\\71\\75\\65\\6E\\63\\65\\46\\69\\6C\\65\\4E\\61\\6D\\65\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\22\\22\\25\\43\\75\\73\\74\\6F\\6D\\65\\72\\53\\65\\71\\25\\5C\\58\\58\\58\\58\\58\\5F\\43\\45\\53\\54\\5F\\52\\65\\76\\31\\34\\31\\36\\22\\22\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\38\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\32\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\31\\30\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\33\\32\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\64\\46\\72\\65\\65\\5B\\31\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\32\\0A\\23\\23\\23\\20\\41\\53\\43\\43\\4F\\4E\\56\\20\\45\\4E\\44\\20\\23\\23\\23\\22";
    m_ValidCESTCustomTagSingleOffset                              = "36\\23\\23\\23\\20\\41\\53\\43\\43\\4F\\4E\\56\\20\\42\\45\\47\\49\\4E\\20\\23\\23\\23\\0A\\75\\6C\\56\\65\\72\\73\\69\\6F\\6E\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\30\\78\\31\\34\\62\\34\\34\\62\\36\\0A\\74\\53\\65\\71\\75\\65\\6E\\63\\65\\46\\69\\6C\\65\\4E\\61\\6D\\65\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\22\\22\\25\\43\\75\\73\\74\\6F\\6D\\65\\72\\53\\65\\71\\25\\5C\\58\\58\\58\\58\\58\\5F\\43\\45\\53\\54\\5F\\52\\65\\76\\31\\34\\31\\36\\22\\22\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\38\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\34\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\31\\30\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\33\\32\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\64\\46\\72\\65\\65\\5B\\31\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\32\\0A\\23\\23\\23\\20\\41\\53\\43\\43\\4F\\4E\\56\\20\\45\\4E\\44\\20\\23\\23\\23\\22";
    m_ValidCESTCustomTagListOffset                                = "36\\23\\23\\23\\20\\41\\53\\43\\43\\4F\\4E\\56\\20\\42\\45\\47\\49\\4E\\20\\23\\23\\23\\0A\\75\\6C\\56\\65\\72\\73\\69\\6F\\6E\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\30\\78\\31\\34\\62\\34\\34\\62\\36\\0A\\74\\53\\65\\71\\75\\65\\6E\\63\\65\\46\\69\\6C\\65\\4E\\61\\6D\\65\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\22\\22\\25\\43\\75\\73\\74\\6F\\6D\\65\\72\\53\\65\\71\\25\\5C\\58\\58\\58\\58\\58\\5F\\43\\45\\53\\54\\5F\\52\\65\\76\\31\\34\\31\\36\\22\\22\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\38\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\33\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\6C\\46\\72\\65\\65\\5B\\31\\30\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\33\\32\\0A\\73\\57\\69\\50\\4D\\65\\6D\\42\\6C\\6F\\63\\6B\\2E\\61\\64\\46\\72\\65\\65\\5B\\31\\5D\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\20\\3D\\20\\32\\0A\\23\\23\\23\\20\\41\\53\\43\\43\\4F\\4E\\56\\20\\45\\4E\\44\\20\\23\\23\\23\\22";
  }

  void tearDown() override
  {
  }

  void ValidPropertyParsedToPropertyList_Success()
  {
    mitk::CustomTagParser tagParser(m_PathToModule);
    auto tsproperty = mitk::TemporoSpatialStringProperty::New();
    tsproperty->SetValue(0, 0, m_ValidCESTCustomTag);
    auto parsedPropertyList = tagParser.ParseDicomProperty(tsproperty);

    std::string sampling = "";
    std::string offset = "";
    std::string offsets = "";
    std::string measurements = "";
    std::string revision = "";
    std::string jsonRevision = "";
    bool hasRevision = parsedPropertyList->GetStringProperty("CEST.Revision", revision);
    parsedPropertyList->GetStringProperty("CEST.Offset", offset);
    parsedPropertyList->GetStringProperty(mitk::CEST_PROPERTY_NAME_OFFSETS().c_str(), offsets);
    parsedPropertyList->GetStringProperty("CEST.measurements", measurements);
    parsedPropertyList->GetStringProperty("CEST.SamplingType", sampling);
    parsedPropertyList->GetStringProperty("CEST.revision_json", jsonRevision);

    bool offsetsMatch =( offsets == "-300 -2 -1.86667 -1.73333 -1.6 -1.46667 -1.33333 -1.2 -1.06667 -0.933333 -0.8 -0.666667 -0.533333 -0.4 -0.266667 -0.133333 0 0.133333 0.266667 0.4 0.533333 0.666667 0.8 0.933333 1.06667 1.2 1.33333 1.46667 1.6 1.73333 1.86667 2");

    CPPUNIT_ASSERT_MESSAGE("Verify we found a revision.", hasRevision);
    CPPUNIT_ASSERT_MESSAGE("Verify the revision is the one we expect.", revision == "1416");
    CPPUNIT_ASSERT_MESSAGE("Verify the revision and the json revision match.", revision == jsonRevision);
    CPPUNIT_ASSERT_MESSAGE("Verify a couple of resulting properties match our expectation.", offset == "2" && sampling == "1" && measurements == "32");
    CPPUNIT_ASSERT_MESSAGE("Verify offsets are correctly parsed.", offsetsMatch);
  }

  void ValidPropertyMissingParametersParsedToEmptyPropertiesPropertyList_Success()
  {
    mitk::CustomTagParser tagParser(m_PathToModule);
    auto parsedPropertyList = tagParser.ParseDicomPropertyString(m_ValidCESTCustomTagAllParametersMissing);
    std::string revision = "";
    std::string jsonRevision = "";
    bool hasRevision = parsedPropertyList->GetStringProperty("CEST.Revision", revision);
    bool hasJsonRevision = parsedPropertyList->GetStringProperty("CEST.revision_json", jsonRevision);

    auto propertyMap = parsedPropertyList->GetMap();
    bool propertiesEmpty = true;
    for (auto const &prop : *propertyMap)
    {
      std::string key = prop.first;
      if (key != "CEST.Revision" && key != "CEST.revision_json")
      {
        propertiesEmpty = propertiesEmpty && prop.second->GetValueAsString() == "";
      }
    }

    CPPUNIT_ASSERT_MESSAGE("Verify we found a revision.", hasRevision);
    CPPUNIT_ASSERT_MESSAGE("Verify we found a json revision.", hasJsonRevision);
    CPPUNIT_ASSERT_MESSAGE("Property list properties are empty but for the revision information", propertiesEmpty);
  }

  void ValidPropertyRevisionVeryLow_UseDefault_Success()
  {
    mitk::CustomTagParser tagParser(m_PathToModule);
    auto parsedPropertyList = tagParser.ParseDicomPropertyString(m_ValidCESTCustomTagUnsupportedRevisionTooLow);
    std::string revision = "";
    std::string jsonRevision = "";
    bool hasRevision = parsedPropertyList->GetStringProperty("CEST.Revision", revision);
    bool hasJsonRevision = parsedPropertyList->GetStringProperty("CEST.revision_json", jsonRevision);
    bool usedDefault = (jsonRevision == "default mapping, corresponds to revision 1416");

    CPPUNIT_ASSERT_MESSAGE("Verify we found a revision.", hasRevision);
    CPPUNIT_ASSERT_MESSAGE("Verify we found a json revision.", hasJsonRevision);
    CPPUNIT_ASSERT_MESSAGE("Verify we used default mapping.", usedDefault);
  }

  void ValidPropertyNoExactRevisionMatchUseInternal_Success()
  {
    mitk::CustomTagParser tagParser(m_PathToModule);
    auto parsedPropertyList = tagParser.ParseDicomPropertyString(m_ValidCESTCustomTagUnsupportedRevisionNoExactJSONUseInternal);
    std::string revision = "";
    std::string jsonRevision = "";
    bool hasRevision = parsedPropertyList->GetStringProperty("CEST.Revision", revision);
    bool hasJsonRevision = parsedPropertyList->GetStringProperty("CEST.revision_json", jsonRevision);
    bool usedInternal = (jsonRevision == "1416");

    CPPUNIT_ASSERT_MESSAGE("Verify we found a revision.", hasRevision);
    CPPUNIT_ASSERT_MESSAGE("Verify we found a json revision.", hasJsonRevision);
    CPPUNIT_ASSERT_MESSAGE("Verify we used internal mapping.", usedInternal);
  }

  void ValidPropertyNoExactRevisionMatchUseExternal_Success()
  {
    std::string externalMappingString =
    "{\n"
      "  \"external mapping for test\" : \"revision_json\",\n"
      "  \"sWiPMemBlock.alFree[1]\" : \"AdvancedMode\"\n"
      "}";

    // we assume the test library will be in the same location as the MitkCEST library on windows
    // on linux the test driver should have a relative path of ../bin/
#ifdef _WIN32
    std::string dirname = m_PathToModule + "/CESTRevisionMapping";
#else
    std::string dirname = m_PathToModule + "/../lib/CESTRevisionMapping";
#endif

    //bool dirWasThere = itksys::SystemTools::FileIsDirectory(dirname);

    std::string filename = dirname + "/118.json";

    itk::FileTools::CreateDirectory(dirname);

    std::ofstream externalFile(filename.c_str());

    if (externalFile.is_open())
    {
      externalFile << externalMappingString;
      externalFile.close();
    }

    mitk::CustomTagParser tagParser(m_PathToModule);
    auto parsedPropertyList = tagParser.ParseDicomPropertyString(m_ValidCESTCustomTagUnsupportedRevisionNoExactJSONUseExternal);
    std::string revision = "";
    std::string jsonRevision = "";
    bool hasRevision = parsedPropertyList->GetStringProperty("CEST.Revision", revision);
    bool hasJsonRevision = parsedPropertyList->GetStringProperty("CEST.revision_json", jsonRevision);
    bool usedExternal = (jsonRevision == "external mapping for test");

    CPPUNIT_ASSERT_MESSAGE("Verify we found a revision.", hasRevision);
    CPPUNIT_ASSERT_MESSAGE("Verify we found a json revision.", hasJsonRevision);
    CPPUNIT_ASSERT_MESSAGE("Verify we used external mapping.", usedExternal);

    bool wasError = std::remove(filename.c_str());

    if (wasError)
    {
      MITK_ERROR << "Could not delete test revision file";
    }
  }

  void ValidPropertyWindowsLineEndings_Success()
  {
    mitk::CustomTagParser tagParser(m_PathToModule);
    auto parsedPropertyList = tagParser.ParseDicomPropertyString(m_ValidCESTCustomTagWindowsLineEndings);
    std::string sampling = "";
    std::string offset = "";
    std::string offsets = "";
    std::string measurements = "";
    std::string revision = "";
    std::string jsonRevision = "";
    bool hasRevision = parsedPropertyList->GetStringProperty("CEST.Revision", revision);
    parsedPropertyList->GetStringProperty("CEST.Offset", offset);
    parsedPropertyList->GetStringProperty(mitk::CEST_PROPERTY_NAME_OFFSETS().c_str(), offsets);
    parsedPropertyList->GetStringProperty("CEST.measurements", measurements);
    parsedPropertyList->GetStringProperty("CEST.SamplingType", sampling);
    parsedPropertyList->GetStringProperty("CEST.revision_json", jsonRevision);

    bool offsetsMatch = (offsets == "-300 -2 -1.86667 -1.73333 -1.6 -1.46667 -1.33333 -1.2 -1.06667 -0.933333 -0.8 -0.666667 -0.533333 -0.4 -0.266667 -0.133333 0 0.133333 0.266667 0.4 0.533333 0.666667 0.8 0.933333 1.06667 1.2 1.33333 1.46667 1.6 1.73333 1.86667 2");

    CPPUNIT_ASSERT_MESSAGE("Verify we found a revision.", hasRevision);
    CPPUNIT_ASSERT_MESSAGE("Verify the revision is the one we expect.", revision == "1416");
    CPPUNIT_ASSERT_MESSAGE("Verify the revision and the json revision match.", revision == jsonRevision);
    CPPUNIT_ASSERT_MESSAGE("Verify a couple of resulting properties match our expectation.", offset == "2" && sampling == "1" && measurements == "32");
    CPPUNIT_ASSERT_MESSAGE("Verify offsets are correctly parsed.", offsetsMatch);
  }

  void InvalidPropertyInvalidRevision_Failure()
  {
    mitk::CustomTagParser tagParser(m_PathToModule);
    auto parsedPropertyList = tagParser.ParseDicomPropertyString(m_InvalidCESTCustomTagRevisionNoNumber);
    std::string revision = "";
    std::string jsonRevision = "";
    bool hasRevision = parsedPropertyList->GetStringProperty("CEST.Revision", revision);
    bool hasJsonRevision = parsedPropertyList->GetStringProperty("CEST.revision_json", jsonRevision);
    bool usedDefault = (jsonRevision == "default mapping, corresponds to revision 1416");

    CPPUNIT_ASSERT_MESSAGE("Verify we found a revision.", hasRevision);
    CPPUNIT_ASSERT_MESSAGE("Verify we found a json revision.", hasJsonRevision);
    CPPUNIT_ASSERT_MESSAGE("Verify we used default mapping.", usedDefault);
  }

  void InvalidPropertyNoCESTSequence_Failure()
  {
    mitk::CustomTagParser tagParser(m_PathToModule);
    auto parsedPropertyList = tagParser.ParseDicomPropertyString(m_NonCESTCustomTag);

    auto size = parsedPropertyList->GetMap()->size();

    CPPUNIT_ASSERT_MESSAGE("Property list is empty", mitk::Equal(size, 0));
  }

  void InvalidPropertyGarbageInDelimiters_Failure()
  {
    mitk::CustomTagParser tagParser(m_PathToModule);
    auto parsedPropertyList = tagParser.ParseDicomPropertyString(m_GarbageWithinDelimiters);

    auto size = parsedPropertyList->GetMap()->size();

    CPPUNIT_ASSERT_MESSAGE("Property list is empty", mitk::Equal(size, 0));
  }

  void ValidPropertyAlternatingOffset_Success()
  {
    mitk::CustomTagParser tagParser(m_PathToModule);
    auto parsedPropertyList = tagParser.ParseDicomPropertyString(m_ValidCESTCustomTagAlternatingOffset);
    std::string offsets = "";
    parsedPropertyList->GetStringProperty(mitk::CEST_PROPERTY_NAME_OFFSETS().c_str(), offsets);

    bool offsetsMatch = (offsets == "-300 2 -2 1.86667 -1.86667 1.73333 -1.73333 1.6 -1.6 1.46667 -1.46667 1.33333 -1.33333 1.2 -1.2 1.06667 -1.06667 0.933333 -0.933333 0.8 -0.8 0.666667 -0.666667 0.533333 -0.533333 0.4 -0.4 0.266667 -0.266667 0.133333 -0.133333 0");

    CPPUNIT_ASSERT_MESSAGE("Verify offsets are correctly parsed.", offsetsMatch);
  }
  void ValidPropertySimpleOffset_Success()
  {
    mitk::CustomTagParser tagParser(m_PathToModule);
    auto parsedPropertyList = tagParser.ParseDicomPropertyString(m_ValidCESTCustomTagSingleOffset);
    std::string offsets = "";
    parsedPropertyList->GetStringProperty(mitk::CEST_PROPERTY_NAME_OFFSETS().c_str(), offsets);

    bool offsetsMatch = (offsets == "-300 2 2 2 2 2 2 2 2 2 2 2 2 2 2 2 2 2 2 2 2 2 2 2 2 2 2 2 2 2 2 2");

    CPPUNIT_ASSERT_MESSAGE("Verify offsets are correctly parsed.", offsetsMatch);
  }
  void ValidPropertyListOffset_Success()
  {

    std::string offsetList = "-300\n -100 \n -50 \n -35\n -25 \n -17\n -12\n -9.5 \n -8.25\n -7\n -6.1 \n -5.4  \n -4.7  \n -4\n -3.3\n -2.7\n -2\n -1.7\n -1.5  \n -1.1 \n -0.9\n -300\n -0.6 \n -0.4\n -0.2\n 0 \n 0.2\n 0.4\n 0.6\n 0.95 \n 1.1 \n 1.25 \n 1.4\n 1.55\n 1.7\n 1.85 \n 2 \n 2.15 \n 2.3\n 2.45 \n 2.6\n 2.75 \n 2.9 \n 3.05\n -300 \n 3.2\n 3.35 \n 3.5\n 3.65 \n 3.8 \n 3.95\n 4.1 \n 4.25\n 4.4 \n 4.7\n 5.2\n 6\n 7\n 9\n 12 \n 17\n 25\n 35\n 50 \n 100\n -300" ;
    std::string filename = m_PathToModule + "/" + "LIST.txt";
    std::ofstream externalFile(filename.c_str());

    if (externalFile.is_open())
    {
      externalFile << offsetList;
      externalFile.close();
    }

    mitk::CustomTagParser tagParser(m_PathToModule);
    auto parsedPropertyList = tagParser.ParseDicomPropertyString(m_ValidCESTCustomTagListOffset);
    std::string offsets = "";
    parsedPropertyList->GetStringProperty(mitk::CEST_PROPERTY_NAME_OFFSETS().c_str(), offsets);
    std::string referenceString = "-300 -100 -50 -35 -25 -17 -12 -9.5 -8.25 -7 -6.1 -5.4 -4.7 -4 -3.3 -2.7 -2 -1.7 -1.5 -1.1 -0.9 -300 -0.6 -0.4 -0.2 0 0.2 0.4 0.6 0.95 1.1 1.25 1.4 1.55 1.7 1.85 2 2.15 2.3 2.45 2.6 2.75 2.9 3.05 -300 3.2 3.35 3.5 3.65 3.8 3.95 4.1 4.25 4.4 4.7 5.2 6 7 9 12 17 25 35 50 100 -300";
    bool offsetsMatch = (offsets == referenceString);

    CPPUNIT_ASSERT_MESSAGE("Verify offsets are correctly parsed.", offsetsMatch);

    bool wasError = std::remove(filename.c_str());

    if (wasError)
    {
      MITK_ERROR << "Could not delete test offset list file";
    }
  }

  void ExtractRevision()
  {
    std::string empty = "";
    std::string invalidRule1a = "CESaaaaaaa";
    std::string invalidRule1b = "aaaCESTaaa";
    std::string invalidRule2a = "CESTaaaa";
    std::string invalidRule2b = "aaa_CESTaaa";
    std::string invalidRule3a = "CESTaaa_REVaaaa";
    std::string valid1 = "CEST_REV12345";
    std::string valid2 = "aaa_CESTaaaa_REV2";
    std::string valid3 = "CESTaaaa_REV3_c";
    std::string valid4 = "cest_rev4";
    std::string valid5 = "aaa_cestaaaa_rev5";
    std::string valid6 = "cestaaaa_rev6_c";

    CPPUNIT_ASSERT_THROW_MESSAGE("Verify exception on empty", mitk::CustomTagParser::ExtractRevision(empty), mitk::Exception);
    CPPUNIT_ASSERT_THROW_MESSAGE("Verify exception on invalidRule1a", mitk::CustomTagParser::ExtractRevision(invalidRule1a), mitk::Exception);
    CPPUNIT_ASSERT_THROW_MESSAGE("Verify exception on invalidRule1b", mitk::CustomTagParser::ExtractRevision(invalidRule1b), mitk::Exception);
    CPPUNIT_ASSERT_THROW_MESSAGE("Verify exception on invalidRule2a", mitk::CustomTagParser::ExtractRevision(invalidRule2a), mitk::Exception);
    CPPUNIT_ASSERT_THROW_MESSAGE("Verify exception on invalidRule2b", mitk::CustomTagParser::ExtractRevision(invalidRule2b), mitk::Exception);
    CPPUNIT_ASSERT_MESSAGE("Verify empty revision on invalidRule3a", mitk::CustomTagParser::ExtractRevision(invalidRule3a) == "");
    CPPUNIT_ASSERT_MESSAGE("Extract revision from valid1.", mitk::CustomTagParser::ExtractRevision(valid1) == "12345");
    CPPUNIT_ASSERT_MESSAGE("Extract revision from valid2.", mitk::CustomTagParser::ExtractRevision(valid2) == "2");
    CPPUNIT_ASSERT_MESSAGE("Extract revision from valid3.", mitk::CustomTagParser::ExtractRevision(valid3) == "3");
    CPPUNIT_ASSERT_MESSAGE("Extract revision from valid4.", mitk::CustomTagParser::ExtractRevision(valid4) == "4");
    CPPUNIT_ASSERT_MESSAGE("Extract revision from valid5.", mitk::CustomTagParser::ExtractRevision(valid5) == "5");
    CPPUNIT_ASSERT_MESSAGE("Extract revision from valid6.", mitk::CustomTagParser::ExtractRevision(valid6) == "6");

  }
};

MITK_TEST_SUITE_REGISTRATION(mitkCustomTagParser)
