/*===================================================================

The Medical Imaging Interaction Toolkit (MITK)

Copyright (c) German Cancer Research Center,
Division of Medical and Biological Informatics.
All rights reserved.

This software is distributed WITHOUT ANY WARRANTY; without
even the implied warranty of MERCHANTABILITY or FITNESS FOR
A PARTICULAR PURPOSE.

See LICENSE.txt or http://www.mitk.org for details.

===================================================================*/
#ifndef __mitkThreadedToFRawDataReconstruction_h
#define __mitkThreadedToFRawDataReconstruction_h

// mitk includes
#include "mitkImageSource.h"
#include <MitkPMDExports.h>

// itk includes
#include <itkBarrier.h>
#include <itkFastMutexLock.h>

// vtk includes
#include "vtkShortArray.h"

namespace mitk
{
  struct ThreadDataStruct
  {
    std::vector<std::vector<short> > m_InputData;
    std::vector<float*> m_OutputData;
    unsigned int m_DataSize;
    unsigned int m_LineWidth;
    unsigned int m_FrameHeight;
    unsigned int m_ModulationFrequency;
    itk::Barrier::Pointer m_Barrier;              ///< barrier to synchronize ends of threads
    itk::FastMutexLock::Pointer m_ImageDataMutex; ///< mutex for coordinated access to image data
    itk::FastMutexLock::Pointer m_ThreadDataMutex; ///< mutex to control access to images
  };

  class MITKPMD_EXPORT ThreadedToFRawDataReconstruction : public itk::ProcessObject
  {
   public:

    mitkClassMacroItkParent( ThreadedToFRawDataReconstruction ,  itk::ProcessObject );
    itkFactorylessNewMacro(Self)
    itkCloneMacro(Self)

    itkGetMacro(Init, bool);

    void SetChannelData(vtkShortArray* sourceData);

    void Initialize(int width, int height, int modulationFrequency, int sourceDataSize );

    void GetDistances(float* dist);
    void GetAmplitudes(float* ampl);
    void GetIntensities(float* inten);
    void GetAllData(float* dist, float* ampl, float* inten);

    void Update();

   protected:
    /*!
    \brief standard constructor
    */
    ThreadedToFRawDataReconstruction();
    /*!
    \brief standard destructor
    */
    ~ThreadedToFRawDataReconstruction();

    /*!
    \brief method generating the outputs of this filter. Called in the updated process of the pipeline.
    This method generates the two outputs of the ToFImageSource: The distance and the intensity image
    */
    virtual void GenerateData();

    /*!
    \brief method configures the camera output and prepares the thread data struct for threaded data
          generation
    */
    virtual void BeforeThreadedGenerateData();

    /*!
    \brief threader callback function for multi threaded data generation
    */
    static ITK_THREAD_RETURN_TYPE ThreadedGenerateDataCallbackFunction(void* data);
    /*!
    \brief Method to reduce image size from original controller size to output size (200x200px)
    and rotate the image. This is needed due to defect pixel rows at the rim of the chip
    */
    void ResizeOutputImage(float* in, float* out);

    // member variables
    int m_StackSize;                ///<
    int m_Width;
    int m_Height;
    int m_ImageSize;
    int m_SourceDataSize;
    vtkShortArray* m_SourceData;
    bool m_Init;

    float* m_CISDist;               ///< holds the distance information from for one distance image slice
    float* m_CISAmpl;               ///< holds the amplitude information from for one amplitude image slice
    float* m_CISInten;              ///< holds the intensity information from for one intensity image slice
    float* m_ThreadedCISDist;
    float* m_ThreadedCISAmpl;
    float* m_ThreadedCISInten;

    itk::MultiThreader::Pointer m_Threader;
    ThreadDataStruct* m_ThreadData;

  };
} //end mitk namespace

#endif // __mitkThreadedToFRawDataReconstruction_h
