/*===================================================================

The Medical Imaging Interaction Toolkit (MITK)

Copyright (c) German Cancer Research Center,
Division of Medical and Biological Informatics.
All rights reserved.

This software is distributed WITHOUT ANY WARRANTY; without
even the implied warranty of MERCHANTABILITY or FITNESS FOR
A PARTICULAR PURPOSE.

See LICENSE.txt or http://www.mitk.org for details.

===================================================================*/

#include "mitkVtkGLMapperWrapper.h"

//mitk includes
#include "mitkDataNode.h"
#include "vtkGLMapperProp.h"
#include "mitkGL.h"

// constructor LocalStorage
mitk::VtkGLMapperWrapper::LocalStorage::LocalStorage()
{
  m_GLMapperProp = vtkSmartPointer<vtkGLMapperProp>::New();
}
// destructor LocalStorage
mitk::VtkGLMapperWrapper::LocalStorage::~LocalStorage()
{
}

// constructor VtkGLMapperWrapper
mitk::VtkGLMapperWrapper::VtkGLMapperWrapper(GLMapper::Pointer mitkGLMapper)
{
  m_MitkGLMapper = mitkGLMapper;
}

// destructor
mitk::VtkGLMapperWrapper::~VtkGLMapperWrapper()
{
}

// returns propassembly
vtkProp* mitk::VtkGLMapperWrapper::GetVtkProp(mitk::BaseRenderer * renderer)
{
  LocalStorage *ls = m_LSH.GetLocalStorage(renderer);
  return ls->m_GLMapperProp;
}

void mitk::VtkGLMapperWrapper::GenerateDataForRenderer(mitk::BaseRenderer *renderer)
{
  LocalStorage *ls = m_LSH.GetLocalStorage(renderer);
  ls->m_GLMapperProp->SetBaseRenderer(renderer);
  ls->m_GLMapperProp->SetWrappedGLMapper(m_MitkGLMapper);
}

void mitk::VtkGLMapperWrapper::ApplyColorAndOpacityProperties(mitk::BaseRenderer *renderer, vtkActor *actor)
{
  m_MitkGLMapper->ApplyColorAndOpacityProperties(renderer,actor);
}

void mitk::VtkGLMapperWrapper::MitkRender(mitk::BaseRenderer *renderer, mitk::VtkPropRenderer::RenderType type)
{
  if(type != mitk::VtkPropRenderer::Opaque)
    return;
  Enable2DOpenGL(renderer);
  Superclass::MitkRender(renderer,type);
  Disable2DOpenGL();
}

void mitk::VtkGLMapperWrapper::Update(mitk::BaseRenderer* renderer)
{
  Superclass::Update(renderer);
  m_MitkGLMapper->Update(renderer);
}

void mitk::VtkGLMapperWrapper::SetDataNode(mitk::DataNode *node)
{
  m_MitkGLMapper->SetDataNode(node);
}

mitk::DataNode *mitk::VtkGLMapperWrapper::GetDataNode() const
{
  return m_MitkGLMapper->GetDataNode();
}

/*!
\brief
Enable2DOpenGL() and Disable2DOpenGL() are used to switch between 2D rendering (orthographic projection) and 3D rendering (perspective projection)
*/
void mitk::VtkGLMapperWrapper::Enable2DOpenGL(mitk::BaseRenderer *renderer)
{
  GLint iViewport[4];

  // Get a copy of the viewport
  glGetIntegerv( GL_VIEWPORT, iViewport );

  // Save a copy of the projection matrix so that we can restore it
  // when it's time to do 3D rendering again.
  glMatrixMode( GL_PROJECTION );
  glPushMatrix();
  glLoadIdentity();

  // Set up the orthographic projection
  const mitk::DisplayGeometry* displayGeometry = renderer->GetDisplayGeometry();

  float displayGeometryWidth = displayGeometry->GetSizeInDisplayUnits()[0];
  float displayGeometryHeight = displayGeometry->GetSizeInDisplayUnits()[1];
  float viewportWidth = iViewport[2];
  float viewportHeight = iViewport[3];

  /*
     The following makes OpenGL mappers draw into the same viewport
     that is used by VTK when someone calls vtkRenderer::SetViewport().

     The parameters of glOrtho describe what "input" coordinates
     (display coordinates generated by the OpenGL mappers) are transformed
     into the region defined by the viewport. The call has to consider
     that the scene is fit vertically and centered horizontally.

     Problem: this is a crude first step towards rendering into viewports.
      - mitkViewportRenderingTest demonstrates the non-interactive rendering
        that is now possible
      - interactors that measure mouse movement in pixels will
        probably run into problems with display-to-world transformation

     A proper solution should probably modify the DisplayGeometry to
     correctly describe the viewport.
  */
  // iViewport is (x,y,width,height)
  // glOrtho expects (left,right,bottom,top,znear,zfar)
  glOrtho( 0
           - 0.5 * (viewportWidth/viewportHeight-1.0)*displayGeometryHeight
           + 0.5 * (displayGeometryWidth - displayGeometryHeight)
           ,
           displayGeometryWidth
           + 0.5 * (viewportWidth/viewportHeight-1.0)*displayGeometryHeight
           - 0.5 * (displayGeometryWidth - displayGeometryHeight)
           ,
           0, displayGeometryHeight,
      -1.0, 1.0
      );

  glMatrixMode( GL_MODELVIEW );
  glPushMatrix();
  glLoadIdentity();

  // Make sure depth testing and lighting are disabled for 2D rendering until
  // we are finished rendering in 2D
  glPushAttrib( GL_DEPTH_BUFFER_BIT | GL_LIGHTING_BIT );
  glDisable( GL_DEPTH_TEST );
  glDisable( GL_LIGHTING );
  // disable the texturing here so crosshair is painted in the correct colors
  // vtk will reenable texturing every time it is needed
  glDisable( GL_TEXTURE_1D );
  glDisable( GL_TEXTURE_2D );
  glLineWidth(1.0);
}

/*!
\brief Initialize the VtkPropRenderer

Enable2DOpenGL() and Disable2DOpenGL() are used to switch between 2D rendering (orthographic projection) and 3D rendering (perspective projection)
*/
void mitk::VtkGLMapperWrapper::Disable2DOpenGL()
{
  glPopAttrib();
  glMatrixMode( GL_PROJECTION );
  glPopMatrix();
  glMatrixMode( GL_MODELVIEW );
  glPopMatrix();
}
