/*===================================================================

The Medical Imaging Interaction Toolkit (MITK)

Copyright (c) German Cancer Research Center,
Division of Medical and Biological Informatics.
All rights reserved.

This software is distributed WITHOUT ANY WARRANTY; without
even the implied warranty of MERCHANTABILITY or FITNESS FOR
A PARTICULAR PURPOSE.

See LICENSE.txt or http://www.mitk.org for details.

===================================================================*/

#ifndef __itkPointShell_txx__
#define __itkPointShell_txx__

#include "vnl/vnl_vector_fixed.h"
#include "vnl/vnl_matrix_fixed.h"

#define _USE_MATH_DEFINES
#include <math.h>

namespace itk
{
#define DIST_POINTSHELL_PI M_PI

template<int NPoints, class TMatrixType >
TMatrixType *
PointShell<NPoints, TMatrixType >
::DistributePointShell()
{

  auto theta
      = new vnl_vector_fixed<double,NPoints>();

  auto phi
      = new vnl_vector_fixed<double,NPoints>();

  double C = sqrt(4*DIST_POINTSHELL_PI);

  (*phi)(0) = 0.0;
  (*phi)(NPoints-1) = 0.0;

  for(int i=0; i<NPoints; i++)
  {
    (*theta)(i) = acos(-1.0+2.0*i/(NPoints-1.0)) - DIST_POINTSHELL_PI / 2.0;
    if( i>0 && i<NPoints-1)
    {
      (*phi)(i) = ((*phi)(i-1) + C /
                   sqrt(NPoints*(1-(-1.0+2.0*i/(NPoints-1.0))*(-1.0+2.0*i/(NPoints-1.0)))));
      // % (2*DIST_POINTSHELL_PI);
    }
  }

  auto pointshell
      = new vnl_matrix_fixed<double, 3, NPoints>();

  for(int i=0; i<NPoints; i++)
  {
    (*pointshell)(0,i) = cos((*theta)(i)) * cos((*phi)(i));
    (*pointshell)(1,i) = cos((*theta)(i)) * sin((*phi)(i));
    (*pointshell)(2,i) = sin((*theta)(i));
  }

  return pointshell;
}

// generate by n-fold subdivisions of an icosahedron

// 1 - 12
// 2 - 42
// 3 - 92
// 4 - 162
// 5 - 252
// 6 - 362
// 7 - 492
// 8 - 642
// 9 - 812
// 10 - 1002

template<> class PointShell<12, vnl_matrix_fixed<double, 3, 12> >
{
public:
  static vnl_matrix_fixed<double, 3, 12>* DistributePointShell()
  {
    double coords[3*12] = {
      0.250254,0.320072,0.437594,
      0.550563,0.623196,0.975676,
      -0.250253,-0.320073,-0.437593,
      -0.550563,-0.6231950000000001,-0.975676,
      -0.916162,-0.078054,-0.598061,
      0.758025,0.436642,-0.177816,
      0.916161,0.078054,0.598061,
      -0.758024,-0.436643,0.177817,
      -0.313083,-0.944171,0.671442,
      -0.349682,0.648821,-0.128204,
      0.313082,0.944172,-0.671442,
      0.349683,-0.648822,0.128204 };

    auto      pointshell
        = new vnl_matrix_fixed<double, 3, 12>(coords);

    return pointshell;
  }
};


template<> class PointShell<42, vnl_matrix_fixed<double, 3, 42> >
{
public:
  static vnl_matrix_fixed<double, 3, 42>* DistributePointShell()
  {
    double coords[3*42] = {
      0.9756767549555488,0.9397937974509784,0.6231965669156312,
      0.8971014254962764,0.6899186612252972,0.5505632701289798,
      0.4375940376503738,0.06907711216819172,-0.3200730131503601,
      0.6235168560971455,0.1781713819192711,0.3200724178002418,
      0.7616224484252742,0.5117469206751248,0.2502538949373057,
      0.4043068999720243,0.7205836671095894,0.8306996157029517,
      -0.4375932291383154,-0.06907715072573903,0.06640211724156461,
      -0.6231952788307174,-0.2192092047974508,-0.1781713502362079,
      0.3352293490184233,-0.2502532651867688,-0.4043060780204903,
      0.1765178736510398,-0.975676581463901,-0.9397935597648752,
      -0.8306992108488227,-0.6235158879179115,0.21921067448898,
      -0.7616223457476226,-0.3352294641675141,-0.7205834943870382,
      -0.5117470172320778,-0.06640153973221416,-0.5505634949474448,
      -0.176517304179968,-0.6899183532182501,-0.8971016370404784,
      -0.1778161375899129,0.1521341729788544,0.4366423972091847,
      0.3410385404813853,0.7022080829625521,0.7580253719184177,
      -0.5980610514568759,-0.3056524524744078,0.07805400320688782,
      -0.09487972868743588,0.3025309778153461,-0.07805410188638827,
      -0.1503967869358663,0.3996773221825281,-0.9161616153729887,
      -0.8900379255247358,-0.6430242175142256,-0.4560492515462431,
      0.5980613131647216,0.3056526230838691,0.797087559189558,
      -0.4366431953633789,-0.7951587738972527,-0.3025315118038733,
      -0.5843857415998933,0.9161619708326186,0.8900381161951118,
      0.9840623116657442,0.1778171059718252,-0.1521342296531137,
      0.4560499397602033,0.09487917246041,0.7951586652134927,
      0.1503972383762518,0.584385354546882,0.6430245879023526,
      -0.399676350745083,-0.7970870947583816,-0.7580246814516085,
      -0.9840624137666333,-0.7022083677250226,-0.3410374453295617,
      -0.1282040992012935,0.3060111300013609,0.6488215902264565,
      -0.2808945465452747,0.1758296024995542,-0.3496821715684524,
      0.6714420577705413,0.9496341036793501,0.9441720387917811,
      0.7760312926982584,0.9363385958760152,-0.9441722324566724,
      -0.6303269409870232,-0.7605084662990487,-0.3130828685601688,
      0.2106381298878163,-0.2593820240207756,0.3193074202631062,
      -0.671442351589465,-0.9496340459616871,-0.6002017842451572,
      -0.648822290296783,-0.5653935344751363,-0.9363384293729856,
      -0.7389956620788968,0.3130823317650696,-0.2106389019045293,
      -0.02151295053653907,0.1282040764044601,-0.3060118317848469,
      -0.3193074905789999,-0.7760321386045594,0.5653931175085022,
      0.6303269573377657,0.7389959159190775,0.2593815856416469,
      0.7605089118538748,0.6002024649156598,0.3496833143594963,
      0.02151295276860444,-0.175829673802738,0.2808952005684321 };

    auto      pointshell
        = new vnl_matrix_fixed<double, 3, 42>(coords);

    return pointshell;
  }
};


template<> class PointShell<92, vnl_matrix_fixed<double, 3, 92> >
{
public:
  static vnl_matrix_fixed<double, 3, 92>* DistributePointShell()
  {
    double coords[3*92] = {
      0.9756767549555487,0.9881046683022421,0.8528238339140162,
      0.6231965669156312,0.9602286065935876,0.9016229543668698,
      0.68966672282401,0.7970712198937139,0.6617902410567804,
      0.5505632701289798,0.4375940376503739,0.2130515669413184,
      -0.07773888316805401,-0.3200730131503601,0.575075565563677,
      0.3107083252171692,-0.006505365777837463,0.6463093264667951,
      0.3555189519163455,0.3200724178002418,0.6201479032959075,
      0.8717669667532965,0.4569904905821254,0.7744719807282636,
      0.5454521871592555,0.2502538949373057,0.3600408236839538,
      0.4319414682718342,0.5665551499276381,0.697797754038698,
      0.7103565350090897,0.8449704612165639,0.9168712118355732,
      -0.4375932291383153,-0.213051272338593,0.07773856031405026,
      -0.1245894223683107,0.181648017057727,0.2546621526726748,
      -0.6231952788307172,-0.3823142165271753,-0.04708691012811009,
      -0.3555183995893311,-0.02217694060430831,0.3149362814931866,
      0.006504983859799645,0.3417324928044827,-0.2502532651867688,
      -0.3600400745299253,-0.4319406005716204,0.01921174674137865,
      -0.05758606025521316,0.3265627679363224,-0.9756765814639009,
      -0.988104613901425,-0.8528232434185344,-0.9168708930751739,
      -0.854235225343551,-0.6463082287051728,-0.7103560109842373,
      -0.5750746905699001,0.3823157740702657,0.0470881327099307,
      0.3873829237054168,-0.6201479767677171,-0.8717667933239607,
      -0.3417327749428254,-0.6485010946606566,-0.8449702640289916,
      -0.3149361858566655,-0.5665549223826651,0.854235631268382,
      -0.4569906864326022,-0.1816479322770386,0.1245901828483064,
      -0.5454522916908328,-0.2546618495309371,-0.5505634949474448,
      0.6485010946606566,0.05758689641252078,-0.01921097131735245,
      -0.3265625234832059,-0.3107079508004592,-0.3873820250532686,
      -0.6896660583043277,-0.6617902107749167,-0.6977972953381247,
      -0.9016230372124288,-0.9602286916122059,-0.7970714969272577,
      0.02217737847260294,-0.774471897882643,-0.1778161375899129,
      0.03109142233089637,0.2669187829195654,0.4366423972091846,
      0.1544374088220477,0.4265382311030135,0.6260922126282031,
      0.5136107375104821,0.7494380454402638,0.7580253719184178,
      -0.5980610514568761,-0.4291113361140411,-0.1696203080333493,
      0.07805400320688781,-0.2914862882507431,-0.03496909012717316,
      0.2274959035287995,0.1056298518860434,0.3651211825280409,
      -0.07805410188638827,-0.1281592432697842,-0.1664476243728032,
      0.2310143008479173,0.2106389321639752,0.5518992169467314,
      -0.9161616153729886,-0.9327747060225072,-0.8106886470978539,
      -0.771485918473268,-0.7097589298055721,-0.5273136421678301,
      -0.4881111164997092,-0.3660248731570757,0.5980613131647216,
      0.4291115157626469,0.1696204410073287,0.7499962909367706,
      0.5360957563846484,0.8113900261915854,-0.4366431953633788,
      -0.6867851479477911,-0.8708228680516066,-0.365121777041269,
      -0.6002019152270697,-0.7331976694123558,-0.2274963530521035,
      -0.4115345048612673,0.9161619708326186,0.9327749805257903,
      0.8106887957474585,0.9941684586829005,0.9531387910864955,
      0.9334760373935599,0.1778171059718252,-0.03109104889813258,
      -0.266919222145987,0.3660256667153609,0.1422988506889488,
      -0.1056306282458797,0.5273142067726671,0.2914860445503094,
      0.6867847261588661,0.8708230332223915,0.8854287054759334,
      0.1281594836202007,0.1664482855753907,0.4115339986401247,
      0.4916319868779721,0.4881116080953447,0.7331975726799166,
      0.7714862700692517,-0.1422987785572308,-0.2310136280273065,
      -0.5360950867028047,-0.749995864312011,-0.551898166785698,
      -0.8113895155772442,-0.7580246814516086,-0.4916319868779721,
      -0.9531387450388693,-0.9941684373181742,-0.9334760519103793,
      0.03496867572798409,-0.885428988462601,-0.6260926899993671,
      -0.7494380619781833,0.7097593165624652,-0.4265380492676017,
      -0.154436271103469,-0.513609764630649,0.6002015742230613,
      -0.2106379566061765,-0.1282040992012935,0.1506070646973712,
      0.448827218017063,0.6488215902264565,-0.232615798766268,
      0.07170206110196441,0.3638246180723771,-0.3176184514965804,
      -0.01939827024983318,-0.3496821715684524,0.6714420577705416,
      0.8777656242085763,0.9824386073167399,0.9441720387917811,
      0.7644107780855932,0.9498618317294465,0.9737573075945113,
      0.7557291769626011,0.8604026946131842,-0.9441722324566724,
      -0.7739455965390631,-0.4607792790679794,-0.8589482430980673,
      -0.5965101770495287,-0.6307845637432553,-0.3130828685601688,
      0.01737679733452725,0.3952347219739472,-0.2895250242862085,
      0.09654302677976184,0.4661908578614318,-0.2185691140597737,
      0.1592889611288683,-0.671442351589465,-0.8777656078901646,
      -0.982438609905361,-0.649602216294724,-0.8243817913354027,
      -0.5261115978506249,-0.6488222902967828,-0.6181925269683732,
      -0.4893366483035592,-0.8604026705465684,-0.7995410210009178,
      -0.6026908960383706,-0.9737572051250643,-0.8449013283633929,
      0.3130823317650696,-0.01737758434298078,-0.3952353653550799,
      0.1061130743462423,-0.2970021727032599,-0.1482539922220807,
      0.1282040764044601,-0.1506074987015245,-0.4488280788164578,
      -0.159288972427813,-0.5000332157715695,-0.7557300072673856,
      -0.4661910177108468,-0.7644115292814133,0.6181920323103048,
      0.4893362367199527,0.2568082513095248,0.7739454978673308,
      0.4607793683387657,0.8449014608184471,0.5811577408091089,
      0.2185687785328751,0.6026910636921159,0.2895245326714229,
      0.5000325428335932,0.8589483198537152,0.8243822455102904,
      0.6496025629973337,0.6307853921801013,0.5261125320735169,
      0.3496833143594963,-0.5811577408091089,0.2970021583546204,
      -0.1061134148983228,0.1482544392795429,-0.9498619694604108,
      -0.2568086311939767,-0.3638250562458945,0.01939866440958929,
      -0.0965434988571485,-0.07170210104888378,0.2326162031595532,
      0.3176193294845107,0.7995412648414353,0.5965106290979496 };

    auto      pointshell
        = new vnl_matrix_fixed<double, 3, 92>(coords);

    return pointshell;
  }
};


template<> class PointShell<162, vnl_matrix_fixed<double, 3, 162> >
{
public:
  static vnl_matrix_fixed<double, 3, 162>* DistributePointShell()
  {
    double coords[3*162] = {
      0.9756767549555488,0.9968737649581846,0.9397937974509784,
      0.7989268505692991,0.6231965669156312,0.9764792677814802,
      0.9657129396229355,0.8567905682573015,0.679558714000995,
      0.8971014254962764,0.8343457228904876,0.6899186612252972,
      0.7377425905504456,0.6387690340353133,0.5505632701289798,
      0.4375940376503738,0.2787444080680243,0.06907711216819172,
      -0.1467487371786262,-0.3200730131503601,0.5436064763029955,
      0.3641181930198064,0.1299861910861632,-0.09463328683104604,
      0.6235168560971455,0.421472539809791,0.1781713819192711,
      0.6478378002177454,0.4350913076613168,0.3200724178002418,
      0.543583002477908,0.7616224484252742,0.9117594243665534,
      0.4242146396073646,0.6694499319211135,0.8720427891122669,
      0.5117469206751248,0.7406754453578036,0.5536544075493044,
      0.2502538949373057,0.3336803609057232,0.4043068999720243,
      0.438887490788423,0.4847692553550146,0.5913899314506929,
      0.6492812914734455,0.6425800233106177,0.7205836671095894,
      0.8155579470101821,0.8306996157029517,0.8921547074343706,
      0.9447583574027534,-0.4375932291383154,-0.278743955138478,
      -0.06907715072573903,0.1467483093137361,-0.2140239599986307,
      -0.001406336898152031,0.2327251986251803,0.06640211724156461,
      0.3039508137298482,0.3409081262151013,-0.6231952788307174,
      -0.4546947222788382,-0.2192092047974508,0.03581973661208818,
      -0.4350905112236564,-0.2089153242611869,0.0609953219830862,
      0.3006812043050254,-0.1781713502362079,0.08257031568712175,
      0.3352293490184233,0.09463277722176672,0.3398901915912972,
      -0.2502532651867688,-0.3336796480610522,-0.4043060780204903,
      -0.4388866188864613,-0.05621340666038893,-0.1197552832780665,
      -0.1776465335448071,0.1765178736510398,0.1277105869872963,
      0.3935116760630417,-0.975676581463901,-0.9968737272214637,
      -0.9397935597648752,-0.7989260596435309,-0.9447580743829742,
      -0.9308031701048347,-0.8218804228775519,-0.6478366382006043,
      -0.8306992108488227,-0.7645261844287355,-0.6235158879179115,
      -0.6425794288509028,-0.543605629081954,0.4546962564744617,
      0.21921067448898,-0.03581866880653414,0.4779576265881325,
      0.2080468003445843,0.4555126388719097,-0.543583197615373,
      -0.7616223457476226,-0.9117592443356049,-0.3398905505309818,
      -0.5766491384642831,-0.7792417673233771,-0.8921544952781482,
      -0.3352294641675141,-0.5550737397870624,-0.7205834943870382,
      -0.3006809838192767,-0.4847689570296102,0.8218810203164033,
      0.9308033931946534,0.7645267056644409,-0.4242149221746557,
      -0.2327253235991419,0.00140664545831139,0.2140247669633664,
      -0.5117470172320778,-0.3039507303469009,-0.06640153973221416,
      -0.5536545418185335,-0.3409079709201472,-0.5505634949474448,
      0.7792418677670352,0.5766490604842006,0.5550737528785307,
      0.1776474313398276,0.1197561650892531,0.05621421006516834,
      -0.1277100030888262,-0.176517304179968,-0.3935115910448358,
      -0.1299862309792015,-0.3641176806190365,-0.4214718858755578,
      -0.2080457563809749,-0.477956563666828,-0.6795578679273953,
      -0.4555121490743896,-0.6899183532182501,-0.6387691005409376,
      -0.6492806993353901,-0.5913894852863821,-0.8155575864896008,
      -0.8567903368462096,-0.9657130735429578,-0.9764792804081987,
      -0.834345897568434,-0.8971016370404784,-0.7377428728490714,
      -0.06099477310985241,0.2089161290281741,-0.08257033948198818,
      -0.8720426859708773,-0.6694498734011487,-0.740675476302381,
      -0.1778161375899129,-0.02718233037986957,0.1521341729788544,
      0.3178872248051531,0.4366423972091847,0.06305939500089614,
      0.2592762165236786,0.4491542846588709,0.5806638189836385,
      0.3410385404813853,0.5484671467389891,0.7022080829625521,
      0.5886123721388867,0.761147081998915,0.7580253719184177,
      -0.5980610514568759,-0.4818747107897384,-0.3056524524744078,
      -0.102179801407443,0.07805400320688782,-0.3811861908796716,
      -0.2105722545208477,-0.001641189844283069,0.1883562467232593,
      -0.09487972868743588,0.1091686599118738,0.3025309778153461,
      0.1998858588754617,0.3897335832229792,-0.07805410188638827,
      -0.1156801299839266,-0.1503967869358663,-0.1717048170733874,
      0.1470965334761953,0.1310545667952879,0.100226360425432,
      0.3996773221825281,0.3894173756029716,0.6166247865338805,
      -0.9161616153729887,-0.93968222548459,-0.8900379255247358,
      -0.7610423024299814,-0.8216809647863927,-0.8059784419089511,
      -0.7076799521032789,-0.5537209611836944,-0.6430242175142256,
      -0.577817181345379,-0.4560492515462431,-0.4070385900414059,
      -0.3177184789930138,0.5980613131647216,0.4818748986235024,
      0.3056526230838691,0.1021798945147061,0.7166389355513205,
      0.5797448495418049,0.3708138719979267,0.797087559189558,
      0.6291765311790336,0.8064720598658887,-0.4366431953633789,
      -0.6250679008877252,-0.7951587738972527,-0.894357332660672,
      -0.3897342152274581,-0.5770900179397438,-0.7252694054566711,
      -0.7936687210308288,-0.3025315118038733,-0.4662800296194353,
      -0.5843857415998933,-0.1883566309643827,-0.3230016809603438,
      0.9161619708326186,0.9396825386256142,0.8900381161951118,
      0.7610424573388179,0.98459889389352,0.985272860446651,
      0.8869743499570832,0.9840623116657442,0.9364058649128151,
      0.8957919700911449,0.1778171059718252,0.02718289859111821,
      -0.1521342296531137,-0.3178878068603045,0.31771932061924,
      0.1597779786826127,-0.03010074909289372,-0.1998866892639353,
      0.4560499397602033,0.2896405921772683,0.09487917246041,
      0.5537214556082512,0.3811861035860583,0.6250673427588886,
      0.7951586652134927,0.8943575917970892,0.7872122199392447,
      0.9353918603084618,0.8865248509843126,0.1156802730367007,
      0.1503972383762518,0.1717055726654157,0.3230012075937738,
      0.3862979863651042,0.4171265008378126,0.4070391829394841,
      0.584385354546882,0.6452876111171104,0.6430245879023526,
      0.7936687790069655,0.8216813253645509,0.03010042800399446,
      -0.159777601870924,-0.2896404384341761,-0.1470960585711635,
      -0.3708132697350703,-0.5797443521912047,-0.7166385361913998,
      -0.399676350745083,-0.6291757405426562,-0.7970870947583816,
      -0.6166237809711579,-0.8064715143730887,-0.7580246814516085,
      -0.417126245587897,-0.3862981411371962,-0.6452875761129986,
      -0.8869742038831786,-0.9852727972566903,-0.9845987728990332,
      -0.9364058061885231,-0.9840624137666333,-0.8957918541675716,
      0.001640881330767951,0.2105720065923513,-0.109169268647464,
      -0.9353919526972201,-0.7872126141915534,-0.5806643774075225,
      -0.8865251005609038,-0.7022083677250226,-0.7611469342493191,
      0.707680317140174,0.8059787882713181,0.577817672809353,
      -0.4491544873380491,-0.2592757174096171,-0.063058272648828,
      -0.5484668466671915,-0.3410374453295617,-0.5886114742048517,
      0.7252693007743299,0.5770896067011005,0.4662795459560942,
      -0.1002253860034311,-0.1310538163159151,-0.3894163023521484,
      -0.1282040992012935,0.07418771901881034,0.3060111300013609,
      0.5105522497698926,0.6488215902264565,-0.2061837828129686,
      0.01320461245153456,0.2533186742384155,0.4483631157317138,
      -0.2808945465452747,-0.05523589088936585,0.1758296024995542,
      -0.3305621657917769,-0.1123798946526014,-0.3496821715684524,
      0.6714420577705413,0.8307215647087252,0.9496341036793501,
      0.983882155708098,0.9441720387917811,0.7477894668983293,
      0.9072360592139173,0.9915144459980433,0.9775308002021622,
      0.7760312926982584,0.9002461340540874,0.9363385958760152,
      0.7350862725086435,0.8116669810370153,-0.9441722324566724,
      -0.8313456723553714,-0.6303269409870232,-0.3731115220313701,
      -0.8935348618728204,-0.7312055040636086,-0.4790574606800003,
      -0.7605084662990487,-0.5474979380988695,-0.5596790773588694,
      -0.3130828685601688,-0.07519795113003093,0.2106381298878163,
      0.4776948653078811,-0.299731815412124,-0.02562616156907596,
      0.2786088475509629,0.5296074119458033,-0.2593820240207756,
      0.03150460300683353,0.3193074202631062,-0.1959070295313489,
      0.08053951964108649,-0.671442351589465,-0.8307216077305004,
      -0.9496340459616871,-0.9838822098556939,-0.6637939293172054,
      -0.8147968652653733,-0.8990751104656249,-0.6002017842451572,
      -0.7153675946298389,-0.4830988161196208,-0.648822290296783,
      -0.6344626299567083,-0.5653935344751363,-0.4459169294668335,
      -0.8116671044980045,-0.7895071237697285,-0.6857578728711079,
      -0.5288485365725882,-0.9363384293729856,-0.8807752703984302,
      -0.7389956620788968,-0.9775307754986758,-0.8832579304805722,
      0.3130823317650696,0.07519720122433592,-0.2106389019045293,
      -0.4776954195825031,0.1655447705464922,-0.1220494268493898,
      -0.4262841911656161,-0.02151295053653907,-0.3268548028226648,
      -0.2066526242811376,0.1282040764044601,-0.07418801789892716,
      -0.3060118317848469,-0.51055312502423,-0.08053951945115613,
      -0.3287498381001709,-0.5688642328312289,-0.7350870708018485,
      -0.3193074905789999,-0.5758541835274278,-0.7760321386045594,
      -0.5296076162752261,-0.7477901272839138,0.6344620803203709,
      0.5653931175085022,0.4459164955009389,0.389683753785766,
      0.2859346018498913,0.08112906023855893,0.831345524857385,
      0.6303269573377657,0.3731116142445962,0.8832579654623612,
      0.7198955735651289,0.467747528435525,0.1959067638111305,
      0.7389959159190775,0.5248781214125594,0.2593815856416469,
      0.5288485749239231,0.2997313094215869,0.5688633866559212,
      0.3287493895928871,0.5758535688432742,0.8935348059013593,
      0.8990753265127792,0.8147972186077681,0.6637941002839092,
      0.7605089118538748,0.7153683254339585,0.6002024649156598,
      0.5596800524090022,0.4830998363360157,0.3496833143594963,
      -0.4677475887278177,-0.7198955529773995,-0.5248781506021705,
      0.4262841209617862,0.1220490717256491,-0.165545217365127,
      0.326855199205205,0.02151295276860444,0.2066532886754674,
      -0.9915144412787341,-0.9072363224101446,-0.9002463663906001,
      -0.2859350592025221,-0.3896842610441149,-0.4483636748755489,
      -0.0811290830782055,-0.175829673802738,0.1123805173372347,
      -0.2786093002807504,0.02562556437243592,-0.03150492197167008,
      -0.253319097564816,-0.01320461850800903,0.2061840662709727,
      0.05523623192861417,0.2808952005684321,0.3305631346577989,
      0.6857580324047194,0.78950721141067,0.8807755242176112,
      0.4790578522946484,0.7312056921497674,0.5474986596025796 };

    auto      pointshell
        = new vnl_matrix_fixed<double, 3, 162>(coords);

    return pointshell;
  }
};


template<> class PointShell<252, vnl_matrix_fixed<double, 3, 252> >
{
public:
  static vnl_matrix_fixed<double, 3, 252>* DistributePointShell()
  {
    double coords[3*252] = {
      0.9756767549555488,0.9977154378498742,0.9738192119472443,
      0.8915721200771204,0.7646073555341725,0.6231965669156312,
      0.9817040172417226,0.9870396762453547,0.9325589150767597,
      0.8173592116492303,0.6708930871960926,0.9399233672993689,
      0.9144882783890762,0.8267930935417315,0.6931818659696647,
      0.8407280774774689,0.782394344826989,0.6762337155773353,
      0.7005607434301688,0.6228579759074076,0.5505632701289798,
      0.4375940376503738,0.3153040621970065,0.1569517536476641,
      -0.01984099037382634,-0.1857690950088067,-0.3200730131503601,
      0.5232435944036425,0.3889403678268736,0.2135250052622625,
      0.02420694871807206,-0.1448539951504302,0.5971534158422009,
      0.4482053228282282,0.2597018771197477,0.06677517278138323,
      0.6404616222418184,0.4782876117785159,0.2868761951248767,
      0.6459894362878276,0.4789651252338281,0.3200724178002418,
      0.4973180497018747,0.6793811951363423,0.8323587928990375,
      0.9308933612987835,0.4036036036586492,0.5984781165037405,
      0.7817280923310203,0.9140795130247613,0.4809905907165384,
      0.6759621154318279,0.8390728924802671,0.5347729120192694,
      0.7094340284155564,0.5560356639783846,0.2502538949373057,
      0.3171352000240629,0.3793963897789465,0.4231100429674418,
      0.4410301813437042,0.4357529867703999,0.5208717223808415,
      0.5850086433327374,0.611055499882272,0.6009463532173235,
      0.6305067000562991,0.7188806066405239,0.7654898954879897,
      0.7616477696596397,0.7997756996573342,0.8700831379830764,
      0.8872031228985237,0.9155019734809123,0.9568003701205341,
      -0.4375932291383153,-0.3153035222278598,-0.1569515927579475,
      0.0198407706589918,0.1857686171195431,-0.2644927501381796,
      -0.1064219080255857,0.07849995612144045,0.2583107784678281,
      -0.04938676750055992,0.1358448755096817,0.3243479900672576,
      0.1811879481039926,0.3692668145365748,0.3890115016151001,
      -0.6231952788307174,-0.4943551945928708,-0.319458133528771,
      -0.1156489798772063,0.08328895892415776,-0.4789641985801549,
      -0.3127252940830145,-0.1059392282183739,0.1077444781964869,
      0.2912280153186658,-0.2868758523956744,-0.08856892011805101,
      0.1287405357080231,0.3245517154572714,-0.06677541204276306,
      0.1413542883070481,0.3408430926744944,0.1448534358763926,
      0.3374016489097037,-0.2502532651867688,-0.3171345072414974,
      -0.3793956104585266,-0.4231091882680272,-0.4410293135613324,
      -0.09929959410007272,-0.1535127609134815,-0.2052877394623771,
      -0.2436963810571767,0.08175409117371149,0.04056025153798869,
      -0.006048944565669369,0.2686152102237028,0.2319923070602857,
      0.430309819720559,-0.975676581463901,-0.9977153903038788,
      -0.9738191090293654,-0.8915716840571059,-0.7646064477130079,
      -0.9568001079664734,-0.9598482725023617,-0.9044523389503778,
      -0.7901672201648241,-0.6459882395962464,-0.8872027729137049,
      -0.8582754834679532,-0.7705800268610806,-0.6404605781008121,
      -0.7616472974254324,-0.7008201753656432,-0.5971525097007422,
      -0.6009457148226922,-0.5232427588825813,0.4943566966479628,
      0.3194596781650836,0.1156503154178581,-0.0832879858164388,
      0.5222841738261358,0.3225497922064885,0.1018140973507329,
      0.5217885230992481,0.3044789836562512,0.4873191346491355,
      -0.4973183240635209,-0.6793811856410323,-0.8323586364840968,
      -0.9308931819742911,-0.3374020539278631,-0.5261951664998159,
      -0.7070125356849136,-0.8417962075837926,-0.9155017573317124,
      -0.3408433114184408,-0.5265312606271311,-0.6896418460594331,
      -0.7997755164970677,-0.3245517106425898,-0.4925847482169691,
      -0.6305065080228541,-0.2912277152063287,-0.4357526334612896,
      0.7901679726328494,0.9044526665335126,0.9598484396937114,
      0.7705806468939737,0.858275831469383,0.7008207681995118,
      -0.4036039458806759,-0.2583110138480089,-0.0784999126587471,
      0.1064223584250461,0.264493571710179,-0.4809907334514471,
      -0.3243480295764106,-0.1358446002697818,0.04938746901646566,
      -0.5347730026038946,-0.3692667658371347,-0.1811875286592425,
      -0.5560358190148772,-0.3890114324926668,-0.5505634949474449,
      0.8417963565884857,0.7070125813068046,0.5261950179989611,
      0.6896418985458221,0.5265311900255359,0.4925848265160583,
      0.2436972866599269,0.2052886581368649,0.153513629451971,
      0.09930039009433847,0.006049691633511915,-0.04055950638179381,
      -0.08175337578691833,-0.2319919155781195,-0.2686148310916902,
      -0.430309819678344,-0.02420720081803753,-0.2135248270679241,
      -0.3889397838050994,-0.2597016312374675,-0.4482046405142344,
      -0.4782867918076852,-0.1018130528605821,-0.322548598821141,
      -0.5222830294256716,-0.6708921376896406,-0.304478224282928,
      -0.5217878437313506,-0.6931813485878851,-0.4873188675145023,
      -0.6762335873429084,-0.6228580878699612,-0.6110548409057,
      -0.5850080622199078,-0.5208712693637837,-0.7654894328832393,
      -0.7188802647693375,-0.8700828159137221,-0.8173587433845655,
      -0.9325588839421305,-0.9870397834787261,-0.9817039872478999,
      -0.8267930492778305,-0.9144884914916022,-0.9399235077793813,
      -0.7823945479956939,-0.8407283372889187,-0.7005610213599369,
      -0.1077438933887955,0.1059400956623477,0.3127262866621893,
      -0.1287403742204129,0.08856921814263634,-0.1413545191115968,
      -0.9140794058749131,-0.7817279594934516,-0.5984781448346268,
      -0.8390728949381593,-0.6759620794963979,-0.709434131000089,
      -0.1778161375899129,-0.06053925384414331,0.07929679392711581,
      0.222673458561735,0.3458247516791153,0.4366423972091846,
      0.01030826616734189,0.1591522280204451,0.3173816763430465,
      0.4549463955350546,0.5521270265729551,0.2292788658415479,
      0.3973400932411465,0.5502139834879405,0.6594089221868847,
      0.4476465561008348,0.6096570464011057,0.7343998566036512,
      0.629214796874201,0.7646693979379596,0.7580253719184178,
      -0.5980610514568761,-0.5101530988159087,-0.382225667160838,
      -0.2244621267538426,-0.06301328229424107,0.07805400320688782,
      -0.4311039309963852,-0.3079662136138592,-0.1501157132113724,
      0.01750888497279251,0.1650825345160538,-0.2148810450151756,
      -0.06090095222676627,0.1073128739652992,0.2584097661066967,
      0.02655484252908358,0.1901297170957776,0.3420822257932489,
      0.2531835106264871,0.4022303494272352,-0.07805410188638827,
      -0.1080255529483224,-0.1376217050758367,-0.1609000070073124,
      -0.1740018618448228,0.09827676798573926,0.083291898217249,
      0.06127443921955168,0.03526739273256396,0.2991139104294396,
      0.2941068360088736,0.2692865316145088,0.4942032775296958,
      0.4857723178878524,0.6512069539966677,-0.9161616153729886,
      -0.9396953110011561,-0.9204280785344878,-0.8462030522374957,
      -0.7293237120999879,-0.8470541513588044,-0.8482966176587544,
      -0.7977006542517769,-0.6951661565374421,-0.566558592627622,
      -0.7243096319272092,-0.6931460376496088,-0.6140043047773551,
      -0.5016343691560573,-0.5520254073275178,-0.4928644880867128,
      -0.403575153350467,-0.3587591578566765,-0.2886351685087218,
      0.5980613131647216,0.5101532951859686,0.382225843595672,
      0.2244622808787926,0.06301334452030186,0.6944632949786616,
      0.5955168212825119,0.4473425940100297,0.2700417838303327,
      0.7724043956082883,0.6553545192922715,0.4871408620353512,
      0.8097301284690857,0.6725220182496192,0.8002534097038426,
      -0.4366431953633789,-0.5869882376922511,-0.7332080507197046,
      -0.8450980113065225,-0.9041113586460733,-0.4022310083998925,
      -0.554596445154436,-0.6925605687496104,-0.7854318984598006,
      -0.8250621271173465,-0.3420827953668352,-0.4840440064641756,
      -0.6033456975789954,-0.6777531805937266,-0.2584102557043402,
      -0.3819753792546441,-0.4821906665520286,-0.1650828712784331,
      -0.270790845781693,0.9161619708326184,0.9396956424389374,
      0.9204283182965946,0.8462032095340455,0.7293238793541417,
      0.9749588444840027,0.9879501207294071,0.942053498973333,
      0.8348196077814718,0.9950795014807369,0.9818515654328379,
      0.9027098746674149,0.9581801446138297,0.9118246030313639,
      0.8703772282258925,0.1778171059718252,0.06053992567271226,
      -0.07929659020903117,-0.2226737578340799,-0.345825401239635,
      0.2886360377097776,0.1672516508448342,0.02000533874392893,
      -0.1285435155191929,-0.2531843553864728,0.403575906447316,
      0.2774342678683828,0.1245598363284875,-0.02655554762561945,
      0.5016349858535857,0.3695530582277636,0.2148806720954671,
      0.5665590425344393,0.431103930292903,0.5869876102086139,
      0.7332077514676827,0.845098078457225,0.9041116580482536,
      0.7182616282077119,0.8617334421407644,0.9490975365686583,
      0.8223898048944452,0.9416915744235097,0.8729720010540123,
      0.1080256414522809,0.1376220280275969,0.1609005865750696,
      0.1740026689030255,0.2707904196202965,0.3196768235430837,
      0.3552546724685221,0.3677018240803483,0.3587598208776521,
      0.4821901792282771,0.5389508449256169,0.5637713635689835,
      0.5520258363563475,0.6777529577987501,0.7231337276202411,
      0.724309982145211,0.8250622687013296,0.8470545173149734,
      0.1285429999155006,-0.02000532948058562,-0.1672511147059996,
      -0.1245600244829796,-0.2774338902981233,-0.3695528631494325,
      -0.09827641615811868,-0.2700412859530667,-0.4473420975374328,
      -0.5955164071695848,-0.6944629164413806,-0.2991130971968019,
      -0.4871400501186961,-0.6553538941234454,-0.7724039524031648,
      -0.4942022299541438,-0.6725212074710563,-0.8097296395344389,
      -0.6512059956089504,-0.8002528392148971,-0.7580246814516085,
      -0.3677014077761052,-0.3552545716101517,-0.3196770257819652,
      -0.5637712030900536,-0.5389510214534028,-0.7231336172569296,
      -0.8348194119106425,-0.9420533966954356,-0.9879499956150448,
      -0.9749586635216289,-0.9027097279159257,-0.9818515951566739,
      -0.9950795477220543,-0.9118244750171576,-0.9581802235578871,
      -0.8703770126934449,-0.0175091339170676,0.1501155140512474,
      0.3079660822386824,-0.1073133727582037,0.06090046334304851,
      -0.1901304002696938,-0.9490974969653682,-0.8617336589899791,
      -0.7182621005240754,-0.5521276321758419,-0.941691783045487,
      -0.8223901593137167,-0.6594093292610237,-0.872972144171723,
      -0.7343999908188845,-0.7646691446910742,0.6951665021597787,
      0.7977009700656229,0.8482969664746548,0.6140047811934269,
      0.6931464276818936,0.4928650597255946,-0.4549467775084718,
      -0.3173815862988101,-0.1591515620353438,-0.01030716362341688,
      -0.5502140363721867,-0.3973395475484636,-0.2292777334167206,
      -0.609656670182737,-0.4476455277450017,-0.6292139442700462,
      0.7854319364049284,0.6925603649758249,0.5545959620739339,
      0.6033453603619342,0.4840435291285519,0.3819748711371402,
      -0.03526641653115874,-0.06127364342066123,-0.0832913202753871,
      -0.2692854573778917,-0.2941058574917593,-0.4857712605383084,
      -0.1282040992012934,0.02998172476739921,0.2130449739264662,
      0.394354771181159,0.5438573645627299,0.6488215902264565,
      -0.1901340637026595,-0.02057293935230464,0.1720544722828635,
      0.3534794142829396,0.4950335464190314,-0.252933321812349,
      -0.07636778766011496,0.1169519253626288,0.2909961752861106,
      -0.304612640171253,-0.1271903099934383,0.0580070042064605,
      -0.3366056805211806,-0.1653138037361849,-0.3496821715684524,
      0.6714420577705413,0.8002044514563711,0.9106424580428781,
      0.9742808059046055,0.9805708386415104,0.9441720387917811,
      0.7350956003099328,0.8682639008659977,0.9653353535299492,
      0.9995536316680411,0.9755844796257857,0.7728091190204586,
      0.8918537226509272,0.9597077065970592,0.9637247890765801,
      0.767530944505584,0.857374860312736,0.8948082473172733,
      0.7201359303293944,0.7802583897718675,-0.9441722324566724,
      -0.8608166107545396,-0.7207644955095487,-0.5303678229575245,
      -0.3211867088850157,-0.9096404828216634,-0.7967975927156801,
      -0.620601831095295,-0.4039985827676406,-0.8241231220089414,
      -0.6757043639889994,-0.4726959329165278,-0.6854057579633669,
      -0.5106159168102177,-0.5164821811548767,-0.3130828685601688,
      -0.128054626578418,0.09418349997750548,0.3239109228229815,
      0.523048087763098,-0.3043330399192493,-0.09531787499064681,
      0.146419102006115,0.3786227553496849,0.5638039035645359,
      -0.2789925774668332,-0.05252850546893189,0.1924160430438771,
      0.4101897544477446,-0.2358533016570041,-0.006318969895916147,
      0.2236016867495729,-0.1820659309330632,0.03496843200875603,
      -0.6714423515894649,-0.8002045390283683,-0.9106424117173109,
      -0.9742807748705438,-0.9805709251787237,-0.6691519386893557,
      -0.79626257796142,-0.8909109722488041,-0.9275521423149625,
      -0.6332080201962388,-0.7430051304270751,-0.8108589038018792,
      -0.5581290590099237,-0.6413705283620924,-0.4563600901355626,
      -0.648822290296783,-0.6411378559950974,-0.600293640880965,
      -0.5219527418637842,-0.4191009430775375,-0.7802586188950914,
      -0.7711197529973893,-0.713538182957094,-0.6094980396194888,
      -0.4842093859996422,-0.8948081394501657,-0.8705497953564927,
      -0.7870195954857328,-0.6597854273844109,-0.9637246412193355,
      -0.9132982945459158,-0.8070428410352181,-0.975584505681221,
      -0.9015722073987464,0.3130823317650696,0.1280539101236855,
      -0.09418429615654819,-0.3239116283455282,-0.523048586255087,
      0.1989845274738189,-0.01970764286946627,-0.2653151882168217,
      -0.4936479477555078,0.05597369301027853,-0.1852629737758222,
      -0.4302072668465533,-0.09867461327158224,-0.3387557568094476,
      -0.2393260112020272,0.1282040764044601,-0.0299819504088954,
      -0.2130455201840074,-0.3943555879655132,-0.5438582278251758,
      -0.03496843048443223,-0.2252069693946209,-0.4261053308619027,
      -0.5992598174372795,-0.720136706807033,-0.2236017161594696,
      -0.4317330442416767,-0.6250530132529536,-0.7675317913865697,
      -0.4101898771205939,-0.6101488498350025,-0.7728099228904255,
      -0.5638041319099237,-0.7350961954485662,0.6411372723067146,
      0.6002931843810706,0.5219523372221417,0.4191004905746099,
      0.4596949806286311,0.3916338931244087,0.2980734064957148,
      0.226741584116328,0.1432114770939381,-0.02097489882147943,
      0.8608164411414747,0.7207644427956729,0.5303678926086439,
      0.3211867913977836,0.9015721838250796,0.7879881821713033,
      0.6114960278478284,0.3951892122332402,0.1820657113417612,
      0.8070430398170311,0.6574928275930984,0.4544842943197335,
      0.2358529185889448,0.6597856586149884,0.4841878538357612,
      0.2789921022280572,0.4842093252521232,0.3043325272261384,
      0.5992589358516202,0.4261046359672609,0.2252066549797059,
      0.6250522113657903,0.4317325950511361,0.6101482870567641,
      0.9096403689902206,0.9275522217134882,0.8909112253661301,
      0.796262827475376,0.6691520068054228,0.8241233338640371,
      0.810859375773786,0.7430057321681839,0.6332085061147845,
      0.6854064426268304,0.6413714065577412,0.5581299045184589,
      0.5164832226272315,0.4563611494403301,0.3496833143594963,
      -0.3951892821849063,-0.6114960336943951,-0.787988199289983,
      -0.4544844137443082,-0.657492739431111,-0.484187939006181,
      0.4936478319326018,0.2653148405479006,0.01970714938077021,
      -0.1989850169013517,0.4302075642722875,0.1852629793843341,
      -0.0559739158243807,0.3387563694841473,0.09867487876932232,
      0.2393267951217032,-0.999553621201999,-0.9653354239158236,
      -0.8682642090770526,-0.9597077173597477,-0.8918540989344099,
      -0.8573751662344773,-0.2980738893651726,-0.3916343988495664,
      -0.4596955428592778,-0.4950341577852201,-0.1432117197792371,
      -0.2267418620329016,-0.2909964852939082,0.02097514873862574,
      -0.05800679989935065,0.1653145532988453,-0.3786231842883476,
      -0.1464197032303796,0.09531724619007391,-0.1924163631703616,
      0.05252803743712917,0.006318730357784829,-0.3534800054422614,
      -0.1720548071373146,0.02057294660420643,0.190134278339324,
      -0.1169519894866824,0.07636807502743861,0.2529338262925594,
      0.1271908635410245,0.3046134343217798,0.3366066958443542,
      0.6094980941008995,0.7135382519498201,0.7711196978950583,
      0.7870198804193677,0.8705500304441893,0.9132984713369965,
      0.403998910419839,0.62060207699311,0.7967976318501995,
      0.4726965405256068,0.6757048258462731,0.5106167801856609 };

    auto      pointshell
        = new vnl_matrix_fixed<double, 3, 252>(coords);

    return pointshell;
  }

};

template<> class PointShell<362, vnl_matrix_fixed<double, 3, 362> >
{
public:
  static vnl_matrix_fixed<double, 3, 362>* DistributePointShell()
  {
    double coords[3*362] = {
      0.9756767549555487,0.9966345256373878,0.9881046683022421,
      0.9397937974509782,0.8528238339140162,0.7412214378486838,
      0.6231965669156312,0.9834769262997924,0.9946239689739521,
      0.9675363014408187,0.8953435882073797,0.7874413200090197,
      0.6642104349774659,0.9602286065935876,0.9526601679743949,
      0.9016229543668698,0.8082746868815032,0.68966672282401,
      0.8971014254962764,0.865591182877164,0.7933984257386667,
      0.6899186612252972,0.7970712198937139,0.7447486127431006,
      0.6617902410567804,0.6754324093484999,0.6115790799976598,
      0.5505632701289798,0.4375940376503739,0.3383775196739809,
      0.2130515669413184,0.06907711216819173,-0.07773888316805401,
      -0.2106409255960856,-0.3200730131503601,0.5092547181142852,
      0.4023388431564537,0.265404014577164,0.1102235177567159,
      -0.04300669133613336,-0.1770183828831563,0.575075565563677,
      0.4585983658816588,0.3107083252171692,0.1482373789939068,
      -0.006505365777837463,0.6235168560971455,0.4966123011343984,
      0.3414318272800055,0.178171381919271,0.6463093264667951,
      0.5114333072914884,0.3555189519163455,0.6437455132939696,
      0.5064909697503934,0.3200724178002418,0.4666605048128079,
      0.6201479032959075,0.7616224484252742,0.8717669667532965,
      0.9417225609211535,0.3896493566867455,0.5498790773446007,
      0.7090922337204692,0.8433688828589941,0.9352336066902967,
      0.4569904905821254,0.6220231833138386,0.7744719807282636,
      0.8905764752647638,0.5117469206751248,0.6692307929647209,
      0.8035074327650433,0.5454521871592555,0.6853580997057199,
      0.5566669166856135,0.2502538949373057,0.3059448167491349,
      0.3600408236839538,0.4043068999720242,0.4319414682718342,
      0.441694708443188,0.4034206272561209,0.4730271557866856,
      0.532848297654098,0.5712181400125941,0.5831430197408345,
      0.573108039637807,0.5665551499276381,0.6430548267994817,
      0.697797754038698,0.7197945422057391,0.7103565350090897,
      0.7205836671095895,0.7916313033238946,0.830001177910469,
      0.8306996157029517,0.8449704612165639,0.8994200667307424,
      0.9168712118355732,0.9290744770112713,0.9630120083465428,
      -0.4375932291383153,-0.3383769257894378,-0.213051272338593,
      -0.069077150725739,0.07773856031405026,0.210640419502912,
      -0.2966224671809508,-0.1723655417522876,-0.02500201334267453,
      0.1301781584808292,0.2729791005675229,-0.1245894223683107,
      0.02220568982766716,0.181648017057727,0.3325661017840546,
      0.06640211724156465,0.224593525159963,0.3797737583307059,
      0.2546621526726748,0.4084582460890006,0.4194124862841989,
      -0.6231952788307172,-0.5191377313825043,-0.3823142165271753,
      -0.2192092047974507,-0.04708691012811009,0.1137784201343049,
      -0.5064899646320268,-0.3766634899185586,-0.2148504892229182,
      -0.03595624973804722,0.1367370476543766,0.2846552253706529,
      -0.3555183995893311,-0.2005508680670486,-0.02217694060430831,
      0.1572376520619428,0.3149362814931866,-0.1781713502362079,
      -0.007356907961155389,0.1715373930872541,0.3352293490184234,
      0.006504983859799645,0.1777752170237141,0.3417324928044827,
      0.1770178009419758,0.3352469806136632,-0.2502532651867688,
      -0.305944136422603,-0.3600400745299253,-0.4043060780204903,
      -0.4319406005716204,-0.4416938460694254,-0.1269352612818317,
      -0.1739928009393344,-0.2202529758737042,-0.2586227459701909,
      -0.284108486127532,0.01921174674137865,-0.0178650781598001,
      -0.05758606025521316,-0.0946046496972474,0.1765178736510398,
      0.1461529703905971,0.1077831731751904,0.3265627679363224,
      0.2967152689909496,0.4533499532981318,-0.9756765814639009,
      -0.9966344649740785,-0.988104613901425,-0.9397935597648753,
      -0.8528232434185344,-0.7412204556286736,-0.9630117602017364,
      -0.9724898559386537,-0.9443983719046766,-0.872205391539895,
      -0.7653064930405823,-0.6437442957669286,-0.9168708930751739,
      -0.9063844772960285,-0.854235225343551,-0.7619985358162277,
      -0.6463082287051728,-0.8306992108488227,-0.7961776618828416,
      -0.7239846967363567,-0.6235158879179115,-0.7103560109842373,
      -0.6562122899574033,-0.5750746905699001,-0.57310737178428,
      -0.5092538887995088,0.5191392058051657,0.3823157740702657,
      0.21921067448898,0.0470881327099307,-0.1137775092906953,
      0.5472470006444878,0.3931697400830757,0.2142753542688524,
      0.03384607877300972,0.5571877569557013,0.3873829237054168,
      0.1993990574966068,0.5423114310655514,0.3634170963876792,
      0.5045541233354154,-0.4666608333315146,-0.6201479767677171,
      -0.7616223457476226,-0.8717667933239607,-0.9417223829869568,
      -0.3352474166348553,-0.4910400536375052,-0.6475846838452103,
      -0.7818611891844719,-0.8763941477411082,-0.9290742611527086,
      -0.3417327749428254,-0.4990082491682611,-0.6485010946606566,
      -0.7675612257958812,-0.8449702640289916,-0.3352294641675141,
      -0.4847082676062016,-0.6189847486203324,-0.7205834943870382,
      -0.3149361858566655,-0.4500011545486986,-0.5665549223826651,
      -0.2846548708446658,-0.4034202330037945,0.7653073532861547,
      0.872205859792842,0.9443985741044771,0.9724900050827093,
      0.761999264311147,0.854235631268382,0.9063847463761845,
      0.7239853440497003,0.7961780994118357,0.656212928522132,
      -0.3896497406159066,-0.2729794041072619,-0.1301782703739063,
      0.02500220716165068,0.1723660797382231,0.2966232936614225,
      -0.4569906864326022,-0.3325662253633467,-0.1816479322770386,
      -0.02220527620210722,0.1245901828483064,-0.5117470172320778,
      -0.379773775950072,-0.2245933008307449,-0.0664015397322142,
      -0.5454522916908328,-0.4084582325525932,-0.2546618495309371,
      -0.5566670856360056,-0.4194124724547443,-0.5505634949474448,
      0.8763943198737287,0.7818612997205676,0.6475846717443501,
      0.4910398461057182,0.7675613256844861,0.6485010946606566,
      0.4990081234589473,0.6189848075516286,0.4847082388524531,
      0.450001299655162,0.2841093866952762,0.2586236728290179,
      0.2202538772848093,0.1739936456875063,0.1269360432434686,
      0.09460546886511037,0.05758689641252078,0.01786589383219402,
      -0.01921097131735245,-0.1077825867383485,-0.1461523912078012,
      -0.176517304179968,-0.2967150069434516,-0.3265625234832059,
      -0.4533500040307881,0.04300632519905166,-0.1102235647973928,
      -0.2654036978528873,-0.4023382153059064,-0.148237402516883,
      -0.3107079508004592,-0.4585976694770509,-0.341431363569028,
      -0.4966114955495211,-0.5114323842282982,-0.0338450669354788,
      -0.2142741518944354,-0.3931684703578905,-0.5472458119265531,
      -0.6642094200020999,-0.1993981773021475,-0.3873820250532686,
      -0.5571869441797634,-0.6896660583043277,-0.3634165630969035,
      -0.5423109973324143,-0.6899183532182501,-0.504553984174398,
      -0.6617902107749167,-0.6115792176515682,-0.5831423229868272,
      -0.5712174808640838,-0.5328477301676857,-0.4730266887482617,
      -0.7197940112087894,-0.6977972953381247,-0.643054488612089,
      -0.8300007800598662,-0.7916309980553373,-0.8994197666839766,
      -0.787440689562144,-0.8953433656420341,-0.9675363415293142,
      -0.9946240407011416,-0.9834768689460589,-0.8082744359221968,
      -0.9016230372124288,-0.9526603585193832,-0.9602286916122059,
      -0.7933984756354565,-0.8655914258905276,-0.8971016370404785,
      -0.7447488265146829,-0.7970714969272577,-0.6754326817805886,
      -0.1367364480221615,0.03595711502889169,0.2148515425621942,
      0.3766645834143537,-0.1572373865760062,0.02217737847260294,
      0.2005514111744157,-0.1715374471229988,0.007356909834687842,
      -0.1777755545457471,-0.9352334959045954,-0.8433687324667886,
      -0.7090921450939843,-0.5498791863597017,-0.8905764592918793,
      -0.774471897882643,-0.6220231689320465,-0.8035075122026962,
      -0.6692308066909014,-0.6853582437416091,-0.1778161375899129,
      -0.08196112014272428,0.03109142233089637,0.1521341729788544,
      0.2669187829195654,0.3632859224072412,0.4366423972091846,
      -0.02374119024204515,0.09471244259515242,0.2248569419991813,
      0.350706344617783,0.4558820271112225,0.5328176919569597,
      0.1544374088220477,0.290680604004569,0.4265382311030135,
      0.5423794137964371,0.6260922126282031,0.3410385404813853,
      0.4823536424782295,0.6082030290912097,0.702208082962552,
      0.5136107375104821,0.6447862719396159,0.7494380454402638,
      0.6543855332308085,0.7656972306146125,0.7580253719184178,
      -0.5980610514568761,-0.5275667073977156,-0.4291113361140411,
      -0.3056524524744078,-0.1696203080333493,-0.0376422258299667,
      0.07805400320688781,-0.4626069507186271,-0.367865039277211,
      -0.2460698157210856,-0.1075922772772709,0.02954542570643959,
      0.1497986786591435,-0.2914862882507431,-0.1726261297241735,
      -0.03496909012717316,0.1043290137555473,0.2274959035287995,
      -0.09487972868743588,0.03929513959897087,0.1777727700059916,
      0.3025309778153461,0.1056298518860434,0.2403183339972744,
      0.3651211825280409,0.2871566925727233,0.4096380159360284,
      -0.07805410188638827,-0.1029113333575652,-0.1281592432697842,
      -0.1503967869358663,-0.1664476243728032,-0.1752005995696255,
      0.06662046905832243,0.05250750966401235,0.0344560317246887,
      0.01402339491018058,-0.006131193542750579,0.2310143008479173,
      0.2261291767278549,0.2106389321639752,0.1852638684253743,
      0.399677322182528,0.3973696146518023,0.3769369479489949,
      0.5518992169467314,0.5439427436094942,0.6724578932085956,
      -0.9161616153729886,-0.9381742633287281,-0.9327747060225072,
      -0.8900379255247358,-0.8106886470978539,-0.7076732091095549,
      -0.8620450993411092,-0.8700248897395222,-0.8443283587296551,
      -0.7791770527382307,-0.6830500709829336,-0.5739187050780551,
      -0.771485918473268,-0.7582565243728648,-0.7097589298055721,
      -0.6279538602818827,-0.5273136421678301,-0.6430242175142256,
      -0.6070333936049677,-0.5418820315598687,-0.456049251546243,
      -0.4881111164997092,-0.4360363777707778,-0.3660248731570757,
      -0.327027482762229,-0.2694021154175107,0.5980613131647216,
      0.5275669112241526,0.4291115157626469,0.305652623083869,
      0.1696204410073287,0.03764226488103771,0.6790262210552894,
      0.60193444501539,0.4907540154202648,0.3522765038549025,
      0.2045239960649213,0.7499962909367706,0.6619943399357987,
      0.5360957563846484,0.3850393813839648,0.7970875591895581,
      0.6947571795971019,0.5562797487334057,0.8113900261915854,
      0.6959590446635378,0.7949388873021738,-0.4366431953633788,
      -0.5614655245500564,-0.6867851479477911,-0.7951587738972525,
      -0.8708228680516066,-0.908932652052906,-0.4096386948450996,
      -0.5369984265979351,-0.6595626659529629,-0.757774542842855,
      -0.8188523117857327,-0.8439728473440037,-0.365121777041269,
      -0.4879068833210152,-0.6002019152270697,-0.6843307714263435,
      -0.7331976694123558,-0.3025315118038733,-0.4144630434096209,
      -0.5126750515159538,-0.5843857415998934,-0.2274963530521035,
      -0.3262252144152584,-0.4115345048612673,-0.1497989805512191,
      -0.2366659840339851,0.9161619708326186,0.938174604388516,
      0.9327749805257903,0.890038116195112,0.8106887957474585,
      0.7076733877280226,0.9671526283488294,0.9837044593162763,
      0.9631630917506131,0.8980117463611794,0.7967295682829974,
      0.9941684586829005,0.9959257775132996,0.9531387910864955,
      0.8656231437836883,0.9840623116657442,0.9635372003220445,
      0.8983859076909864,0.9334760373935599,0.8907539689815491,
      0.8525641076255658,0.1778171059718252,0.08196185466921782,
      -0.03109104889813258,-0.1521342296531137,-0.266919222145987,
      -0.363286609556101,0.2694030024488408,0.1709841140913856,
      0.0528882307987244,-0.07296161244603722,-0.1901866620094973,
      -0.2871575400870702,0.3660256667153609,0.2648098022646842,
      0.1422988506889488,0.01311012528268693,-0.1056306282458797,
      0.4560499397602033,0.3508815236279322,0.2250316837343615,
      0.09487917246041001,0.5273142067726671,0.4179974732249477,
      0.2914860445503094,0.5739191244872464,0.4626070035784272,
      0.5614648567136933,0.6867847261588661,0.7951586652134927,
      0.8708230332223915,0.9089329717882446,0.6702235465196067,
      0.7988296386378929,0.8970417421055762,0.9520780267590907,
      0.7664410264477003,0.8854287054759334,0.9628651295567006,
      0.8322645377129818,0.9304765454032002,0.8591275181397264,
      0.1029113872119272,0.1281594836202007,0.1503972383762519,
      0.1664482855753907,0.175201438535584,0.2366655991467124,
      0.2755134432217821,0.308440535504026,0.3288733874144261,
      0.3341527600467577,0.3270281951640686,0.4115339986401247,
      0.4596635416815941,0.4916319868779721,0.5005292304537458,
      0.4881116080953447,0.584385354546882,0.6313194611701424,
      0.6517522812978667,0.6430245879023526,0.7331975726799166,
      0.7681409935232665,0.7714862700692517,0.8439730388551088,
      0.8620454677591289,0.1901860445525563,0.0729613782704526,
      -0.05288800289873952,-0.1709834836318114,-0.01311052982462483,
      -0.1422987785572308,-0.2648092917366038,-0.2250317714436444,
      -0.3508811505850218,-0.4179972559636759,-0.06662019868190015,
      -0.2045235871714226,-0.3522760475882291,-0.490753594798392,
      -0.6019340751777634,-0.6790258583886756,-0.2310136280273065,
      -0.385038640575524,-0.5360950867028047,-0.6619938111098648,
      -0.749995864312011,-0.399676350745083,-0.5562788600968236,
      -0.6947565047867294,-0.7970870947583818,-0.551898166785698,
      -0.6959582378852542,-0.8113895155772442,-0.6724569726942554,
      -0.794938298671674,-0.7580246814516086,-0.3341522378920435,
      -0.3288731104321892,-0.3084405200297279,-0.2755136565752797,
      -0.5005289572181015,-0.4916319868779721,-0.4596638193626508,
      -0.651752116850578,-0.6313195665821806,-0.7681408256666581,
      -0.7967293465331413,-0.8980116136367456,-0.9631629650010809,
      -0.9837042844506774,-0.9671524096287046,-0.8656229381767455,
      -0.9531387450388693,-0.9959257826316365,-0.9941684373181742,
      -0.8983857314821276,-0.9635372279543751,-0.9840624137666333,
      -0.890753770890823,-0.9334760519103793,-0.8525638190917326,
      -0.0295456393046314,0.1075921183812993,0.24606969704266,
      0.3678649858572621,-0.1043294399383607,0.03496867572798409,
      0.1726257663402015,-0.1777733562873886,-0.03929575512161578,
      -0.2403190522322838,-0.9520779048267128,-0.8970418148986564,
      -0.7988299587745943,-0.6702240741753134,-0.5328183289734231,
      -0.9628652449155908,-0.885428988462601,-0.7664414449873368,
      -0.6260926899993671,-0.9304767531770004,-0.8322648150388383,
      -0.7022083677250226,-0.8591275565599557,-0.7494380619781833,
      -0.7656969051450071,0.6830504023375936,0.7791773456982055,
      0.8443286725827212,0.8700252447721738,0.6279543165738012,
      0.7097593165624652,0.7582568868087332,0.5418825825925072,
      0.6070338354963091,0.4360370100657923,-0.4558825077011868,
      -0.3507064981157044,-0.2248566368863532,-0.0947116843681468,
      0.02374227578387531,-0.5423796577050666,-0.4265380492676017,
      -0.2906799053833736,-0.154436271103469,-0.6082029717256227,
      -0.4823530727762694,-0.3410374453295618,-0.6447858429341417,
      -0.513609764630649,-0.654384710247139,0.8188524345231695,
      0.757774488968496,0.6595623604908243,0.5369978886010214,
      0.6843305795053973,0.6002015742230613,0.4879063887999532,
      0.512674598571755,0.4144625345167917,0.3262247228510809,
      0.006132171174653799,-0.01402256634368312,-0.03445538312060447,
      -0.05250705338074756,-0.1852627984273255,-0.2106379566061765,
      -0.226128333686081,-0.376935865064754,-0.397368560322806,
      -0.5439417227815399,-0.1282040992012935,0.001413185228105039,
      0.1506070646973712,0.3060111300013609,0.448827218017063,
      0.5644591381617006,0.6488215902264565,-0.1794700290349442,
      -0.04186542201104336,0.1153813721057172,0.2745267180089526,
      0.4148587047410423,0.5243374917023425,-0.232615798766268,
      -0.08912570230038545,0.07170206110196441,0.2291650104977662,
      0.3638246180723771,-0.2808945465452747,-0.1344874258557153,
      0.02465792853592251,0.1758296024995542,-0.3176184514965804,
      -0.1720469916489061,-0.01939827024983318,-0.3399568124040331,
      -0.1991953311157892,-0.3496821715684524,0.6714420577705416,
      0.7792137212760166,0.8777656242085763,0.9496341036793504,
      0.9824386073167399,0.9768385042055774,0.9441720387917811,
      0.7257096121901333,0.8383309419108317,0.9322072488654631,
      0.9880661303797563,0.9986378183907035,0.9727408945826257,
      0.7644107780855932,0.871715411213502,0.9498618317294465,
      0.9834333410851046,0.9737573075945113,0.7760312926982584,
      0.8670825303083295,0.9229415742963546,0.9363385958760152,
      0.7557291769626011,0.8250351935145946,0.8604026946131842,
      0.7093185095742097,0.7587249788041119,-0.9441722324566724,
      -0.8784288421462625,-0.7739455965390631,-0.6303269409870232,
      -0.4607792790679794,-0.2871643574027738,-0.9185505385856864,
      -0.8335922034949309,-0.7042733744470981,-0.5371520006668183,
      -0.3539781199209927,-0.8589482430980673,-0.7496350811244772,
      -0.5965101770495287,-0.4153923937207508,-0.7605084662990487,
      -0.6278754136763828,-0.4607541022738985,-0.6307845637432553,
      -0.4841596501592634,-0.4877728218435191,-0.3130828685601688,
      -0.1619469688877697,0.01737679733452725,0.2106381298878163,
      0.3952347219739472,0.5514566289762698,-0.3068061997519851,
      -0.1389316419038192,0.05641200521180441,0.2580561121281895,
      0.4397576822043592,0.5849482839226374,-0.2895250242862085,
      -0.1073663493645557,0.09654302677976184,0.2959218923430848,
      0.4661908578614318,-0.2593820240207756,-0.06950063773914905,
      0.1321435149365393,0.3193074202631061,-0.2185691140597737,
      -0.03026253166221194,0.1592889611288683,-0.1728399308158635,
      0.005511078744538343,-0.671442351589465,-0.7792138411724322,
      -0.8777656078901646,-0.949634045961687,-0.982438609905361,
      -0.9768386118322019,-0.6715195478039778,-0.7797211321494723,
      -0.8709393754318948,-0.9267982045118057,-0.9400278430381637,
      -0.649602216294724,-0.7491798190236064,-0.8243817913354027,
      -0.8608974751546269,-0.6002017842451572,-0.6832790131816587,
      -0.7391378312821431,-0.5261115978506249,-0.5905953516181053,
      -0.4383666636546263,-0.6488222902967828,-0.6444008694878965,
      -0.6181925269683732,-0.5653935344751363,-0.4893366483035592,
      -0.4011183181354504,-0.7587252832301369,-0.7548226978527568,
      -0.7202890787466467,-0.6515250496520278,-0.557479929039536,
      -0.4546220799908838,-0.8604026705465684,-0.8495447148476215,
      -0.7995410210009178,-0.7120166543368723,-0.6026908960383706,
      -0.9363384293729855,-0.9100364067183718,-0.8412723782021747,
      -0.7389956620788968,-0.9737572051250643,-0.9284250544292145,
      -0.8449013283633929,-0.9727409539931392,-0.9119203221721993,
      0.3130823317650696,0.1619462783427865,-0.01737758434298078,
      -0.2106389019045294,-0.3952353653550799,-0.5514570904847834,
      0.220234495302766,0.04529946956154991,-0.1542902631671154,
      -0.3559342336221437,-0.5333895041467447,0.1061130743462423,
      -0.08838939227573586,-0.2970021727032599,-0.4916772652886296,
      -0.02151295053653907,-0.2241324894823635,-0.4257764066297352,
      -0.1482539922220807,-0.3442635849044243,-0.2600158884245429,
      0.1282040764044601,-0.001413366553346585,-0.1506074987015245,
      -0.3060118317848468,-0.4488280788164578,-0.564459985716308,
      -0.005511078361168723,-0.1582021264863434,-0.3245220334380661,
      -0.4836676111525484,-0.6149268292273564,-0.7093192714417703,
      -0.159288972427813,-0.3291548996161899,-0.5000332157715695,
      -0.6474460255720145,-0.7557300072673856,-0.319307490579,
      -0.4929333495466833,-0.6520789064271625,-0.7760321386045594,
      -0.4661910177108468,-0.6282225265671552,-0.7644115292814133,
      -0.5849485267562503,-0.7257101604516547,0.6444002635563192,
      0.6181920323103048,0.5653931175085022,0.4893362367199527,
      0.4011178519784379,0.5013193772200437,0.4552897581943395,
      0.3865257888849735,0.3039767654183579,0.319546485626849,
      0.2568082513095248,0.1820185654081009,0.1150393454476789,
      0.04627539851448641,-0.08558647206366617,0.8784286613131731,
      0.7739454978673308,0.6303269573377658,0.4607793683387657,
      0.2871644290583872,0.9119202617664459,0.8264212051537363,
      0.6967772336314446,0.5296559032972297,0.3468071953669221,
      0.1728397432053184,0.8449014608184471,0.7346429035325667,
      0.5811577408091089,0.4004001175275866,0.2185687785328751,
      0.7389959159190774,0.6053871763277086,0.4382657239603054,
      0.2593815856416469,0.6026910636921159,0.4554759872637936,
      0.2895245326714229,0.4546219464457031,0.3068056829967603,
      0.6149259495773405,0.4836668020692608,0.3245214821530512,
      0.1582018910739125,0.6474451599919397,0.5000325428335932,
      0.329154569384261,0.6520781574644046,0.4929329083973684,
      0.6282220041094108,0.9185503953325266,0.9400278438555837,
      0.9267983622223734,0.8709396068783997,0.7797212987315195,
      0.6715195494525621,0.8589483198537152,0.8608977587445145,
      0.8243822455102904,0.7491802985670569,0.6496025629973337,
      0.7605089118538747,0.7391384910218132,0.6832797730640632,
      0.6002024649156599,0.6307853921801013,0.5905963116881646,
      0.5261125320735169,0.4877738980761407,0.4383677443133593,
      0.3496833143594963,-0.3468072634833415,-0.5296559121107985,
      -0.6967772517279487,-0.8264212573361066,-0.4004002676068469,
      -0.5811577408091089,-0.734642815176818,-0.4382658852809146,
      -0.6053870894221652,-0.4554761269845107,0.5333893556910446,
      0.3559338950225972,0.1542897676170885,-0.04530002222683179,
      -0.2202350051083672,0.491677469651672,0.2970021583546204,
      0.08838916973894111,-0.1061134148983228,0.4257769268822734,
      0.2241327483663198,0.02151295276860445,0.3442643233016264,
      0.1482544392795429,0.2600167460395648,-0.998637827838979,
      -0.988066142434594,-0.9322073703652781,-0.8383312666748716,
      -0.9834332923270373,-0.9498619694604108,-0.8717158495436891,
      -0.9229416329138649,-0.8670829638030551,-0.8250355565045274,
      -0.3039772599799364,-0.3865262864971092,-0.4552902929786847,
      -0.5013199694054605,-0.5243381301130434,-0.1820189194073548,
      -0.2568086311939767,-0.319546898969995,-0.3638250562458945,
      -0.04627540884770265,-0.1150393837809054,-0.1758296738027379,
      0.08558690678521951,0.01939866440958929,0.1991961595715477,
      -0.4397580914639075,-0.2580566866135989,-0.05641266799084353,
      0.1389310087452558,-0.295922215665352,-0.0965434988571485,
      0.107365815243832,-0.1321437542333658,0.06950025525528913,
      0.03026233882278601,-0.4148593732748149,-0.2745272477919344,
      -0.1153816305493603,0.04186543327713237,0.1794701997231261,
      -0.2291653183673072,-0.07170210104888378,0.08912594410177796,
      0.2326162031595532,-0.02465773800362392,0.134487905062478,
      0.2808952005684321,0.1720476740810845,0.3176193294845107,
      0.3399578552959914,0.5574798958328108,0.6515250645580214,
      0.7202890442609949,0.754822534930785,0.7120168974234773,
      0.7995412648414353,0.8495448706478895,0.841272643210163,
      0.9100366384710021,0.9284251625234463,0.3539783956892765,
      0.5371522584244614,0.7042734954121332,0.8335921603240184,
      0.4153929051801631,0.5965106290979496,0.7496353473630305,
      0.4607548496345096,0.6278760663083329,0.4841605931432954 };

    auto      pointshell
        = new vnl_matrix_fixed<double, 3, 362>(coords);

    return pointshell;
  }
};


template<> class PointShell<492, vnl_matrix_fixed<double, 3, 492> >
{
public:
  static vnl_matrix_fixed<double, 3, 492>* DistributePointShell()
  {
    double coords[3*492] = {
      0.9756767549555487,0.995129146382046,0.9942990656144199,
      0.9656349505028294,0.906669314667086,0.8226336999568686,
      0.7243638464104109,0.6231965669156312,0.9839703767589784,
      0.997142038124067,0.984428692964357,0.9391662760580684,
      0.862670051560969,0.7648456045484533,0.6590519021583431,
      0.9707167715611407,0.9718838371878742,0.9426229117731289,
      0.8796306184740903,0.7892458129328142,0.6846076300785651,
      0.929183358966584,0.9136522029381524,0.8666502644050813,
      0.7898345918065129,0.693320589737796,0.8580670030245058,
      0.8250351905308677,0.7641558269670571,0.6811699046825979,
      0.7636773150775941,0.7169778531093051,0.6492336193335421,
      0.6574101272386674,0.6032570204852985,0.5505632701289798,
      0.4375940376503738,0.3542026794336651,0.2512745383622963,
      0.1321836977435469,0.005435183516526082,-0.1177256014717769,
      -0.2278158095071639,-0.3200730131503601,0.499121216713679,
      0.4104288008617515,0.2993922246212043,0.1713800199472592,
      0.03766821044034947,-0.08889959511906904,-0.1993010113619456,
      0.5575360887326264,0.462310883286662,0.3429526003657482,
      0.2075487278198905,0.06972486181048379,-0.0574641852590465,
      0.6055762148233742,0.5027272059054465,0.3761219220443625,
      0.2365775369888062,0.09858215717061569,0.6366252883681098,
      0.5264243511029411,0.3955623729759449,0.2563797979489069,
      0.647928555739847,0.5327472238507116,0.4019285426857001,
      0.6416958910015911,0.5252922545125212,0.3200724178002418,
      0.4449423089720693,0.5765874507842912,0.7038338496229432,
      0.8135085046154491,0.8958808454008315,0.9485590875281112,
      0.3796302423021757,0.5151344445922483,0.6532516657507655,
      0.7794467577011209,0.879719713309462,0.9471999912418547,
      0.4385611209938041,0.579827292674625,0.7171156368030697,
      0.8342797026222076,0.9195293587283735,0.4904847582024935,
      0.6301148467885526,0.7583069705424822,0.8604123883748889,
      0.529043067100619,0.6594466179023598,0.7726806309784415,
      0.5508150816257275,0.6670356654438421,0.5566867904077125,
      0.2502538949373057,0.2979150767790733,0.345210193321538,
      0.3870703453412754,0.4184100968812649,0.4364489070347424,
      0.4418245296642042,0.3805829325031832,0.4390055479929592,
      0.492568207092815,0.5338014037295256,0.5572819316169835,
      0.5624691657697151,0.5532743164301828,0.5199151685430116,
      0.5855041443032848,0.6393053209231115,0.6727852191382864,
      0.6825747626046904,0.6719797747984303,0.6571141852151331,
      0.7228163905998002,0.7689467902788032,0.7886244561063042,
      0.78247330591778,0.7784686816166657,0.8360899632174316,
      0.8684468597631155,0.8724880558103147,0.8732117959842211,
      0.9170839626929869,0.9340377000206754,0.9378324283560583,
      0.9666143691402809,-0.4375932291383153,-0.354202049210883,
      -0.2512741515648044,-0.1321835946181268,-0.005435354322950186,
      0.1177252152663303,0.2278152856398797,-0.318790682468547,
      -0.2170414107297249,-0.09666371763400751,0.03477626912480049,
      0.1650597498899593,0.28228599025285,-0.17643804815428,
      -0.05685397486912016,0.07657633376040601,0.2119799408472924,
      0.3357313012364012,-0.01650847491202961,0.1157420722211241,
      0.2531713417578513,0.3818912918445931,0.1487980970433173,
      0.2844888970254793,0.4153506930815835,0.3048153313129787,
      0.4341873267571942,0.4402832445658493,-0.6231952788307174,
      -0.5360175915304849,-0.4245963103234196,-0.291418513965459,
      -0.1453010498480997,0.0007922214273259519,0.13494147117507,
      -0.5252911977287607,-0.4194174780522606,-0.28881747505953,
      -0.1402907332818909,0.01290161533377887,0.1562152529527386,
      0.2798596737299269,-0.401927847344762,-0.2767587723714138,
      -0.1302687984978551,0.02582668527333206,0.1758198972364496,
      0.3070530822616677,-0.2563795458672579,-0.1157654558096325,
      0.03830398324407951,0.191055688005941,0.3280904342412125,
      -0.09858232519641157,0.04907792605008903,0.199937541973602,
      0.3397702575045667,0.05746372336207268,0.2022279185194857,
      0.3410562736291031,0.1993004179530413,0.3334923558881231,
      -0.2502532651867688,-0.2979144048694595,-0.34520946527015,
      -0.3870695532799813,-0.4184092501028168,-0.4364480352337543,
      -0.4418236723862358,-0.1460995017681994,-0.1875439456120633,
      -0.2289599148396085,-0.2657355106897256,-0.2936735229963843,
      -0.3110073744636102,-0.02437365804953178,-0.05828840766257726,
      -0.0937912788361675,-0.1272711110222873,-0.1553588546931559,
      0.1088504388442424,0.08137944753513068,0.04932387614619547,
      0.01557150971513331,0.2428173001617129,0.2183408313117627,
      0.1859840000400707,0.3656411198910443,0.3402213332299013,
      0.4690651309969549,-0.975676581463901,-0.995129073697949,
      -0.9942990262685066,-0.9656348165534084,-0.9066689404406876,
      -0.8226329948202892,-0.7243628133079894,-0.9666141311314619,
      -0.9785291857720615,-0.9649167587949544,-0.9193242448617983,
      -0.8431576679717276,-0.7462319572506099,-0.6416946598622301,
      -0.9340374019521771,-0.9328600498682232,-0.9022446524809692,
      -0.8392521444630035,-0.7502214356485383,-0.6479274205124774,
      -0.8724876912825506,-0.8541267298323965,-0.8060828752150751,
      -0.7303087484697733,-0.6366242668463477,-0.7824728542655722,
      -0.7469876674129576,-0.6861081367635697,-0.6055752927616058,
      -0.6719792111143933,-0.6239138376203725,-0.557535231009602,
      -0.5532736276163508,-0.4991203912134258,0.5360190436422174,
      0.4245978595344713,0.2914200439268497,0.1453024287888095,
      -0.0007910862586512385,-0.1349406039583337,0.5628633831872754,
      0.4391923979565908,0.2932083572735901,0.1374730553820866,
      -0.01276978562268766,0.5775069821113988,0.4414551280568763,
      0.2853595486273291,0.1249280534487668,0.5745152767417482,
      0.4284746793321108,0.2676940340983546,0.5524169045150343,
      0.4015572805868465,0.5149954726706612,-0.4449426764484256,
      -0.576587591800914,-0.7038338094577924,-0.8135083597704743,
      -0.8958806666660788,-0.9485589107573548,-0.3334928142377668,
      -0.4656561029153702,-0.6013830911884229,-0.7267009436209915,
      -0.8278508961406972,-0.8977211598944568,-0.9378322143189233,
      -0.3410565997728103,-0.4760902226557944,-0.6097779474907786,
      -0.7269418806406593,-0.8157918899746622,-0.8732115893121841,
      -0.3397704479470293,-0.4718778429932776,-0.5973004074862646,
      -0.7021751105216565,-0.7784685034185206,-0.3280904656572098,
      -0.4519721780114144,-0.5652060578754561,-0.6571140026135045,
      -0.3070529166634529,-0.4196426265222307,-0.5199149039850575,
      -0.2798592799108112,-0.3805825072580722,0.7462328929439237,
      0.843158260373911,0.9193245436545445,0.9649169034529495,
      0.9785293294285902,0.7502222547730358,0.8392526460505537,
      0.9022449400962008,0.9328602810500073,0.7303094694378339,
      0.8060833454941833,0.8541270795356295,0.6861088083158182,
      0.7469881692208658,0.6239145070358707,-0.3796306565856781,
      -0.2822863401756764,-0.1650599593464782,-0.03477624931230538,
      0.09666402587967657,0.2170420055376572,0.3187915103535204,
      -0.4385613640703605,-0.3357314877557171,-0.2119799800663603,
      -0.07657612650429085,0.05685448230997714,0.1764388387711721,
      -0.4904848834824611,-0.3818913632934865,-0.2531712491023975,
      -0.1157417111569205,0.01650914517247642,-0.5290431556806968,
      -0.415350713896117,-0.2844887227712293,-0.1487976298306422,
      -0.5508152020150771,-0.4341873428629178,-0.3048151129115739,
      -0.5566869690240907,-0.4402832689070376,-0.5505634949474448,
      0.8977213434708383,0.8278510409608931,0.7267010116232644,
      0.601383022760614,0.4656558477896065,0.8157920246508805,
      0.7269419391910862,0.6097779075806529,0.4760900512385949,
      0.7021751903445496,0.5973004014138014,0.4718777509809839,
      0.5652061482232076,0.4519722029963607,0.4196428289231057,
      0.3110082673965054,0.2936744473798271,0.2657364265527945,
      0.2289607907051673,0.1875447674114613,0.1461002696873639,
      0.1553597055562715,0.1272719901065957,0.09379215148049876,
      0.05828924862952556,0.02437445585198169,-0.01557080948102478,
      -0.04932316861007474,-0.08137874963697227,-0.1088497589109511,
      -0.1859835450151495,-0.2183403902096033,-0.2428168640462141,
      -0.3402211599784607,-0.3656409687130195,-0.4690652148440055,
      0.08889916251124211,-0.03766840286064398,-0.1713799135166342,
      -0.2993918156335824,-0.4104281434194711,-0.06972505514639399,
      -0.2075485942610162,-0.3429521440427875,-0.4623101760624211,
      -0.2365773504687261,-0.3761213898309075,-0.5027264164274929,
      -0.3955617526602613,-0.5264234573415938,-0.5327462324847574,
      0.0127707607707569,-0.1374718849546943,-0.293207061528236,
      -0.4391910893180505,-0.5628621680946149,-0.6590508424134647,
      -0.1249271271017331,-0.2853585453187341,-0.4414541323690568,
      -0.5775060764024141,-0.684606860838456,-0.2676933323589697,
      -0.4284740156087684,-0.5745147048698985,-0.6933201342296018,
      -0.4015569080051182,-0.5524166274339118,-0.6811697288917756,
      -0.5149954132531107,-0.6492336477442618,-0.6032571744726157,
      -0.562468445445845,-0.557281226121636,-0.5338007588723378,
      -0.4925676465433573,-0.4390050662086606,-0.6825741836716002,
      -0.6727846814032147,-0.6393048713560345,-0.5855037965960378,
      -0.7886240006308879,-0.7689463943189565,-0.7228160987485833,
      -0.8684464995603698,-0.8360896738847699,-0.9170836780332227,
      -0.7648448608123986,-0.8626696659610523,-0.939166188119448,
      -0.9844287505675849,-0.9971420785412131,-0.9839703005564542,
      -0.789245396544295,-0.8796305387421426,-0.9426230395503291,
      -0.9718839912389975,-0.9707168152563411,-0.7898344765948023,
      -0.8666504070362878,-0.9136524365050784,-0.9291835215938483,
      -0.7641559277074473,-0.8250354441736533,-0.8580672517338752,
      -0.7169780714779912,-0.7636775971947915,-0.6574103946704876,
      -0.1562146458746545,-0.01290076927461595,0.1402917844716378,
      0.2888186351042548,0.4194186302321976,-0.1758195663775944,
      -0.02582617528460154,0.1302694498290071,0.2767594860031721,
      -0.1910556254438784,-0.03830382499862148,0.11576568242264,
      -0.1999377238417288,-0.04907809969461731,-0.2022283181311396,
      -0.9471998764213244,-0.8797195604896335,-0.7794466205193276,
      -0.653251640029333,-0.5151346189117816,-0.9195293275646416,
      -0.834279604589424,-0.717115548266178,-0.579827326676834,
      -0.8604124463046277,-0.7583069490956061,-0.6301148416214684,
      -0.7726807603671683,-0.659446675397405,-0.6670358344072718,
      -0.1778161375899129,-0.09682873617143042,-0.002564331870712152,
      0.1001489444293402,0.20294037487074,0.2966901028193483,
      0.375180936757441,0.4366423972091846,-0.0474532387078642,
      0.05034809081619229,0.1589075235585833,0.2695165933494903,
      0.3711623656745905,0.4552971059046723,0.5189584406200277,
      0.1017827048364344,0.2144158936234644,0.3316684820340907,
      0.4414792935657015,0.5327981708248342,0.6005401860053867,
      0.2614395442328129,0.3824105897655243,0.4989141911429241,
      0.5982545824011208,0.6726053169812281,0.4179945008346566,
      0.5376874271465311,0.6438148252422833,0.7263687759217851,
      0.5575576702299878,0.6670994939031171,0.7570605631322944,
      0.6714338802491097,0.7658357564785362,0.7580253719184177,
      -0.5980610514568761,-0.539304606284799,-0.4601049879074139,
      -0.3610209854418669,-0.247915131130675,-0.1308224218325103,
      -0.01993206276454632,0.07805400320688782,-0.4842131513641215,
      -0.4078776799069966,-0.3108032946618485,-0.1973076501762547,
      -0.07725012597615898,0.03770515913931162,0.1390339189204285,
      -0.3436782750638498,-0.2488690439489935,-0.1366770376390136,
      -0.015847433302958,0.1014607675543575,0.2051261004110744,
      -0.1810587170687898,-0.07134455272126199,0.04823642170967721,
      0.1661580441221543,0.2713649127044082,-0.007965415159176724,
      0.1085527127599039,0.2253294012222397,0.3313524432512411,
      0.1602444452922388,0.2740270626018594,0.3797664442036205,
      0.3106168108763466,0.4144914982680714,-0.07805410188638828,
      -0.09926944144321027,-0.1210902608980977,-0.1414695137250296,
      -0.1581584914998737,-0.1696765129666563,-0.1759038243562996,
      0.0445080871572717,0.0312941062385701,0.01557512261039578,
      -0.001683488527681644,-0.01888610824676229,-0.03445251803458384,
      0.1827599102095679,0.1772109890337862,0.1655328468312471,
      0.1477041886655562,0.1255191042072879,0.3281955954096453,
      0.327054627020386,0.3149499526725762,0.2920106184972662,
      0.4680615585920147,0.4660213421080895,0.4487906967476448,
      0.5899485877914471,0.5822989875106862,0.6867607929057539,
      -0.9161616153729887,-0.9364062548880398,-0.9378298619231508,
      -0.9131962255692416,-0.8599818711884049,-0.7829080825969388,
      -0.6920503887795537,-0.8718421840682521,-0.8825595832157751,
      -0.8702495449678662,-0.8290948069847427,-0.7603667866829383,
      -0.6729204873538853,-0.5786150802505966,-0.8013843264777428,
      -0.7976661081468692,-0.768502718877635,-0.711654457383761,
      -0.632841917808473,-0.5431812243043028,-0.7022906139785883,
      -0.681473218889239,-0.6365518584536976,-0.5697323001067327,
      -0.4894330041487118,-0.5787744062647625,-0.5426164975113815,
      -0.4876750583595674,-0.4191311379161569,-0.4417943029188641,
      -0.3959648506658267,-0.3385129506001341,-0.3046660637665696,
      -0.2557948010777292,0.5980613131647217,0.5393048163248674,
      0.4601051714264852,0.3610211606830505,0.2479152916167419,
      0.1308225338647039,0.01993208413931726,0.6677553214177145,
      0.6047094356064554,0.5171428148173597,0.40713632090741,
      0.2835896681332588,0.1591265698738985,0.7315641787700664,
      0.661547900814459,0.5636788840886751,0.4428493496799737,
      0.3112182413099586,0.7806188517072566,0.7008301889050352,
      0.5922663576988024,0.4633278474811566,0.8073788931114407,
      0.7168049773549268,0.6000284935406993,0.8094700983598709,
      0.7101312760741833,0.7906351929054215,-0.4366431953633788,
      -0.5432543873236343,-0.6518847469150993,-0.7519687641278953,
      -0.8321863440327827,-0.8854204339944604,-0.9116068551607237,
      -0.4144921923404347,-0.5235089742164267,-0.6316176685916446,
      -0.7265196152590857,-0.7972596530206874,-0.8395279581409318,
      -0.8565153620209197,-0.3797670590739394,-0.4868622160190148,
      -0.5894554430684077,-0.6751509712448159,-0.7353253350307362,
      -0.7689936060115737,-0.3313530028134794,-0.4321132275126733,
      -0.5253715109567975,-0.600556355906943,-0.6522238398741106,
      -0.271365416599326,-0.3629934344088176,-0.4458145633979653,
      -0.5120186635901344,-0.2051265147794646,-0.2871867138062492,
      -0.3608173514139882,-0.1390341944715653,-0.2127048766307628,
      0.9161619708326185,0.9364066014623644,0.937830159535626,
      0.9131964504670906,0.8599820357483486,0.7829082325289186,
      0.6920505768643884,0.9609824530756677,0.9781541365013002,
      0.9704616345084118,0.9310013912281646,0.86057880605657,
      0.7685149559210446,0.989767208703788,0.9980900358901049,
      0.9758827628822744,0.91903449699185,0.833265855013368,
      0.9934762859646802,0.9871926244298611,0.9476218597261858,
      0.8754517847749737,0.9670219266787334,0.9434641845143519,
      0.8885228243698201,0.9127512333378954,0.8739366103094172,
      0.8395063271314147,0.1778171059718252,0.09682951222050769,
      0.002564819628494808,-0.1001488152988667,-0.2029406081721617,
      -0.2966906282753529,-0.3751816472846526,0.2557957005565683,
      0.173079435577695,0.07531211221520523,-0.03133676316818776,
      -0.1369434620566336,-0.2318708914823758,-0.3106176573163182,
      0.3385137719481263,0.2540232415826708,0.1530285137711279,
      0.04321731529158302,-0.06436010707835081,-0.160245257281258,
      0.4191318730425461,0.3321294046295875,0.2281310249483762,
      0.1162846541659367,0.007964747877728702,0.489433642057663,
      0.3991899134973828,0.2930621280324591,0.1810582896385999,
      0.5431817494666105,0.4500346846996958,0.3436781221558973,
      0.5786154775493708,0.4842132402051962,0.5432536936373913,
      0.6518842443443043,0.751968519322874,0.8321863638644724,
      0.8854206569589087,0.9116071863986105,0.6355394193271248,
      0.7490598502163163,0.8459479265844492,0.9147022337905197,
      0.9515590961968758,0.7217468924595103,0.832492462220627,
      0.9181881173363158,0.9702103552386692,0.7903983024315012,
      0.8899272054685274,0.9588415071652456,0.8327634000595739,
      0.9155844937451473,0.847341552439817,0.09926947140308087,
      0.1210904447270627,0.141469873017204,0.1581590349215481,
      0.16967722932441,0.1759046848916014,0.2127045253884476,
      0.2445425491615283,0.273585758469152,0.2957341763491218,
      0.308047355491858,0.310289867067379,0.304666812316337,
      0.360816856533433,0.40111044364496,0.4328607304643095,
      0.4506895618367096,0.4528028309124678,0.441794849646433,
      0.5120181966189056,0.5550964104131831,0.5826403298719719,
      0.5901407115935625,0.5787748141912227,0.6522235657335231,
      0.6906215449122578,0.7078523050511999,0.7022909666912099,
      0.7689935992922883,0.7968842810746374,0.8013846828119295,
      0.8565155859292075,0.8718425534235449,0.2318702160237238,
      0.1369430725775421,0.03133674940539507,-0.07531173762971051,
      -0.1730787436687962,0.06435957029576898,-0.0432174738186679,
      -0.1530282625973764,-0.2540226452878776,-0.1162849524413907,
      -0.2281309051554154,-0.3321289161485715,-0.2930621449512896,
      -0.3991895474466183,-0.4500344545042632,-0.04450787402063819,
      -0.1591262308672862,-0.2835892617928241,-0.4071359133189035,
      -0.5171424431454641,-0.6047090919043497,-0.667754971050827,
      -0.182759348567292,-0.311217588457912,-0.4428487008416941,
      -0.5636783165490287,-0.6615474307561048,-0.7315637666610734,
      -0.3281947296279775,-0.4633269810702052,-0.5922656048068207,
      -0.7008296027194382,-0.7806184022682794,-0.4680605237949423,
      -0.6000275832292742,-0.7168042826134902,-0.8073784119772466,
      -0.5899475572826254,-0.7101304786227719,-0.8094695688943716,
      -0.6867599019180602,-0.7906345906969057,-0.7580246814516085,
      -0.3102892714845059,-0.3080469554975349,-0.2957339957005241,
      -0.2735857903437153,-0.2445427612771611,-0.4528024643637317,
      -0.4506894194251309,-0.4328608289948512,-0.4011107629731985,
      -0.5901404834522559,-0.5826403283348299,-0.5550966534493702,
      -0.7078521147132451,-0.6906215734113883,-0.7968840729809553,
      -0.7685147196433009,-0.8605786508071993,-0.9310012591522379,
      -0.9704614726381049,-0.9781539258256538,-0.9609822090426239,
      -0.833265614060389,-0.9190343924750819,-0.9758827239891746,
      -0.9980899940222691,-0.9897671312659951,-0.8754515637483313,
      -0.9476218214663272,-0.9871926808705815,-0.9934763548505058,
      -0.8885226077999228,-0.9434641720148271,-0.9670220212347258,
      -0.8739363526795478,-0.912751179432779,-0.8395059839899685,
      -0.03770535005341785,0.07724999366628089,0.1973075542157886,
      0.3108032352691305,0.4078776804650779,-0.1014611429922413,
      0.0158470793272554,0.1366767120232094,0.2488687826864697,
      -0.1661585633922329,-0.04823695834758815,0.07134403839749669,
      -0.2253300338935004,-0.1085533974058879,-0.2740278002009307,
      -0.951558921251018,-0.9147022049495029,-0.8459481068009407,
      -0.7490602493222091,-0.6355399873284351,-0.5189591001755303,
      -0.9702103859734443,-0.9181883003327364,-0.8324928057608664,
      -0.7217473579669872,-0.6005407102760519,-0.9588416857098183,
      -0.8899274903619968,-0.7903986578533415,-0.6726056915299506,
      -0.9155846529394258,-0.8327635907539088,-0.7263689563436138,
      -0.8473415025692198,-0.7570604878450786,-0.7658353788323373,
      0.6729208082532415,0.7603670653263657,0.8290950909162599,
      0.8702498673585751,0.8825599429670654,0.6328423544382739,
      0.7116548337644016,0.7685030738837739,0.7976664655237659,
      0.569732829878993,0.6365523011473486,0.6814736060981027,
      0.4876756652068531,0.542616991447444,0.3959655290681676,
      -0.4552976478369847,-0.3711626697331859,-0.2695165436861755,
      -0.158907070465767,-0.05034727738276638,0.04745431047424936,
      -0.5327985288361234,-0.4414793439189789,-0.3316681166200877,
      -0.2144151021287253,-0.1017815730335519,-0.5982547186038516,
      -0.498913942879685,-0.3824098828497664,-0.261438418974126,
      -0.6438146830008175,-0.5376868414254443,-0.4179934509733508,
      -0.6670990265838406,-0.5575567398179423,-0.6714330779441176,
      0.8395281342551513,0.797259699808284,0.7265194573448021,
      0.631617280616129,0.5235083955500395,0.7353252547527041,
      0.6751507612001276,0.5894550707609513,0.4868616963131356,
      0.6005560087932349,0.5253710768222695,0.4321127152346745,
      0.4458140656510558,0.3629929231163813,0.287186249855225,
      0.03445349641709906,0.01888696113562014,0.001684189644027077,
      -0.01557458737398696,-0.03129373726645494,-0.1255180400739348,
      -0.1477032144715439,-0.1655319877909157,-0.1772102701265833,
      -0.2920095247225022,-0.3149489017570188,-0.3270536482990907,
      -0.4487896352785713,-0.466020275040162,-0.5822980026295906,
      -0.1282040992012934,-0.01849805050225584,0.1065963991903489,
      0.2398318813191588,0.3698188179209382,0.48502451338496,
      0.5783911156884447,0.6488215902264565,-0.1719025532007262,
      -0.05632783998118585,0.07515681888535472,0.2129025876416392,
      0.343567868763417,0.4557584300457211,0.5443645168181878,
      -0.2175987830988393,-0.0973017552494557,0.03818460718077296,
      0.1770480397960358,0.3053148472212455,0.4131148482333044,
      -0.2612807879100861,-0.1378469909045712,-0.001396095557054295,
      0.1351028209250271,0.2586670590048458,-0.2983313855304687,
      -0.1738078395074255,-0.03960230940919204,0.09162948390440753,
      -0.3254633663003954,-0.2022894544581522,-0.07317794255704746,
      -0.3420357716053446,-0.2226579469368842,-0.3496821715684524,
      0.6714420577705413,0.7639967300466739,0.8515659143445876,
      0.9231420898872643,0.968766507749211,0.9843912721595947,
      0.9735002156201846,0.9441720387917811,0.7186206475410257,
      0.8155881299141562,0.9020895786249314,0.9652453470205394,
      0.9962999166711242,0.9953266714812502,0.9700250905310301,
      0.7556710613817772,0.8510774619026286,0.9293561756854016,
      0.9780963063206267,0.9923931460327553,0.9770471587096806,
      0.7749162464509299,0.8614959728515891,0.9253137561812753,
      0.9572996257002351,0.9574145614312103,0.7711321510425843,
      0.8432637257218606,0.8903690639450959,0.9080060338759378,
      0.7446544865975486,0.8006806881904648,0.8332051934736653,
      0.701244308549226,0.7431452383031477,-0.9441722324566724,
      -0.8900404033987204,-0.8080124134651047,-0.6961352518091236,
      -0.5596336341615111,-0.4106183042575765,-0.2632365894868439,
      -0.9240670480583062,-0.8565379051211715,-0.7569806316871794,
      -0.6264662143927534,-0.4751173969978469,-0.3187870772043735,
      -0.8799221319939898,-0.7952336612815645,-0.6770111077910543,
      -0.5311881497579278,-0.3724388713810448,-0.8072870326747171,
      -0.7042659659540528,-0.5707688373923591,-0.4176365891727677,
      -0.7078360053862648,-0.5898764843905154,-0.4489449331750177,
      -0.5903925893983768,-0.4647486526822006,-0.4673964384898752,
      -0.3130828685601689,-0.185445228664084,-0.03612578735626812,
      0.1274723552911619,0.2921714771415931,0.4433590934587491,
      0.5708391580621228,-0.3082982931616353,-0.1684123240900115,
      -0.006503141132676928,0.1663065314632002,0.3335853689877698,
      0.4804232043289857,0.5992429554740794,-0.2957556065313156,
      -0.1446156178201561,0.0263111709560856,0.2022572179054991,
      0.3655171685702938,0.5033858756710694,-0.2738409047359961,
      -0.1145893425106214,0.05935120233887466,0.2312500236673017,
      0.3849556597532595,-0.2429129440661442,-0.08075215189426678,
      0.08929215655126845,0.2511845570441078,-0.2057157098095482,
      -0.04657083216475545,0.1139410251668723,-0.1662796617532138,
      -0.01502268661531953,-0.671442351589465,-0.7639968739619373,
      -0.8515659293219712,-0.9231420361206036,-0.9687664657212128,
      -0.984391303457987,-0.973500337776358,-0.6726628661443731,
      -0.7663025019644482,-0.850423091629073,-0.9127051151953394,
      -0.9446333569667822,-0.9460408841414493,-0.6585464808986428,
      -0.7477446091210315,-0.8224367335791617,-0.8711767663146228,
      -0.8890600090241656,-0.624789307379154,-0.7038755707064229,
      -0.7649345287347953,-0.7996789024113027,-0.570962738954507,
      -0.6365977473321981,-0.6837028657947305,-0.5018423792963483,
      -0.5542516901414858,-0.4254957770577467,-0.648822290296783,
      -0.6461886815965452,-0.6283026738755576,-0.5912852336175236,
      -0.5351200741127089,-0.4647903048152701,-0.3882830681102161,
      -0.7431455981686927,-0.7416382763983708,-0.7194745213161431,
      -0.6726721034791168,-0.603498627991077,-0.5203744826993632,
      -0.4336588493758105,-0.8332052486459366,-0.828474359608384,
      -0.7972278976396602,-0.7372273383122565,-0.6545105158801499,
      -0.5606846159713372,-0.9080059008549388,-0.8943581597155409,
      -0.8500103413150977,-0.7764211405289977,-0.6833452492402967,
      -0.957414401308411,-0.9304983201224565,-0.8725103749365023,
      -0.7889189186796404,-0.9770470988810096,-0.9362839635412132,
      -0.8680388569279747,-0.9700251729575815,-0.918444056010861,
      0.3130823317650696,0.1854445580462805,0.03612501838448451,
      -0.1274731492430557,-0.2921722054174978,-0.4433596869123763,
      -0.5708395935644333,0.2348779693026915,0.08967582567032931,
      -0.07603665789539318,-0.2502419786720302,-0.4161249577436337,
      -0.5591593471944528,0.1405944428773656,-0.02046318130327168,
      -0.1971193271182073,-0.3730652190141688,-0.5305955532581022,
      0.03400663448859465,-0.1372155522896737,-0.3155629354201782,
      -0.4830545524260225,-0.07686580556220674,-0.2494045186454276,
      -0.4194486169997139,-0.1821860518397986,-0.3471084060868513,
      -0.2742390008535104,0.1282040764044601,0.01849789837253189,
      -0.1065967544617238,-0.2398324745610959,-0.3698196073694229,
      -0.4850253879196206,-0.5783919486276667,0.0150226853040112,
      -0.1119113111904829,-0.2515228306741237,-0.3922511186515919,
      -0.5199342604026214,-0.6239982016489247,-0.7012450602090677,
      -0.1139410284132941,-0.2554296773980814,-0.4031586053186698,
      -0.542022233559877,-0.6580467871742169,-0.7446552996299098,
      -0.251184596582457,-0.4001969364777949,-0.5460646790820674,
      -0.673147094708149,-0.7711330012738366,-0.384955766757247,
      -0.5316548106571598,-0.6658605062472989,-0.7749170668851217,
      -0.5033860614642302,-0.6389055531086461,-0.7556717637535928,
      -0.5992432078251344,-0.718621161033365,0.6461880602442403,
      0.6283021483764009,0.5912847908960939,0.5351196688462319,
      0.4647898820036706,0.3882825918191143,0.5284642450992455,
      0.4960034056042855,0.4454222319691665,0.3800276076057463,
      0.3072006168941325,0.3815325789989955,0.3347740585202575,
      0.2747735598444634,0.2075688995229241,0.2126093137747108,
      0.156329837645353,0.09467242600265936,0.03661534005366113,
      -0.02137253427038529,-0.1295837823093738,0.890040216351073,
      0.808012285288426,0.696135219402544,0.5596336911375484,
      0.4106183982029223,0.263236651429863,0.9184439709263976,
      0.8505076927737054,0.7506591839147216,0.6200379306780087,
      0.468796032975448,0.3127569623094897,0.1662794974092329,
      0.868038934491271,0.7825908968870643,0.6639295465452925,
      0.5181065730349851,0.3597960374562562,0.2057154129295492,
      0.7889191397304282,0.6849812234203291,0.5511465043193342,
      0.3983516721236944,0.2429125431986682,0.6833454958120994,
      0.5645910307529023,0.4236593282043838,0.2738404383453708,
      0.5606847158751149,0.4345981000668111,0.2957551060736421,
      0.4336586612857497,0.3082977736038213,0.6239973336558853,
      0.5199334023086634,0.3922504194657625,0.2515223878822021,
      0.1119111251755824,0.6580459058109899,0.5420214442749262,
      0.4031580569915867,0.2554294261048449,0.6731462609902346,
      0.5460640349170377,0.4001965955151434,0.6658598068272309,
      0.5316543804516458,0.6389050615461086,0.9240668881257947,
      0.9460408367508209,0.9446334423554673,0.9127052977657962,
      0.8504232826059466,0.7663026047198382,0.672662821600414,
      0.8799221274953846,0.8890601671224364,0.8711770865982926,
      0.8224371418546537,0.7477449864099799,0.6585467268780121,
      0.8072873085337188,0.7996793702817474,0.7649351423426319,
      0.7038762137269466,0.6247898512026207,0.7078366234465509,
      0.6837036520525703,0.6365986074780202,0.5709635410688519,
      0.5903935068116462,0.5542526992524547,0.5018433659765416,
      0.4673975349060661,0.4254968708628413,0.3496833143594963,
      -0.3127570262656423,-0.4687960400735756,-0.6200379371399192,
      -0.7506592271179455,-0.8505077714673068,-0.3597961933958567,
      -0.5181066147650901,-0.6639295189615018,-0.7825908374998334,
      -0.3983518694010773,-0.5511465125252933,-0.684981089854631,
      -0.4236595256885799,-0.5645909758908611,-0.4345982861939252,
      0.5591591752824973,0.4161246264414231,0.250241497477734,
      0.07603608646389884,-0.08967640497055197,-0.2348784900769719,
      0.5305956825248687,0.3730651765875013,0.1971191044524291,
      0.02046282792852898,-0.1405948497186229,0.4830549755697658,
      0.3155631609030148,0.1372155601346925,-0.03400679840825552,
      0.4194492775199674,0.2494049520909613,0.07686599366344003,
      0.3471092245505432,0.1821866253122765,0.2742399078697009,
      -0.9953267028881266,-0.996299919654964,-0.9652453855328264,
      -0.9020897348256817,-0.8155884604797163,-0.9923930940648169,
      -0.9780963403965233,-0.9293563919654181,-0.851077922468494,
      -0.9572995816653358,-0.9253139445404677,-0.8614964761454963,
      -0.8903691794182322,-0.843264195535517,-0.8006810953746416,
      -0.3072011182546949,-0.3800281004182731,-0.4454227426500373,
      -0.49600396162629,-0.5284648561961354,-0.5443651710562558,
      -0.2075693133952311,-0.2747739903004954,-0.3347745172062594,
      -0.3815330693216025,-0.4131153608788349,-0.09467260192740248,
      -0.156330035009575,-0.2126095377726251,-0.2586673060001281,
      0.02137271473539107,-0.03661518331511127,-0.09162936047901911,
      0.1295843445482322,0.07317846937762097,0.2226588286505751,
      -0.4804235981897006,-0.3335859124413625,-0.1663071857914592,
      0.006502456545087328,0.1684116947040793,-0.3655174937182495,
      -0.2022576822989345,-0.02631172536408047,0.1446150543700356,
      -0.2312502717555327,-0.05935158437557636,0.1145888807063153,
      -0.08929234550841479,0.08075182856000082,0.04657066968406794,
      -0.4557591367849598,-0.3435685084916551,-0.2129030384298065,
      -0.07515702237270272,0.05632785157141965,0.171902693522397,
      -0.3053152988366306,-0.1770483103705765,-0.03818462683571519,
      0.09730196068160382,0.2175991175749613,-0.1351028913474505,
      0.001396275074369462,0.1378474039188301,0.2612813355040192,
      0.03960267796661452,0.1738084474791471,0.2983321411535529,
      0.2022902215891641,0.3254642982361922,0.3420368325512359,
      0.5203743808147369,0.603498584268205,0.6726720548002406,
      0.719474396239089,0.741638033276861,0.6545106949482685,
      0.7372275485364457,0.7972280664875324,0.8284744266242357,
      0.7764214244136933,0.8500106167743341,0.8943583778920716,
      0.8725105875877179,0.9304985104221231,0.9362840195367765,
      0.3187873126271887,0.4751176460530464,0.6264663831892676,
      0.7569806648964799,0.8565378137633573,0.3724393069541885,
      0.5311885746144091,0.6770114116120104,0.7952337966915513,
      0.4176372345893317,0.5707694457794332,0.7042664250854263,
      0.4489457715859918,0.5898772631309003,0.4647496441635906 };

    auto      pointshell
        = new vnl_matrix_fixed<double, 3, 492>(coords);

    return pointshell;
  }
};


template<> class PointShell<642, vnl_matrix_fixed<double, 3, 642> >
{
public:
  static vnl_matrix_fixed<double, 3, 642>* DistributePointShell()
  {
    double coords[3*642] = {
      0.9756767549555488,0.9936273045965159,0.9968737649581846,
      0.9799612120178374,0.9397937974509784,0.8775847359538656,
      0.7989268505692991,0.7116707502187636,0.6231965669156312,
      0.9839439883131955,0.997606269193783,0.9928389431247303,
      0.9642927845075879,0.9105363959854476,0.8355633282696746,
      0.7474316403407526,0.6549945735140008,0.9764792677814802,
      0.98203616585628,0.9657129396229355,0.9235205478703636,
      0.8567905682573015,0.7723352820545265,0.679558714000995,
      0.9483175267842808,0.9421385440112533,0.9121470670895898,
      0.8569526200134567,0.7808692882316954,0.6923761132322002,
      0.8971014254962764,0.8773049187483365,0.8343457228904876,
      0.7697920647699087,0.6899186612252972,0.8248449061833976,
      0.7923517869921962,0.7399265426789414,0.6712800260218211,
      0.7377425905504456,0.6958798370509692,0.6387690340353133,
      0.6438863846345996,0.5968935863094567,0.5505632701289798,
      0.4375940376503738,0.3657089664269339,0.2787444080680243,
      0.1782978006253255,0.06907711216819172,-0.04176369281672855,
      -0.1467487371786262,-0.2403654598146251,-0.3200730131503601,
      0.4914658799359538,0.415730180051685,0.322900981562039,
      0.2155490838668689,0.09999803429967301,-0.01516787889370763,
      -0.1220279512352428,-0.2156209889399338,0.5436064763029955,
      0.4631956410318647,0.3641181930198064,0.2503784412304452,
      0.1299861910861632,0.01243715490308354,-0.09463328683104604,
      0.5892511755529848,0.5032632034506475,0.398083833846048,
      0.2794416963056139,0.1566100659016669,0.03909742267488438,
      0.6235168560971455,0.5315692770219257,0.421472539809791,
      0.300467209121473,0.1781713819192711,0.6431587521468424,
      0.5460109940101698,0.4333214055893837,0.3130665794035795,
      0.6478378002177454,0.547462575328819,0.4350913076613168,
      0.6399334332955604,0.5389210845670883,0.3200724178002418,
      0.4287826750526064,0.543583002477908,0.6574820925009807,
      0.7616224484252742,0.84789985457486,0.9117594243665534,
      0.9532148993238304,0.3721063927495636,0.4892591428502506,
      0.6101418466998186,0.7256202393614284,0.8256057937005235,
      0.9026705854615993,0.9545776898360481,0.4242146396073646,
      0.5469136809977159,0.6694499319211135,0.7811234805255704,
      0.8720427891122669,0.9369519951934821,0.4722731360977363,
      0.595952912761318,0.7145554586697968,0.8172157093145741,
      0.8959095653312069,0.5117469206751248,0.6311048134383331,
      0.7406754453578036,0.8310759079383635,0.5392182936767531,
      0.6497579584113991,0.7472675269245613,0.5536544075493044,
      0.6528511281517397,0.5564810307882812,0.2502538949373057,
      0.2918876561725356,0.3336803609057232,0.3723595102048779,
      0.4043068999720243,0.4267717639621923,0.438887490788423,
      0.4417451579814044,0.3636248847448722,0.4137579266740147,
      0.4613876866690337,0.5016743933949674,0.5302454776050456,
      0.5449783839335098,0.5467236437808158,0.5384586718996058,
      0.4847692553550146,0.5414178984405463,0.5913899314506929,
      0.6286876371986284,0.6492812914734455,0.6528721916210272,
      0.6425800233106177,0.6067853073481849,0.6657991451456106,
      0.7129452846304364,0.742280692335758,0.7515124484581964,
      0.7428160470273413,0.7205836671095894,0.7764326916323942,
      0.8155579470101821,0.8335749126408774,0.8306996157029517,
      0.8174816717930142,0.8650995515806045,0.8929631510323212,
      0.8991001659430972,0.8921547074343706,0.9286306144549926,
      0.9447583574027534,0.943906557149077,0.9688828514077716,
      -0.4375932291383154,-0.3657083101776577,-0.278743955138478,
      -0.1782975892372044,-0.06907715072573903,0.04176343253822806,
      0.1467483093137361,0.2403649238452659,-0.334979155028856,
      -0.249113234922861,-0.1483246071802934,-0.03654010765740752,
      0.07901069585299947,0.1897435516910367,0.2886438308249422,
      -0.2140239599986307,-0.114043112737167,-0.001406336898152031,
      0.1172166109444462,0.2327251986251803,0.3367145075141528,
      -0.07787968046152194,0.03376382657255556,0.153308912419789,
      0.2719508349268641,0.3804163499145621,0.06640211724156461,
      0.1844665658852621,0.3039508137298482,0.4155682770935452,
      0.2091265361205084,0.3268694132891478,0.4395588586722926,
      0.3409081262151013,0.4522359229770948,0.4554687573578106,
      -0.6231952788307174,-0.5482281060337162,-0.4546947222788382,
      -0.3434835709283097,-0.2192092047974508,-0.08979359660521438,
      0.03581973661208818,0.1504629769958484,-0.538919991494099,
      -0.4498208454617149,-0.3413996198032652,-0.2168595871258214,
      -0.08365072681681855,0.04833087340999961,0.1701142201456619,
      0.2762195992594308,-0.4350905112236564,-0.3310154747857772,
      -0.2089153242611869,-0.07495576041016457,0.0609953219830862,
      0.1886252092359959,0.3006812043050254,-0.3130661573786132,
      -0.1955639168315415,-0.06402306061993944,0.07274927837670857,
      0.2040627320686719,0.3211588705998562,-0.1781713502362079,
      -0.05170714345574682,0.08257031568712175,0.2147106704082434,
      0.3352293490184233,-0.03909771946676634,0.08986774257262679,
      0.2197779802638865,0.3414374432933526,0.09463277722176672,
      0.2196681717134536,0.3398901915912972,0.2156203893390849,
      0.3320690152456851,-0.2502532651867688,-0.2918869903084482,
      -0.3336796480610522,-0.3723587421891308,-0.4043060780204903,
      -0.4267709034446925,-0.4388866188864613,-0.4417443051644412,
      -0.1601455684487161,-0.1971059987993399,-0.2343857490409392,
      -0.2689083414095461,-0.2974793723400317,-0.3179763028669095,
      -0.3300715220338927,-0.05621340666038893,-0.08741483463411388,
      -0.1197552832780665,-0.1507029759524935,-0.1776465335448071,
      -0.1988689416472687,0.05815084571042724,0.03249724572817093,
      0.004031222699520391,-0.02530412585141796,-0.053215898827698,
      0.1765178736510398,0.1546289422059015,0.1277105869872963,
      0.09748683381429475,0.2907448784422404,0.269907109295262,
      0.2420435658170933,0.3935116760630417,0.371277608971055,
      0.4804450046073554,-0.975676581463901,-0.9936272216648101,
      -0.9968737272214637,-0.9799611311880234,-0.9397935597648752,
      -0.8775842433730664,-0.7989260596435309,-0.7116696803542213,
      -0.9688826210954523,-0.9815699711830299,-0.9760365766523995,
      -0.9470636842643547,-0.893307090661199,-0.8187603857237982,
      -0.7313944919484914,-0.6399321926850757,-0.9447580743829742,
      -0.9484314744303296,-0.9308031701048347,-0.8881406289392687,
      -0.8218804228775519,-0.7387299092709896,-0.6478366382006043,
      -0.8990998280625567,-0.8904516044143657,-0.8590773924437412,
      -0.8038827695098977,-0.7291818553952708,-0.6431576879244336,
      -0.8306992108488227,-0.808389020899373,-0.7645261844287355,
      -0.700875860490056,-0.6235158879179115,-0.7428155559569007,
      -0.7083401623912103,-0.6559147788974127,-0.5892502820769048,
      -0.6425794288509028,-0.599662429956479,-0.543605629081954,
      -0.5384579674699886,-0.4914650570197842,0.5482295401850756,
      0.4546962564744617,0.3434851238806179,0.21921067448898,
      0.08979489147188233,-0.03581866880653414,-0.1504621418996611,
      0.5734097649981191,0.4708926749454921,0.3496406892388488,
      0.2164317110563474,0.08116185373706966,-0.04652577585176784,
      0.5899999525047667,0.4779576265881325,0.3476203004228133,
      0.2080468003445843,0.07035894736836774,0.5939040461937424,
      0.4730191144791596,0.3362467154220247,0.1942772308316476,
      0.5828267863796578,0.4555126388719097,0.3164089231485616,
      0.5575911119751512,0.4276808797198417,0.5218578151430378,
      -0.4287830717591774,-0.543583197615373,-0.6574821123382336,
      -0.7616223457476226,-0.8478996902559745,-0.9117592443356049,
      -0.953214723425574,-0.332069490448325,-0.446630150856391,
      -0.5654761604524486,-0.6798202351943792,-0.7798056854765759,
      -0.8580045844734643,-0.9119481688022204,-0.943906345398722,
      -0.3398905505309818,-0.4575823132651822,-0.5766491384642831,
      -0.6870731075595852,-0.7792417673233771,-0.8476201697382868,
      -0.8921544952781482,-0.3414376861695784,-0.4585532096219049,
      -0.5734800429475671,-0.6761401742385725,-0.7585095042345053,
      -0.8174814835779161,-0.3352294641675141,-0.4479051019866874,
      -0.5550737397870624,-0.6478759583634458,-0.7205834943870382,
      -0.3211588327497382,-0.4264290282814993,-0.5239384830804104,
      -0.6067851042454853,-0.3006809838192767,-0.3970753303611698,
      -0.4847689570296102,-0.2762191758252563,-0.3636244354414296,
      0.7313954822136002,0.8187610780655567,0.8933074970084837,
      0.9470638862390964,0.9760366948920106,0.9815701139230523,
      0.7387307994522452,0.8218810203164033,0.8881409840733967,
      0.9308033931946534,0.9484316859524741,0.7291826478779969,
      0.8038833043893007,0.8590777515266519,0.8904519038272285,
      0.7008765828222415,0.7645267056644409,0.808389434193021,
      0.6559154701692661,0.7083407113717404,0.5996631212693852,
      -0.3721068299668963,-0.2886442143937944,-0.1897438268706112,
      -0.07901079510230862,0.0365402424927288,0.1483250002713656,
      0.249113868623751,0.3349799829238154,-0.4242149221746557,
      -0.3367147430533525,-0.2327253235991419,-0.1172165489233448,
      0.00140664545831139,0.1140436844250324,0.2140247669633664,
      -0.4722732967958943,-0.3804164687039511,-0.2719508356433764,
      -0.1533087105593531,-0.03376336567881106,0.07788040640138937,
      -0.5117470172320778,-0.415568329019231,-0.3039507303469009,
      -0.1844662592618953,-0.06640153973221416,-0.5392183881102406,
      -0.4395588911842847,-0.3268692828205596,-0.2091261596739739,
      -0.5536545418185335,-0.4522359631907089,-0.3409079709201472,
      -0.5564812164049491,-0.4554688095204276,-0.5505634949474448,
      0.911948358176072,0.8580047480914766,0.779805799941989,
      0.6798202570778734,0.5654760412600151,0.4466298574551351,
      0.847620327622159,0.7792418677670352,0.6870731334410928,
      0.5766490604842006,0.457582102461728,0.7585096116294638,
      0.6761402059852791,0.5734799960054555,0.458553071765391,
      0.6478760427254714,0.5550737528785307,0.4479050525999325,
      0.523938612103898,0.42642910652745,0.3970755800133287,
      0.330072407434952,0.3179772211416992,0.2974802928098944,
      0.268909235983653,0.2343866000755754,0.1971068007343541,
      0.1601463238235379,0.1988698061348988,0.1776474313398276,
      0.1507038783156887,0.1197561650892531,0.08741568024186514,
      0.05621421006516834,0.05321666456243916,0.02530491044308729,
      -0.004030439768117324,-0.03249647940113914,-0.05815010397008495,
      -0.09748624755051107,-0.1277100030888262,-0.1546283656290311,
      -0.176517304179968,-0.2420432126719758,-0.2699067748969105,
      -0.2907445498638314,-0.3712774990883263,-0.3935115910448358,
      -0.4804451115014433,0.1220274770108985,0.01516759063021488,
      -0.09999808166532918,-0.2155488634876903,-0.3229005078918639,
      -0.4157295012949034,-0.01243745747761582,-0.1299862309792015,
      -0.2503781938406115,-0.3641176806190365,-0.4631949245146199,
      -0.1566100780036334,-0.2794414004304853,-0.3980832592651996,
      -0.5032624248731086,-0.3004668503638805,-0.4214718858755578,
      -0.531568413027702,-0.4333206720101446,-0.5460100361038957,
      -0.5474615360697464,0.04652671746835695,-0.08116072547370122,
      -0.2164304342992144,-0.349639340464782,-0.4708913462876021,
      -0.5734085330236464,-0.6549934813681663,-0.07035800928464173,
      -0.2080457563809749,-0.3476192098203675,-0.477956563666828,
      -0.5899989806009898,-0.6795578679273953,-0.1942764317499744,
      -0.3362459041917389,-0.4730183457316648,-0.5939033648561503,
      -0.6923755409289452,-0.3164083741856072,-0.4555121490743896,
      -0.5828263835645211,-0.6899183532182501,-0.4276806208272582,
      -0.5575909387515523,-0.6712799366842756,-0.5218578089727968,
      -0.6387691005409376,-0.5968937515401853,-0.5467229076134115,
      -0.5449776495480215,-0.5302447815086285,-0.5016737622437388,
      -0.4613871281827766,-0.4137574314122305,-0.6528715777859069,
      -0.6492806993353901,-0.6286871066630142,-0.5913894852863821,
      -0.5414175353843733,-0.751511947518686,-0.7422802285333175,
      -0.7129448986517994,-0.6657988547064399,-0.8335745044290338,
      -0.8155575864896008,-0.7764324183665692,-0.8929628157948554,
      -0.865099271542177,-0.9286303415753032,-0.7474308144934858,
      -0.8355628131811949,-0.910536173478481,-0.964292764650283,
      -0.9928389973443023,-0.9976062839252843,-0.9839438984088483,
      -0.7723347377989339,-0.8567903368462096,-0.9235205630598494,
      -0.9657130735429578,-0.9820362841720375,-0.9764792804081987,
      -0.7808690244142814,-0.8569526309007678,-0.9121472452874007,
      -0.9421387497441781,-0.94831764697765,-0.7697920349927029,
      -0.834345897568434,-0.877305173541843,-0.8971016370404784,
      -0.7399266733270721,-0.7923520436827605,-0.8248451738745377,
      -0.6958800576642888,-0.7377428728490714,-0.6438866478040335,
      -0.1701136087322269,-0.04833004892467691,0.08365175264713676,
      0.2168607561552602,0.3414008427548977,0.4498220342255287,
      -0.1886248334720497,-0.06099477310985241,0.07495646481991973,
      0.2089161290281741,0.3310163070583758,-0.2040625875734068,
      -0.07274901933354507,0.06402341598693255,0.1955643282951042,
      -0.2147107402706368,-0.08257033948198818,0.05170715566660035,
      -0.2197782438454324,-0.0898680254606548,-0.2196686109085476,
      -0.9545775709750491,-0.9026704336737921,-0.8256056375543351,
      -0.7256201416180276,-0.6101418826154756,-0.4892593689607926,
      -0.9369519509430696,-0.8720426859708773,-0.7811233602703923,
      -0.6694498734011487,-0.5469137667342426,-0.895909603120296,
      -0.8172156710096443,-0.7145554011223244,-0.595952924378129,
      -0.8310760190279151,-0.740675476302381,-0.631104833094877,
      -0.747267688498931,-0.6497580495472918,-0.6528513129029779,
      -0.1778161375899129,-0.107732211558371,-0.02718233037986957,
      0.06111658446407255,0.1521341729788544,0.2395836558807668,
      0.3178872248051531,0.3837865324362482,0.4366423972091847,
      -0.06488531255731433,0.01813723762238458,0.1103935391881494,
      0.2069070831033206,0.3006175096647589,0.3845630878227619,
      0.4542522939636228,0.5085532875085977,0.06305939500089614,
      0.1581936380649512,0.2592762165236786,0.3589842883205384,
      0.4491542846588709,0.5237530566427051,0.5806638189836385,
      0.2011334080862844,0.3049347792866978,0.4093092940034627,
      0.5055265527628903,0.5860660664928973,0.6473011578337731,
      0.3410385404813853,0.4476590365909358,0.5484671467389891,
      0.6350798717696985,0.7022080829625521,0.4729450152996533,
      0.5757634189631973,0.6671532416651529,0.7406455836915592,
      0.5886123721388867,0.682355586840312,0.761147081998915,
      0.6837147822732754,0.765634519258382,0.7580253719184177,
      -0.5980610514568759,-0.5477295977070786,-0.4818747107897384,
      -0.4002556178808948,-0.3056524524744078,-0.2038805884680659,
      -0.102179801407443,-0.006890393767211274,0.07805400320688782,
      -0.4999225208961594,-0.4363106684051463,-0.3565940723019476,
      -0.2625351309487317,-0.1594214993217877,-0.0549134894376971,
      0.04356549235716502,0.1310565626039044,-0.3811861908796716,
      -0.3032604317093944,-0.2105722545208477,-0.107535321253739,
      -0.001641189844283069,0.09898039401487412,0.1883562467232593,
      -0.2440298576445608,-0.1531593370054226,-0.05138446179122957,
      0.05448758952315854,0.1561816678501543,0.2469078854412431,
      -0.09487972868743588,0.004642223416678918,0.1091686599118738,
      0.2108696280700075,0.3025309778153461,0.05649572496138416,
      0.1584211910021343,0.2589814921571267,0.3510585742974243,
      0.1998858588754617,0.298075413103841,0.3897335832229792,
      0.3277592945902335,0.4178993211010856,-0.07805410188638827,
      -0.09654989477487182,-0.1156801299839266,-0.1341453490447077,
      -0.1503967869358663,-0.1631208476237052,-0.1717048170733874,
      -0.1763517237002561,0.02821687967586119,0.015882211907606,
      0.001803084777968777,-0.01336575229369984,-0.02858037077889998,
      -0.04271047578209198,-0.0549244526833743,0.1470965334761953,
      0.140993103021656,0.1310545667952879,0.117197412734933,
      0.100226360425432,0.0816416498682333,0.273572290929994,
      0.2720829158426912,0.2637397329328206,0.2481180890825449,
      0.2264390031249597,0.3996773221825281,0.3995925724178936,
      0.3894173756029716,0.3691632841122737,0.5164083739773968,
      0.514049378265622,0.4992114966670869,0.6166247865338805,
      0.609293992020038,0.6970151178651957,-0.9161616153729887,
      -0.9347343784055889,-0.93968222548459,-0.9257960724039628,
      -0.8900379255247358,-0.8334050053153379,-0.7610423024299814,
      -0.6802788134766146,-0.8787071877689653,-0.8904371961034525,
      -0.8856636768163644,-0.8596394026846126,-0.8111262103564944,
      -0.743728138896018,-0.6646636793709766,-0.581842306375343,
      -0.8216809647863927,-0.8231596650234216,-0.8059784419089511,
      -0.7670189419545808,-0.7076799521032789,-0.6339122309388853,
      -0.5537209611836944,-0.7427083018941957,-0.7314571731747196,
      -0.7012133398208896,-0.6514023484259069,-0.5859175101354265,
      -0.511730458968065,-0.6430242175142256,-0.6188528748169413,
      -0.577817181345379,-0.5218263992188138,-0.4560492515462431,
      -0.5282587532766855,-0.4937121240962341,-0.4464002188664107,
      -0.3896719563417937,-0.4070385900414059,-0.3663927387120716,
      -0.3177184789930138,-0.2880885834764299,-0.2456792441590282,
      0.5980613131647216,0.5477298128889157,0.4818748986235024,
      0.4002557953563768,0.3056526230838691,0.2038807357406828,
      0.1021798945147061,0.006890401333784563,0.6591961567279382,
      0.6058950950047038,0.5342799125746878,0.4447327915277979,
      0.3416191699708766,0.2325993338657758,0.1260188711040966,
      0.7166389355513205,0.658631931443462,0.5797448495418049,
      0.481678385980196,0.3708138719979267,0.256391213430893,
      0.7645097528661115,0.6997519467043368,0.6125989595293794,
      0.5067270065580872,0.3904111886835731,0.797087559189558,
      0.724147921463701,0.6291765311790336,0.5179208049695498,
      0.8109706900869452,0.7300074788244177,0.6294473765659205,
      0.8064720598658887,0.7194304887783171,0.7871548899782018,
      -0.4366431953633789,-0.5296368489080719,-0.6250679008877252,
      -0.7161969218969775,-0.7951587738972527,-0.8554711457309036,
      -0.894357332660672,-0.9132140384692056,-0.4179000270237177,
      -0.5130217272422377,-0.6088499338150114,-0.697440817161931,
      -0.770571681316741,-0.8228096498409901,-0.8533624251889405,
      -0.8654069300758386,-0.3897342152274581,-0.4841961940355134,
      -0.5770900179397438,-0.659947012990183,-0.7252694054566711,
      -0.769475954957455,-0.7936687210308288,-0.3510591528670133,
      -0.441802940329269,-0.528708833131731,-0.6037960859260583,
      -0.6611957546281005,-0.6992451996033126,-0.3025315118038733,
      -0.3871149210682015,-0.4662800296194353,-0.5333768673612823,
      -0.5843857415998933,-0.2469083611836815,-0.3241948947822677,
      -0.3955149325344624,-0.4558201674611825,-0.1883566309643827,
      -0.2585114083922621,-0.3230016809603438,-0.131056817682024,
      -0.1949865131648375,0.9161619708326186,0.934734728353398,
      0.9396825386256142,0.9257963253168833,0.8900381161951118,
      0.8334051579656884,0.7610424573388179,0.6802790093210535,
      0.95606107640987,0.9727986783919703,0.9719596947667498,
      0.9481265951474337,0.8996133652092656,0.830024065837643,
      0.7470250693356176,0.98459889389352,0.9957514224668438,
      0.985272860446651,0.9487272931457502,0.8869743499570832,
      0.8065039787728745,0.9954874564049441,0.9969183323524673,
      0.9737757915263076,0.9239648191561672,0.851378736753008,
      0.9840623116657442,0.9728010399724041,0.9364058649128151,
      0.8757746689964236,0.9495572492185336,0.9251912324727678,
      0.8778794078487933,0.8957919700911449,0.8605602916932662,
      0.8295641154807277,0.1778171059718252,0.1077330169440829,
      0.02718289859111821,-0.06111631614771156,-0.1521342296531137,
      -0.2395840100830868,-0.3178878068603045,-0.3837872586817543,
      0.2456801528075706,0.1743614211692323,0.0913009251307558,
      -9.150547636147518e-005,-0.09380226130703311,-0.1828695547719357,
      -0.2617550281067236,-0.3277601386896922,0.31771932061924,
      0.2451950112624202,0.1597779786826127,0.06571158217888019,
      -0.03010074909289372,-0.1203656141836725,-0.1998866892639353,
      0.3896727248396884,0.31551207359359,0.2277347108654265,
      0.1315171099293629,0.03437982180295363,-0.05649645964725484,
      0.4560499397602033,0.379603102905924,0.2896405921772683,
      0.1921815993207149,0.09487917246041,0.5117310565618495,
      0.432706773459909,0.3413166137711556,0.2440295330914618,
      0.5537214556082512,0.4726319336973814,0.3811861035860583,
      0.5818426869442771,0.4999226354150897,0.5296361374668247,
      0.6250673427588886,0.7161965757306298,0.7951586652134927,
      0.8554712518134584,0.8943575917970892,0.9132143767653008,
      0.6095437192932451,0.7099831823616649,0.80114225201565,
      0.874273305605062,0.9239434255089938,0.9498851810027883,
      0.6864630907143527,0.7872122199392447,0.8728983750907815,
      0.9353918603084618,0.9717433027190985,0.7529076094482211,
      0.8481358928542101,0.9232232087908666,0.9723005987504904,
      0.8019213565969565,0.8865248509843126,0.9481833022791925,
      0.8298631139894471,0.9011831013407439,0.8376467778090524,
      0.09654990728233075,0.1156802730367007,0.1341456406128616,
      0.1503972383762518,0.163121458312337,0.1717055726654157,
      0.1763525997352119,0.1949861891736041,0.2217704348994384,
      0.2472029328538624,0.2686947191514402,0.2839095008331465,
      0.2917169688378368,0.2925778501087149,0.2880893594612404,
      0.3230012075937738,0.3570186650821726,0.3862979863651042,
      0.4071207791101573,0.4171265008378126,0.4163707105097114,
      0.4070391829394841,0.4558196682853914,0.4939031399298617,
      0.5227372695226704,0.5383590521019039,0.5395474601226077,
      0.5282592036287527,0.584385354546882,0.6217221861558012,
      0.6452876111171104,0.652151703940053,0.6430245879023526,
      0.6992450234541641,0.7309800986736478,0.7458180696675939,
      0.7427086516187589,0.7936687790069655,0.8166219571778315,
      0.8216813253645509,0.8654071763861386,0.8787075572966508,
      0.2617543171369139,0.1828690638929441,0.09380207139517749,
      9.165794599607253e-005,-0.09130044825545007,-0.1743606863382632,
      0.1203649946594492,0.03010042800399446,-0.06571155052616354,
      -0.159777601870924,-0.2451943556628112,-0.03438026636045293,
      -0.1315172003159161,-0.2277344419143756,-0.3155115060038017,
      -0.1921818096757193,-0.2896404384341761,-0.3796026338863787,
      -0.3413165804505149,-0.4327064149448353,-0.4726316954005857,
      -0.02821670874193007,-0.1260185867435537,-0.2325989754896013,
      -0.3416187871825309,-0.444732423910872,-0.5342795725490167,
      -0.6058947680736913,-0.6591958160975573,-0.1470960585711635,
      -0.2563906423085273,-0.3708132697350703,-0.4816778179392841,
      -0.5797443521912047,-0.6586314997511467,-0.7166385361913998,
      -0.2735715277331494,-0.3904103835191622,-0.5067262463544535,
      -0.6125983088644244,-0.6997514194040853,-0.7645093154710443,
      -0.399676350745083,-0.5179198795056302,-0.6291757405426562,
      -0.7241473051258307,-0.7970870947583816,-0.516407321060813,
      -0.6294464664763966,-0.7300067759514388,-0.8109701936315704,
      -0.6166237809711579,-0.7194297010446339,-0.8064715143730887,
      -0.697014250125597,-0.7871542772926711,-0.7580246814516085,
      -0.2925772011093052,-0.2917164801053564,-0.2839091960301776,
      -0.2686946036173254,-0.2472029920942969,-0.2217706416368722,
      -0.4163702688077253,-0.417126245587897,-0.407120730642943,
      -0.3862981411371962,-0.3570189974757899,-0.5395471670390704,
      -0.5383589526996108,-0.5227373833229371,-0.493903460730213,
      -0.6521514808212688,-0.6452875761129986,-0.6217223634380962,
      -0.7458178526549705,-0.730980062865163,-0.8166217206889562,
      -0.7470248243274916,-0.8300238938854384,-0.8996132253994521,
      -0.9481264427436803,-0.9719595014156921,-0.9727984419199607,
      -0.9560608144734745,-0.8065037168656155,-0.8869742038831786,
      -0.9487272157391016,-0.9852727972566903,-0.9957513357128307,
      -0.9845987728990332,-0.8513784827612589,-0.9239647227267276,
      -0.9737757945994621,-0.9969183688503204,-0.9954874793720167,
      -0.8757744250303561,-0.9364058061885231,-0.9728010979637555,
      -0.9840624137666333,-0.8778791504147609,-0.9251911695017921,
      -0.9495573085262712,-0.8605599852861939,-0.8957918541675716,
      -0.8295637303264062,-0.04356566771769104,0.05491337401809196,
      0.1594214223252114,0.2625350815284993,0.3565940574819209,
      0.4363107079656693,-0.09898073214385947,0.001640881330767951,
      0.107535036331357,0.2105720065923513,0.3032602493938258,
      -0.1561821374463407,-0.0544880626634626,0.05138399870775862,
      0.1531589210125483,-0.2108702003917341,-0.109169268647464,
      -0.004642833898907788,-0.2589821529532703,-0.158421915656623,
      -0.2980761626942871,-0.9498849702325406,-0.9239433259414138,
      -0.8742733756226419,-0.8011425227232617,-0.7099836412348224,
      -0.6095443176654687,-0.508553964054817,-0.9717432666228709,
      -0.9353919526972201,-0.8728986221002711,-0.7872126141915534,
      -0.6864635931358892,-0.5806643774075225,-0.9723007209152063,
      -0.9232234443044223,-0.8481362314873311,-0.7529080174160984,
      -0.647301592079592,-0.948183485298304,-0.8865251005609038,
      -0.8019216417165289,-0.7022083677250226,-0.9011831985532854,
      -0.829863222791688,-0.7406456753648975,-0.8376466557989706,
      -0.7611469342493191,-0.7656341022868733,0.6646639924076627,
      0.7437284087719479,0.8111264735787829,0.8596396929680696,
      0.8856640083476979,0.8904375589329932,0.6339126501414293,
      0.707680317140174,0.7670192843558462,0.8059787882713181,
      0.8231600236208387,0.5859180191150533,0.651402785083716,
      0.7012137302542939,0.7314575393289862,0.5218269830903716,
      0.577817672809353,0.6188532957478536,0.446400869266141,
      0.4937126651205253,0.3663934528166599,-0.4542528774843695,
      -0.3845634904875965,-0.3006176376093279,-0.2069068719953542,
      -0.11039298336186,-0.01813638876408994,0.06488637319187028,
      -0.5237534893339546,-0.4491544873380491,-0.358984163651417,
      -0.2592757174096171,-0.1581927886880057,-0.063058272648828,
      -0.5860663263409258,-0.5055265231286077,-0.4093088881435918,
      -0.3049339801812209,-0.2011322717929374,-0.6350799196461491,
      -0.5484668466671915,-0.4476583293687135,-0.3410374453295617,
      -0.6671530327423928,-0.5757628211692016,-0.4729440070010416,
      -0.6823550906513535,-0.5886114742048517,-0.6837139951578899,
      0.8533626369422408,0.8228097648480324,0.7705716341879386,
      0.6974405664026642,0.6088494821992544,0.513021117997808,
      0.7694759566268259,0.7252693007743299,0.6599467606818161,
      0.5770896067011005,0.4841956488153404,0.6611955122025592,
      0.6037957522821978,0.5287083985893539,0.441802418256347,
      0.5333764308745213,0.4662795459560942,0.387114402450423,
      0.3955144253601533,0.3241943959991313,0.2585109731392027,
      0.05492543133462732,0.04271134692005163,0.02858111208036886,
      0.01336634929270618,-0.001802636489293655,-0.01588190769759986,
      -0.08164059192530317,-0.1002253860034311,-0.1171965412658538,
      -0.1310538163159151,-0.1409924872128895,-0.2264379065659517,
      -0.2481170428345225,-0.2637387553592617,-0.2720820325977006,
      -0.3691621981499088,-0.3894163023521484,-0.3995915326289882,
      -0.4992104641187212,-0.5140483226358935,-0.6092930383066186,
      -0.1282040992012935,-0.03314438343075717,0.07418771901881034,
      0.1895805529161077,0.3060111300013609,0.4152646181118619,
      0.5105522497698926,0.5884153641804594,0.6488215902264565,
      -0.1662652220900749,-0.06672909617768248,0.04565193884691998,
      0.1653145024136672,0.2838178720038453,0.392358453375388,
      0.4847688072125908,0.5588878800183537,-0.2061837828129686,
      -0.1028578719693947,0.01320461245153456,0.135055834380525,
      0.2533186742384155,0.3594175117480762,0.4483631157317138,
      -0.2454367954185786,-0.1392470619788042,-0.0212985877993455,
      0.1003748549375823,0.2162630814588294,0.3187734758296685,
      -0.2808945465452747,-0.1730244679184136,-0.05523589088936585,
      0.0639822904424641,0.1758296024995542,-0.309764254307227,
      -0.2016807651564305,-0.08611192469610081,0.02876188488837943,
      -0.3305621657917769,-0.2239243298387429,-0.1123798946526014,
      -0.3434303716667939,-0.2398370895892076,-0.3496821715684524,
      0.6714420577705413,0.7524953419593941,0.8307215647087252,
      0.8988911695232938,0.9496341036793501,0.9781045443142695,
      0.983882155708098,0.9706580078492304,0.9441720387917811,
      0.7131190377206373,0.7979983822227008,0.8766844493347566,
      0.9405391525406509,0.982132974036711,0.9983759032186432,
      0.9915700817355307,0.967642375326243,0.7477894668983293,
      0.8327562120383802,0.9072360592139173,0.9621573628309809,
      0.9915144459980433,0.9950116576093856,0.9775308002021622,
      0.7702158662916526,0.8504518596256061,0.9159087827490561,
      0.958615376965025,0.9752335996497555,0.9682499097067426,
      0.7760312926982584,0.8470020976881563,0.9002461340540874,
      0.9301900108044431,0.9363385958760152,0.7636458947692996,
      0.8226631878609201,0.863227169508762,0.8824665400301602,
      0.7350862725086435,0.7819435892168385,0.8116669810370153,
      0.6950244929211268,0.7313851393296726,-0.9441722324566724,
      -0.8982335639429714,-0.8313456723553714,-0.7414326155290212,
      -0.6303269409870232,-0.504437732215241,-0.3731115220313701,
      -0.2455227591385536,-0.9277610900314076,-0.8719938339703784,
      -0.7922901462160301,-0.6879655695561273,-0.5635230570387244,
      -0.4282074607020305,-0.2928561738545139,-0.8935348618728204,
      -0.8252311012303484,-0.7312055040636086,-0.6132787902902372,
      -0.4790574606800003,-0.3397875802758924,-0.8379237951960332,
      -0.7555203588771385,-0.6479597593660587,-0.5201883296648473,
      -0.3821929206720374,-0.7605084662990487,-0.6648552402767216,
      -0.5474979380988695,-0.4159703173399101,-0.6652563588950895,
      -0.5599712780007364,-0.4386104476633646,-0.5596790773588694,
      -0.4500513701337736,-0.452215421939787,-0.3130828685601688,
      -0.2026653349757201,-0.07519795113003093,0.06519223482424771,
      0.2106381298878163,0.3511437862186094,0.4776948653078811,
      0.5848777234045024,-0.3092743464263315,-0.189541493899996,
      -0.05216564154993419,0.09665867971286488,0.2468076262033025,
      0.3871266155400391,0.5092302530840672,0.6095258724352348,
      -0.299731815412124,-0.1710988752936458,-0.02562616156907596,
      0.1281943739657924,0.2786088475509629,0.4146241489277275,
      0.5296074119458033,-0.2831889282507517,-0.1472477576556685,
      0.002978788421451789,0.1571443739074394,0.3031991278501129,
      0.4316900017871718,-0.2593820240207756,-0.1190520671608967,
      0.03150460300683353,0.1812458940061291,0.3193074202631062,
      -0.229491624834242,-0.08860645786522432,0.05782435035907074,
      0.1994358494400003,-0.1959070295313489,-0.05832274784344905,
      0.08053951964108649,-0.1613858092941266,-0.03012854522533278,
      -0.671442351589465,-0.7524955042654247,-0.8307216077305004,
      -0.8988911324269672,-0.9496340459616871,-0.9781045247295723,
      -0.9838822098556939,-0.9706581405182357,-0.6732379910932457,
      -0.7555353929734253,-0.8321927576733623,-0.8949176301047361,
      -0.9365114268652753,-0.9538841305299445,-0.949106939734586,
      -0.6637939293172054,-0.7437729138119192,-0.8147968652653733,
      -0.8684734875620626,-0.8990751104656249,-0.906028082404147,
      -0.6398902977415805,-0.7135875679260879,-0.7753830615613055,
      -0.8180895331241328,-0.8383689548572043,-0.6002017842451572,
      -0.6645162706136697,-0.7153675946298389,-0.747703849697991,
      -0.5464363016035276,-0.6002045214061834,-0.6407682965760649,
      -0.4830988161196208,-0.5271645300902241,-0.4158550110968205,
      -0.648822290296783,-0.6472486014147985,-0.6344626299567083,
      -0.6075203746111165,-0.5653935344751363,-0.5100061066600627,
      -0.4459169294668335,-0.3786832614419487,-0.7313855414086173,
      -0.7310744930343959,-0.7160642832118092,-0.6830433558919348,
      -0.6318400429346884,-0.5662582509809229,-0.492781618346183,
      -0.418059300054038,-0.8116671044980045,-0.8099276517898805,
      -0.7895071237697285,-0.74756375917153,-0.6857578728711079,
      -0.6101862708902825,-0.5288485365725882,-0.882466459585,
      -0.8755369303175729,-0.8463852653947486,-0.7938122127531423,
      -0.7219269876119373,-0.6386807736786417,-0.9363384293729856,
      -0.920580474049937,-0.8807752703984302,-0.8181738478934094,
      -0.7389956620788968,-0.9682497764059059,-0.9417119830617079,
      -0.8917766410561483,-0.8219783739569782,-0.9775307754986758,
      -0.9407006676234038,-0.8832579304805722,-0.9676424743883053,
      -0.9228815898031446,0.3130823317650696,0.2026646799488628,
      0.07519720122433592,-0.06519302988114704,-0.2106389019045293,
      -0.3511444697687643,-0.4776954195825031,-0.5848781397282268,
      0.2455618355531619,0.1217043818280624,-0.01891222872758066,
      -0.1695413327023869,-0.3196901877897432,-0.4582042349710876,
      -0.5770670118160844,0.1655447705464922,0.0289439344448441,
      -0.1220494268493898,-0.2778581225722938,-0.4262841911656161,
      -0.5567786600361208,0.07498802093385774,-0.07139865292607843,
      -0.2274740800198117,-0.3816395081442781,-0.5218451060581218,
      -0.02151295053653907,-0.1724761631672423,-0.3268548028226648,
      -0.4727737792825113,-0.1175093448034605,-0.2667795638863601,
      -0.413210185645996,-0.2066526242811376,-0.3486960301430706,
      -0.2845979196243639,0.1282040764044601,0.03314425179116901,
      -0.07418801789892716,-0.1895810572320515,-0.3060118317848469,
      -0.4152654547344749,-0.51055312502423,-0.5884161844630592,
      0.03012854220925541,-0.07822075478791662,-0.1975265605100549,
      -0.3210457437412529,-0.4395492890997685,-0.5442335497807274,
      -0.6297193044589443,-0.6950252371327137,-0.08053951945115613,
      -0.2008908777469321,-0.3287498381001709,-0.454849657793549,
      -0.5688642328312289,-0.6631668267266261,-0.7350870708018485,
      -0.1994358711312821,-0.3279452265433291,-0.4583916832350944,
      -0.5800652917395807,-0.683455813944916,-0.7636467367251844,
      -0.3193074905789999,-0.4498985165051642,-0.5758541835274278,
      -0.6869055692486817,-0.7760321386045594,-0.4316901383846216,
      -0.5576908306091364,-0.6732598101682218,-0.7702166526708257,
      -0.5296076162752261,-0.6457739738857279,-0.7477901272839138,
      -0.6095261314476621,-0.7131195245736661,0.6472479688324123,
      0.6344620803203709,0.6075199046816795,0.5653931175085022,
      0.5100057007389891,0.4459164955009389,0.3786827774326808,
      0.5474099886510654,0.5236257914257873,0.4857185197867951,
      0.4345152489333596,0.3738198763531287,0.3091172837171857,
      0.4250511511939136,0.389683753785766,0.3423573476615138,
      0.2859346018498913,0.2253099468416643,0.2835632478824897,
      0.2385758256608239,0.1860028309402729,0.1299534657091715,
      0.1312824242313411,0.08112906023855893,0.02887591783637235,
      -0.02047837605993849,-0.07041365620347247,-0.1612832799773341,
      0.8982333732257657,0.831345524857385,0.7414325451852105,
      0.6303269573377657,0.5044378109359756,0.3731116142445962,
      0.2455228128124526,0.9228814872695982,0.8667983517234625,
      0.7868465046935174,0.6823837598597929,0.5579412946135831,
      0.4227639330815517,0.2876608055402039,0.1613856625640601,
      0.8832579654623612,0.8143439687072217,0.7198955735651289,
      0.6018165967179313,0.467747528435525,0.3289004093680582,
      0.1959067638111305,0.8219785498827571,0.7387750958937861,
      0.6307665078242817,0.5029950256220953,0.3654474931658011,
      0.2294912586325793,0.7389959159190775,0.6425282428469168,
      0.5248781214125594,0.3936431094566906,0.2593815856416469,
      0.6386809855643202,0.5327535820451708,0.4113926019080829,
      0.2831884462269615,0.5288485749239231,0.4188791724037085,
      0.2997313094215869,0.418059069948,0.3092738247890606,
      0.6297184498311313,0.5442326731425132,0.4395485037972626,
      0.3210451478850294,0.1975261966743493,0.07822060158766293,
      0.6631659475585863,0.5688633866559212,0.4548489689298922,
      0.3287493895928871,0.2008906793070462,0.6834549460780991,
      0.5800645299836926,0.4583911438911752,0.3279449596345045,
      0.6869047733719595,0.5758535688432742,0.449898169624651,
      0.6732591535970365,0.5576904114999332,0.6457735063412144,
      0.9277609198709332,0.949106860839256,0.9538841631800392,
      0.9365115581247451,0.8949178072878723,0.8321929059126493,
      0.755535446210326,0.6732379126875105,0.8935348059013593,
      0.9060281564869469,0.8990753265127792,0.8684738109832391,
      0.8147972186077681,0.7437732084300421,0.6637941002839092,
      0.8379239537972928,0.8383692759036656,0.8180900037577743,
      0.7753836155369005,0.7135881068095284,0.639890731966599,
      0.7605089118538748,0.747704461889213,0.7153683254339585,
      0.664517027376214,0.6002024649156598,0.6652570996829634,
      0.6407691682823238,0.6002054473372921,0.5464371824658241,
      0.5596800524090022,0.5271655706244485,0.4830998363360157,
      0.4522165310007144,0.4158561136906575,0.3496833143594963,
      -0.2876608652754351,-0.4227639382532795,-0.5579412897306785,
      -0.6823837835510913,-0.7868465717409772,-0.8667984500087526,
      -0.3289005616515215,-0.4677475887278177,-0.6018165999573247,
      -0.7198955529773995,-0.8143439414326642,-0.3654477029697049,
      -0.5029950893382186,-0.6307664561929963,-0.7387749669914671,
      -0.3936433402523272,-0.5248781506021705,-0.6425281057326593,
      -0.4113928310116375,-0.5327535685472626,-0.4188793967911134,
      0.577066822550539,0.4582039092088336,0.3196897246958904,
      0.169540766109225,0.01891161845986538,-0.1217049732030892,
      -0.2455623627393322,0.5567787306370593,0.4262841209617862,
      0.2778578974521733,0.1220490717256491,-0.02894436515050694,
      -0.165545217365127,0.5218454423524129,0.3816396895816639,
      0.2274740807377008,0.07139849210836301,-0.07498829123147716,
      0.4727743520975624,0.326855199205205,0.1724763529977552,
      0.02151295276860444,0.4132109394322543,0.2667801205874505,
      0.1175096785325448,0.3486969033343195,0.2066532886754674,
      0.2845988618411943,-0.9915701323913838,-0.9983759139464571,
      -0.9821329817122836,-0.9405392168410572,-0.8766846298252051,
      -0.7979987142017131,-0.9950116201913212,-0.9915144412787341,
      -0.9621574590524813,-0.9072363224101446,-0.8327566769720253,
      -0.9752335225013484,-0.9586154363208594,-0.9159090584602802,
      -0.8504523952727223,-0.9301899969464084,-0.9002463663906001,
      -0.8470026365747616,-0.863227339500068,-0.8226636840857314,
      -0.7819440310902068,-0.3091177896659217,-0.3738203674077975,
      -0.4345157440029583,-0.4857190441849699,-0.5236263640902669,
      -0.5474106128468904,-0.5588885443532919,-0.2253103954606045,
      -0.2859350592025221,-0.3423578252345214,-0.3896842610441149,
      -0.4250516888435538,-0.4483636748755489,-0.1299537462879293,
      -0.1860031284708927,-0.2385761459990237,-0.2835635916740772,
      -0.3187738370923436,-0.02887592341002545,-0.0811290830782055,
      -0.1312824709082232,-0.175829673802738,0.07041398450649188,
      0.02047868355036482,-0.02876160928094873,0.1612839336168132,
      0.1123805173372347,0.2398380094723457,-0.5092306348673886,
      -0.3871271308997349,-0.2468082566335953,-0.09665937383397012,
      0.05216495245602301,0.189540870507144,-0.4146244745670597,
      -0.2786093002807504,-0.1281949271302634,0.02562556437243592,
      0.1710982989127918,-0.3031993859046618,-0.1571447546490307,
      -0.00297925984664287,0.147247252029971,-0.1812460903983632,
      -0.03150492197167008,0.119051661271318,-0.05782450627681404,
      0.08860617741886109,0.05832260658459812,-0.4847695337402862,
      -0.3923591556376936,-0.2838184503258238,-0.1653148824643805,
      -0.04565210375030839,0.06672910665806144,0.1662653402185898,
      -0.3594180507425287,-0.253319097564816,-0.1350560618895416,
      -0.01320461850800903,0.1028580486723768,0.2061840662709727,
      -0.2162633298539993,-0.1003749112365929,0.02129875586186771,
      0.139247419944916,0.2454372621958313,-0.06398217348632655,
      0.05523623192861417,0.17302500577526,0.2808952005684321,
      0.08611242071710691,0.2016814632806757,0.3097650809527805,
      0.2239251562628219,0.3305631346577989,0.3434314452768145,
      0.4927814627145429,0.5662581542385627,0.6318399645986254,
      0.683043240780333,0.7160640841376876,0.7310741891309572,
      0.6101863849438733,0.6857580324047194,0.7475639112793486,
      0.78950721141067,0.8099276375885235,0.7219272504872226,
      0.7938124902720932,0.8463855099577528,0.8755371018531016,
      0.8181741141101887,0.8807755242176112,0.9205806914487843,
      0.8917768010350154,0.9417121277771469,0.9407006846750625,
      0.2928563777418555,0.4282076937835696,0.5635232482078705,
      0.6879656610506216,0.7922901195775008,0.8719937126447483,
      0.3397879564876633,0.4790578522946484,0.6132791099941464,
      0.7312056921497674,0.8252311496217807,0.3821934817691987,
      0.520188888877212,0.6479602207299933,0.7555206677931847,
      0.415971059154168,0.5474986596025796,0.6648558465528478,
      0.4386113475958364,0.5599721413103722,0.4500523932965443 };

    auto      pointshell
        = new vnl_matrix_fixed<double, 3, 642>(coords);

    return pointshell;
  }
};


template<> class PointShell<812, vnl_matrix_fixed<double, 3, 812> >
{
public:
  static vnl_matrix_fixed<double, 3, 812>* DistributePointShell()
  {
    double coords[3*812] = {
      0.9756767549555487,0.9922508097125681,0.9977223655964853,
      0.9881046683022422,0.9605519470607568,0.9146195529015363,
      0.8528238339140162,0.7799959774035472,0.7017828679367573,
      0.6231965669156313,0.9837002757017411,0.9971982264747434,
      0.9970195212545997,0.9790578176231792,0.9411837637692063,
      0.8844913295435498,0.8132898830804115,0.733693935416079,
      0.6517368229295557,0.9797764734635961,0.9875546637606186,
      0.9791028733313679,0.9508425500024201,0.9020818068674743,
      0.8358206809420552,0.757892552927626,0.6749593629473696,
      0.9602286065935876,0.9595713376770644,0.9407948278222147,
      0.9016229543668696,0.8432733179563544,0.7704382634077368,
      0.6896667228240099,0.9226923908191266,0.9116587851060941,
      0.8819863001926875,0.833225536765196,0.7683765355083914,
      0.693030195174853,0.8672949635738141,0.8455181303237055,
      0.8062955359163166,0.7509515682311381,0.6835651897016059,
      0.7970712198937139,0.7659650471206068,0.7200326020746473,
      0.6617902410567805,0.7171844959174759,0.6793953685069648,
      0.6300938559406523,0.6333774065241421,0.5918819525326566,
      0.5505632701289798,0.4375940376503738,0.3744427273176402,
      0.2993381563954681,0.2130515669413183,0.1182813882301629,
      0.01954824200640591,-0.07773888316805401,-0.1686715778074173,
      -0.2499271804752164,-0.32007301315036,0.4854882574576871,
      0.4194229010537701,0.3399338752288831,0.2482942679442456,
      0.1481384200976902,0.04502062388977755,-0.05499873367858239,
      -0.1469877513574829,-0.2280773868570659,0.5324004751734927,
      0.4628532125512143,0.3786136701220479,0.2817283713716202,
      0.1769156386035929,0.07062416136199023,-0.03081255771979229,
      -0.1228131994141411,0.5750755655636769,0.5013629074790578,
      0.4122166091807719,0.3107083252171691,0.2025911492637849,
      0.09481562164394035,-0.006505365777837463,0.6099587706972414,
      0.5315747218240751,0.4378921540503243,0.3330794318970117,
      0.2235852429888137,0.1162930447896056,0.6341450324413492,
      0.5511580300049011,0.4540605894555947,0.3478844246654598,
      0.2392124945153528,0.6463093264667951,0.5595982440953389,
      0.46086512448968,0.3555189519163455,0.647046909800018,
      0.5581741842011717,0.4598430939447178,0.6384375755240163,
      0.5492419252084272,0.3200724178002418,0.4163027723501219,
      0.5178612432225475,0.6201479032959074,0.7171164359379398,
      0.8025495773806206,0.8717669667532965,0.9228269911622408,
      0.9565659576168886,0.3662566345969678,0.4693191708546967,
      0.5762858482292442,0.6811797034275579,0.7768921407658641,
      0.8570710368648269,0.9180184145424319,0.9594305591455454,
      0.4128244293574089,0.5208884118196857,0.6304558785383466,
      0.7343501607073976,0.8250439179808262,0.8969571451990621,
      0.9480541165102998,0.4569904905821253,0.5671264332790738,
      0.6755415775491402,0.7744719807282637,0.8569290887080647,
      0.9189090911818834,0.4955267041670984,0.6040846468052238,
      0.7074267562469234,0.7981205092043148,0.8705859012982401,
      0.5255624261223592,0.6289645166670731,0.7241496615411406,
      0.804855834362468,0.5454521871592556,0.6410312785579098,
      0.7264644091774437,0.5551982228864044,0.641627582268501,
      0.5561969667017818,0.2502538949373057,0.2872027754172267,
      0.3245344712904296,0.3600408236839537,0.3911506315330521,
      0.4155633223527059,0.4319414682718341,0.4402542183539166,
      0.4415840008368697,0.3505479885894233,0.3943500788462172,
      0.43685649380825,0.4748347886538213,0.5050141859852789,
      0.5250961060909611,0.5345072780254662,0.5344003893389805,
      0.5269837056592495,0.4574833953114167,0.5069750581222974,
      0.5525393550046852,0.5900243847437713,0.6159403270481748,
      0.6286926742607352,0.6290385665487288,0.6194911060068102,
      0.5665551499276381,0.6191962274832306,0.6644606027057701,
      0.6977977540386979,0.7162925017359711,0.7197189132272425,
      0.7103565350090897,0.6716249671619864,0.7237430894816727,
      0.7648128117488485,0.7907287689487973,0.7998964507309693,
      0.7936885378432519,0.7661563418365048,0.813819159923423,
      0.8474217166041815,0.864080531689031,0.8638072256614686,
      0.8449704612165639,0.8851005033495888,0.9095132315221558,
      0.9168712118355733,0.9055760485508874,0.9366559154796388,
      0.9518640240021961,0.9483465315553231,0.9704010313875515,
      -0.4375932291383154,-0.3744420515634606,-0.2993376542433732,
      -0.213051272338593,-0.1182813170567518,-0.01954838456202594,
      0.07773856031405028,0.1686711209781245,0.249926635788548,
      -0.3473076758598704,-0.2731773531779083,-0.1869784560548725,
      -0.09084077595186728,0.01090483422160223,0.1124324367516528,
      0.2079533392573423,0.2932321828182562,-0.2423880400933935,
      -0.156942680061891,-0.06052677114529338,0.04301840739087177,
      0.1478309256372904,0.2474621200563187,0.336722136779245,
      -0.1245894223683107,-0.02900257303954696,0.07490364945070915,
      0.181648017057727,0.2845287164429474,0.3775439714321544,
      0.00186176882165046,0.104598795853549,0.2116013903531214,
      0.3164139369537128,0.4125877669907546,0.1306304412032901,
      0.2361087796289779,0.3411560209375105,0.4393820957485916,
      0.2546621526726749,0.358132074959863,0.4568650785571312,
      0.3679945821324933,0.465540692618106,0.46700138061564,
      -0.6231952788307172,-0.5574600324007678,-0.4770927236000206,
      -0.3823142165271753,-0.2756046525485073,-0.1617836973833182,
      -0.0470869101281101,0.06243545366421368,0.1623223370722767,
      -0.5492408053824432,-0.4724705019699099,-0.3803275558898422,
      -0.2743443251752258,-0.1587631541423281,-0.04000763487086941,
      0.07495625978661888,0.180495590370777,0.2733676095225912,
      -0.4598422233986804,-0.3712294758837851,-0.2676544688598244,
      -0.1524290150215411,-0.03159943316739083,0.08740013657156358,
      0.1978753137579477,0.2954972412516565,-0.3555183995893311,
      -0.2556130138286056,-0.1427706895826486,-0.02217694060430834,
      0.09888849629266881,0.2130604246988013,0.3149362814931866,
      -0.2392122924568965,-0.1303824105317197,-0.01228273317409482,
      0.1085468771341984,0.2246722751170698,0.3298926161068582,
      -0.116293170907997,-0.002544855969777311,0.1157810363722986,
      0.2317919381153067,0.3389908350437634,0.006504983859799631,
      0.1204470178133221,0.2342680443317885,0.3417324928044827,
      0.122812659812205,0.23268986296373,0.338624103921372,
      0.2280767837862029,0.330902937710777,-0.2502532651867688,
      -0.2872021140968067,-0.3245337697854004,-0.3600400745299252,
      -0.3911498325365141,-0.4155624805143979,-0.4319406005716203,
      -0.4402533481355587,-0.4415831518511793,-0.1708721571930995,
      -0.2041871265271164,-0.2379683866812855,-0.2700974862342881,
      -0.2982093646817494,-0.3203587126138262,-0.3356190104303782,
      -0.3442372407872382,-0.08038054194912378,-0.1091996275570321,
      -0.1389306574098392,-0.1677554863057991,-0.1936713782240709,
      -0.2150838382608105,-0.2312629415986541,0.01921174674137868,
      -0.004986017519443271,-0.03105772275158658,-0.05758606025521312,
      -0.08288952042427381,-0.1055085254918716,0.123925135248786,
      0.1034734912606135,0.07972411299125522,0.05380820853884998,
      0.02732027507363822,0.2282811487692264,0.2098640891542395,
      0.1865989201629623,0.1596028170589836,0.3265627679363224,
      0.3082244742822114,0.2838117948729975,0.4142824694706087,
      0.3944807806873461,0.4890558390601624,-0.975676581463901,
      -0.9922507181739645,-0.9977223240044052,-0.988104613901425,
      -0.9605517932270823,-0.9146192114249708,-0.8528232434185343,
      -0.7799951208906594,-0.7017817703686649,-0.9704008071255719,
      -0.9831225967982262,-0.9822981052334858,-0.9639034131304861,
      -0.9258761899238439,-0.8693365652843744,-0.798567800167315,
      -0.7196174154484367,-0.6384363279989543,-0.951863752574358,
      -0.9581118567763953,-0.948488121885417,-0.9195867065770826,
      -0.8708257888235816,-0.8052054331206489,-0.7284489979298084,
      -0.6470457280542714,-0.9168708930751738,-0.9141083375203286,
      -0.8939111910264987,-0.854235225343551,-0.7963893637954717,
      -0.7249746765437297,-0.6463082287051728,-0.8638068526880216,
      -0.8504293560497204,-0.8194747846295519,-0.7707138739098191,
      -0.7071466871144221,-0.6341440232385527,-0.7936880971710701,
      -0.7697465167330589,-0.7297586716006348,-0.675179694541184,
      -0.609957839140293,-0.7103560109842373,-0.6776367931781411,
      -0.631704229014305,-0.5750746905699001,-0.6194904872158699,
      -0.5808664875389803,-0.53239963484432,-0.5269829891706682,
      -0.4854874364134862,0.5574614519339011,0.4770942414262169,
      0.3823157740702657,0.2756061719864941,0.1617850988203112,
      0.0470881327099307,-0.06243443855232068,-0.1623215265517549,
      0.5809489294799253,0.4937832424572646,0.3911366273799454,
      0.276734760526323,0.1567997280124966,0.03849904347704674,
      -0.07201766541821289,0.5981392304579348,0.5035962293740359,
      0.3933108814734469,0.2724812113143264,0.1485413749948833,
      0.02903407175199371,0.6059866911555354,0.5040926955212129,
      0.3873829237054168,0.2624333853186085,0.1373130258988978,
      0.6022642769367629,0.4940448560674484,0.3732152095431146,
      0.2472094891883707,0.5864999399564288,0.4740709200953988,
      0.352163126792074,0.5602791768912673,0.4464581629930486,
      0.5266502279007443,-0.4163031917414612,-0.5178614819829612,
      -0.6201479767677171,-0.7171163801074496,-0.8025494402189062,
      -0.8717667933239608,-0.9228268112486158,-0.9565657823609164,
      -0.3309034260843112,-0.4319023018909033,-0.5371520139920846,
      -0.6408948133012854,-0.7362003328755568,-0.8167859635709049,
      -0.8788842105927732,-0.9220131313955344,-0.9483463221383831,
      -0.338624488411881,-0.4426206857782847,-0.5490725801393385,
      -0.6512627192307555,-0.7419563768480812,-0.8155735468033498,
      -0.8697858892475217,-0.9055758335701712,-0.3417327749428255,
      -0.4462719473348893,-0.5509105300296661,-0.6485010946606566,
      -0.7322978310246682,-0.7980541850077586,-0.8449702640289916,
      -0.3389910093648414,-0.4413180005751814,-0.5412519758066311,
      -0.6319456221810869,-0.7078189355434928,-0.7661561658452967,
      -0.3298926669562023,-0.427540138135818,-0.5206911591272092,
      -0.6034312473993601,-0.6716247883748602,-0.3149361858566656,
      -0.4062273552356623,-0.49166038715387,-0.566554922382665,
      -0.2954969768757202,-0.37970711664501,-0.4574830674709519,
      -0.2733671630067541,-0.3505475201547475,0.7196184465472099,
      0.7985685715325342,0.8693370709616777,0.9258764738945257,
      0.9639035601847371,0.9822982118031284,0.9831227403545156,
      0.7284499433836592,0.8052061147659351,0.8708262265695167,
      0.9195869704437927,0.9484883092395549,0.9581120573066315,
      0.7249755313208718,0.796389971071354,0.8542356312683819,
      0.8939114816111898,0.9141086068192968,0.7071474645768509,
      0.7707144394707161,0.8194752023181837,0.8504297125393642,
      0.675180421967678,0.7297592328398797,0.7697469803916117,
      0.6317049359907486,0.6776373777282386,0.5808671950649753,
      -0.366257089693115,-0.2932325919216836,-0.2079536613378413,
      -0.1124326200188124,-0.01090482797043428,0.09084100501255125,
      0.1869789126919778,0.2731780147607579,0.3473085031912251,
      -0.4128247445431558,-0.3367224114404735,-0.2474623076665533,
      -0.1478309677445388,-0.04301824976707492,0.06052715939621055,
      0.1569432970144201,0.2423888563057751,-0.4569906864326022,
      -0.377544132020254,-0.2845287871856594,-0.1816479322770386,
      -0.07490335342398057,0.02900310611676332,0.1245901828483064,
      -0.4955268212110568,-0.4125878505007722,-0.3164139219124497,
      -0.2116012050517462,-0.1045983865863539,-0.001861117423562203,
      -0.5255625148012287,-0.4393821442371415,-0.3411559538741524,
      -0.2361085232789399,-0.1306299482392742,-0.5454522916908328,
      -0.4568651248612463,-0.3581319814379551,-0.2546618495309371,
      -0.5551983685993169,-0.4655407524301317,-0.3679944751189687,
      -0.5561971575770934,-0.4670014539093825,-0.5505634949474449,
      0.9220133236751157,0.8788843849049742,0.8167861054780646,
      0.7362004119831233,0.6408947895214253,0.5371518506414063,
      0.4319019774051972,0.8697860623176014,0.8155736760665241,
      0.7419564499373366,0.6512627126170119,0.5490724656572447,
      0.4426204410195734,0.7980543166386451,0.73229789903684,
      0.6485010946606566,0.5509104500833818,0.446271772176749,
      0.7078190327552072,0.6319456504394643,0.5412519405164845,
      0.4413178995840423,0.6034313501235972,0.5206912029562619,
      0.4275401340932171,0.4916605546162393,0.4062274818767118,
      0.379707405130632,0.3442381194341675,0.3356199216704505,
      0.3203596326360576,0.2982102696268391,0.2700983579505985,
      0.2379692158357015,0.2041879117102824,0.1708729017037819,
      0.2312638113027555,0.2150847428759935,0.1936722946488835,
      0.1677563920072661,0.1389315354365512,0.1092004686440709,
      0.08038134335000687,0.1055093293250425,0.08289035011594331,
      0.05758689641252075,0.03105854840479935,0.004986821056754707,
      -0.01921097131735246,-0.02731960194972041,-0.05380752683403425,
      -0.0797234336008201,-0.103472821018762,-0.1239244773230862,
      -0.1596023291970502,-0.1865984417823142,-0.209863618928254,
      -0.2282806816594251,-0.2838115205510092,-0.3082242217152255,
      -0.3265625234832059,-0.39448071777243,-0.41428243288548,
      -0.4890559628033717,0.1469872495241008,0.05499837995147388,
      -0.04502078247419547,-0.1481383500660343,-0.2482939617934105,
      -0.3399333541527281,-0.4194222062209155,0.03081218235410985,
      -0.07062432464607897,-0.1769155567176647,-0.281728039903757,
      -0.3786131170062194,-0.462852487954783,-0.09481577398373708,
      -0.2025910390450985,-0.3107079508004592,-0.4122160055133201,
      -0.5013621351348974,-0.2235850902780548,-0.3330789997683635,
      -0.4378914824575998,-0.5315738812374622,-0.3478839295034498,
      -0.4540598420421491,-0.5511571070104784,-0.4608643070543677,
      -0.5595972383031447,-0.5581731097866064,0.07201857797974105,
      -0.03849795696968416,-0.1567984874655333,-0.2767334157256128,
      -0.3911352491971175,-0.4937819036034042,-0.5809476860617484,
      -0.6517357063102128,-0.02903313746554977,-0.1485403228794575,
      -0.2724800804665481,-0.3933097299991974,-0.503595119108857,
      -0.598138209964731,-0.6749584587129783,-0.1373121735202105,
      -0.2624324872189829,-0.3873820250532686,-0.5040918441619855,
      -0.6059859240149241,-0.6896660583043277,-0.2472088184389502,
      -0.3732145606963823,-0.4940442637379011,-0.6022637655195766,
      -0.6930297734586861,-0.3521626999463403,-0.4740705579739117,
      -0.5864996568405977,-0.6835649861113746,-0.4464579863091031,
      -0.5602790756561028,-0.6617902107749168,-0.5266502592949665,
      -0.6300939487796368,-0.5918821259544012,-0.53439964190375,
      -0.5345065247441787,-0.525095375735572,-0.5050135034697202,
      -0.4748341677209202,-0.4368559346659283,-0.3943495717901673,
      -0.6290379264397734,-0.6286920435345361,-0.6159397381647077,
      -0.5900238619718436,-0.552538907016351,-0.5069746781213621,
      -0.7197183758293895,-0.7162919864381756,-0.6977972953381246,
      -0.6644602226954861,-0.6191959297333463,-0.7998960023058911,
      -0.7907283519716943,-0.7648124613084771,-0.7237428249931476,
      -0.8640801553848605,-0.8474213786178793,-0.8138188945925402,
      -0.9095129144421652,-0.8851002301776055,-0.936655652047691,
      -0.7336930479121847,-0.8132892661843129,-0.8844909854341375,
      -0.9411836430482039,-0.9790578295745921,-0.9970195644563975,
      -0.9971982203541059,-0.9837001754183022,-0.7578919095374027,
      -0.8358203200836417,-0.9020816987419813,-0.9508426122816337,
      -0.9791029969688041,-0.9875547507725127,-0.9797764622248338,
      -0.7704378741700656,-0.8432731964302291,-0.9016230372124288,
      -0.9407950105701793,-0.959571511273145,-0.960228691612206,
      -0.768376378413949,-0.8332256052212422,-0.8819865069139035,
      -0.9116590219002929,-0.9226925651801682,-0.7509515941742244,
      -0.8062957285440923,-0.8455183951051963,-0.8672952051616537,
      -0.7200327517213486,-0.7659653038152823,-0.7970714969272577,
      -0.6793955903243286,-0.717184776407749,-0.633377666107338,
      -0.1804949761426579,-0.07495545509988207,0.04000862921312944,
      0.1587643045074381,0.2743455685619207,0.380328817183559,
      0.472471714864844,-0.1978749051329674,-0.08739956539043256,
      0.0316001612578226,0.152429864698119,0.2676553828144042,
      0.3712303926692266,-0.213060219387871,-0.0988881700848137,
      0.02217737847260294,0.1427712082619221,0.2556135701396871,
      -0.2246722609068409,-0.1085467937789958,0.01228287704184508,
      0.130382595337822,-0.2317920983902931,-0.1157811822543603,
      0.002544721985696509,-0.2342683636555594,-0.1204473722877317,
      -0.23269032918574,-0.9594304364836954,-0.9180182641419954,
      -0.8570708740276961,-0.7768920058412835,-0.6811796536024748,
      -0.5762859389154773,-0.4693194383350184,-0.9480540609614943,
      -0.8969570397650206,-0.8250437852464129,-0.7343500529080861,
      -0.6304558624488807,-0.5208885452010862,-0.9189091107086019,
      -0.8569290407920915,-0.774471897882643,-0.6755415222905497,
      -0.5671264763802179,-0.8705859926593716,-0.798120523611733,
      -0.7074267366705677,-0.6040846614078772,-0.8048559824957469,
      -0.724149733213467,-0.6289645629177542,-0.7264645919207081,
      -0.641031394884048,-0.6416277774997261,-0.1778161375899129,
      -0.1160626578000681,-0.04588612344347009,0.03109142233089637,
      0.1117410720994994,0.1918123956592439,0.2669187829195654,
      0.3336638559940376,0.3902934987610119,0.4366423972091847,
      -0.07822810590231034,-0.006235377192768587,0.07354985884216768,
      0.1581390650418906,0.2429944925392092,0.3229913994840846,
      0.3938351936689208,0.4531165220472979,0.5004647043037878,
      0.03352103754179751,0.1154299733980676,0.2032831078056842,
      0.292541229210957,0.3775430167942871,0.4530586788103156,
      0.5157866661887469,0.5648910982101242,0.1544374088220477,
      0.2443626306102227,0.3370001516124326,0.4265382311030134,
      0.5070037257166236,0.5740673187258024,0.6260922126282031,
      0.2792615471857886,0.3736358874527184,0.4664608463431823,
      0.5514626265288498,0.6234114376019896,0.6796182090584544,
      0.4012129841179684,0.495438498091003,0.5837000128597972,
      0.660290805551725,0.7214982694759079,0.513610737510482,
      0.6032356918969084,0.6833069969058528,0.7494380454402638,
      0.6115888872401081,0.6933734762473292,0.7634088105496576,
      0.6929698693584984,0.7653064136814648,0.7580253719184178,
      -0.5980610514568759,-0.5540609356997265,-0.4979043682596813,
      -0.4291113361140411,-0.3489872892412604,-0.26088135031786,
      -0.1696203080333493,-0.08026917121509158,0.003104503401481844,
      0.07805400320688781,-0.5118464583535911,-0.4574805454176465,
      -0.3903649047262822,-0.3111481867341126,-0.2226772699131629,
      -0.1297540437661504,-0.03794114332033659,0.04796410149079763,
      0.1249140518997611,-0.4093046623238462,-0.3436365672482747,
      -0.2657018883722788,-0.177734023626004,-0.08420287536469411,
      0.009136975046692321,0.09689317432239272,0.1753846695482919,
      -0.2914862882507431,-0.214943022039254,-0.1281283439422551,
      -0.03496909012717313,0.05893398726743186,0.1478453618978381,
      0.2274959035287995,-0.16207395039147,-0.07691226494625918,
      0.01500850865584403,0.108539699421764,0.1979267183009546,
      0.2784559688242085,-0.02723965791063066,0.06265634702765752,
      0.1549020715540746,0.2440506450225836,0.3251843877273637,
      0.1056298518860434,0.1957006847953421,0.2838067303057802,
      0.3651211825280409,0.2298299119587134,0.3160346825162738,
      0.3968842416588859,0.3408204777877171,0.4204156838417109,
      -0.07805410188638827,-0.09444407992200232,-0.1114341761737927,
      -0.1281592432697841,-0.1435433307961642,-0.1565435248513201,
      -0.1664476243728032,-0.1730570234043899,-0.1766548755042983,
      0.0157271412771055,0.00421518985213093,-0.008591614873188899,
      -0.02222682113317407,-0.03596894129854906,-0.0489918802882604,
      -0.0605924265125066,-0.07036414931943043,0.1197931219386519,
      0.1133598979509488,0.1043544464570979,0.09273871068686244,
      0.07893799958180921,0.06380141477309535,0.04835884529075852,
      0.2310143008479173,0.2288491935457802,0.2221994711972146,
      0.2106389321639752,0.1945980336985477,0.1753190140959641,
      0.3442626575810957,0.344448149200232,0.3378594888639801,
      0.3240587639448084,0.3038950785717152,0.4532138881334763,
      0.4531600279320766,0.4442184199518987,0.4263949183927991,
      0.5518992169467315,0.5491648599857102,0.536164631289481,
      0.636238087652084,0.6292447926211703,0.7047142955122331,
      -0.9161616153729886,-0.9332409689531019,-0.9400448551193769,
      -0.9327747060225071,-0.9086729333885725,-0.8672206083002895,
      -0.8106886470978539,-0.7435546757108702,-0.671104187615456,
      -0.8837679892439161,-0.8957169226742606,-0.8953622468139258,
      -0.879022816612988,-0.8447970322719913,-0.7936799886395396,
      -0.7295529133114078,-0.6579138707033989,-0.5841830474046977,
      -0.8362110030564622,-0.8405992754866302,-0.8309565568491305,
      -0.8043503638067647,-0.7603455612152211,-0.7016495768401146,
      -0.6333375620907219,-0.5611246469661415,-0.771485918473268,
      -0.766275808658604,-0.7462153282219795,-0.709758929805572,
      -0.6582056983470901,-0.595590067087498,-0.5273136421678301,
      -0.6896210625953493,-0.6739687249276908,-0.6440754938657285,
      -0.6000706653666524,-0.544661671880628,-0.4823592409207763,
      -0.5934057828726733,-0.5681859910978439,-0.5308212768379395,
      -0.4828430686319266,-0.4275962761242672,-0.4881111164997092,
      -0.4557380987684553,-0.4142857087129486,-0.3660248731570757,
      -0.3801362714084162,-0.3437506203025199,-0.3015400644213536,
      -0.2753204578199928,-0.2378722763394139,0.5980613131647216,
      0.5540611551609296,0.4979045604542198,0.4291115157626469,
      0.3489874637290572,0.2608815137134187,0.1696204410073288,
      0.08026924777144263,-0.003104506871877828,0.6524878671692068,
      0.6063306836302526,0.5460446037762663,0.4714063018804604,
      0.3845537031663028,0.2900121666992721,0.1936208245443922,
      0.1008859431994594,0.7044824728062356,0.6549957846833561,
      0.589454559810383,0.5082654506345897,0.4147343369684993,
      0.3146157758763655,0.2144661074447615,0.7499962909367706,
      0.6957170016368869,0.6239253864871747,0.5360957563846484,
      0.4368631914867457,0.3329288310787005,0.7847921789285285,
      0.7244174460120194,0.6460542235032618,0.5525231471367963,
      0.4495787672324474,0.805637417939517,0.7386336021807628,
      0.6544795745347518,0.5572396030423862,0.8113900261915854,
      0.7383768115126219,0.6502709550316551,0.8032920019996154,
      0.725915395913506,0.7843092736443379,-0.4366431953633788,
      -0.5190819485847166,-0.6039640604128714,-0.686785147947791,
      -0.7620470421725769,-0.8245340389659808,-0.8708228680516067,
      -0.9001617337332903,-0.9142381355016868,-0.4204163992491611,
      -0.5047005830710198,-0.5903445107000899,-0.6720308549719619,
      -0.7437914180445663,-0.8006802874293815,-0.8402926072696949,
      -0.8631751150784488,-0.8720236313262308,-0.396884888208038,
      -0.4811289714781835,-0.5652543720979706,-0.6434244315426998,
      -0.7097591671683677,-0.7601772892083588,-0.7935642249312238,
      -0.8115619523188943,-0.3651217770412689,-0.4471759195096291,
      -0.5274838962988779,-0.6002019152270698,-0.6601534291553377,
      -0.7044750126107846,-0.7331976694123559,-0.3251849417564136,
      -0.4031031327086847,-0.4778781266280965,-0.5442129229008882,
      -0.5980262207943586,-0.6376204486797759,-0.2784564800311357,
      -0.3509705292257924,-0.4194890855987771,-0.4796201828812963,
      -0.5284049821532861,-0.2274963530521034,-0.2942151694937769,
      -0.3567023183694048,-0.4115345048612673,-0.1753850281553566,
      -0.2366295712717834,-0.2938643950650254,-0.1249142906614369,
      -0.1813653192898883,0.9161619708326186,0.9332413210693342,
      0.9400451789678279,0.9327749805257902,0.9086731502239969,
      0.8672207777405768,0.8106887957474585,0.7435548367414557,
      0.6711043898508244,0.9520728550100875,0.9680084878195204,
      0.9709706641436524,0.9568547607732569,0.923414887485804,
      0.8715118657296428,0.8051612260268248,0.7302053387950368,
      0.9795688039876239,0.9918158125836303,0.9881921235557868,
      0.9648780850920352,0.9208732655385489,0.8588851060453266,
      0.7845540741828551,0.9941684586829005,0.9997711640715929,
      0.9870068177874959,0.9531387910864955,0.8989972043871354,
      0.829085472566247,0.9920538154161512,0.9884396081113127,
      0.9651307683693848,0.9211259732003716,0.8591326593451197,
      0.9714466929952836,0.9573448137030386,0.9239099284934242,
      0.8720020075968424,0.9334760373935599,0.9093872922332051,
      0.8679349808976941,0.881888023912356,0.8497898935937275,
      0.8217573530818993,0.1778171059718252,0.116063484932519,
      0.04588675059291181,-0.03109104889813256,-0.111740984528746,
      -0.1918125910017186,-0.266919222145987,-0.3336644772286424,
      -0.3902942361971635,0.2378731920177505,0.1751982200024925,
      0.103165135794194,0.0237726939459323,-0.05924605897326656,
      -0.1410802144925054,-0.217121271652078,-0.2841551297870086,
      -0.3408213192163667,0.3015409214864257,0.2379996307470747,
      0.1642142712273472,0.08265050914794721,-0.002351577711465992,
      -0.08556215486494216,-0.1623583634318567,-0.2298307519128458,
      0.3660256667153609,0.3013724622732556,0.225787563312025,
      0.1422988506889489,0.05578339303902819,-0.02833333511696591,
      -0.1056306282458797,0.42759700108315,0.3613584700754575,
      0.2839225264985345,0.1989204411880135,0.1115820483497766,
      0.0272390133777002,0.4823598903354967,0.4141193140607006,
      0.3350423572261403,0.2492664112715316,0.1620734932467959,
      0.5273142067726671,0.4570511845454862,0.376979581956603,
      0.2914860445503094,0.5611251168568853,0.4893610105925254,
      0.409304623993076,0.5841834148888068,0.5118465920502688,
      0.5190812242961226,0.6039634624796284,0.686784726158866,
      0.7620468277893128,0.8245340313583351,0.8708230332223915,
      0.9001620167170649,0.9142384783990968,0.5894211704148156,
      0.6789524307604198,0.7632448457303311,0.8359266838325403,
      0.891894617970833,0.9289011503005559,0.9478965192118636,
      0.6583452758445511,0.749524790850048,0.8315532424890266,
      0.8978881068207238,0.944448070806156,0.9707810672151929,
      0.7208184417027227,0.809677174255781,0.8854287054759336,
      0.9423468561715556,0.9781178066356491,0.7716444357268979,
      0.854135980832467,0.9204708008966339,0.9665675908847857,
      0.8070418390872103,0.8801658592323851,0.9356914498708749,
      0.8258654425765057,0.8883525351377126,0.8296779096104938,
      0.09444407913201958,0.1114342885457886,0.1281594836202007,
      0.1435437104478973,0.1565440479837881,0.1664482855753907,
      0.1730578081089095,0.176655763191492,0.1813650175735936,
      0.2043807284477892,0.226758533434972,0.2468100045265798,
      0.262820145857554,0.2735753518716192,0.2787599021207723,
      0.2789608606213531,0.2753212552695638,0.2938639456031617,
      0.3229737187957749,0.3493473093663221,0.3704627992219873,
      0.3842636510528081,0.3899007538969458,0.3879754312808241,
      0.380136902581416,0.4115339986401249,0.4448996490238289,
      0.4726025527270443,0.4916319868779722,0.5002042468552221,
      0.4984303277487671,0.4881116080953447,0.5284045295040382,
      0.5629551752469137,0.5885431838636429,0.6023440210047828,
      0.6035085783646522,0.5934061805946154,0.6376201472701202,
      0.6697547537752425,0.6900358414869175,0.6965200475426346,
      0.6896214177247305,0.7331975726799168,0.7598360876893205,
      0.7728363883884936,0.7714862700692517,0.8115620579167974,
      0.8309267868599184,0.8362113667997926,0.8720238939031298,
      0.8837683585624158,0.2841543957726532,0.2171207121163834,
      0.1410798976495498,0.0592460337919352,-0.02377241605014863,
      -0.1031645846803213,-0.175197453605406,0.1623576902420928,
      0.08556172001198789,0.002351438316001383,-0.08265033236073645,
      -0.1642138033298737,-0.2379989318886596,0.02833279000440243,
      -0.05578364441956794,-0.1422987785572308,-0.2257871853292439,
      -0.301371837576038,-0.1115824064790156,-0.1989204755639271,
      -0.2839222453034268,-0.3613579271786332,-0.2492665519223285,
      -0.3350421794323328,-0.4141188622036356,-0.376979512041707,
      -0.4570508332511692,-0.489360766920074,-0.01572700265409594,
      -0.1008857021547371,-0.1936205087916404,-0.2900118132468701,
      -0.3845533472228392,-0.4714059641664385,-0.5460442847763476,
      -0.6063303681777003,-0.6524875344052207,-0.1197927153790677,
      -0.214465606765001,-0.3146152265921756,-0.4147337903295822,
      -0.5082649460772711,-0.5894541111166113,-0.65499537962445,
      -0.7044820845747036,-0.2310136280273066,-0.3329280970872507,
      -0.43686246029571,-0.5360950867028047,-0.6239248102620224,
      -0.695716515229827,-0.749995864312011,-0.3442617649825689,
      -0.4495778732913127,-0.5525223259640154,-0.6460535267660109,
      -0.7244168848320439,-0.7847917261432807,-0.4532128635509181,
      -0.5572386535616057,-0.654478768383796,-0.7386329688076397,
      -0.805636940817698,-0.5518981667856981,-0.6502700535877015,
      -0.7383761054919119,-0.811389515577244,-0.6362371067112403,
      -0.7259146170102994,-0.8032914429700343,-0.7047134464025514,
      -0.7843086526571425,-0.7580246814516085,-0.2789601712242363,
      -0.2787593471791244,-0.2735749526850533,-0.2628199127188,
      -0.2468099337172354,-0.2267586092205901,-0.2043809286570809,
      -0.3879749291048343,-0.3899004091962853,-0.3842634838788619,
      -0.3704628145053439,-0.349347495161076,-0.3229740508586004,
      -0.498429974108997,-0.5002040604834036,-0.4916319868779722,
      -0.4726027411534949,-0.4449000101574501,-0.6035083146581218,
      -0.602343922563195,-0.588543271739299,-0.5629554529176811,
      -0.6965198150492191,-0.690035767365203,-0.6697548637373957,
      -0.7728361486877732,-0.7598360011487315,-0.8309265297968929,
      -0.7302050882445748,-0.8051610414501039,-0.8715117174305623,
      -0.9234147395568944,-0.9568545837269061,-0.9709704452151936,
      -0.968008232338148,-0.9520725798532584,-0.7845537997567457,
      -0.8588849308275506,-0.9208731569645157,-0.9648780014111388,
      -0.9881920288336453,-0.9918156880829223,-0.9795686493343514,
      -0.8290851961543591,-0.8989970625213397,-0.9531387450388693,
      -0.987006817440241,-0.9997711657617464,-0.9941684373181741,
      -0.8591323903215224,-0.9211258673075536,-0.9651307788218259,
      -0.9884396774035085,-0.9920538947738091,-0.8720017371830424,
      -0.9239098369784071,-0.9573448480949935,-0.9714467930697923,
      -0.8679346859597882,-0.9093871792307242,-0.9334760519103793,
      -0.8497895472965013,-0.8818878542954445,-0.8217569347483581,
      -0.04796426561714673,0.03794103867724843,0.1297539800598756,
      0.2226772328227925,0.3111481732455522,0.3903649242566535,
      0.4574806142481337,-0.09689348391076305,-0.00913724963965745,
      0.08420262555973922,0.177733799656962,0.265701704660737,
      0.3436364458954788,-0.1478457927031966,-0.05893441045550545,
      0.0349686757279841,0.1281279562907054,0.214942691058949,
      -0.1979272450610679,-0.1085402471877293,-0.01500906268915838,
      0.07691173856262062,-0.2440512506935033,-0.154902725197284,
      -0.06265701957101971,-0.2838074100306656,-0.1957014339320081,
      -0.3160354401730464,-0.9478962831266258,-0.9289010005915933,
      -0.8918946048405405,-0.8359268445086131,-0.7632451895269284,
      -0.6789529354540513,-0.5894217923304083,-0.5004653941257039,
      -0.970780979775399,-0.9444480894583087,-0.8978882630448816,
      -0.8315535470087182,-0.7495252275742876,-0.6583458081952406,
      -0.5648916828872914,-0.9781178706298981,-0.942347026709516,
      -0.885428988462601,-0.8096775549851075,-0.7208188890120822,
      -0.6260926899993671,-0.9665677527611446,-0.9204710448766127,
      -0.8541362909342279,-0.7716447840125762,-0.6796185647672793,
      -0.9356916072835824,-0.8801660576917285,-0.8070420537147268,
      -0.7214984744500784,-0.8883525705162149,-0.8258654786670103,
      -0.7494380619781833,-0.8296777286633932,-0.7634086046890135,
      -0.7653059660649493,0.6579141776573929,0.7295531777521024,
      0.7936802384922247,0.8447972973738501,0.8790231171992372,
      0.8953625859168114,0.8957172874110191,0.6333379663382935,
      0.7016499314834187,0.7603458913147364,0.8043506945733514,
      0.8309569030556652,0.8405996366357998,0.5955905567158142,
      0.6582061258811204,0.7097593165624652,0.7462156952614298,
      0.7662761679440523,0.5446622343557217,0.6000711505073423,
      0.6440759195726991,0.6739691095309485,0.4828436947701487,
      0.5308218111264984,0.5681864487641957,0.4142863939154359,
      0.455738680653885,0.3437513626484632,-0.4531171355531167,
      -0.3938356644953573,-0.3229916528986729,-0.2429944658048455,
      -0.1581387295403868,-0.07354922969450257,0.00623625010153865,
      0.07822915744532465,-0.5157871513702247,-0.4530589859436308,
      -0.3775430647931653,-0.2925409610470896,-0.2032825122447724,
      -0.1154290866147241,-0.0335199252992554,-0.5740676626100093,
      -0.5070038470041614,-0.4265380492676018,-0.3369996244991067,
      -0.244361769966899,-0.154436271103469,-0.6234116143878676,
      -0.5514625314967724,-0.4664604103168303,-0.3736350915397741,
      -0.2792604271482146,-0.6602907797777987,-0.5836996709958147,
      -0.4954377928330375,-0.4012119230675125,-0.6833067346898886,
      -0.6032350844987688,-0.5136097646306489,-0.6933729576191783,
      -0.6115880145737621,-0.6929690938467825,0.8631753518491425,
      0.8402927698803195,0.8006803216023103,0.7437912797965405,
      0.6720305274074914,0.5903440100883541,0.504699950277095,
      0.7935642870226529,0.7601772654056832,0.7097590166001092,
      0.6434241301081504,0.5652539229757697,0.4811284030348103,
      0.7044748595212692,0.66015319194957,0.6002015742230613,
      0.5274834493042921,0.4471753838390926,0.5980258661419307,
      0.5442125068737823,0.4778776501574548,0.4031026076549349,
      0.4796197003718685,0.4194885812207204,0.3509700147775985,
      0.356701818393901,0.2942146894919281,0.2366291626721505,
      0.07036512794891495,0.06059331156828225,0.04899265284320812,
      0.0359695874535567,0.02222733419962072,0.008591995504757999,
      -0.004214935230486583,-0.048357793313131,-0.06380043950324864,
      -0.07893711715811484,-0.09273793551230918,-0.1043537896571689,
      -0.113359365885021,-0.1753179190717106,-0.1945969916350631,
      -0.2106379566061765,-0.2221985793343545,-0.2288484037077831,
      -0.3038939791712426,-0.3240576891045761,-0.3378584512937831,
      -0.3444471709858402,-0.4263938526473902,-0.4442173491846191,
      -0.4531589664405656,-0.5361636274888475,-0.5491638241821553,
      -0.6292438651304321,-0.1282040992012934,-0.04435978008441287,
      0.04944233877845754,0.1506070646973712,0.2546642688007949,
      0.3559200448438602,0.4488272180170631,0.5294097717623608,
      0.5959629108387098,0.6488215902264566,-0.1619077855794933,
      -0.07454406204984147,0.02329146842828509,0.1282100848553425,
      0.234790969615194,0.3366773586411712,0.4283146113627389,
      0.5063384506251003,0.5698896326340842,-0.1972673366002622,
      -0.1068255930288844,-0.005791503701260298,0.1015779223878756,
      0.2090686111987675,0.3100671263842886,0.3994533679902369,
      0.4746871659657975,-0.232615798766268,-0.1396773162324554,
      -0.03655119363532085,0.07170206110196441,0.1784301917623365,
      0.2772572737517244,0.3638246180723772,-0.2658039506626713,
      -0.1710978759257405,-0.06718962048721892,0.04030107224694054,
      0.1447607652202159,0.2404750267534745,-0.2946652126306246,
      -0.1990974281683128,-0.09582172900178909,0.009369859926838612,
      0.110357956539064,-0.3176184514965803,-0.222270660707132,
      -0.1210148748258565,-0.01939827024983316,-0.3340739915348134,
      -0.2401148801885883,-0.1420870179889694,-0.3444209939391287,
      -0.2529463331304582,-0.3496821715684524,0.6714420577705413,
      0.7435113472506464,0.8139336018335218,0.8777656242085762,
      0.9296329303259336,0.9651729312878542,0.9824386073167399,
      0.9823984726130219,0.9682596586238317,0.9441720387917811,
      0.7087413879135209,0.7840892682826439,0.8556051669023013,
      0.9173531285160066,0.9635713995106854,0.9905236148375803,
      0.9977653075434614,0.9879747192712225,0.9655968026264566,
      0.7409584518952473,0.8171173803566061,0.8865969745674289,
      0.942889039924638,0.9806175261526792,0.9974611488769433,
      0.9948177215231417,0.976809672217213,0.7644107780855932,
      0.8381137943506111,0.9020313711806934,0.9498618317294466,
      0.9774884200770568,0.984455050704485,0.9737573075945112,
      0.7756818501512159,0.8435121923354123,0.8989022227634631,
      0.9366307840854883,0.95437657834853,0.9533825051676448,
      0.7727341579527024,0.8320450757851201,0.8774020340365609,
      0.9052158359967212,0.9148948007536834,0.7557291769626011,
      0.805326671094437,0.8408668603662851,0.8604026946131843,
      0.7269859063883545,0.7671790270453136,0.7943722223707224,
      0.690100618808278,0.7222076989157948,-0.9441722324566724,
      -0.904307645384738,-0.8481757819865325,-0.773945596539063,
      -0.6820112385376135,-0.5756807280718292,-0.4607792790679794,
      -0.3441534701740746,-0.2318935611202809,-0.9303809621015218,
      -0.8830185434308674,-0.8172030379805079,-0.7317787780884728,
      -0.6286054699715032,-0.5128635622007778,-0.3918810385950385,
      -0.2730236045989769,-0.9028984430498929,-0.8460642977744163,
      -0.7691784804069041,-0.6724056610478215,-0.5595277701998983,
      -0.4374897246213904,-0.3144118544998256,-0.8589482430980673,
      -0.7911988721488439,-0.703044075433466,-0.5965101770495288,
      -0.4772883218837695,-0.3533685405348139,-0.7974562734423089,
      -0.7186356601273992,-0.6208045991538126,-0.507926737136379,
      -0.3869469856202812,-0.7199870192241779,-0.6317037484916127,
      -0.5275199172214167,-0.4127886377572592,-0.6307845637432554,
      -0.5361873333707976,-0.4298568952019761,-0.5356827691130239,
      -0.4385943873688064,-0.4404800743875191,-0.3130828685601688,
      -0.2158144102235012,-0.1048477339184343,0.01737679733452727,
      0.145995491636905,0.2742909798999176,0.3952347219739472,
      0.503291831296798,0.595502090315309,-0.3099520106140219,
      -0.2053757769283784,-0.08650347275041102,0.04302105714473008,
      0.1768577004771411,0.3071581921441303,0.4266784695958356,
      0.5306275177659158,0.6172668313578871,-0.302423711672288,
      -0.1907069702272193,-0.06489577644534418,0.0699408153488305,
      0.2061362196609274,0.3353108895502397,0.4507704673548409,
      0.5489716751760349,-0.2895250242862085,-0.1715150632893158,
      -0.04067912706104116,0.09654302677976182,0.2317116972020255,
      0.3567592436499814,0.4661908578614318,-0.2708185619737357,
      -0.1481961478813345,-0.01510368122142407,0.1210917442402347,
      0.2520105777387078,0.3706589640013763,-0.2467266437111619,
      -0.1219141274081689,0.01025701144397722,0.142223084729963,
      0.2664554400639934,-0.2185691140597738,-0.09433919811625824,
      0.03395634318261122,0.1592889611288683,-0.1882244284110535,
      -0.06716551971534636,0.055031530590314,-0.1575991801780021,
      -0.0416967436508152,-0.671442351589465,-0.7435115240281642,
      -0.8139336689380561,-0.8777656078901646,-0.9296328738783285,
      -0.9651728842355707,-0.9824386099053611,-0.982398544792382,
      -0.9682597992070626,-0.673525768984227,-0.7468180740978238,
      -0.8166237491391959,-0.8772252002603724,-0.9230382082945876,
      -0.9503956493652942,-0.9587838051377746,-0.9507033782544355,
      -0.6670475871543912,-0.7391545962936478,-0.8055307765029669,
      -0.860125367790516,-0.8978538005513599,-0.9163947908554175,
      -0.9168546729772998,-0.649602216294724,-0.7177302448625567,
      -0.7778859526921634,-0.8243817913354027,-0.853342815897189,
      -0.8640711446818524,-0.6197562179675709,-0.6813798175903147,
      -0.733374932687383,-0.7711033603736286,-0.7922438176365504,
      -0.5778269971598609,-0.6314056112449312,-0.6747362861851252,
      -0.7045760418412901,-0.5261115978506249,-0.571437749107807,
      -0.6069777467397017,-0.4682968578205298,-0.5062792722326205,
      -0.4083732040408349,-0.6488222902967828,-0.6479137619538795,
      -0.6384433779259612,-0.6181925269683732,-0.5859405951200543,
      -0.5421896843793962,-0.4893366483035593,-0.4310576147663105,
      -0.3712412833748336,-0.722208134089623,-0.7225295469502088,
      -0.7119299887749709,-0.6878297181787101,-0.6492831334094801,
      -0.5977545202455957,-0.5369263387902474,-0.4715400964562942,
      -0.4060110053588667,-0.7943724032793758,-0.7941684890748284,
      -0.780287498377037,-0.7501802425240992,-0.7037353198781235,
      -0.6438103021069973,-0.5754139215508255,-0.5040323183695714,
      -0.8604026705465683,-0.8571439110045493,-0.8374826979355753,
      -0.7995410210009178,-0.744592852017412,-0.6769935095953296,
      -0.6026908960383707,-0.9148946566640305,-0.9058191273231,
      -0.8783402703714518,-0.831895347992814,-0.7693419318067468,
      -0.6961400902573952,-0.9533823404746037,-0.9363830473278872,
      -0.9003464103777271,-0.846307850369865,-0.7783786922776838,
      -0.9737572051250643,-0.9481191644195498,-0.904368254349328,
      -0.8449013283633929,-0.9768096756732072,-0.9433249035584995,
      -0.8938553761971309,-0.9655969141863968,-0.9260722794540685,
      0.3130823317650696,0.2158137676495382,0.1048470017130739,
      -0.01737758434298081,-0.1459962827220426,-0.2742917196047844,
      -0.3952353653550799,-0.5032923546149073,-0.5955024919545937,
      0.2536926972719504,0.1458327290767235,0.02422841952626041,
      -0.1071275628186857,-0.241611507146674,-0.3712645460380625,
      -0.4889532492383237,-0.5901702171834956,0.1843472992213911,
      0.06615765451161945,-0.06461113970496464,-0.2021592879969449,
      -0.338354586304306,-0.4648175094943233,-0.575319351774491,
      0.1061130743462423,-0.02080286326993523,-0.1576482143840719,
      -0.2970021727032601,-0.4300387818650189,-0.549076707052971,
      0.0217206853443193,-0.1108168656978456,-0.2493324402368818,
      -0.3855277139058982,-0.5110231660272195,-0.06464549315404491,
      -0.1986151347691201,-0.3340228839844541,-0.4627520281251212,
      -0.1482539922220807,-0.2793070464170456,-0.4076024215152396,
      -0.2250412157123533,-0.3496313063989884,-0.2924709198142461,
      0.1282040764044601,0.04435966353085222,-0.04944259604098935,
      -0.1506074987015245,-0.2546648874595455,-0.3559208170725224,
      -0.4488280788164576,-0.5294106421507812,-0.5959637203467131,
      0.04169673907303984,-0.0526834259769666,-0.1563566027108192,
      -0.2651887425550946,-0.3731530321841731,-0.4736563198547184,
      -0.5613802828138158,-0.6335666007603626,-0.690101357380803,
      -0.05503152918262894,-0.1593042550239766,-0.2709316440867437,
      -0.3840952777948459,-0.4915861222634885,-0.5867907021484375,
      -0.6655838183420531,-0.7269866926442972,-0.159288972427813,
      -0.2712574169662021,-0.3872244295124012,-0.5000332157715695,
      -0.6022061061584294,-0.6881925170266414,-0.7557300072673857,
      -0.2664554858018714,-0.3823479651751564,-0.4978444298202386,
      -0.6053352646595813,-0.6982069960901036,-0.7727350088749903,
      -0.3706590624880138,-0.4857936740008229,-0.5959856542636139,
      -0.6942612161795336,-0.7756826781990721,-0.4661910177108468,
      -0.5761185565808596,-0.6773744620468557,-0.7644115292814133,
      -0.5489718931615636,-0.6504766905631515,-0.7409590768696484,
      -0.6172670952623341,-0.7087418537741278,0.6479131208665134,
      0.6384428093318625,0.6181920323103047,0.5859401592476801,
      0.5421892777726929,0.4893362367199527,0.4310571708515276,
      0.3712407933301676,0.5613207863628051,0.5433246784685077,
      0.5142659275004758,0.4739667199004282,0.4241908010905809,
      0.3683211596170611,0.3103315078005186,0.4569583774922522,
      0.4296550077229934,0.3922062664218491,0.345761387535644,
      0.293177934819726,0.2382039928591278,0.3364534206151339,
      0.3005219256748686,0.2568082513095248,0.2076321770193636,
      0.1563031965788156,0.2045546517133144,0.1623927536580789,
      0.1159478851139529,0.06807760796624844,0.0685659565224499,
      0.02376600432590625,-0.02150913216682495,-0.06350995748665327,
      -0.1072608130272718,-0.1851326652875042,0.9043074523981195,
      0.8481756214456443,0.7739454978673308,0.6820112173362239,
      0.5756807770328707,0.4607793683387657,0.3441535580133543,
      0.2318936078193602,0.926072164038092,0.8784582627879883,
      0.8124335550546075,0.7268690803364112,0.6236462465254911,
      0.5079539512218285,0.3871116910406285,0.2684634496083129,
      0.1575990472160092,0.8938553783036348,0.8365254960181361,
      0.7592600076242926,0.6622795368519036,0.5494016575639665,
      0.4275712710943025,0.3048730417540333,0.1882241880032878,
      0.8449014608184471,0.7764699938313133,0.6878549375110379,
      0.5811577408091089,0.4620991215146014,0.3386395224598848,
      0.2185687785328751,0.7783789236415784,0.6987988931237948,
      0.6005524285298199,0.4876744928391774,0.3671099975843117,
      0.2467262336464146,0.6961403422326237,0.6071556637949365,
      0.5027238349931351,0.3882403289629487,0.2708181010489455,
      0.602691063692116,0.5075711336386517,0.4012405518966602,
      0.2895245326714229,0.5040323033369615,0.4066733092394736,
      0.3024232018417092,0.4060107420397416,0.3099514873609258,
      0.6335657588222857,0.5613794019471703,0.4736554861193561,
      0.3731523315866652,0.2651882329560032,0.1563562968227162,
      0.05268329572084096,0.665582947801096,0.5867898301880646,
      0.4915853474795021,0.3840946840958189,0.2709312717864595,
      0.159304093053227,0.6881916390051439,0.6022052797777839,
      0.500032542833593,0.387223979093218,0.2712572035078997,
      0.6982061514387302,0.6053345332894541,0.4978439026516569,
      0.3823476853537574,0.694260458246256,0.5959850669993192,
      0.4857933245168793,0.6773738416456014,0.5761181477179906,
      0.6504762420694124,0.9303807852901133,0.9507032776506229,
      0.9587837990451511,0.9503957355401843,0.9230383566605702,
      0.8772253580218827,0.8166238578878361,0.7468180882094819,
      0.6735256647351061,0.90289835288087,0.9168546892221431,
      0.9163949287729053,0.8978540461202875,0.860125673826823,
      0.8055310756657073,0.7391548242370243,0.6670477005822436,
      0.8589483198537152,0.8640713573235344,0.8533431666375708,
      0.8243822455102904,0.7778864433741867,0.7177306948097215,
      0.6496025629973337,0.7974565860481383,0.7922442814347184,
      0.7711039549446601,0.7333755999307914,0.6813804770418425,
      0.6197567932818368,0.7199875994402858,0.704576762533665,
      0.6747371020412667,0.6314064480391776,0.5778277738330908,
      0.6307853921801013,0.6069786776259655,0.5714387199953477,
      0.5261125320735169,0.5356837831672402,0.5062803340418997,
      0.4682979008420118,0.4404811918375181,0.4083743128719785,
      0.3496833143594963,-0.2684635055945434,-0.3871116949027418,
      -0.5079539380311434,-0.6236462513912529,-0.7268691253470475,
      -0.8124336419036672,-0.8784583757439789,-0.3048731870534707,
      -0.4275713388616055,-0.5494016757834509,-0.6622795348064975,
      -0.7592600049270052,-0.8365254973182974,-0.3386397327605397,
      -0.4620992154744972,-0.5811577408091089,-0.6878548720792258,
      -0.7764698875810475,-0.3671102436707482,-0.487674577809656,
      -0.6005523742168883,-0.6987987332107705,-0.3882405864046344,
      -0.5027238913366621,-0.6071555453420947,-0.4012408083880579,
      -0.5075711618348809,-0.4066735651215305,0.590170014682346,
      0.4889529277025601,0.3712641002824293,0.2416109555838913,
      0.1071269463449765,-0.02422904936538466,-0.1458333255362707,
      -0.2536932284372023,0.575319376406684,0.4648174146687293,
      0.3383543572464268,0.2021589358241457,0.06461070043643219,
      -0.06615813268612159,-0.1843477715691807,0.5490769699623794,
      0.4300389185128004,0.2970021583546204,0.157648053896879,
      0.02080258943368662,-0.1061134148983228,0.5110236542957048,
      0.3855280620563327,0.2493326170112359,0.110816873466397,
      -0.02172081454080515,0.4627527059506328,0.3340234043580559,
      0.1986154658537707,0.06464563879740347,0.4076032405543511,
      0.2793076930534321,0.1482544392795429,0.3496322190693472,
      0.2250419477534466,0.2924718882891986,-0.9879747859642711,
      -0.9977653310206788,-0.990523615974921,-0.9635714188487002,
      -0.9173532159549495,-0.85560536501624,-0.7840895998008963,
      -0.994817702995987,-0.997461134800403,-0.9806175623759372,
      -0.9428891811827549,-0.886597265826606,-0.8171178418357974,
      -0.9844549713337977,-0.9774884174060016,-0.9498619694604108,
      -0.9020317021121872,-0.8381143412526206,-0.9543765048804849,
      -0.9366308742795815,-0.8989025406729466,-0.8435127700618501,
      -0.9052158630007405,-0.8774023054284241,-0.8320456365430426,
      -0.8408670789696687,-0.805327187943167,-0.7671794966406347,
      -0.3103320171372975,-0.3683216507471385,-0.4241912872571255,
      -0.4739672217049487,-0.5142664654627133,-0.5433252645626975,
      -0.5613214202090539,-0.5698903038330552,-0.2382044630883786,
      -0.2931784077953506,-0.3457618730212876,-0.3922067754962655,
      -0.4296555471996706,-0.4569589463091218,-0.4746877559159965,
      -0.1563035449296971,-0.2076325381624787,-0.2568086311939766,
      -0.300522327960348,-0.336453843995229,-0.3638250562458944,
      -0.06807774532319075,-0.1159480367576914,-0.1623929246512774,
      -0.2045548436199081,-0.2404752368196474,0.02150927301351948,
      -0.02376587784419006,-0.06856585200941642,-0.1103578775111495,
      0.1072612554376189,0.06351038126206604,0.0193986644095893,
      0.1851333868984673,0.142087712340153,0.2529472816242414,
      -0.5306278899290351,-0.4266789610910172,-0.3071587951012373,
      -0.1768583830726642,-0.0430217688399398,0.08650278658817297,
      0.2053751597977648,-0.4507707926372612,-0.3353113300288001,
      -0.2061367616665949,-0.06994142151255993,0.06489515773394869,
      0.1907063885440197,-0.3567595103770218,-0.231712075682342,
      -0.09654349885714854,0.04067860126609572,0.171514533036228,
      -0.2520107854074638,-0.1210920629807046,0.01510327295162653,
      0.1481956904535304,-0.1422232452547296,-0.01025728491155416,
      0.1219137656087612,-0.03395647625215152,0.09433895004400768,
      0.06716539409874363,-0.5063391876138633,-0.4283153498076878,
      -0.3366780195432001,-0.2347914812056201,-0.1282104074092338,
      -0.0232916058381637,0.07454407090624685,0.1619078867980062,
      -0.3994539622273614,-0.310067650345868,-0.2090689910564566,
      -0.101578111712534,0.005791506035575475,0.1068257468493955,
      0.1972675814170221,-0.2772576433388097,-0.1784304214682155,
      -0.07170210104888376,0.0365513498207285,0.1396776293169983,
      0.2326162031595532,-0.1447608377351292,-0.04030095729466294,
      0.06718993398467658,0.1710983522944223,0.2658045221403088,
      -0.009369596982657442,0.09582219059763358,0.1990980586833061,
      0.2946659462714916,0.1210154650283388,0.2222714245712332,
      0.3176193294845107,0.2401157501920485,0.3340749869695284,
      0.3444220768905359,0.4715398981500615,0.5369261966393798,
      0.5977544079195963,0.6492830104914913,0.6878295422870739,
      0.7119297300805439,0.7225291958412279,0.5754139764385705,
      0.6438104077520811,0.703735439041779,0.7501803284162304,
      0.7802875102236209,0.7941684049065568,0.6769937335137454,
      0.7445931056463463,0.7995412648414352,0.8374828910500066,
      0.8571440245657689,0.7693422116354997,0.8318956310273866,
      0.8783405275925528,0.9058193343789728,0.8463080799206776,
      0.900346626617332,0.936383240514816,0.9043683688324304,
      0.948119268339247,0.9433248910510675,0.2730237834281818,
      0.3918812540743772,0.512863760525918,0.6286055997510109,
      0.7317788088844818,0.817202970027369,0.883018402482024,
      0.3144121837974857,0.4374900830135051,0.559528090412838,
      0.6724058856901879,0.7691785827023693,0.8460642869454081,
      0.3533690330364157,0.4772888327771027,0.5965106290979496,
      0.7030444104060292,0.7911990697093608,0.3869476434981415,
      0.5079274002472546,0.6208051861357136,0.718636116718038,
      0.4127894497989522,0.5275207205127158,0.631704463913464,
      0.4298578384111184,0.536188255170367,0.4385954324163403 };

    auto      pointshell
        = new vnl_matrix_fixed<double, 3, 812>(coords);

    return pointshell;
  }
};


template<> class PointShell<1002, vnl_matrix_fixed<double, 3, 1002> >
{
public:
  static vnl_matrix_fixed<double, 3, 1002>* DistributePointShell()
  {
    double coords[3*1002] = {
      0.9756767549555488,0.9910244201339067,0.9977154378498742,
      0.9927800993607139,0.9738192119472443,0.9397937974509784,
      0.8915721200771204,0.8318778673129345,0.7646073555341725,
      0.693869346809198,0.6231965669156312,0.9833704607135893,
      0.9964241736681655,0.9989802553190876,0.987902775761947,
      0.9610864540427623,0.9183225885811028,0.8616492668438125,
      0.7948923552800862,0.7226196323725681,0.6490709211350905,
      0.9817040172417226,0.9905693687720445,0.9870396762453547,
      0.9681598062324672,0.9325589150767597,0.881205326354676,
      0.8173592116492303,0.7456638277324369,0.6708930871960926,
      0.9679134019101711,0.9705588332310378,0.959200825309353,
      0.9315264271335991,0.8872988802116832,0.8287690756632178,
      0.7601362175694236,0.6863342915062906,0.9399233672993689,
      0.934650496070032,0.9144882783890762,0.8781852416959071,
      0.8267930935417315,0.7635949206829796,0.6931818659696647,
      0.8971014254962764,0.8830745457598012,0.8543282347786828,
      0.8108509720355672,0.754782849204541,0.6899186612252973,
      0.8407280774774689,0.8182890571891552,0.782394344826989,
      0.7341199823031558,0.6762337155773353,0.7738549713477869,
      0.7444263912611687,0.7036087793266443,0.6531781708866408,
      0.7005607434301688,0.6661985543964391,0.6228579759074076,
      0.6249824933683414,0.5878380796842572,0.5505632701289798,
      0.4375940376503738,0.3812939184075813,0.3153040621970065,
      0.2399906033000567,0.1569517536476641,0.06907711216819173,
      -0.01984099037382634,-0.1058738447453135,-0.1857690950088067,
      -0.2574501777461976,-0.3200730131503601,0.4806956770236153,
      0.4221182950629059,0.3527542025586833,0.2732412963568581,
      0.1857299193159225,0.09380763985613347,0.001855404825389049,
      -0.08593912303271299,-0.1664330628904374,-0.2378914640002027,
      0.5232435944036425,0.4619857831524665,0.3889403678268736,
      0.305177254463085,0.2135250052622625,0.1182658918935892,
      0.02420694871807206,-0.06444621903648747,-0.1448539951504302,
      0.5629325029626137,0.4984853033494325,0.4215263456365218,
      0.3337322884960374,0.2386638182445053,0.1411593054699449,
      0.04617547921974115,-0.04233031957129997,0.5971534158422009,
      0.5290500940108692,0.4482053228282282,0.3570208135159443,
      0.2597018771197477,0.1613609902081595,0.06677517278138323,
      0.6235168560971456,0.5515679242489461,0.4673132571654228,
      0.3738575901923395,0.2758011201251887,0.1781713819192711,
      0.6404616222418184,0.5649656062523414,0.4782876117785159,
      0.3840417218310146,0.2868761951248767,0.6476575150457713,
      0.5694506961858633,0.481712063543859,0.388259258354853,
      0.6459894362878276,0.566293293292668,0.4789651252338281,
      0.6371661773752719,0.5573232120043787,0.3200724178002418,
      0.4063802566221164,0.4973180497018747,0.5897273522875156,
      0.6793811951363423,0.7616224484252742,0.8323587928990375,
      0.889001584581435,0.9308933612987835,0.9590813334864968,
      0.3615817483188572,0.4535168742898277,0.5491779429163389,
      0.64449992308253,0.7344437631005388,0.8139835008962493,
      0.8793283338489147,0.9287763706314454,0.9627864725173401,
      0.4036036036586492,0.499949320334643,0.5984781165037405,
      0.6942590478190055,0.7817280923310203,0.8559922020830166,
      0.9140795130247613,0.9554674311422222,0.4441835036049274,
      0.5429866860840018,0.6418227066741695,0.7351609508563529,
      0.8174230627001258,0.8844224346157006,0.9343673632346869,
      0.4809905907165384,0.5797158369815773,0.6759621154318279,
      0.7640818148143264,0.8390728924802671,0.897874775015538,
      0.5117469206751248,0.6076796078041039,0.6986831899155831,
      0.7795497616193484,0.8462988042859838,0.5347729120192694,
      0.6255452713743446,0.7094340284155564,0.7820975288082114,
      0.5493993272212266,0.633404689923654,0.7093243648499448,
      0.5560356639783846,0.6325607438661257,0.5558948726075613,
      0.2502538949373057,0.2834596649308375,0.3171352000240629,
      0.3497321770767574,0.3793963897789465,0.4043068999720242,
      0.4231100429674418,0.4352503773152589,0.4410301813437042,
      0.4413950384816501,0.3401627625801983,0.3789979847088327,
      0.4171897539368579,0.4525063773118708,0.4825397797856335,
      0.5052684266816663,0.5196090830694199,0.5256607773912181,
      0.5245227828309349,0.5178400854028183,0.4357529867703999,
      0.479500221024681,0.5208717223808415,0.5569598687552583,
      0.5850086433327374,0.6031754713474787,0.611055499882272,
      0.6096654752354952,0.6009463532173235,0.5339523560360242,
      0.5809954188619362,0.6233304915365264,0.6575940739282847,
      0.6811006607903708,0.6926539121832439,0.6928333109300957,
      0.6836092937706954,0.6305067000562991,0.6783846661286849,
      0.7188806066405239,0.7486166997908829,0.7654898954879897,
      0.7692993175005894,0.7616477696596397,0.7205836671095895,
      0.7663950141411601,0.8022874547238388,0.8253952737866894,
      0.8345810205360885,0.8306996157029517,0.7997756996573342,
      0.8408319454787708,0.8700831379830764,0.8855671363619817,
      0.8872031228985237,0.8650978792558638,0.8995240516802058,
      0.921218286608597,0.9292366279325103,0.9155019734809123,
      0.9425022788739567,0.9568003701205341,0.951723763021908,
      0.9714657198391669,-0.4375932291383153,-0.3812932275173999,
      -0.3153035222278598,-0.2399902437364576,-0.1569515927579475,
      -0.06907715072573904,0.0198407706589918,0.1058734766109667,
      0.1857686171195431,0.257449626507278,-0.3570040493097855,
      -0.2918648787974354,-0.2168311251285635,-0.1330995445648626,
      -0.04332608784719146,0.04859599442048472,0.1382857743932946,
      0.2218613031927873,0.2966853245585511,-0.2644927501381796,
      -0.1901399993818987,-0.1064219080255857,-0.01561852722247663,
      0.07849995612144045,0.1712924506084217,0.2583107784678281,
      0.3362909838861173,-0.1610782390926968,-0.0780603807507026,
      0.01281177251559359,0.1081011253085713,0.2031694108248999,
      0.2931782765634202,0.3742486038435638,-0.04938676750055992,
      0.04056537812597379,0.1358448755096817,0.232090324110116,
      0.3243479900672576,0.4082538491780348,0.06640211724156458,
      0.1604512568352431,0.2565833049862117,0.3500388266409877,
      0.436217636937124,0.1811879481039926,0.2758835926122972,
      0.3692668145365748,0.4568072375603044,0.2900013554556395,
      0.3820078910424991,0.4697464271137876,0.3890115016151001,
      0.4757292848583335,0.4760519631420977,-0.6231952788307174,
      -0.5646794654796425,-0.4943551945928708,-0.4121616924667699,
      -0.319458133528771,-0.2192092047974507,-0.1156489798772063,
      -0.01344431006605927,0.08328895892415776,0.1716736688754058,
      -0.5573220717056252,-0.4899613221250736,-0.4101426439397438,
      -0.3185871044669488,-0.2177607050610849,-0.1117914195951733,
      -0.005729665044898873,0.09558913699521315,0.1885288856066103,
      0.2710751964644532,-0.4789641985801549,-0.402057686424166,
      -0.3127252940830145,-0.2127814923512051,-0.1059392282183739,
      0.002692533596869102,0.1077444781964869,0.2048204657664428,
      0.2912280153186658,-0.3882586050506625,-0.3019152874380811,
      -0.2041697091828281,-0.09809683500613878,0.01149943312218195,
      0.1190413522149791,0.2195138260022026,0.3094968646292108,
      -0.2868758523956744,-0.1923492526670043,-0.08856892011805101,
      0.02025987059937257,0.1287405357080231,0.2315279747463358,
      0.3245517154572714,-0.1781713502362079,-0.07790941981152795,
      0.02852799618806018,0.136265045008317,0.2399985475821452,
      0.3352293490184233,-0.06677541204276306,0.035950103618558,
      0.1413542883070481,0.2445218382684434,0.3408430926744944,
      0.04232987969953258,0.1440992946503647,0.2452457212250337,
      0.3413681152295799,0.1448534358763926,0.2427634183836315,
      0.3374016489097037,0.2378908589603879,0.3299350928268919,
      -0.2502532651867688,-0.2834590071461157,-0.3171345072414974,
      -0.3497314427767591,-0.3793956104585266,-0.4043060780204903,
      -0.4231091882680272,-0.43524950627169,-0.4410293135613324,
      -0.4413941927799012,-0.1793269230918291,-0.2096296374650262,
      -0.2404488793458187,-0.2702796532141552,-0.2973713527283103,
      -0.3200999575733405,-0.3373822412281449,-0.3489197313348205,
      -0.3551542383465527,-0.09929959410007272,-0.1260192341590947,
      -0.1535127609134815,-0.1804461251889914,-0.2052877394623771,
      -0.2266616403246165,-0.2436963810571767,-0.2561842894548281,
      -0.01142151773467507,-0.034316887225212,-0.05856032689489409,
      -0.08307746147223684,-0.1065840014642066,-0.1278836167145054,
      -0.1461545898453977,0.08175409117371149,0.06228642306456245,
      0.04056025153798869,0.01740518175433122,-0.006048944565669369,
      -0.02862806146519492,0.1765178736510397,0.1594436875323896,
      0.1389956691289133,0.1158878972138483,0.09125781311736192,
      0.2686152102237028,0.2525244737680689,0.2319923070602857,
      0.2077893717181683,0.3541399826172473,0.3376577748776239,
      0.3159635830269856,0.430309819720559,0.4124399032037845,
      0.4957938830064738,-0.975676581463901,-0.9910243213530433,
      -0.9977153903038788,-0.9927800571163499,-0.9738191090293654,
      -0.9397935597648752,-0.8915716840571059,-0.8318771966689975,
      -0.7646064477130079,-0.6938682276631534,-0.9714655004610564,
      -0.983887723902267,-0.9858981447798461,-0.9744145976009966,
      -0.9473804480134012,-0.9046164166427493,-0.8481606115014303,
      -0.7818095106057968,-0.7100822624036758,-0.6371649246085965,
      -0.9568001079664734,-0.9644051676944579,-0.9598482725023617,
      -0.9402907381732713,-0.9044523389503778,-0.8533359449908623,
      -0.7901672201648241,-0.7194988284538555,-0.6459882395962464,
      -0.9292363237529666,-0.9300944286029187,-0.9173973041924147,
      -0.8890014167596085,-0.844773720173384,-0.7869651249304563,
      -0.7196711542084638,-0.6476563908917461,-0.8872027729137049,
      -0.8798270414245473,-0.8582754834679532,-0.8214852287823353,
      -0.7705800268610806,-0.7087709559526547,-0.6404605781008121,
      -0.8306992108488226,-0.8145452502402749,-0.7846557072154475,
      -0.7411783184941589,-0.6862531911113832,-0.6235158879179116,
      -0.7616472974254324,-0.73736034048619,-0.7008201753656432,
      -0.6531910402117445,-0.5971525097007422,-0.6836087421746132,
      -0.652851924507533,-0.612034208511092,-0.5629316404694558,
      -0.6009457148226922,-0.5659076360456838,-0.5232427588825813,
      -0.5178393593330568,-0.4806948574024101,0.5646808729466086,
      0.4943566966479628,0.4121632456350594,0.3194596781650836,
      0.21921067448898,0.1156503154178581,0.013445471844618,
      -0.0832879858164388,-0.1716728777368211,0.5865775784632071,
      0.5109645672417243,0.4225383974740667,0.323390011818691,
      0.2174206286797427,0.1096806787773226,0.005232361633134523,
      -0.09191315247018395,0.6036999320472409,0.5222841738261358,
      0.4275626351684279,0.3225497922064885,0.2120884454656291,
      0.1018140973507329,-0.003178626653090429,0.6137659240175044,
      0.5263406106031267,0.4258274919194899,0.3162311593161161,
      0.2031293654582096,0.09233653282973062,0.6148617703985678,
      0.5217885230992481,0.4167137837086616,0.3044789836562512,
      0.190984387847104,0.6060496757760786,0.5084224141934828,
      0.4006853449688581,0.2881416501332511,0.5877496225758772,
      0.4873191346491355,0.3791778875849738,0.5616447830131567,
      0.460498372897877,0.5301563765120954,-0.4063806941036804,
      -0.4973183240635209,-0.5897274724251101,-0.6793811856410323,
      -0.7616223457476226,-0.8323586364840968,-0.8890014069516364,
      -0.9308931819742911,-0.9590811587055613,-0.3299355917757612,
      -0.4201916721510849,-0.5144019383498233,-0.6086443620415749,
      -0.6980092714824531,-0.7775489279696631,-0.8434725272922868,
      -0.8939999530932097,-0.9294606883512702,-0.9517235558231533,
      -0.3374020539278631,-0.4303972015599547,-0.5261951664998159,
      -0.6201746406216091,-0.7070125356849136,-0.7819076179156583,
      -0.8417962075837926,-0.8859147779324957,-0.9155017573317124,
      -0.3413684282280401,-0.4354200733480757,-0.5306961815651272,
      -0.6221165659419365,-0.7043785843697545,-0.7732956289121359,
      -0.8268002829198581,-0.8650976752328781,-0.3408433114184408,
      -0.4339783095259441,-0.5265312606271311,-0.6133559560669121,
      -0.6896418460594331,-0.752136864699832,-0.7997755164970677,
      -0.3352294641675141,-0.4255076064827776,-0.5134721858345982,
      -0.594338662260662,-0.6641265172984932,-0.7205834943870382,
      -0.3245517106425898,-0.4104116666038574,-0.4925847482169691,
      -0.5669637398252093,-0.6305065080228541,-0.3094967204309826,
      -0.3899712787954628,-0.4658908667926167,-0.5339521035735905,
      -0.2912277152063287,-0.3659561925975448,-0.4357526334612896,
      -0.2710747314715586,-0.3401622786069621,0.7100833249889532,
      0.78181034476627,0.8481612032420389,0.9046167870637368,
      0.9473806530146282,0.9744147135590244,0.9858982469324079,
      0.9838878689052019,0.7194998173208004,0.7901679726328494,
      0.8533364642620689,0.9044526665335126,0.9402909454666769,
      0.9598484396937114,0.9644053616541416,0.7196720611840924,
      0.7869658012318539,0.8447741887331299,0.8890017361264828,
      0.9173975522215362,0.930094678876582,0.7087717858204172,
      0.7705806468939737,0.8214856791906156,0.858275831469383,
      0.8798273607029399,0.6862539610403643,0.741178910976439,
      0.7846561726417634,0.8145456545188131,0.6531917738872118,
      0.7008207681995118,0.7373608444418822,0.6120349280184139,
      0.6528525365910808,0.5659083559960189,-0.3615822177172413,
      -0.2966857537063793,-0.22186166029524,-0.1382860190443388,
      -0.04859608414266816,0.04332618727116257,0.1330998494814693,
      0.2168316303130393,0.2918655611474421,0.3570048758530226,
      -0.4036039458806759,-0.3362912903737734,-0.2583110138480089,
      -0.1712925696560921,-0.0784999126587471,0.01561876770079114,
      0.1064223584250461,0.1901406492265948,0.264493571710179,
      -0.4441837318029271,-0.3742488010631976,-0.2931784021968726,
      -0.2031694140087586,-0.1081009568659323,-0.01281139938873339,
      0.07806096659640296,0.1610790216932629,-0.4809907334514471,
      -0.4082539641847306,-0.3243480295764106,-0.2320902314782855,
      -0.1358446002697818,-0.04056489081311081,0.04938746901646566,
      -0.5117470172320779,-0.4362177044221099,-0.3500388090292672,
      -0.2565831412886956,-0.1604508981073868,-0.06640153973221416,
      -0.5347730026038946,-0.4568072901222747,-0.3692667658371347,
      -0.2758833806070636,-0.1811875286592425,-0.5493994424693911,
      -0.4697464869291438,-0.3820078284652705,-0.2900011116348868,
      -0.5560358190148772,-0.4757293607590583,-0.3890114324926668,
      -0.5558950675560154,-0.4760520530085997,-0.5505634949474449,
      0.9294608819831129,0.8940001336574592,0.8434726858300888,
      0.7775490434174117,0.6980093127768466,0.6086442955222793,
      0.5144017370522948,0.4201913220833389,0.8859149609435977,
      0.8417963565884857,0.7819077242323556,0.7070125813068046,
      0.6201746011540189,0.5261950179989611,0.4303969275523253,
      0.8268004336113636,0.7732957269571541,0.7043786253344242,
      0.622116538987203,0.5306960716781189,0.4354198662954997,
      0.7521369798040806,0.6896418985458221,0.6133559483764451,
      0.5265311900255359,0.4339781689134173,0.664126617868866,
      0.5943387021482599,0.5134721735844615,0.4255075445403705,
      0.5669638671854947,0.4925848265160583,0.4104117066697837,
      0.4658910697532281,0.3899714480143811,0.3659565132385014,
      0.3551551111068512,0.3489206356503204,0.3373831583083507,
      0.3201008675109025,0.2973722384562898,0.2702805035644438,
      0.2404496897688697,0.2096304085552757,0.1793276582557128,
      0.2561851603269178,0.2436972866599269,0.2266625621335284,
      0.2052886581368649,0.1804470242976833,0.153513629451971,
      0.1260200668693211,0.09930039009433847,0.1461554161525869,
      0.1278844724735265,0.1065848698133528,0.08307832599156778,
      0.05856117448713494,0.03431770949468276,0.01142231063250637,
      0.02862879196054897,0.006049691633511915,-0.01740443052828812,
      -0.04055950638179381,-0.06228569088648633,-0.08175337578691833,
      -0.09125722755264572,-0.1158873114023829,-0.1389950879223139,
      -0.1594431124986218,-0.176517304179968,-0.2077889659882155,
      -0.2319919155781195,-0.2525240921328624,-0.2686148310916902,
      -0.3159633704917529,-0.3376575857943411,-0.3541398039589991,
      -0.4124398762523879,-0.430309819678344,-0.4957940195469321,
      0.1664325418685087,0.08593872296651928,-0.001855644740443331,
      -0.09380768650859528,-0.1857297549973136,-0.2732409243006708,
      -0.3527536454451026,-0.4221175876781037,0.06444579340302205,
      -0.02420720081803753,-0.1182659371646478,-0.2135248270679241,
      -0.3051768592760364,-0.3889397838050994,-0.4619850515450297,
      -0.04617573152762199,-0.1411593366285305,-0.238663612601281,
      -0.3337318558644699,-0.4215257202593218,-0.4984845341024346,
      -0.1613609945604779,-0.2597016312374675,-0.3570203295626905,
      -0.4482046405142344,-0.529049270280373,-0.2758008261967574,
      -0.3738570466287534,-0.467312506887279,-0.5515670310282831,
      -0.3840411190982298,-0.4782867918076852,-0.5649646363950814,
      -0.4817111823705308,-0.5694496536902212,-0.5662921921093249,
      0.0919140401962618,-0.005231313174239269,-0.1096794790522038,
      -0.2174193094193431,-0.3233886246408304,-0.4225370035538252,
      -0.510963223483816,-0.5865763269261466,-0.6490697854071827,
      0.003179550365269548,-0.1018130528605821,-0.2120873052250725,
      -0.322548598821141,-0.4275614411441423,-0.5222830294256716,
      -0.6036988748207749,-0.6708921376896406,-0.09233565318963641,
      -0.2031284196976215,-0.316230181653878,-0.4258265246770868,
      -0.5263396945379846,-0.6137650892643368,-0.6863335534851791,
      -0.190983636619622,-0.304478224282928,-0.4167130490396238,
      -0.5217878437313506,-0.6148611676086653,-0.6931813485878851,
      -0.2881410938833685,-0.4006848272509508,-0.5084219568812576,
      -0.6060492919686921,-0.68991835321825,-0.3791775567551729,
      -0.4873188675145023,-0.5877494261961844,-0.6762335873429084,
      -0.4604982573084285,-0.5616447337445059,-0.653178182658568,
      -0.5301564355851682,-0.6228580878699612,-0.5878382593304642,
      -0.5245220270255621,-0.5256600112160642,-0.5196083293421272,
      -0.5052677071807448,-0.4825391100490625,-0.4525057633827704,
      -0.4171891927640138,-0.3789974674618629,-0.6096648147710031,
      -0.6110548409057,-0.6031748400009704,-0.5850080622199078,
      -0.5569593510286511,-0.5208712693637837,-0.4794998244238688,
      -0.6928327439316474,-0.6926533571029836,-0.6811001461047985,
      -0.6575936218265264,-0.6233301120657577,-0.5809951090033914,
      -0.769298834842142,-0.7654894328832393,-0.7486162870521085,
      -0.7188802647693375,-0.6783844022430058,-0.8345806089171193,
      -0.8253948883356401,-0.8022871256496356,-0.766394760848657,
      -0.8855667831280848,-0.8700828159137221,-0.840831684134326,
      -0.9212179835682319,-0.8995237842791114,-0.9425020232333647,
      -0.7226186971374691,-0.794891657666899,-0.8616488184692596,
      -0.9183223647655301,-0.9610863946187903,-0.9879028002301763,
      -0.9989802852434637,-0.9964241505078985,-0.9833703523103508,
      -0.7456631063614969,-0.8173587433845655,-0.8812051016459226,
      -0.9325588839421305,-0.9681598882374777,-0.9870397834787261,
      -0.99056942924076,-0.9817039872478999,-0.7601357246335175,
      -0.82876883364816,-0.887298854659372,-0.931526545874374,
      -0.9592009976026995,-0.9705589758896634,-0.9679134580585838,
      -0.7635946484712315,-0.8267930492778305,-0.8781853684611858,
      -0.9144884914916022,-0.934650707238976,-0.9399235077793813,
      -0.7547827674388909,-0.8108510779697197,-0.8543284576557796,
      -0.8830748007745595,-0.8971016370404786,-0.7341200461470681,
      -0.7823945479956939,-0.8182893261768063,-0.8407283372889187,
      -0.7036089418396923,-0.7444266467538491,-0.7738552525426188,
      -0.6661987769113784,-0.7005610213599369,-0.6249827499272808,
      -0.1885282693989691,-0.09558834947432154,0.005730628355370741,
      0.111792540109273,0.2177619400763412,0.3185883951684442,
      0.4101439293014277,0.4899625516770138,-0.2048200319759604,
      -0.1077438933887955,-0.002691796830558471,0.1059400956623477,
      0.2127824486290439,0.3127262866621893,0.4020586651367561,
      -0.2195135736243867,-0.1190409791973846,-0.01149894225597868,
      0.09809742254017301,0.2041703576935574,0.3019159562443602,
      -0.2315278949190421,-0.1287403742204129,-0.02025963236651961,
      0.08856921814263634,0.1923495858173033,-0.2399986269164587,
      -0.136265086596977,-0.02852800385427971,0.07790943731694504,
      -0.2445220631569563,-0.1413545191115968,-0.03595033950187966,
      -0.2452460806309568,-0.1440996982422414,-0.2427639040507477,
      -0.9627863463864214,-0.9287762212458259,-0.8793281693046819,
      -0.8139833471872829,-0.734443661783315,-0.6444999213728085,
      -0.5491780807749006,-0.4535171752887245,-0.9554673658442766,
      -0.9140794058749131,-0.8559920650689382,-0.7817279594934516,
      -0.6942589693680463,-0.5984781448346268,-0.4999494955348047,
      -0.934367366569149,-0.8844223798763841,-0.8174229679769948,
      -0.7351608585502457,-0.641822673191966,-0.542986764989374,
      -0.8978748469678216,-0.8390728949381593,-0.7640817734701692,
      -0.6759620794963979,-0.5797158650472573,-0.8462989355647894,
      -0.7795498187240726,-0.6986832060861172,-0.6076796358995511,
      -0.7820977027354027,-0.709434131000089,-0.6255453415851245,
      -0.7093245618847172,-0.6334048252011675,-0.6325609463354179,
      -0.1778161375899129,-0.1226310892230631,-0.06053925384414331,
      0.007442324388539504,0.07929679392711581,0.1521341729788544,
      0.222673458561735,0.2879340035685283,0.3458247516791153,
      0.3953823593491522,0.4366423972091846,-0.08876371160394406,
      -0.02528588772613339,0.04476837629156977,0.1195214252181962,
      0.1959986327009231,0.2705464283542062,0.3396119290589926,
      0.4005431057992871,0.4520218529859129,0.4940014731008501,
      0.01030826616734189,0.08198490466663314,0.1591522280204451,
      0.2389097207973864,0.3173816763430465,0.3904924510986985,
      0.4549463955350546,0.5089146064888007,0.5521270265729551,
      0.117472560686524,0.1962648015037478,0.2785512690899858,
      0.3604576080817683,0.4375569230040101,0.505925367029105,
      0.5630823357891753,0.6083330877322986,0.2292788658415479,
      0.3129719884702788,0.3973400932411465,0.4778825649301736,
      0.5502139834879405,0.6111631805503013,0.6594089221868847,
      0.3410385404813853,0.4265108902934403,0.5094170629259336,
      0.5852084170909949,0.6501542610793901,0.7022080829625521,
      0.4476465561008348,0.5314703253828574,0.6096570464011057,
      0.6781972947322877,0.7343998566036512,0.5446711990330303,
      0.6238421950486808,0.6949971116211809,0.7550398975145011,
      0.629214796874201,0.7016703235997708,0.7646693979379596,
      0.7001887122370751,0.7649403475532479,0.7580253719184178,
      -0.5980610514568761,-0.5589881370818129,-0.5101530988159087,
      -0.4510385045651341,-0.382225667160838,-0.3056524524744078,
      -0.2244621267538426,-0.1424014047662905,-0.06301328229424107,
      0.01100431499766755,0.07805400320688782,-0.5212000928328268,
      -0.473821384591564,-0.4161517087436187,-0.3483435180237136,
      -0.2719386249037586,-0.1899104644827079,-0.106168437105403,
      -0.0246774460389087,0.05138083746707222,0.120041421580981,
      -0.4311039309963852,-0.3746268308434739,-0.3079662136138592,
      -0.2322443529582202,-0.1501157132113724,-0.06545131715451497,
      0.01750888497279251,0.09514231706859014,0.1650825345160538,
      -0.3282706848689272,-0.2627160067380389,-0.1880137321428399,
      -0.1064226243048281,-0.0215868970434456,0.06217585792644167,
      0.1409092129286223,0.2118443492085483,-0.2148810450151756,
      -0.1414239428971924,-0.06090095222676627,0.0234070437450894,
      0.1073128739652992,0.1866888371989906,0.2584097661066967,
      -0.0948797286874359,-0.01589086440904075,0.06724059985094104,
      0.1506371707417032,0.2301937869064764,0.3025309778153461,
      0.02655484252908358,0.1079004271192526,0.1901297170957776,
      0.269350641258243,0.3420822257932489,0.1440569199487702,
      0.2244719783102178,0.3027669323023652,0.3755350385655634,
      0.2531835106264871,0.3299301683341083,0.4022303494272352,
      0.3510968726936461,0.4223460594339758,-0.07805410188638827,
      -0.09276641502761757,-0.1080255529483224,-0.123231540300325,
      -0.1376217050758367,-0.1503967869358663,-0.1609000070073124,
      -0.1687715154704865,-0.1740018618448228,-0.1768698604837748,
      0.005852711686888685,-0.004907458921365185,-0.01667363978400245,
      -0.02910230703937302,-0.04167551826680154,-0.05377893240309974,
      -0.06483570957325965,-0.07443642513334246,-0.08240206216970784,
      0.09827676798573926,0.09169789424305551,0.083291898217249,
      0.07306224170584404,0.06127443921955168,0.04845159286147566,
      0.03526739273256396,0.0223825152995957,0.1971676395871742,
      0.1943681810797974,0.1884952036420767,0.1792304305890973,
      0.1667127428266929,0.1515792114414989,0.134812448209863,
      0.2991139104294396,0.2989413652639448,0.2941068360088736,
      0.2841377396140656,0.2692865316145088,0.2505276366498019,
      0.399677322182528,0.4003768622965933,0.3947504029769399,
      0.382445062756354,0.3640665537863402,0.4942032775296958,
      0.4938403319910914,0.4857723178878524,0.470018015874229,
      0.5788262679961029,0.575792452883696,0.5642398722628517,
      0.6512069539966677,0.6445542311848552,0.7107016666079528,
      -0.9161616153729886,-0.9319303956122561,-0.9396953110011561,
      -0.9366313083669698,-0.9204280785344878,-0.8900379255247357,
      -0.8462030522374957,-0.7914228667351423,-0.7293237120999879,
      -0.663758722300958,-0.8876449455755545,-0.8994356085395798,
      -0.9017511736022054,-0.8917607021504116,-0.8675634598042835,
      -0.8289706205471806,-0.7778214867506247,-0.7175692482251322,
      -0.6523369554154551,-0.5859517695425136,-0.8470541513588044,
      -0.8530863623730024,-0.8482966176587544,-0.8301999149225319,
      -0.7977006542517769,-0.7517266504248017,-0.6951661565374421,
      -0.6320680091083589,-0.566558592627622,-0.7927543534178244,
      -0.7914101005217242,-0.7783641365944748,-0.7518828443693624,
      -0.7119690859200297,-0.660654211393362,-0.6015113260745434,
      -0.5386394371453509,-0.7243096319272092,-0.7146076494872069,
      -0.6931460376496088,-0.6592387683821411,-0.6140043047773551,
      -0.5602361841518015,-0.5016343691560573,-0.6430242175142256,
      -0.6250295478554301,-0.596219325041805,-0.5569826617529877,
      -0.5092509168566325,-0.4560492515462431,-0.5520254073275178,
      -0.5269450103103401,-0.4928644880867128,-0.4509854395387125,
      -0.403575153350467,-0.4557100264506188,-0.425580431270515,
      -0.3887439926860071,-0.3468035231429711,-0.3587591578566765,
      -0.3258932157449829,-0.2886351685087218,-0.2651897825092151,
      -0.2316682604407079,0.5980613131647216,0.5589883601394342,
      0.5101532951859686,0.451038686721687,0.382225843595672,
      0.3056526230838691,0.2244622808787926,0.1424015236007947,
      0.06301334452030186,-0.01100432747032814,0.6470944588058076,
      0.6063945120633992,0.5544956308560397,0.4909814474854842,
      0.4168791394819704,0.3348509827825754,0.248806364862682,
      0.163021328012756,0.08119217116689301,0.6944632949786616,
      0.6513144930113939,0.5955168212825119,0.5269608417915713,
      0.4473425940100297,0.3601678525102263,0.2700417838303327,
      0.1815453704759045,0.7372825381163968,0.6906294333400347,
      0.6300883699674214,0.5561262995663098,0.4712906267013536,
      0.3798989213097872,0.2870043862751919,0.7724043956082883,
      0.7211854504544825,0.6553545192922715,0.576197864204669,
      0.4871408620353512,0.3930729578844198,0.797087559189558,
      0.7405927447841627,0.6695504697315396,0.5861540209201834,
      0.4945084290202527,0.8097301284690857,0.7479263796731634,
      0.6725220182496192,0.5864764631401966,0.8103038862615974,
      0.7439345925004668,0.6656398165194758,0.8002534097038426,
      0.7306537225524881,0.7819506470598,-0.4366431953633789,
      -0.5106666754492736,-0.5869882376922511,-0.6624740500734984,
      -0.7332080507197046,-0.7951587738972527,-0.8450980113065225,
      -0.8813674368391481,-0.9041113586460733,-0.9149201193897882,
      -0.4223467825896815,-0.4979651901691212,-0.5751701148400569,
      -0.6502749910785526,-0.7189488051634834,-0.7771253015026822,
      -0.8220318406135049,-0.8528138420803982,-0.8704523073313828,
      -0.877132051866122,-0.4022310083998925,-0.4781163288807728,
      -0.554596445154436,-0.6275642260210416,-0.6925605687496104,
      -0.7458582025116769,-0.7854318984598006,-0.8112889263329699,
      -0.8250621271173465,-0.3755356470489176,-0.4501425958887488,
      -0.5241864933378524,-0.5934002650897944,-0.6535679922027033,
      -0.7016275314245141,-0.7364042366101387,-0.758599491836005,
      -0.3420827953668352,-0.4139041914442195,-0.4840440064641756,
      -0.5484062706939985,-0.6033456975789954,-0.6466104559972755,
      -0.6777531805937266,-0.3025315118038733,-0.3703991963484116,
      -0.435725075678865,-0.4948721352375582,-0.5449289606848956,
      -0.5843857415998934,-0.2584102557043402,-0.321701018732575,
      -0.3819753792546441,-0.436205800680254,-0.4821906665520286,
      -0.2118447747227839,-0.2704914294051139,-0.3260202911877148,
      -0.3760152101544279,-0.1650828712784331,-0.2194154020550029,
      -0.270790845781693,-0.1200416470582943,-0.1705734571487906,
      0.9161619708326184,0.9319307491762596,0.9396956424389374,
      0.9366315992939006,0.9204283182965946,0.8900381161951118,
      0.8462032095340455,0.7914230156292639,0.7293238793541417,
      0.6637589298557965,0.9487876867424746,0.9638219991954514,
      0.9689402187057976,0.9610351558002814,0.937956150415152,
      0.8993632776707052,0.8470958517094176,0.7847581799295993,
      0.716723246034345,0.9749588444840027,0.9874641413480602,
      0.9879501207294071,0.9733335913695014,0.942053498973333,
      0.8948602911895638,0.8348196077814718,0.7664457510261931,
      0.9913973712326389,0.9992329380155185,0.9930645438165779,
      0.9702883664051513,0.9303746073596573,0.8753546246993716,
      0.8093341962935008,0.9950795014807369,0.9961777933609237,
      0.9818515654328379,0.9504460903479695,0.9027098746674149,
      0.841806420960272,0.9840623116657442,0.9769921932350635,
      0.9540532376241087,0.9148166158751114,0.8612136882276913,
      0.9581801446138297,0.9425905156295621,0.9118246030313639,
      0.8666310652479987,0.9192101668893,0.895902096029785,
      0.8590657328509213,0.8703772282258925,0.8409823115598557,
      0.8154720767160899,0.1778171059718252,0.1226319330808614,
      0.06053992567271226,-0.007441870151939188,-0.07929659020903117,
      -0.1521342296531137,-0.2226737578340799,-0.2879345044307138,
      -0.345825401239635,-0.3953831051073846,0.2316691816765674,
      0.1757724390286309,0.1122685379972423,0.04238946123155791,
      -0.03147431095673338,-0.1060223690705309,-0.1777018008392572,
      -0.2435073681639464,-0.3015367950655638,-0.3510977114791443,
      0.2886360377097776,0.2320884584878828,0.1672516508448342,
      0.09562800512668546,0.02000533874392893,-0.05595525429524186,
      -0.1285435155191929,-0.1948426136371608,-0.2531843553864728,
      0.3468043362453595,0.289467434229511,0.2232553503091939,
      0.1500085463918315,0.07290895867983387,-0.004119533631953385,
      -0.07735131409521774,-0.1440577225948253,0.403575906447316,
      0.3451254769459608,0.2774342678683828,0.2027388030014913,
      0.1245598363284875,0.04693326425220518,-0.02655554762561945,
      0.4560499397602033,0.396110678421097,0.3269267724177788,
      0.2511351438644352,0.1724665161635576,0.09487917246041,
      0.5016349858535857,0.4399931207714222,0.3695530582277636,
      0.2932656162238703,0.2148806720954671,0.5386399743309223,
      0.4754121349188742,0.4042569522642996,0.3282705050396187,
      0.5665590425344393,0.5022156935301549,0.431103930292903,
      0.5859521265200787,0.5212002413653404,0.5106659414837471,
      0.5869876102086139,0.6624735709935894,0.7332077514676827,
      0.7951586652134928,0.845098078457225,0.8813676435997579,
      0.9041116580482536,0.9149204654298684,0.5734213362282777,
      0.6539109564930851,0.7314600073324019,0.8014445481530106,
      0.8596212046135118,0.9032173146992399,0.9315553799266565,
      0.9459093111069366,0.6355985255876813,0.7182616282077119,
      0.7953081408089798,0.8617334421407644,0.9136023622650965,
      0.9490975365686583,0.9687717311497821,0.6936983436880695,
      0.7758024684863688,0.8493585337677274,0.9095263441824443,
      0.9532437439744202,0.9799603432594377,0.7438876747925507,
      0.8223898048944452,0.8896840618492347,0.9416915744235097,
      0.9765940848309838,0.782878718417177,0.8550853688846947,
      0.9142324281909144,0.9574084820329141,0.808813197110931,
      0.8729720010540123,0.9233179132328828,0.8216808702627202,
      0.8772096742129835,0.8230696091572214,0.09276640377310574,
      0.1080256414522809,0.1232317408385954,0.1376220280275969,
      0.1503972383762519,0.1609005865750696,0.1687722156036723,
      0.1740026689030255,0.1768707572388351,0.1705731738763045,
      0.1906930126322413,0.2105464412580206,0.2289927972013699,
      0.2447929403327111,0.2568964854662051,0.2647265853183172,
      0.2683098464673112,0.2681884415185128,0.2651905971794372,
      0.2707904196202965,0.2960475246517403,0.3196768235430837,
      0.3399487493807457,0.3552546724685221,0.3645596525653644,
      0.3677018240803483,0.3653636154505258,0.3587598208776521,
      0.3760147097386288,0.4053026883769021,0.4310210766569808,
      0.4509771335793986,0.4634949414172432,0.4679374232163439,
      0.464858993637328,0.4557105558116591,0.4821901792282771,
      0.5135270414103618,0.5389508449256169,0.5561389504462149,
      0.5637713635689835,0.5619411906656432,0.5520258363563475,
      0.584385354546882,0.6152086674144305,0.6377766977965403,
      0.6500821302058097,0.651519250667168,0.6430245879023526,
      0.6777529577987501,0.7055014818762047,0.7231337276202411,
      0.7293239492942762,0.724309982145211,0.7585994584260806,
      0.7813166545183711,0.7928692947394784,0.7927547081501097,
      0.8250622687013296,0.8417304368393459,0.8470545173149734,
      0.8771323267159262,0.887645314498864,0.3015360453845148,
      0.2435067604258691,0.1777013915256129,0.1060222058462941,
      0.0314744189369969,-0.04238908731366174,-0.1122679310637554,
      -0.1757716484888774,0.1948419043847709,0.1285429999155006,
      0.05595498503480459,-0.02000532948058562,-0.0956277167931003,
      -0.1672511147059996,-0.2320877267540867,0.07735069910586773,
      0.004119161966720001,-0.07290904878978868,-0.1500083477710563,
      -0.2232548905422917,-0.2894667665085293,-0.04693373219192027,
      -0.1245600244829796,-0.2027386973931088,-0.2774338902981233,
      -0.3451248789428226,-0.1724667992141356,-0.2511351336125937,
      -0.3269264832443289,-0.3961101570586181,-0.2932657019670711,
      -0.3695528631494325,-0.4399926839835791,-0.4042568549159931,
      -0.475411790223567,-0.5022154461218424,-0.005852598553161901,
      -0.08119196505793361,-0.163021049159767,-0.2488060411927053,
      -0.3348506437156618,-0.4168788074510346,-0.4909811315875961,
      -0.554495326274857,-0.6063942048702905,-0.6470941325142202,
      -0.09827641615811868,-0.1815449292413942,-0.2700412859530667,
      -0.3601673374231551,-0.4473420975374328,-0.5269603854089682,
      -0.5955164071695848,-0.6513141073946994,-0.6944629164413806,
      -0.1971670439169432,-0.2870037215224481,-0.3798982345693028,
      -0.4712899665970914,-0.5561257020948373,-0.6300878481113522,
      -0.6906289766699522,-0.7372821213716372,-0.2991130971968019,
      -0.3930721160778319,-0.4871400501186961,-0.5761971323316351,
      -0.6553538941234454,-0.7211849292226609,-0.7724039524031648,
      -0.399676350745083,-0.4945074863044202,-0.5861531681464745,
      -0.6695497477032463,-0.7405921623591333,-0.7970870947583817,
      -0.4942022299541438,-0.5864755076757882,-0.6725212074710563,
      -0.74792573591698,-0.8097296395344389,-0.5788252298638613,
      -0.665638926578608,-0.7439338855424372,-0.810303362756081,
      -0.6512059956089504,-0.7306529513600663,-0.8002528392148971,
      -0.7107008326980349,-0.7819500193454378,-0.7580246814516085,
      -0.2681877205946842,-0.2683092406063016,-0.2647261130437045,
      -0.2568961585702784,-0.2447927610385159,-0.2289927579868598,
      -0.2105465274520093,-0.1906932062912961,-0.3653630641630742,
      -0.3677014077761052,-0.3645593896043551,-0.3552545716101517,
      -0.339948807237406,-0.3196770257819652,-0.2960478502452225,
      -0.464858585918057,-0.4679371612123701,-0.4634948428974661,
      -0.4509772049977542,-0.4310213111152231,-0.4053030678481073,
      -0.5619408834098191,-0.5637712030900536,-0.5561389551781383,
      -0.5389510214534028,-0.5135273825534448,-0.651518994572369,
      -0.6500820157815153,-0.6377767462064369,-0.6152088877366557,
      -0.7293237033806105,-0.7231336172569296,-0.7055015320014856,
      -0.7928690367045447,-0.7813165283080149,-0.841730164442513,
      -0.7167229918086124,-0.7847579855913325,-0.8470956951046914,
      -0.8993631303768833,-0.9379559853910195,-0.9610349555582707,
      -0.968939979478583,-0.9638217293378932,-0.9487874014792429,
      -0.7664454690716618,-0.8348194119106425,-0.8948601580568658,
      -0.9420533966954356,-0.9733334885435185,-0.9879499956150448,
      -0.9874639861492262,-0.9749586635216289,-0.8093339051128812,
      -0.8753544488679451,-0.9303745202617085,-0.9702883306150087,
      -0.9930645219898977,-0.9992329031534651,-0.9913973109989458,
      -0.8418061322197662,-0.9027097279159257,-0.9504460536566606,
      -0.9818515951566739,-0.996177846854893,-0.9950795477220543,
      -0.8612134015838815,-0.9148164899330201,-0.9540532347842757,
      -0.9769922654221862,-0.9840624137666333,-0.8666307679449418,
      -0.9118244750171576,-0.9425905155806792,-0.9581802235578871,
      -0.8590654047339893,-0.8959019370925372,-0.9192101353217356,
      -0.8409819323578077,-0.8703770126934449,-0.8154716315810106,
      -0.05138099314083437,0.02467734836797703,0.1061683820305035,
      0.1899104376062115,0.2719386182644327,0.3483435331583406,
      0.4161517553821363,0.4738214760951626,-0.09514260413767517,
      -0.0175091339170676,0.06545109551523358,0.1501155140512474,
      0.2322441809962577,0.3079660822386824,0.3746267572069504,
      -0.1409096123632751,-0.06217624169087836,0.02158652471546835,
      0.1064222699680223,0.1880134132638135,0.262715746535069,
      -0.1866893275259466,-0.1073133727582037,-0.02340754682308935,
      0.06090046334304851,0.1414234968618731,-0.2301943507183196,
      -0.1506377672576039,-0.06724121511703012,0.01589026002363016,
      -0.2693512698554454,-0.1901304002696938,-0.1079011406857555,
      -0.3027676257747358,-0.224472743157967,-0.3299309317589397,
      -0.9459090564395943,-0.9315551937272754,-0.9032172391195878,
      -0.8596212762960382,-0.8014447869733361,-0.7314604096467703,
      -0.6539114969940219,-0.5734219768623257,-0.4940021735847102,
      -0.9687716041275991,-0.9490974969653682,-0.913602441262072,
      -0.8617336589899791,-0.7953084953825064,-0.7182621005240754,
      -0.6355990826652251,-0.5521276321758419,-0.9799603552305356,
      -0.9532438509855897,-0.909526560381993,-0.8493588583184724,
      -0.77580288428704,-0.6936988224175965,-0.6083335981356588,
      -0.9765942083818028,-0.941691783045487,-0.8896843516207826,
      -0.8223901593137167,-0.7438880688474475,-0.6594093292610237,
      -0.9574086493790075,-0.9142326541123561,-0.8550856370062082,
      -0.782879006233477,-0.7022083677250226,-0.9233180317911697,
      -0.872972144171723,-0.8088133452877488,-0.7343999908188845,
      -0.877209652594128,-0.8216808432264592,-0.7550398503916647,
      -0.8230693796089513,-0.7646691446910742,-0.7649398754101678,
      0.6523372575426411,0.7175695092343044,0.7778217282247372,
      0.828970867566289,0.867563733535706,0.8917610129438806,
      0.9017515185532632,0.8994359744121508,0.6320684006265465,
      0.6951665021597787,0.7517269700406279,0.7977009700656229,
      0.8302002440404257,0.8482969664746548,0.8530867259579973,
      0.6015117982089927,0.6606546288416104,0.7119694660426552,
      0.7518832062338756,0.7783644942261146,0.7914104587099922,
      0.5602367267427266,0.6140047811934269,0.6592391933644046,
      0.6931464276818936,0.7146080169943935,0.509251521131838,
      0.5569831878642361,0.5962197855090862,0.6250299573281404,
      0.4509860998418013,0.4928650597255946,0.5269455040116824,
      0.3887447062136717,0.4255810481267848,0.3258939808872747,
      -0.4520224891160809,-0.4005436259417036,-0.3396122734793814,
      -0.2705465411303475,-0.1959984769042034,-0.1195209939316481,
      -0.04476769320458044,0.02528677771990007,0.08876475562044144,
      -0.5089151305978357,-0.4549467775084718,-0.3904926248184883,
      -0.3173815862988101,-0.2389093389601502,-0.1591515620353438,
      -0.08198399284624955,-0.01030716362341688,-0.5630827400946593,
      -0.5059255957776372,-0.4375569069570671,-0.3604572977868911,
      -0.2785506506829446,-0.1962638993317844,-0.1174714262849324,
      -0.6111634481983498,-0.5502140363721867,-0.477882336850574,
      -0.3973395475484636,-0.3129711292234176,-0.2292777334167206,
      -0.6501543660224874,-0.5852082672248219,-0.509416602839924,
      -0.4265101014752001,-0.3410374453295618,-0.6781972070318585,
      -0.609656670182737,-0.531469622469496,-0.4476455277450017,
      -0.6949968061057114,-0.6238415797897515,-0.5446702557332809,
      -0.701669787076208,-0.6292139442700462,-0.7001879458668289,
      0.8704525623995767,0.8528140389888319,0.8220319347625164,
      0.7771252518482044,0.7189485862927419,0.6502746013906258,
      0.5751695760111849,0.497964538802267,0.8112890338044526,
      0.7854319364049284,0.7458581338540565,0.6925603649758249,
      0.6275638768849461,0.5545959620739339,0.4781157400838166,
      0.7364041563355311,0.7016273774526178,0.6535677410698937,
      0.5933999047634313,0.5241860281758185,0.4501420452269545,
      0.6466101827787859,0.6033453603619342,0.5484058620979402,
      0.4840435291285519,0.4139036582576368,0.5449285342030792,
      0.4948716695409166,0.4357245760359055,0.3703986729586991,
      0.4362052982975687,0.3819748711371402,0.3217005151313638,
      0.3260198057983744,0.270490970101615,0.2194150172838141,
      0.08240304061283914,0.07443732100834362,0.06483650679882089,
      0.05377961812205672,0.04167608470977979,0.02910275219613371,
      0.01667396676186753,0.004907674513220617,-0.0223814688467947,
      -0.03526641653115874,-0.04845070073398356,-0.06127364342066123,
      -0.07306155181298434,-0.0832913202753871,-0.09169743029697446,
      -0.1348113569049385,-0.1515781730059853,-0.1667117684090611,
      -0.1792295328710818,-0.1884943959488113,-0.1943674753097082,
      -0.2505265309683513,-0.2692854573778917,-0.2841367059314813,
      -0.2941058574917593,-0.2989404603106648,-0.3640654661632947,
      -0.3824439829710035,-0.394749339616747,-0.4003758331308906,
      -0.4700169740867185,-0.4857712605383084,-0.4938392688789908,
      -0.5642388946704648,-0.5757914389111062,-0.6445533256821902,
      -0.1282040992012934,-0.05321855554427612,0.02998172476739921,
      0.1197175263736552,0.2130449739264662,0.3060111300013609,
      0.394354771181159,0.4744188270546212,0.5438573645627299,
      0.6018455943795508,0.6488215902264565,-0.1584409684968982,
      -0.08061941460724659,0.005851663594817331,0.09880655117780221,
      0.1946724526862442,0.2889433394395604,0.3771263430050333,
      0.4557536219427536,0.5229695128140037,0.5785062522661573,
      -0.1901340637026595,-0.1097761406561385,-0.02057293935230464,
      0.07475784175875169,0.1720544722828635,0.2664447755897635,
      0.3534794142829396,0.4301062418911859,0.4950335464190314,
      -0.2221347426415571,-0.1396269276604669,-0.04840420657561294,
      0.04825793538561177,0.1457553988926505,0.2391262056337631,
      0.3242394390900903,0.3985926425219686,-0.252933321812349,
      -0.168751250747966,-0.07636778766011496,0.02046791155100532,
      0.1169519253626288,0.2083323398966038,0.2909961752861106,
      -0.2808945465452747,-0.1956215915798094,-0.1030413667410095,
      -0.007197896555794293,0.08719109671857094,0.1758296024995542,
      -0.304612640171253,-0.218957329454675,-0.1271903099934383,
      -0.03341079168503205,0.0580070042064605,-0.3232363968742333,
      -0.2380131586083366,-0.1480327683071742,-0.05721040323642156,
      -0.3366056805211806,-0.2526623104055841,-0.1653138037361849,
      -0.3451559795786212,-0.2632729700486575,-0.3496821715684524,
      0.6714420577705413,0.7363064649907652,0.8002044514563711,
      0.8596329319699905,0.9106424580428781,0.9496341036793503,
      0.9742808059046055,0.9841303617506333,0.9805708386415104,
      0.9662289123338648,0.9441720387917811,0.7051823376417343,
      0.772857969148872,0.8380824708147486,0.8966582333398196,
      0.9442212565687949,0.9773096449869836,0.994346428784018,
      0.9959947242778775,0.9847131791126237,0.9638452720539006,
      0.7350956003099328,0.8038805096388253,0.8682639008659977,
      0.9235417715931376,0.9653353535299492,0.9908225390539687,
      0.9995536316680411,0.9933753693114809,0.9755844796257857,
      0.7585152302781616,0.8261311652202954,0.8871112537106268,
      0.9366413319148751,0.9708622908203767,0.988032496009293,
      0.9889451040531339,0.9763862533615466,0.7728091190204586,
      0.8367229328771276,0.8918537226509272,0.9338031104143576,
      0.9597077065970592,0.9690768333338338,0.9637247890765801,
      0.7760312926982586,0.8339786000659982,0.8815310666159923,
      0.9151715385907231,0.933244106655567,0.9363385958760152,
      0.767530944505584,0.8180289491084085,0.857374860312736,
      0.8831546795136415,0.8948082473172733,0.7481893791144126,
      0.790783305063361,0.8223658994270935,0.8415629406712848,
      0.7201359303293944,0.755286693908195,0.7802583897718675,
      0.6861124167319267,0.7148528823763073,-0.9441722324566724,
      -0.9089826616996974,-0.8608166107545396,-0.798145123044153,
      -0.7207644955095487,-0.6303269409870232,-0.5303678229575245,
      -0.4256669568776663,-0.3211867088850157,-0.221088326716299,
      -0.9323220393450906,-0.8912341788673274,-0.8355390934902637,
      -0.7640503287556382,-0.6773887436471858,-0.5783931938554159,
      -0.4717817419682783,-0.3630890689211784,-0.2573948493808621,
      -0.9096404828216634,-0.8611864915849053,-0.7967975927156801,
      -0.7160071810801653,-0.620601831095295,-0.5147133115844718,
      -0.4039985827676406,-0.2942465140581043,-0.873971359384776,
      -0.8169372490707259,-0.7433257505302604,-0.6537697064615564,
      -0.5513858884167668,-0.4413849791363659,-0.329823034867975,
      -0.8241231220089414,-0.7579997417459179,-0.6757043639889994,
      -0.5791759017751473,-0.4726959329165278,-0.3620176676150795,
      -0.7605084662990487,-0.6858745237990584,-0.5966690200428781,
      -0.4960221196000292,-0.388888002229758,-0.6854057579633669,
      -0.603999039701328,-0.5106159168102177,-0.4091534188855591,
      -0.6025948313141819,-0.5169734519149861,-0.4224835049786242,
      -0.5164821811548767,-0.4294377118759299,-0.4311427045530306,
      -0.3130828685601688,-0.2261777090945995,-0.128054626578418,
      -0.02022860607974414,0.09418349997750548,0.2106381298878162,
      0.3239109228229815,0.4291933772271586,0.523048087763098,
      0.6038168419091856,-0.3104444323030748,-0.2176605469020037,
      -0.1131261690200332,0.0009102704499983739,0.1203702792920805,
      0.2398156109769667,0.3535535822768655,0.4569191625595045,
      0.5471127369113119,0.6232994222041273,-0.3043330399192493,
      -0.2057260225894636,-0.09531787499064681,0.02374459641152675,
      0.146419102006115,0.2666203176913846,0.3786227553496849,
      0.4783283811020137,0.5638039035645359,-0.2939990078566014,
      -0.1900378279544306,-0.07488904582360796,0.04735210953525339,
      0.1708856651832371,0.2894245513177179,0.3977010533927281,
      0.4924894823466497,-0.2789925774668332,-0.1706755754519138,
      -0.05252850546893189,0.07054986219834002,0.1924160430438771,
      0.307105809225674,0.4101897544477446,-0.2593820240207756,
      -0.1482455618465796,-0.02928064957337935,0.0921572380074013,
      0.2100905134498461,0.3193074202631062,-0.2358533016570041,
      -0.1238167822688868,-0.006318969895916147,0.1112788403952597,
      0.2236016867495729,-0.2096044634529125,-0.09868017515402502,
      0.01532894544522089,0.127462173385832,-0.1820659309330632,
      -0.07404772953164979,0.03496843200875603,-0.1545841458628095,
      -0.05083475466396539,-0.6714423515894649,-0.7363066534250458,
      -0.8002045390283683,-0.859632936776908,-0.9106424117173109,
      -0.9496340459616872,-0.9742807748705438,-0.9841303841597815,
      -0.9805709251787237,-0.9662290590683913,-0.6736593131243975,
      -0.739662482666216,-0.8034418824887359,-0.8609423496737078,
      -0.9079287599677793,-0.9410171351552678,-0.9586305009774437,
      -0.9613540496396505,-0.9515175508275152,-0.6691519386893557,
      -0.7345993546337776,-0.79626257796142,-0.8497460401941979,
      -0.8909109722488041,-0.9170267250100911,-0.9275521423149625,
      -0.9240939630879936,-0.656100800090535,-0.7189837013183803,
      -0.7764177384094252,-0.8240374297552895,-0.8582583152457706,
      -0.8773387645254044,-0.8817972923410141,-0.6332080201962388,
      -0.6915533212632716,-0.7430051304270751,-0.7836645345684842,
      -0.8108589038018792,-0.8239068177963524,-0.6002017842451572,
      -0.6525164982995073,-0.6970417319534228,-0.7306820673883421,
      -0.7517816084827516,-0.5581290590099237,-0.6037336945345413,
      -0.6413705283620924,-0.6688591076565297,-0.5092217844350768,
      -0.5482985439178426,-0.5798809609547027,-0.4563600901355626,
      -0.4897211300815704,-0.4024024278645879,-0.648822290296783,
      -0.648349171242789,-0.6411378559950974,-0.6255004973956005,
      -0.600293640880965,-0.5653935344751363,-0.5219527418637842,
      -0.4722400785685584,-0.4191009430775375,-0.3653071673956498,
      -0.7148533441384284,-0.7155197916209453,-0.7077869387160182,
      -0.6896699883604869,-0.6600703681315339,-0.6193375043296356,
      -0.5694127009086171,-0.5133919240157367,-0.4547192979777763,
      -0.3964310803290843,-0.7802586188950914,-0.7808677178913338,
      -0.7711197529973893,-0.7489240139903413,-0.713538182957094,
      -0.6660993109205865,-0.6094980396194888,-0.5475936950081823,
      -0.4842093859996422,-0.841562970549242,-0.8403682541467874,
      -0.8267667446471012,-0.798907464199165,-0.7567804454437204,
      -0.7025296887747676,-0.6399394350221852,-0.5733572200389935,
      -0.8948081394501657,-0.8897668713226807,-0.8705497953564927,
      -0.8359665662613534,-0.7870195954857328,-0.7268354112898293,
      -0.6597854273844109,-0.9363384293729855,-0.9255995665778384,
      -0.8996275962075316,-0.8582152453052212,-0.8034009739634004,
      -0.7389956620788967,-0.9637246412193355,-0.9461585726484955,
      -0.9132982945459158,-0.8659870495929599,-0.8070428410352181,
      -0.9763861801087089,-0.9518769772925728,-0.91299797697185,
      -0.8614408694954784,-0.975584505681221,-0.9449459265141765,
      -0.9015722073987464,-0.963845393304663,-0.9284651475621579,
      0.3130823317650696,0.2261770766623218,0.1280539101236855,
      0.02022783076448256,-0.09418429615654819,-0.2106389019045293,
      -0.3239116283455282,-0.4291939860046181,-0.523048586255087,
      -0.6038172319641657,0.2600841789503569,0.1646285788165745,
      0.05778571619095332,-0.05796842566746966,-0.1783494840893115,
      -0.297794748004312,-0.4106115455276073,-0.5122593289719761,
      -0.6001443622816693,0.1989845274738189,0.09504589509123174,
      -0.01970764286946627,-0.1416365624266124,-0.2653151882168217,
      -0.3845121065161455,-0.4936479477555078,-0.5890080819228373,
      0.1303871207046509,0.01886496318699474,-0.1019485160673513,
      -0.2272413285785661,-0.3507747719153151,-0.4662617951252705,
      -0.5688734433747777,0.05597369301027853,-0.06123892158082981,
      -0.1852629737758222,-0.3104018186002749,-0.4302072668465533,
      -0.5390198361162625,-0.02151295053653907,-0.1416473256500602,
      -0.2654479680509194,-0.3868857125795179,-0.4999829944651056,
      -0.09867461327158224,-0.2185280531038441,-0.3387557568094476,
      -0.4536233831584169,-0.1721555744064095,-0.2886982185503227,
      -0.4027071899557529,-0.2393260112020272,-0.3502029095948441,
      -0.2986531728773539,0.1282040764044601,0.05321845051851246,
      -0.0299819504088954,-0.1197179049294584,-0.2130455201840074,
      -0.3060118317848468,-0.3943555879655132,-0.4744196990222574,
      -0.5438582278251758,-0.6018463947181518,0.0508347486986919,
      -0.03269551088976293,-0.1240992485699228,-0.2207245467966487,
      -0.3185587143239474,-0.4128297421420895,-0.4990447345439135,
      -0.5740017863876323,-0.6362851104978535,-0.6861131509034271,
      -0.03496843048443223,-0.126718664611582,-0.2252069693946209,
      -0.3266628419962321,-0.4261053308619027,-0.5183500501623459,
      -0.5992598174372795,-0.6666016739887244,-0.720136706807033,
      -0.1274621786236507,-0.2261259790523726,-0.329452932630458,
      -0.4326360098383147,-0.5301336109403396,-0.6169837287853274,
      -0.6899929014185586,-0.7481901976756111,-0.2236017161594696,
      -0.3267916497417753,-0.4317330442416767,-0.5329718535250777,
      -0.6250530132529536,-0.703875699754291,-0.7675317913865697,
      -0.3193074905789999,-0.4238069911549194,-0.5267203305462205,
      -0.6225639241960304,-0.7066200241231836,-0.7760321386045594,
      -0.4101898771205939,-0.5125483215775021,-0.6101488498350025,
      -0.6980950818677443,-0.7728099228904255,-0.4924896604754524,
      -0.589718294305585,-0.679698789286976,-0.7585159482037284,
      -0.5638041319099237,-0.6538562109800681,-0.7350961954485662,
      -0.623299689837984,-0.7051827865650066,0.6483485235093655,
      0.6411372723067146,0.6254999813609712,0.6002931843810706,
      0.5653931175085023,0.5219523372221417,0.4722396596033476,
      0.4191004905746099,0.3653066725193389,0.5719394335088436,
      0.5579566936615924,0.5351892752413209,0.5030959515755655,
      0.4623631200726549,0.4149320801413646,0.3635618193945052,
      0.3111391128807077,0.4812077579558775,0.4596949806286311,
      0.4297384718300117,0.3916338931244087,0.3469138437185931,
      0.2980734064957148,0.2479339211516947,0.3769270466275512,
      0.34798863705642,0.3118670040922781,0.2697400287604772,
      0.2237517049369116,0.1764984145646341,0.2618704462020703,
      0.226741584116328,0.1865794537419862,0.1432114770939381,
      0.09893915843486868,0.1407291822657564,0.1016644513337693,
      0.06025214993119726,0.01853073096018258,0.0192767564986509,
      -0.02097489882147943,-0.06089466864900071,-0.09693134249768307,
      -0.1358102942547418,-0.2036925009060082,0.9089824672626068,
      0.8608164411414747,0.7981450033151747,0.7207644427956729,
      0.6303269573377657,0.5303678926086439,0.4256670502613963,
      0.3211867913977836,0.2210883675134422,0.9284650222993554,
      0.8871725500645941,0.8313006928275277,0.7596804192454199,
      0.6729483436994259,0.5739528381046123,0.4674119497069649,
      0.3588508188064044,0.2533333547826946,0.1545840239447369,
      0.9015721838250796,0.8527098639261403,0.7879881821713033,
      0.7069782619885319,0.6114960278478284,0.5056844240896506,
      0.3951892122332402,0.285769898249863,0.1820657113417612,
      0.8614409638909055,0.8038277741655933,0.7297824294607609,
      0.6399926588400225,0.5376088254129453,0.4278416041703199,
      0.3167134480853839,0.2096041546069603,0.8070430398170311,
      0.7402383431039794,0.6574928275930984,0.560806507588693,
      0.4544842943197335,0.3442560602121421,0.2358529185889448,
      0.7389959159190774,0.6636727901334047,0.5740968543042024,
      0.4734498690772693,0.3666860169560138,0.2593815856416469,
      0.6597856586149884,0.5777801683909922,0.4841878538357612,
      0.3829343216652333,0.2789921022280572,0.5733573420809426,
      0.4873055377141242,0.3928154550148792,0.2939985098519586,
      0.4842093252521232,0.3969457605237173,0.3043325272261384,
      0.3964307901608145,0.3104439077532887,0.6362842799460792,
      0.5740009080509757,0.4990438745890764,0.4128289723722984,
      0.3185580939893934,0.2207241066952111,0.1240989860968612,
      0.03269539736236529,0.6666008139591869,0.5992589358516202,
      0.5183492243752909,0.4261046359672609,0.3266623297142192,
      0.2252066549797059,0.1267185286485934,0.689992024373379,
      0.6169828686407737,0.530132851891722,0.4326354224553247,
      0.3294525535291586,0.2261258043934048,0.7038748329124103,
      0.6250522113657903,0.5329711994696135,0.4317325950511361,
      0.3267914216946357,0.7066192072999724,0.622563222965479,
      0.5267198166849954,0.423806701432994,0.6980943593640034,
      0.6101482870567641,0.5125479715352314,0.6796981993053423,
      0.5897178945758851,0.6538557778979411,0.9323218580087335,
      0.9515174346052251,0.9613540145138462,0.9586305496920505,
      0.9410172511751344,0.9079289076765579,0.8609424826856753,
      0.8034419563572277,0.7396624652608896,0.6736591885240169,
      0.9096403689902206,0.9240939382365192,0.9275522217134882,
      0.9170269050764695,0.8909112253661301,0.849746318794698,
      0.796262827475376,0.7345995283284186,0.6691520068054228,
      0.8739713777891581,0.8817974249990892,0.8773390199098727,
      0.8582586769711391,0.8240378550735361,0.7764181680693131,
      0.7189840763729405,0.656101076264991,0.8241233338640371,
      0.8239071624215608,0.810859375773786,0.7836651001194045,
      0.7430057321681839,0.6915538934135402,0.6332085061147845,
      0.7605089118538748,0.7517821894256266,0.730682759920626,
      0.6970424857617678,0.6525172475488723,0.6002024649156599,
      0.6854064426268304,0.6688599095382908,0.6413714065577412,
      0.6037345889195533,0.5581299045184589,0.6025957234227359,
      0.5798819340529643,0.5482995467196561,0.5092227563225322,
      0.5164832226272315,0.4897222069506372,0.4563611494403301,
      0.4311438278222397,0.4024035413263246,0.3496833143594963,
      -0.2533334075588959,-0.3588508219660573,-0.4674119310964882,
      -0.5739528280203016,-0.6729483660875734,-0.7596804843602077,
      -0.8313007955580714,-0.8871726742411541,-0.2857700357309581,
      -0.3951892821849063,-0.5056844492735352,-0.6114960336943951,
      -0.7069782687900093,-0.787988199289983,-0.8527098891878181,
      -0.3167136531290147,-0.4278417135183467,-0.5376088566787567,
      -0.639992634716211,-0.7297823708955478,-0.8038276949870052,
      -0.3442563102744645,-0.4544844137443082,-0.5608065113072425,
      -0.657492739431111,-0.7402381888786644,-0.3666862898302455,
      -0.4734499761180304,-0.5740968114811118,-0.6636726256137523,
      -0.3829346014572981,-0.484187939006181,-0.5777800787254197,
      -0.3928157351219527,-0.4873056046530535,-0.3969460425371419,
      0.600144149407117,0.5122590107240774,0.4106111150772491,
      0.2977942147481387,0.1783488751426739,0.05796778060322072,
      -0.05778635529341983,-0.1646291768340628,-0.260084712697206,
      0.5890080700362191,0.4936478319326018,0.3845118729629886,
      0.2653148405479006,0.1416361235766341,0.01970714938077021,
      -0.09504640342765641,-0.1989850169013517,0.568873645341387,
      0.4662618905156885,0.3507747390771843,0.2272411653349327,
      0.1019482418076114,-0.01886531395275474,-0.1303875092325712,
      0.5390202482542535,0.4302075642722875,0.3104019730711553,
      0.1852629793843341,0.061238796088284,-0.0559739158243807,
      0.4999835950826037,0.3868861858512856,0.2654482825921611,
      0.1416474750285941,0.02151295276860444,0.4536241369952758,
      0.3387563694841473,0.2185284943201295,0.09867487876932232,
      0.4027080566573236,0.2886989329203987,0.1721561104744125,
      0.3502038519557472,0.2393267951217032,0.298654161645374,
      -0.984713259051105,-0.9959947612173518,-0.9943464342167785,
      -0.9773096457316496,-0.9442212908026748,-0.8966583408378461,
      -0.8380826821486993,-0.772858299407996,-0.9933753694301851,
      -0.999553621201999,-0.9908225482912788,-0.9653354239158236,
      -0.9235419454230499,-0.8682642090770526,-0.8038809644055149,
      -0.9889450353591552,-0.9880324674077218,-0.9708623496514911,
      -0.9366415263248317,-0.8871116184521261,-0.8261317121277938,
      -0.9690767381494427,-0.9597077173597477,-0.9338032828335274,
      -0.8918540989344099,-0.8367235291009597,-0.9332440544501512,
      -0.9151716624555581,-0.881531417418751,-0.8339792023300259,
      -0.8831547498989281,-0.8573751662344773,-0.8180295248104962,
      -0.8223661602728524,-0.7907838386634013,-0.7552871860588483,
      -0.3111396248637522,-0.3635623115811258,-0.4149325617880891,
      -0.4623636071651684,-0.503096462804278,-0.5351898259002444,
      -0.5579572907903124,-0.5719400747815568,-0.5785069283639182,
      -0.2479344056308305,-0.2980738893651726,-0.3469143327715778,
      -0.3916343988495664,-0.4297390036046975,-0.4596955428592778,
      -0.4812083484887336,-0.4950341577852201,-0.1764988082863052,
      -0.2237521076309895,-0.2697404459306319,-0.3118674408741246,
      -0.3479890956417611,-0.3769275248330646,-0.3985931343359159,
      -0.09893938901501213,-0.1432117197792371,-0.186579712833973,
      -0.2267418620329016,-0.2618707421483436,-0.2909964852939082,
      -0.01853073419918189,-0.06025216482046628,-0.1016644832117871,
      -0.1407292340032114,-0.1758296738027379,0.06089493100791283,
      0.02097514873862574,-0.01927652693035482,-0.05800679989935065,
      0.1358108258261833,0.09693185715603557,0.0572108907410362,
      0.2036932746994002,0.1653145532988453,0.2632739407438701,
      -0.5471131012749936,-0.4569196340994406,-0.3535541587646317,
      -0.2398162730231354,-0.1203709912173413,-0.0009109877496780188,
      0.1131254888510733,0.2176599356582619,-0.478328705556794,
      -0.3786231842883476,-0.2666208448402719,-0.1464197032303796,
      -0.02374523314186032,0.09531724619007391,0.2057254392916539,
      -0.3977013270678494,-0.2894249268534429,-0.1708861328457008,
      -0.04735264212838212,0.07488848723411734,0.1900372836013431,
      -0.3071060284643041,-0.1924163631703616,-0.07055027067676212,
      0.05252803743712917,0.1706750855887899,-0.2100906838592314,
      -0.0921575105240425,0.02928029003178066,0.1482451448989572,
      -0.1112789754160983,0.006318730357784829,0.123816455924515,
      -0.01532906198457091,0.09867995232819112,0.07404761591340303,
      -0.5229702552542058,-0.455754381535759,-0.3771270572800299,
      -0.2889439451759633,-0.194672902916372,-0.09880682824059986,
      -0.005851780927923981,0.08061942171187554,0.1584410564110822,
      -0.4301068723700084,-0.3534800054422614,-0.2664452641642683,
      -0.1720548071373146,-0.07475800001082551,0.02057294660420643,
      0.1097762759917693,0.190134278339324,-0.324239892584746,
      -0.2391265604455334,-0.1457556026172962,-0.04825796103784657,
      0.04840435106572794,0.1396272041512353,0.2221350978923813,
      -0.2083325524549613,-0.1169519894866824,-0.02046779780351893,
      0.07636807502743861,0.1687516747481361,0.2529338262925594,
      -0.08719102201021951,0.007198147474544054,0.1030417934166896,
      0.1956221602390349,0.2808952005684322,0.03341116907881747,
      0.1271908635410245,0.2189580303556818,0.3046134343217798,
      0.1480334302314487,0.2380139721282078,0.323237313179716,
      0.2526632136782334,0.3366066958443542,0.3451570696887505,
      0.4547190651923899,0.5133917435529424,0.5694125552964621,
      0.6193373643788831,0.6600701990863211,0.6896697595590643,
      0.7077866317530298,0.7155194029861072,0.5475936980371527,
      0.6094980941008995,0.6660993907770576,0.7135382519498201,
      0.748924034855715,0.7711196978950583,0.7808675744788959,
      0.6399396139686974,0.7025299057553986,0.7567806697847073,
      0.7989076596939885,0.8267668794191896,0.8403683088289017,
      0.7268356797799225,0.7870198804193677,0.8359668400797895,
      0.8705500304441893,0.8897670473314109,0.803401239536944,
      0.858215507236093,0.8996278379610205,0.9255997745502057,
      0.8659872391476647,0.9132984713369965,0.9461587349139229,
      0.9129980537560551,0.9518770467134829,0.9449458910860361,
      0.2573950079345842,0.363089267383967,0.4717819390919956,
      0.5783933464143868,0.6773888186481814,0.7640503132419058,
      0.835538996354342,0.8912340245126464,0.2942468056901541,
      0.403998910419839,0.5147136234231619,0.62060207699311,
      0.7160073275453937,0.7967976318501995,0.8611864392751584,
      0.3298234714814622,0.4413854454351217,0.5513863234589741,
      0.6537700563672823,0.7433259842576552,0.8169373645440049,
      0.3620182543235557,0.4726965405256068,0.5791764634315589,
      0.6757048258462731,0.7580000771779366,0.3888887347419664,
      0.4960228623929952,0.5966697046150125,0.685875099677583,
      0.4091542831822379,0.5106167801856609,0.6039998362037971,
      0.422484479771868,0.5169744155067602,0.4294387727293252 };

    auto      pointshell
        = new vnl_matrix_fixed<double, 3, 1002>(coords);

    return pointshell;
  }
};


// ThreeLayerPointShell using a staggered-distribution
#include <itkMacro.h>

template<int NpointsShell1, int NpointsShell2, int NpointsShell3, class TMatrixTypeshell1, class TMatrixTypeshell2, class TMatrixTypeshell3>
TMatrixTypeshell1 *
ThreeLayerPointShell< NpointsShell1,  NpointsShell2,  NpointsShell3,  TMatrixTypeshell1,  TMatrixTypeshell2,  TMatrixTypeshell3>
::DistributePointShell1()
{
  itkGenericExceptionMacro(<<"This type of PointShell is not Supported and should not be used!");
  return 0;
}

template<int NpointsShell1, int NpointsShell2, int NpointsShell3, class TMatrixTypeshell1, class TMatrixTypeshell2, class TMatrixTypeshell3>
TMatrixTypeshell2 *
ThreeLayerPointShell< NpointsShell1,  NpointsShell2,  NpointsShell3,  TMatrixTypeshell1,  TMatrixTypeshell2,  TMatrixTypeshell3>
::DistributePointShell2()
{
itkGenericExceptionMacro(<<"This type of PointShell is not Supported and should not be used!");
return 0;
}

template<int NpointsShell1, int NpointsShell2, int NpointsShell3, class TMatrixTypeshell1, class TMatrixTypeshell2, class TMatrixTypeshell3>
TMatrixTypeshell3 *
ThreeLayerPointShell< NpointsShell1,  NpointsShell2,  NpointsShell3,  TMatrixTypeshell1,  TMatrixTypeshell2,  TMatrixTypeshell3>
::DistributePointShell3()
{
  itkGenericExceptionMacro(<<"This type of PointShell is not Supported and should not be used!");
  return 0;
}



//Distributed Pointshell 7 - 16 - 22

template<> class ThreeLayerPointShell<7,16,22, vnl_matrix_fixed<double, 3, 7>,vnl_matrix_fixed<double, 3, 16>,vnl_matrix_fixed<double, 3, 22> >
{
public:
  static vnl_matrix_fixed<double, 3, 7>* DistributePointShell1()
  {
    double coords[3*7] = {
      0.8134386897688890,-0.4341253640420510,0.3871080808797820,
      -0.4587290424636170,-0.8386718051774130,-0.2935937138306440,
      0.9664216428893120,0.2069240694656700,-0.1523536597226610,
      0.2351246687899070,-0.9681943247699460,0.0855344352275928,
      0.3639220075873960,0.5047564346894090,-0.7828037519284090,
      0.5319989685716170,-0.4881652317616310,-0.6918611160759410,
      0.3581867553682760,0.1086899080235520,0.9273018668009770
    };
    auto      pointshell
        = new vnl_matrix_fixed<double, 3, 7>(coords);

    // List sorted for each row [x,y,z]
    // pointshell need [x...x
    //                  y...y
    //                  z...z]
    int i=0;
    for(int c =0; c < 7; c++)
    {
      for(int r=0; r<3; r++){
        pointshell->put(r,c,coords[i++]);
      }
    }

    return pointshell;
  }

  static vnl_matrix_fixed<double, 3, 16>* DistributePointShell2()
  {
    double coords[3*16] = {
      -0.5452474978820280,0.0761974966412968,0.8348048320170320,
      0.2927980757498900,-0.6773379396203400,0.6748945120446840,
      -0.1417004162163040,-0.9633970491958960,0.2275678308653590,
      -0.6140671213404080,0.1662941114081500,-0.7715360257302810,
      -0.7535977994952050,0.5644302557976330,0.3369107343736630,
      0.6046201505721580,-0.7556729189381400,0.2517794930204040,
      -0.5761191698601790,-0.6394532899960390,0.5091033215692650,
      -0.8771756234198420,-0.0996725024517006,0.4697108876032420,
      0.1777092608486050,-0.9423569168215250,-0.2835187117762620,
      -0.9768429239415920,0.1642087829444060,-0.1371618662353210,
      -0.1487223091187880,0.6561163072920720,0.7398601665691900,
      0.8263922468011810,0.2709995685589460,0.4935940520992520,
      0.0308686015665059,0.1530129849183750,0.9877419480227090,
      0.4225700117384330,0.6462523317724860,0.6354467002495210,
      0.7837589636894810,0.6201599634850410,0.0335187488821074,
      0.0539036789368123,0.3663544585938050,-0.9289126999161520
    };
    auto      pointshell
        = new vnl_matrix_fixed<double, 3, 16>(coords);
    // List sorted for each row [x,y,z]
    // pointshell need [x...x
    //                  y...y
    //                  z...z]
    int i=0;
    for(int c =0; c < 16; c++)
    {
      for(int r=0; r<3; r++){
        pointshell->put(r,c,coords[i++]);
      }
    }

    return pointshell;
  }

  static vnl_matrix_fixed<double, 3, 22>* DistributePointShell3()
  {
    double coords[3*22] = {
      -0.8173635684245140,-0.4833486694613540,0.3135137329404920,
      -0.2868452120883920,-0.8216012153929990,0.4926370541957950,
      -0.7664861123034300,0.2467518351336330,0.5929692837779680,
      -0.6473847770633690,-0.2749696442612980,0.7108337676007140,
      -0.8434413337089970,0.5359848183688930,-0.0364278886191929,
      -0.6982955004575730,-0.6008181795242910,-0.3891026974897790,
      -0.1840600867015900,-0.4748774851820570,-0.8605889021772050,
      -0.5838330320870200,0.5227863025561340,-0.6211549504783510,
      0.2399013062530760,-0.8837050331449780,0.4018865233524250,
      0.2572284397529420,0.1068009420516400,-0.9604306786849050,
      0.0228576846453402,0.6541700548957510,-0.7560020274644400,
      0.2865351008629760,-0.2558878809381770,0.9232654159895800,
      -0.9589554732993380,0.1841748711215180,0.2156015238272050,
      -0.1416546797498710,-0.9826949020935590,-0.1193510833813020,
      0.5319624136210260,0.8404508833828820,-0.1032390580905500,
      -0.9539866018255820,-0.2464051857069770,-0.1708626582784740,
      -0.5967074337720730,-0.2735711735415940,-0.7543865398376490,
      -0.8579620414436700,0.0965216321209641,-0.5045638809650520,
      0.4190874972849550,-0.7580966160880540,-0.4996550713194790,
      -0.1112620377459570,-0.8350080959462270,-0.5388712635322760,
      0.2732684357685670,-0.3457098692283190,-0.8976686740277500,
      -0.5333383595673660,0.8405849409324210,0.0946950436486956
    };
    auto      pointshell
        = new vnl_matrix_fixed<double, 3, 22>();

    // List sorted for each row [x,y,z]
    // pointshell need [x...x
    //                  y...y
    //                  z...z]
    int i=0;
    for(int c =0; c < 22; c++)
    {
      for(int r=0; r<3; r++){
        pointshell->put(r,c,coords[i++]);
      }
    }

    return pointshell;
  }
};

//Distributed Pointshell 15 - 30 - 45

template<> class ThreeLayerPointShell<15,30,45, vnl_matrix_fixed<double, 3, 15>,vnl_matrix_fixed<double, 3, 30>,vnl_matrix_fixed<double, 3, 45> >
{
public:
  static vnl_matrix_fixed<double, 3, 15>* DistributePointShell1()
  {
    double coords[3*15] = {
      -0.8262066830854160,-0.4726354124122490,0.3065913954417130,
      -0.1648544111105980,-0.8594278487393920,-0.4839491666988980,
      -0.0358642514967783,-0.3860637871945480,-0.9217746512470260,
      -0.7684991573547760,0.6362727605963740,-0.0675723261999100,
      0.1470231744209000,-0.2445782513554300,0.9584183142798970,
      -0.4269215415769730,0.7783400701208900,0.4603528348796690,
      0.3354375712062210,0.5671194195694440,-0.7522348036155240,
      0.4782763791345030,0.8758625019989820,-0.0641606012602034,
      0.9898549006984430,0.0293790674196844,0.1390113159452260,
      -0.6942246846812260,0.1045517495672360,-0.7121243001342390,
      -0.5002034766634700,0.0396563572397201,0.8649993383027830,
      -0.3370551539814700,0.7499893159794020,-0.5691307838197430,
      -0.8609008763099680,0.2402687527531810,0.4484647228257290,
      -0.6557865116475140,-0.4999608118385060,-0.5656706088944040,
      -0.0693103556967315,0.9875634181798020,-0.1411189911607090
    };
    auto      pointshell
        = new vnl_matrix_fixed<double, 3, 15>(coords);

    // List sorted for each row [x,y,z]
    // pointshell need [x...x
    //                  y...y
    //                  z...z]
    int i=0;
    for(int c =0; c < 15; c++)
    {
      for(int r=0; r<3; r++){
        pointshell->put(r,c,coords[i++]);
      }
    }

    return pointshell;
  }

  static vnl_matrix_fixed<double, 3, 30>* DistributePointShell2()
  {
    double coords[3*30] = {
      0.2837014644823930,-0.2206420544708420,-0.9331830275189480,
      0.6952056236193340,-0.1366087970901870,-0.7057104062188850,
      -0.3982427056422320,-0.8946622247526970,-0.2024407345454680,
      0.6912813662578750,0.2350369054844600,0.6832918305709010,
      0.3004304980870820,0.2618613453462440,0.9171532868788080,
      -0.8122424463478870,0.2721420733854360,-0.5159466060013130,
      0.9419343376574850,0.3217426374876110,-0.0961320901898529,
      0.7896221636664980,0.5657937790715650,0.2374325971945600,
      0.4752045378048260,-0.5551457462701800,0.6826374203395420,
      0.7035350948936570,-0.6486284012073840,0.2903783177169470,
      -0.9657370911439440,0.2516454388878180,-0.0634542660172643,
      0.4335627021602730,0.7069616605804370,-0.5587741885277380,
      -0.8057237269560320,0.4981857033689590,0.3203440038128640,
      0.9405564452144110,-0.0095010729465114,-0.3395044962566700,
      0.1927525893653990,0.9455549503304530,-0.2622446857393060,
      0.0977862568275677,-0.8610667582542970,-0.4990008875794600,
      -0.9244603975979140,-0.1428771862175540,-0.3534955203841470,
      -0.0587838438498267,0.9919184790835770,0.1124374962135020,
      0.0413372764424745,-0.4921846796379190,0.8695087525183610,
      -0.0416085831127836,0.5985263605884460,0.8000218256348350,
      -0.0355639978656969,-0.0871145642116132,0.9955632851597270,
      0.4117279464778380,0.2997527998928850,-0.8605976743203040,
      -0.3489182251383090,0.9011702877335440,-0.2571928938998360,
      -0.4466362970172200,0.1524928751735680,-0.8816246033363880,
      0.7474002986676580,0.3366743112988240,-0.5727505579769920,
      0.0459516120636567,-0.8054328299457180,0.5909030426342220,
      -0.6709586581555420,-0.7177574233281400,0.1861149115559070,
      0.4102749262480260,0.6499693688374480,0.6396986043952450,
      0.4840608825017460,-0.5520432393197750,-0.6789207052027200,
      0.4932109057185150,-0.8496377203461870,-0.1867049722028180
    };
    auto      pointshell
        = new vnl_matrix_fixed<double, 3, 30>(coords);
    // List sorted for each row [x,y,z]
    // pointshell need [x...x
    //                  y...y
    //                  z...z]
    int i=0;
    for(int c =0; c < 30; c++)
    {
      for(int r=0; r<3; r++){
        pointshell->put(r,c,coords[i++]);
      }
    }

    return pointshell;
  }

  static vnl_matrix_fixed<double, 3, 45>* DistributePointShell3()
  {
    double coords[3*45] = {
      -0.5319244251302650,-0.0924062555950406,-0.8417348097095310,
      -0.1568078023044440,-0.3505682049930730,0.9233164391390330,
      -0.0268033601770698,0.1671912039337100,0.9855600850330810,
      -0.8635644949706830,0.4302558439150240,-0.2629377717312540,
      -0.1232991287092530,-0.6582015219317060,0.7426762965022680,
      0.2376788616301610,0.5073218686335690,0.8283316246168160,
      -0.2566320540992200,0.7153437906464810,0.6499409588511380,
      -0.2043674419674510,0.6455892128571940,-0.7358318536908470,
      0.4526173742466210,0.8360898783655450,-0.3099858510073980,
      0.6836009409422150,-0.5146343588192300,0.5175337962545400,
      -0.2416560443524480,-0.0050416195905518,-0.9703488745290690,
      0.6217071536966980,0.1629149642421430,-0.7661193963530300,
      0.3925487362456630,0.8090879125984540,0.4373582505898420,
      0.9937673576390050,0.0621775579881003,-0.0925223766114162,
      -0.6145262933579910,0.6346672434383510,0.4685668841030610,
      0.8259615790179220,0.5617603544414460,-0.0470401335461508,
      -0.9322660404149650,-0.3254138676429420,-0.1580691134746520,
      0.1164156506796110,-0.9181409097759250,0.3787672980507660,
      0.3168138266596370,-0.3901334055493560,0.8645374052704360,
      0.2176339314862630,0.9758048396839600,-0.0209853929005879,
      -0.8962471871878890,-0.2522993236544900,0.3648095814822890,
      -0.6612497786305060,-0.6092040683197140,0.4377432277075390,
      0.6864998437314790,-0.7035766895855040,-0.1835696228371040,
      0.5089165208574290,0.4070575405636090,0.7584906943850230,
      -0.8813116916999880,0.4680173236355270,0.0651880882519394,
      -0.2012114505332470,-0.8431942495440670,0.4985352642593330,
      0.5022346836643460,-0.7613661632799880,0.4099779115231480,
      0.2920837705654530,0.0626526547841476,-0.9543383654767130,
      0.9371632886212690,-0.1374669688206610,0.3206677454060930,
      0.5505816317240350,0.4677412077906900,-0.6914318689087190,
      -0.7167807102053720,0.3777968874482810,0.5860844011845660,
      -0.2932320783275170,0.9560381526252190,-0.0024492784000661,
      0.6321152281530360,0.6671359233891630,0.3941573265343710,
      0.8208025699405450,0.0279728493062268,-0.5705266522088930,
      -0.2446598581934500,0.4624912733458210,0.8521991409686590,
      0.5981132830055560,-0.3201976866112960,0.7346658711149490,
      0.5635872282435520,-0.8205568040328310,0.0951628473521853,
      0.8170510623081950,0.3573123713720030,0.4524990948560730,
      -0.1077825148871900,-0.7347609483441290,-0.6697083531456270,
      -0.9500761846368700,0.2288974134306240,0.2120406034480560,
      -0.2525195159846320,0.9165943360942510,0.3099818012187580,
      -0.1132939165725980,-0.9559551317814730,-0.2707660881430790,
      0.6286490080714330,0.7713217806218580,0.0993133192936744,
      0.8227143106676380,0.0339170261757165,0.5674423304249230,
      -0.5076617466917020,0.3007946007639900,0.8073426528415060
    };
    auto      pointshell
        = new vnl_matrix_fixed<double, 3, 45>();

    // List sorted for each row [x,y,z]
    // pointshell need [x...x
    //                  y...y
    //                  z...z]
    int i=0;
    for(int c =0; c < 45; c++)
    {
      for(int r=0; r<3; r++){
        pointshell->put(r,c,coords[i++]);
      }
    }

    return pointshell;
  }
};

//Distributed Pointshell 22 - 46 - 67

template<> class ThreeLayerPointShell<22,46,67, vnl_matrix_fixed<double, 3, 22>,vnl_matrix_fixed<double, 3, 46>,vnl_matrix_fixed<double, 3, 67> >
{
public:
  static vnl_matrix_fixed<double, 3, 22>* DistributePointShell1()
  {
    double coords[3*22] = {
      0.25642497332897700,0.69733994966702500,-0.66930055106179500,
      -0.63587617425647900,0.60181880252846700,-0.48319314971980300,
      0.02750479511080110,-0.49812287900990700,-0.86667011235694100,
      -0.35667911834084000,-0.89655924253456900,0.26260527634729500,
      -0.97064232445283100,-0.10849397816068500,0.21466842963899700,
      0.13668042906880200,-0.37888572303413600,0.91529452592620000,
      0.51081097493717800,0.82410825259274000,0.24478099577432200,
      0.56937377655062700,-0.82148556938178300,-0.03122437948083380,
      -0.70712711556887700,0.17758768051548200,-0.68442228058149100,
      0.77448769645207600,-0.17248893945546000,-0.60861841395893200,
      -0.10740168736203100,0.99190238197584200,0.06778305232459120,
      0.33092898900280500,0.01295617559776370,-0.94356671293102500,
      0.96250467033690100,-0.16189196397475400,0.21765971510605600,
      -0.45033217736158200,0.66660174976094400,0.59400592358024700,
      -0.85650276118059000,-0.36171210207403600,-0.36819475186809100,
      0.27465477100258400,0.05835151655784800,0.95977073162340000,
      0.11563788475332800,0.85791228608919200,0.50061401197628500,
      -0.54693056940441200,-0.44061696250608900,-0.71184523922189100,
      -0.84737370301991400,0.46653338404151400,0.25358314022247300,
      0.81343236730821000,0.56190972107245100,-0.15028389527703900,
      -0.18653146496488500,0.84209830314271900,-0.50603997907498000,
      -0.63212021995023600,-0.37635030041830100,0.67733631152118100,
    };
    auto      pointshell
        = new vnl_matrix_fixed<double, 3, 22>(coords);

    // List sorted for each row [x,y,z]
    // pointshell need [x...x
    //                  y...y
    //                  z...z]
    int i=0;
    for(int c =0; c < 22; c++)
    {
      for(int r=0; r<3; r++){
        pointshell->put(r,c,coords[i++]);
      }
    }

    return pointshell;
  }

  static vnl_matrix_fixed<double, 3, 46>* DistributePointShell2()
  {
    double coords[3*46] = {
      -0.23824007219091700,-0.86075578881685600,0.44982345428038400,
      -0.51769500181405200,-0.84172584112854800,0.15326282482449900,
      0.44541900890312100,0.24566021345088800,0.86096049040302900,
      -0.07063199679231950,-0.74435681728880700,-0.66403618093053200,
      0.28462319785348100,0.44431638677441600,-0.84945428581377100,
      -0.64536739671188300,-0.00105803610891677,-0.76387158856768400,
      0.99957798463419400,-0.02741827295649930,0.00959640259292167,
      0.25010658086046200,-0.76013925657241700,-0.59969576355667200,
      0.39532812932947400,-0.74907082575728600,0.53160941315984400,
      -0.34325878056226200,-0.92866133182164700,-0.14057574593845400,
      -0.26920408631492800,0.17274923523146300,0.94746338274272500,
      -0.54467024543405600,0.38808975702695900,-0.74345185737178700,
      0.52496025794034400,0.62807129953803100,-0.57440679860166600,
      -0.78050768202914100,-0.24385280423360000,-0.57562450274541400,
      -0.71694800106757700,-0.61231043102019300,0.33325890810159900,
      -0.79056173292908700,0.40623831712611600,-0.45823855809687000,
      -0.01131173053347560,-0.71410951594821100,0.69994260049274700,
      0.51711028616473900,-0.41981678757221700,-0.74588927919307400,
      0.09293538971980670,0.99313719503650500,-0.07100369830262170,
      0.06305806394427750,0.10620573522794300,0.99234269402071500,
      -0.33630459418638300,0.16690245213556600,-0.92684561357340800,
      -0.91315571223437600,0.04840489638883500,-0.40472658822878600,
      0.93648363925049100,-0.29927660444831100,-0.18284394287491200,
      -0.05612562081659480,-0.20305248576816200,0.97755797920702400,
      -0.37275429556215800,0.89636189167834400,0.23997832045161700,
      -0.67013076748237900,0.05267398640468540,0.74037166722512800,
      -0.71746069173368600,-0.55944913994429100,-0.41505037722258700,
      0.32124017988313500,-0.93693212076753000,0.13770601984920100,
      0.28299823115899100,-0.54968353921141900,0.78597710391644900,
      0.15427673771645000,0.42717011000338400,0.89091210863881800,
      0.73177570493927800,-0.66613283838403200,-0.14412272300735500,
      -0.75153823442269900,-0.65477491803088400,-0.08037467834125450,
      0.48366214065856300,0.15108779957940700,-0.86211565959088600,
      0.93951136227144300,0.29289842100589400,0.17756383397277200,
      -0.87960856796377400,0.45546435899697500,-0.13726246700461500,
      -0.64569564311752200,0.72655108088168600,-0.23494821414239600,
      -0.78338372249755100,-0.28544299520279600,0.55211614703387900,
      -0.21774715968345700,0.47658876151730000,0.85173313123606500,
      -0.02304329061227880,-0.94618817341094000,-0.32279551926107200,
      -0.57374791604915700,0.69666566365718000,0.43066260797792600,
      -0.07108119393689040,0.93003463387864700,-0.36053161255944900,
      0.76657473659095900,-0.39315791656506800,-0.50773026880688800,
      -0.92597907090953600,-0.32929755574671300,0.18473191390972800,
      0.91121096468807600,-0.03396316106949220,-0.41053755190286000,
      -0.43706646286549800,-0.78947199903364200,-0.43093719934587400,
      -0.40196779675371200,-0.59965623521593900,-0.69197853286038600,
    };
    auto      pointshell
        = new vnl_matrix_fixed<double, 3, 46>(coords);
    // List sorted for each row [x,y,z]
    // pointshell need [x...x
    //                  y...y
    //                  z...z]
    int i=0;
    for(int c =0; c < 46; c++)
    {
      for(int r=0; r<3; r++){
        pointshell->put(r,c,coords[i++]);
      }
    }

    return pointshell;
  }

  static vnl_matrix_fixed<double, 3, 67>* DistributePointShell3()
  {
    double coords[3*67] = {
      0.56913096982886400,-0.78778656243185400,-0.23554674957098500,
      -0.83141868259485000,-0.50643545944750500,-0.22861780255797400,
      0.66520728747741500,0.16406099937578400,-0.72841145870363500,
      0.89843444153226500,-0.32337141111972700,0.29706309891846700,
      0.10682918210921600,-0.98310908722825000,0.14860702695065500,
      0.47695410914339100,0.34004755151800200,-0.81048284403671700,
      0.73195863370355600,-0.57860748500094100,-0.35979151858778900,
      -0.34693446874899500,0.36681507780719100,-0.86318200461267500,
      0.69155604307511300,0.71307629015228000,-0.11520609232567700,
      0.46165051573275700,0.03834019742989470,0.88623294374826500,
      -0.97397281635692200,0.09418099894164960,0.20617199721620300,
      -0.29851857051594800,0.88978611904209300,-0.34520620709530000,
      -0.35435497259052800,0.57535527165268900,0.73715592976101400,
      0.81483656941218300,0.03423671864623110,-0.57867885069779600,
      0.09093817184312330,-0.56161837808076500,0.82238375853597700,
      -0.38426863203906100,-0.76843844817080100,0.51170300937524000,
      -0.73642722235150800,0.67410491319471600,-0.05707461945896040,
      -0.54419758869133900,-0.83725684681843000,-0.05338498776051840,
      -0.21793670722020900,0.72064410859729600,-0.65816081651069200,
      -0.86994783943189000,0.48926427520412000,0.06173512514583880,
      0.10775852899571900,-0.66193658502334100,-0.74177358866187400,
      0.64544044927716800,-0.27486614918135400,-0.71263961893168700,
      -0.11695347642120100,0.01614429356247260,0.99300616621366600,
      -0.76804280354387200,0.56311186828017900,-0.30498405815136400,
      -0.89852079402656700,-0.17264968995044600,-0.40354983243941700,
      0.15476743176759600,0.59389894762264400,-0.78951319309861900,
      -0.08940325110166170,0.85346959988929000,0.51341669310343500,
      -0.70606292132486000,-0.42769174553909800,-0.56440669904592900,
      -0.07667441970353810,-0.38529704221355400,0.91960166519238700,
      0.97761562873568900,0.05518196801914950,0.20303357568948200,
      0.27057965262888700,0.90009485795734400,0.34149070011052800,
      -0.07518696164793450,0.30024488370383200,0.95089427940640300,
      0.13690225805945900,0.94909731680809600,-0.28367596825584700,
      -0.50251469982969000,0.85107263931481100,-0.15216549893059300,
      0.21156984762048800,0.24711360232026500,0.94560724781811200,
      0.54655302079371500,0.64091673556772900,0.53898565243470900,
      -0.55776925354389300,-0.75378730237748100,0.34741641091860700,
      0.33541827007282600,-0.94032512320547200,-0.05729962277332840,
      0.67560875362133500,-0.41311684917543700,0.61064497128657700,
      0.89999152407210300,0.19323788395138500,-0.39073568662762600,
      -0.98068938405211500,-0.19462539941972600,0.01921681316436500,
      -0.83571217382768900,-0.43205465164452500,0.33898958761077700,
      0.17146282600716700,-0.94464169246048600,-0.27973661212473500,
      -0.54059787517480200,0.19148492705680600,-0.81919927982545100,
      -0.11323347366714700,-0.98395720927995300,-0.13786366724876500,
      -0.96778324010342800,0.25052610173143600,-0.02514502985004660,
      -0.37464301489130600,0.81614231968582300,0.43994809399632100,
      0.87643936365442000,-0.25855236763220800,-0.40620772398960400,
      0.52679822663262800,-0.75204791987234000,0.39611558241554000,
      -0.83302160989998000,0.19922527352743200,-0.51612429494024400,
      -0.36881247628488900,-0.43365274990848600,-0.82214515131661200,
      -0.68494199482737100,-0.50458868526765800,0.52558988044078300,
      -0.47018702426243700,0.58505538166360200,-0.66078314340008000,
      -0.37416836671127000,0.32358269311843500,0.86907552840178400,
      -0.27722979680714700,-0.21344561523946000,0.93679485966636000,
      -0.42391010237203500,-0.55511199435901700,0.71564718879185400,
      0.14056187290120700,-0.12797911064355400,0.98176560701900200,
      0.65560180621904000,0.70152916060800800,0.27936196680821600,
      0.31456644451092300,0.94702007926960300,-0.06481451571947000,
      0.17066405874813200,0.60471657508100000,0.77794064225615400,
      0.79129808311628700,0.03500830070119940,0.61042752439458600,
      0.89399321972683600,0.44808044136416500,-0.00020285300027376,
      0.62673064388188000,0.22888439553392000,0.74486282864731200,
      0.04867785807210680,0.84066847580244300,-0.53935793303288600,
      -0.61746764544761500,0.52381468179607700,0.58681503556082400,
      0.28885209645735200,0.75743720323072100,0.58553680459405300,
      -0.49988083572550200,0.08893492748180820,0.86151594805214000,
    };
    auto      pointshell
        = new vnl_matrix_fixed<double, 3, 67>();

    // List sorted for each row [x,y,z]
    // pointshell need [x...x
    //                  y...y
    //                  z...z]
    int i=0;
    for(int c =0; c < 67; c++)
    {
      for(int r=0; r<3; r++){
        pointshell->put(r,c,coords[i++]);
      }
    }

    return pointshell;
  }
};

// Distributed Pointshell 30-60-90
template<> class ThreeLayerPointShell<30,60,90, vnl_matrix_fixed<double, 3, 30>,vnl_matrix_fixed<double, 3, 60>,vnl_matrix_fixed<double, 3, 90> >
{
public:
  static vnl_matrix_fixed<double, 3, 30>* DistributePointShell1()
  {
    double coords[3*30] = {
      -0.0384531292653293 ,0.0948025501623591  ,-0.9947531519590260,
      0.5110816729083550,0.5146956026128700,0.6883923011395250,
      -0.1446942037238470,0.9515456918526300,0.2713381354056610,
      -0.4478957210732550,0.8874427193875220,-0.1087880639148770,
      0.0234550198951957,-0.3332336016908820,-0.9425525071558810,
      0.9229881949625620,0.3842551764058190,0.0209940792863651,
      -0.6911807120917000,-0.6043864949901590,0.3962148254494260,
      -0.0316674103891205,-0.6829866700288160,-0.7297440535435660,
      0.8018580068378580,-0.1378458982057450,0.5813968053041460,
      0.3973142766999960,0.4262411104325310,-0.8126868285555020,
      -0.8315080441618440,0.3951380066705710,0.3904616859290430,
      -0.8752414233521170,-0.2348057149272050,-0.4228696336769560,
      -0.2818275522720080,-0.8725715270267310,-0.3989889234084150,
      0.2217711399465410,0.7995959805053020,-0.5580894457401740,
      -0.3678809968698430,-0.0344850464711680,0.9292332073876450,
      0.6642059344014850,0.6635911630050780,0.3442052368680880,
      -0.8248962619923230,0.5292990045320030,-0.1984659183651500,
      -0.1835415051538410,0.8857989535243100,-0.4262308410017950,
      -0.1166555399998580,-0.9878457938197680,0.1027237685247790,
      -0.5153844228415700,-0.8501138123201320,0.1080990415997830,
      0.5451858248749490,-0.3086757610614890,-0.7794175330902750,
      -0.5671903126002800,0.6003393592309330,-0.5638153980254410,
      -0.4890503721480340,-0.1375661511749660,-0.8613392406902020,
      0.3976996460860260,-0.6740608343992310,-0.6224764919513510,
      0.4349956952359570,-0.3316293950165250,0.8371383932702870,
      0.6340600901853650,-0.7334511285536860,-0.2449841710344760,
      0.9845956419603380,-0.1475029048111290,0.0938845828929696,
      0.9610090220995200,0.0558414999946514,-0.2708198410782880,
      -0.7632149908465080,-0.1335295186020360,0.6321967616249480,
      0.0826215834627986,-0.6217188595585800,0.7788705499730220,
    };
    auto      pointshell
        = new vnl_matrix_fixed<double, 3, 30>(coords);

    // List sorted for each row [x,y,z]
    // pointshell need [x...x
    //                  y...y
    //                  z...z]
    int i=0;
    for(int c =0; c < 30; c++)
    {
      for(int r=0; r<3; r++){
        pointshell->put(r,c,coords[i++]);
      }
    }

    return pointshell;
  }

  static vnl_matrix_fixed<double, 3, 60>* DistributePointShell2()
  {
    double coords[3*60] = {
      -0.9757323531821020,0.1141645636029930,0.1868497454401630,
      -0.3081645385608390,-0.9413175847225490,0.1376801433239040,
      0.2268128430936140,0.3605118823065700,-0.9047580433041550,
      0.0053723381344127,-0.9527141369992010,0.3038205245615180,
      0.4997098468405750,0.8591889901243150,0.1099288325223530,
      0.2826871445408290,-0.5384236997311110,0.7938437490332800,
      -0.9169656502561230,0.3927426788378230,0.0701939062145264,
      -0.4122096064130160,-0.6082501787985420,0.6783177429290540,
      0.8689901348066610,0.0966091330875035,-0.4853069348492570,
      0.9016117108528340,-0.0311940714365313,0.4314200421401830,
      0.0602430803117106,-0.3322121265937110,0.9412788504044060,
      -0.7401273323624760,0.6637816179074090,-0.1077288058887970,
      0.2609246136270950,0.8706803367668190,-0.4169341640729010,
      -0.3416095796360020,0.1763315450422400,-0.9231522525152230,
      0.2417187698627780,-0.7721496224459510,-0.5876708235505650,
      0.0613208945746505,0.1158091933659950,0.9913768096039370,
      0.6752331524032520,0.5192825815894980,0.5238375610371130,
      0.1859160712558880,0.0545608595018432,-0.9810496047902950,
      -0.8705635614786970,-0.3824245858905910,-0.3096296522168640,
      -0.7511998187032470,-0.2098339818012360,-0.6258342691652840,
      -0.3238272471878560,0.9093048344891080,0.2613438959560160,
      0.0312467158098265,-0.9020995865833540,-0.4303951424415010,
      0.1997844244448850,-0.2949296496204980,-0.9343996390859500,
      0.0993378318350216,0.6282103369083360,-0.7716759473819500,
      0.7029961904247910,-0.5712633600352550,0.4236207380776580,
      0.6081885222938300,-0.0369636906278732,0.7929315272614730,
      -0.8386908835040670,-0.4071869261928720,0.3616578618871240,
      -0.6722483496885470,-0.0353283318457414,0.7394822954675930,
      0.0999489360975153,-0.5914665380137460,-0.8001109576696270,
      0.1502553918254090,0.7709976382621730,0.6188585937203620,
      -0.2288419561759350,-0.9457701920389410,-0.2305430609326490,
      0.2000892573988740,0.4465173368776440,0.8721161373012560,
      0.6810481798116100,0.7159072717862530,-0.1537860688711920,
      -0.9675337290891160,-0.1992515487286990,0.1554905251265250,
      -0.6915973408017530,0.2529246583829880,0.6765517240963850,
      0.4051983586769120,-0.6954173613750950,0.5934719737474010,
      0.3828531024787670,0.6364092429998630,0.6696318222327270,
      0.5431852686098420,0.7191354072485380,-0.4333520854994610,
      -0.4766435443833670,-0.7756897459394560,-0.4136621201440160,
      -0.2634864894153490,0.9592084977706880,-0.1024398735848210,
      -0.7706589629193440,0.6046810772525520,0.2011108094677480,
      -0.5074055034405280,0.8063323635715170,-0.3039206714513410,
      -0.5457935890943380,0.8354998025106060,0.0636351955151979,
      0.3190257966006890,0.1393790695388820,0.9374412067312650,
      -0.4883735140357740,-0.2538251441534990,0.8349036513178050,
      0.7814334110960810,-0.3206898482390990,0.5352754853896490,
      0.5674042808667500,-0.3889648144620360,0.7257814789346710,
      0.7365968935987790,0.6532051494950900,0.1753512161740710,
      0.5454313517376770,-0.7269270773618580,-0.4172309489236830,
      0.9213613541912860,-0.2944856448350440,0.2537153128782090,
      0.8605683113688990,0.5039629381095560,-0.0737803393843139,
      0.9884254663736010,0.0653918314621231,0.1368904883553190,
      -0.6908004715469640,0.4998012720691420,0.5224877002844740,
      0.6727955231552780,0.3943331152249490,-0.6259772985174470,
      -0.4586010376503450,0.1473697927994620,0.8763373964611290,
      -0.0145240600324916,0.9964844905654110,0.0825088585714514,
      0.5144395480682960,0.3368539248460700,0.7885945629403520,
      0.4119578382931460,-0.5109788900305280,-0.7544476876576700,
      -0.1166393819714400,0.8099047244599660,-0.5748474509561170,
      0.8653655945438620,-0.2118053715810700,-0.4541815411805620,
    };
    auto      pointshell
        = new vnl_matrix_fixed<double, 3, 60>(coords);

    // List sorted for each row [x,y,z]
    // pointshell need [x...x
    //                  y...y
    //                  z...z]
    int i=0;
    for(int c =0; c < 60; c++)
    {
      for(int r=0; r<3; r++){
        pointshell->put(r,c,coords[i++]);
      }
    }

    return pointshell;
  }

  static vnl_matrix_fixed<double, 3, 90>* DistributePointShell3()
  {
    double coords[3*90] = {
      0.7830306271444820,0.5716685391117410,-0.2450675791359720,
      -0.3870553382696590,0.9200004222080310,0.0615417602445897,
      0.9237963183374240,-0.2584997104179900,-0.2824504592668160,
      -0.5842983723039450,-0.7712899205183550,-0.2524029925134570,
      0.3359998847225240,0.7641041306124860,0.5506804473080420,
      -0.9783788169663750,-0.2066075720365220,-0.0093915754081947,
      -0.0445468908163855,-0.8801917861335100,0.4725230091135180,
      0.8775085322085220,-0.2202272889989130,0.4260031890507770,
      -0.5808389302765810,0.4437425367947590,0.6824358564100420,
      0.5331596759297140,0.4366496520639920,-0.7246225509290440,
      0.6151687893470670,-0.1325993633208080,-0.7771645703840200,
      0.4011147552907920,0.8712537751236570,-0.2828848041531790,
      0.7304993029031530,-0.0076263316203289,0.6828708571348050,
      -0.5475051350669790,0.6162612183405430,0.5660920754123990,
      0.8023122911715880,0.2709359026259910,-0.5318728458036140,
      0.2336323077450560,-0.6929349377840680,0.6820975859625270,
      0.1689148924284850,-0.9486250397707260,0.2675411987635920,
      -0.8314637780782760,0.5554909857699080,0.0098868838455222,
      -0.1011517806960660,-0.2148730146317450,0.9713896771353290,
      0.3877486434569820,-0.3420676320990350,-0.8559443466531150,
      -0.3648694025982520,0.8893842897839550,-0.2754376592499350,
      0.1726950806932710,0.9829215274686680,0.0635742081586933,
      -0.3959729941386110,-0.7643798871495520,0.5088504456460080,
      -0.9617290799445310,0.1062410103220330,-0.2525668713723110,
      -0.0327316740125812,-0.4982367497870410,-0.8664229790800670,
      0.1706515883535990,-0.2008209194778680,0.9646496740747160,
      0.2501225023053060,0.6910620770273610,-0.6781385843138260,
      0.3654852701417630,0.4920313286800810,0.7901428281688790,
      0.9153163381057960,-0.3852897569037580,0.1172510316442380,
      -0.6881847644097410,0.7218125899083820,0.0734051434441835,
      -0.6873457829778960,0.2458189841721520,-0.6834682155324220,
      -0.3219850147738870,0.8095000604196670,-0.4909534626027350,
      -0.5632523312637280,-0.5973559275989310,0.5708876483936170,
      -0.9084685052242310,-0.3740631899713040,0.1864449112315510,
      -0.7845198100635990,-0.3605738415896990,-0.5044949676448910,
      -0.2000039873810240,-0.6330205581036100,-0.7478525108936150,
      -0.9148801969792540,0.0534543198667285,0.4001710395103400,
      0.7664323664233560,-0.3341914416639150,-0.5485412546174510,
      0.4872976306941970,-0.8343714235123850,-0.2576147254054740,
      -0.6413753405832620,-0.2298206308844890,0.7319973702900570,
      0.7054813742936480,-0.6006428992790810,-0.3761969405382820,
      -0.1929691654472430,0.9760562116682670,0.1003851226546300,
      -0.9439691512808960,-0.1047955385010510,-0.3129538888403490,
      -0.4276451629474120,0.0186575235064224,-0.9037541211105420,
      -0.7344812627593790,-0.4734265259198780,0.4862145608790640,
      0.6377710213710160,0.3906960864871740,0.6637806055489840,
      0.0474387462958210,0.9621772979130640,0.2682618361425920,
      0.7913486901745450,-0.0748105169885527,-0.6067706626946760,
      -0.0982434620561768,0.1155119940049920,0.9884357345848060,
      -0.2883901942644810,0.3703318191063880,-0.8829979839215070,
      0.3295292140825860,0.2147146585289090,-0.9194063913628870,
      -0.5210574847572160,-0.0609880081835259,0.8513398618861260,
      0.1729068266771900,0.9459249066560850,-0.2744618375222680,
      -0.2062545956858610,0.4593595720405990,0.8639721206920620,
      0.6720182971610780,-0.4607369705129640,0.5797524060176540,
      -0.2554108625029320,0.6389014951165470,0.7256515491979250,
      0.4500266667406290,-0.5361016246739610,0.7141925841425810,
      -0.9983559681192810,-0.0042300087224308,0.0571617699762653,
      -0.8406688842019840,-0.0901045876756096,-0.5340009273535010,
      -0.3909575963129450,0.7939675140678560,0.4655832304648770,
      -0.9713179118089640,0.2320449947116890,0.0519291308258172,
      -0.0612195458720222,0.7850603547539780,0.6163865723689800,
      -0.6242925102191230,-0.1774216962215860,-0.7607761848232110,
      -0.0802679405745239,-0.4570324492769940,0.8858207482463910,
      -0.3609460670444700,-0.2996972708558150,-0.8831191779860200,
      0.1285945459353310,0.8707004556055770,0.4747042862287270,
      0.5951730348219600,-0.7933110389121110,0.1281665095140040,
      0.2033751954557950,0.0131770863036256,0.9790122033304090,
      -0.2052893840274300,0.8803381043882250,0.4276226032002800,
      -0.3731942266334910,-0.9275730588324440,-0.0182835919789316,
      0.2685955896736170,0.5262799019911600,-0.8067749834793190,
      0.9390948774123210,0.2816262184669510,0.1969453840280110,
      -0.9130820847807490,-0.2162790214867180,0.3456942164936930,
      -0.7722185508796150,-0.6352920195758570,0.0090862280741177,
      0.1460103850487270,0.2737759341802480,0.9506459410956610,
      -0.7668676175868570,-0.5322707367226740,-0.3586110984414160,
      0.5327000875787680,-0.1961334068751380,0.8232632042069670,
      -0.8210993253048930,0.4385794796816200,-0.3652997919326650,
      -0.6255762780977020,-0.7798597751409450,-0.0217497447925055,
      0.2991048825976350,-0.1334972919760040,-0.9448358281952090,
      0.0606298744362150,-0.9937831824740320,0.0933756100790349,
      0.8564420120030050,-0.4610622959465800,-0.2322254063032560,
      -0.5455890340730230,-0.6468991032576630,-0.5327796506096080,
      -0.8522969529983180,-0.5007967009121140,-0.1509720777671300,
      -0.0228411934161264,-0.7512799112567760,0.6595883373933590,
      0.5854881600260220,0.7616691389056020,-0.2776035614111240,
      -0.1213541141003080,0.4896966102871260,-0.8634062826179960,
      0.5356517245344040,-0.7170045054372250,0.4460737261776780,
      -0.3912157018163170,-0.8815613582567070,-0.2641966053547140,
      -0.6919685152992150,0.6644155090244570,-0.2823678544069240,
    };
    auto      pointshell
        = new vnl_matrix_fixed<double, 3, 90>(coords);

    // List sorted for each row [x,y,z]
    // pointshell need [x...x
    //                  y...y
    //                  z...z]
    int i=0;
    for(int c =0; c < 90; c++)
    {
      for(int r=0; r<3; r++){
        pointshell->put(r,c,coords[i++]);
      }
    }

    return pointshell;
  }
};

// Optimal PointDistribution For Multishell Scheme 50 - 100 - 150
// Generated by Emmanuel Caruyer
template<> class ThreeLayerPointShell<50,100,150, vnl_matrix_fixed<double, 3, 50>,vnl_matrix_fixed<double, 3, 100>,vnl_matrix_fixed<double, 3, 150> >
{
public:
  static vnl_matrix_fixed<double, 3, 50>* DistributePointShell1()
  {

    double coords[3*50] = {
      0.913310030405731,-0.305436511654557,-0.269394368368340,
      0.274520611983463,-0.931497424367645,-0.238644048726690,
      0.724295307031846,-0.291296816682609,-0.624933974754310,
      0.179546279617146,0.559467539325302,-0.809171926055848,
      0.333306860790720,0.932650091766363,0.138095412226576,
      -0.423224047088266,0.854335111704236,-0.301650332132319,
      -0.057337449880758,-0.311676405763001,-0.948456764924911,
      -0.167285210491163,0.516195511070583,0.839974911947675,
      0.763325168891534,0.512093536284904,-0.393820894102804,
      0.457619183693223,-0.672231259407514,0.581970632069466,
      -0.246793802164352,-0.050216662420050,-0.967766038889902,
      0.927994874653326,0.242210812920042,-0.283124415622830,
      0.613792300977422,0.763214386431347,0.201898022786412,
      0.720951430921191,-0.550579081664088,-0.420822657525251,
      -0.566617724460163,-0.603447657535153,-0.561066198360694,
      0.575968738378302,-0.783648913166458,-0.232710965156210,
      -0.532150073708023,-0.619733926203769,0.576841537828723,
      -0.911227494753416,-0.409216343893830,-0.046972722888692,
      0.243129633174557,0.820980315213879,-0.516603623200345,
      0.429059425848317,0.332137825141317,-0.839995520345857,
      -0.862659530206915,0.023610632091113,0.505233681572638,
      0.837291268298631,-0.163164674031590,0.521843483411485,
      0.803588479073049,0.456397841392963,0.382029536388100,
      0.303630079812031,0.840160692576492,0.449375995445604,
      0.558311152322438,-0.044011396373360,-0.828463429598227,
      0.238930415896487,-0.057120162540996,-0.969355220438215,
      0.246985899213630,-0.326331770359038,0.912417416122670,
      0.837050631211904,-0.547095262685976,0.005745810152941,
      -0.554571420346131,0.053288087401785,-0.830428154313384,
      0.751372177673996,-0.530499886791399,0.392440722570616,
      0.204389904673926,0.615196807070163,0.761418186963024,
      0.106700970850229,-0.867025439314806,0.486705034287282,
      0.717231391359855,0.239339763160166,-0.654442976139587,
      0.069544882069961,0.220289919002015,-0.972952137036535,
      0.648780744604385,-0.754349357811800,0.100202753452702,
      -0.727376323458570,-0.175112683983955,-0.663520332755394,
      0.726060483848279,0.676802250037960,-0.121551997669954,
      0.580156273414571,-0.371836855429485,0.724676514978945,
      -0.060890718603115,0.804268669172008,0.591138078773718,
      -0.386606299473394,0.736073310034347,0.555636258232458,
      0.171832268210408,-0.620770257189966,0.764930035617617,
      0.030029178348917,0.973024956515624,0.228737147060675,
      0.197806050466708,-0.977284832169067,0.076073143822506,
      0.941717594278340,-0.292439497987513,0.166274209195007,
      0.103620708501302,0.975442422157789,-0.194357479466658,
      -0.999930437744010,0.003907201094968,0.011128946611211,
      -0.453284936431437,-0.867544703290094,0.204692340349242,
      -0.939310554852217,-0.109134901287686,-0.325246452500478,
      0.453912788215515,0.373549749748161,0.808964625405526,
      0.446853673336740,-0.414239447718517,-0.792923372451158
    };

    auto      pointshell
        = new vnl_matrix_fixed<double, 3, 50>();


    // List sorted for each row [x,y,z]
    // pointshell need [x...x
    //                  y...y
    //                  z...z]
    int i=0;
    for(int c =0; c < 50; c++)
    {
      for(int r=0; r<3; r++){
        pointshell->put(r,c,coords[i++]);
      }
    }

    return pointshell;
  }

  static vnl_matrix_fixed<double, 3, 100>* DistributePointShell2()
  {
    double coords[3*100] = {
      -0.6783406728914600,-0.6757825732021540,-0.2883952934037510,
      -0.7868810762760640,0.3417432912343670,0.5138381989448860,
      0.1812675472397420,0.9341812435603770,0.3073230881298610,
      0.4392152776431420,0.8556891913232610,0.2736529695387940,
      -0.6296241987663400,0.2128741188301860,0.7471666332620290,
      0.6170474427091470,0.0980582575477010,-0.7807925662894150,
      0.3935013484893500,0.9192019551005550,0.0149818048438275,
      -0.3269299998796330,-0.4401580515926610,0.8362880274145110,
      -0.2869929508839660,0.8309767335503890,0.4765634421992920,
      0.5088992607891730,-0.7072946217461350,-0.4906687889169470,
      0.5806843974003410,0.3583450335309360,-0.7310228912691540,
      0.9491724956627390,-0.1834420594707910,0.2557744793650880,
      0.0153560310858227,0.1752309003794950,0.9844076004691760,
      -0.0479773456177447,0.9905464575577180,-0.1285141615828030,
      0.0693843127112566,-0.5942844228507240,-0.8012564145806050,
      -0.3165721030432440,-0.4164032785066600,-0.8522854059666170,
      0.4896581521634270,0.7211168510820520,0.4901279232051600,
      0.7776461071669160,-0.0320557656217032,-0.6278845115931320,
      0.2300492326525960,0.8027977937856420,0.5500845869944510,
      -0.3766021271868230,-0.7564724591162180,0.5347151170455350,
      -0.1790056555604940,-0.2983809560973960,0.9375104161105410,
      -0.5841988205464850,-0.8089448753434100,-0.0657261494972090,
      -0.8551587253793000,0.4448943753338720,0.2660311057074570,
      -0.7752975795230930,0.6190258715864510,0.1253819504245550,
      -0.6764680634440770,-0.5064314530534920,0.5347131403830920,
      0.7501893689724760,0.6048830482657770,-0.2670812771449510,
      -0.5138286280086930,-0.4945630265573110,-0.7009904092076970,
      0.5804145400895790,0.7956021922609330,-0.1735975613947330,
      -0.1755232810907650,0.3977545839146920,0.9005458726627870,
      0.0397560432054666,0.4921914390519250,-0.8695786590945290,
      0.5538585716719580,-0.8081611673456340,0.2002903147434720,
      0.8992305597521990,0.1140934067118160,-0.4223352873637700,
      0.7698402757004470,0.6378303091348330,0.0227694237612964,
      -0.4412960658520300,0.8971893864056890,0.0175780313033856,
      -0.0811198993955311,0.2753962213738170,-0.9579021260938310,
      -0.8572077886102190,-0.3790625619618600,0.3485776545690900,
      0.6857756541322800,0.5415157578159610,0.4862843162558790,
      0.8549241931498690,-0.2861863354061220,0.4326684705335710,
      -0.4182775751481730,0.7970464185340600,-0.4356155149098700,
      0.3499190445152290,0.8866592889390710,-0.3023110444949170,
      -0.0132033510852791,-0.8749575867614390,-0.4840195170535130,
      0.9116379197252770,-0.0091996476760664,0.4108913114214080,
      -0.6490542475593910,-0.0530061128719349,-0.7588932307797450,
      0.2036777677589340,-0.7982257583158190,0.5668782988277740,
      0.3791318665509280,0.1705982429677800,-0.9094807679450480,
      -0.1470179365181320,0.9220511216311460,-0.3580606868125770,
      -0.3344979114418210,-0.6343147268032640,-0.6969619606560740,
      -0.6076879381877640,0.5109764367904180,0.6079625406438060,
      -0.5844013508588990,-0.7109052658706510,0.3912655927521240,
      0.9862110505381910,-0.1637207304543020,0.0241513191330477,
      -0.2051748270057740,0.7136488133907960,0.6697825479281010,
      0.9069081158736650,-0.4209955960690930,0.0167444752950723,
      -0.1319432828456750,0.9392959075033450,0.3167241201098720,
      0.7549548594926640,-0.1430122887152190,0.6399926916808070,
      0.2084074643614400,-0.1965342739298880,-0.9580921709153480,
      -0.4579812060622150,-0.5603838600979070,0.6900892291838540,
      -0.5044451924143650,-0.2283097986021220,-0.8327122454439420,
      -0.1632507891313170,0.0595367888203028,-0.9847865507941140,
      0.6304967444231040,-0.7732039721298140,-0.0680402289424677,
      -0.4189905487906070,0.2950674433427220,0.8587095690065030,
      -0.1064870135377420,-0.0503578524476393,0.9930380670672570,
      -0.9594694024759230,-0.2706424081220130,-0.0785566842376365,
      0.6572248348109010,-0.6577047684772120,0.3680760166461810,
      0.8938413333977800,-0.1605095289995250,-0.4186697526810690,
      0.9676575561375490,0.2066292219530930,-0.1447177207012450,
      0.8845432302365500,0.4553528053106210,-0.1011785379338700,
      0.3127824838603270,-0.9329319353982300,0.1783399049687880,
      0.0538977574368990,0.9943600559842480,0.0913406306430770,
      -0.8270410405970430,-0.3169506321051900,-0.4642686872666290,
      -0.3836295436457290,0.5570493630789800,0.7365625433970090,
      -0.4341709114900190,0.8573912849439030,0.2763617269416430,
      -0.6794088494430530,-0.3329865853053170,-0.6538528498868610,
      -0.3373613680825300,0.6537103259033100,-0.6773847629913670,
      0.3577142489860180,-0.2395141136991300,0.9025926575212580,
      0.0475306676831556,-0.6921666442661880,0.7201709326228570,
      -0.9238533114270850,-0.2442810637733990,-0.2946554273162900,
      -0.7702042751707540,0.6239784781621260,-0.1320463301238260,
      -0.1058068553327490,-0.9342834914303710,0.3404691865723970,
      -0.6712641876558270,0.6601081560713440,0.3371373795040410,
      -0.9902933383503500,-0.0392378960032633,-0.1333397597723740,
      0.2680611017833540,0.2096519795061000,0.9403134015847410,
      -0.2193780563994170,0.9713963546921800,0.0908976922763060,
      0.4259546539557210,-0.0086224931160314,0.9047034239937000,
      -0.7126278602910880,0.4094408405402170,-0.5696663328955520,
      -0.2175209712545230,0.4840278946263750,-0.8475857622022950,
      0.5687366838665980,-0.5871979617261380,0.5759662647838790,
      0.0944273658455115,0.8302981465345910,-0.5492617412861550,
      0.2132739207069970,0.6628352636620030,-0.7177490146230690,
      0.9572113632369580,-0.2441349912405500,-0.1553850447823740,
      -0.8451592214984760,-0.4858715283504990,-0.2227885729889170,
      -0.8457073503037060,0.4544256488253640,-0.2797792117580030,
      -0.4568229734634650,0.4521433068383050,-0.7660804141846460,
      0.2161513748810380,0.9712225012944180,-0.1000271768893380,
      -0.9727568289959310,0.0233660996838811,0.2306473000651490,
      0.0955550598977780,0.4497094877292240,0.8880487639618950,
      -0.5803054956345510,0.2185728470394220,-0.7845198801008280,
      0.0802960854964209,0.7085380275275620,0.7010894395163160,
      -0.8170365471147090,-0.1054124160886850,-0.5668681532889450,
      -0.7903491223400260,-0.2661789011279690,0.5518125201648250,
      -0.4240807031724980,0.0532021972358502,0.9040603317290260
    };
    auto      pointshell
        = new vnl_matrix_fixed<double, 3, 100>();

    // List sorted for each row [x,y,z]
    // pointshell need [x...x
    //                  y...y
    //                  z...z]
    int i=0;
    for(int c =0; c < 100; c++)
    {
      for(int r=0; r<3; r++){
        pointshell->put(r,c,coords[i++]);
      }
    }

    return pointshell;
  }

  static vnl_matrix_fixed<double, 3, 150>* DistributePointShell3()
  {
    double coords[3*150] = {
      -0.5818137157755120,-0.5030550053652260,0.6390840803153230,
      0.9670851419246870,0.1662861674810110,0.1926012429165080,
      0.0578846316476645,-0.7976809936025330,0.6002952622370840,
      0.4660628327321890,-0.1511747542980830,0.8717405747173640,
      0.6902368719591830,0.3748091607024440,-0.6189435787222700,
      -0.1997955747625600,-0.1526408912355860,0.9678752433181160,
      0.9511193914166630,0.1151598714233260,-0.2865486124289540,
      0.0822444520102396,-0.5510970245321980,0.8303781787025070,
      0.7452339673590670,0.4163359900238880,0.5208557163985940,
      0.7753251365196900,0.6018341048080190,0.1914853596770620,
      0.1116030335825260,-0.9761251513556840,-0.1863449805764000,
      -0.3170679847245510,0.9480932422722550,-0.0242300850265728,
      -0.1215812147248450,0.4013080511364970,-0.9078380121580370,
      0.8680201958339400,-0.2937180126790240,-0.4003381928473620,
      0.7117815464094280,-0.4555489837210150,-0.5346420799205320,
      -0.4221584476961910,-0.8100472693071080,-0.4069467612929700,
      0.0186640877956507,-0.9603335901627470,0.2782284087434000,
      -0.5396677502633030,0.3145543091567380,0.7809060800868890,
      -0.4898321413919820,-0.0679885797782780,0.8691616801718060,
      -0.6750918694436950,0.2756900693390990,-0.6842849943399450,
      -0.4765551297163720,-0.8680896906246820,-0.1389802049651460,
      -0.8007926338965610,0.4138072508780090,-0.4330066011249610,
      0.6252172011926140,0.2326499546399870,-0.7449680865237690,
      0.6824097373557610,-0.7075614150231750,-0.1835042079420600,
      -0.7485107920414520,-0.0179476036655747,-0.6628796857048340,
      0.6619059872141820,-0.0443716919725870,-0.7482724216764290,
      -0.0389254961022405,0.4513344326841560,0.8915054882763560,
      0.6705154952625320,-0.7056948744571640,0.2289185767423150,
      -0.0421968777730212,-0.3582833631874890,0.9326588096240070,
      -0.0859685420644288,0.2858229400217130,0.9544184914033570,
      0.9471108604678190,0.3162982815144110,-0.0541886989596281,
      -0.5550008908723820,-0.7690560878042730,-0.3170595290194690,
      -0.9104090909480310,-0.3561208782301570,0.2105545231234920,
      -0.8278006128591360,-0.4943059810149400,0.2653445731174760,
      0.9608962244735380,-0.0205957379787835,0.2761417414474080,
      0.2202282819057040,-0.6075951351306100,-0.7631039612100540,
      0.6244544517901810,0.4795352369175020,0.6165246095606390,
      -0.1883106968238980,-0.4391326823901580,0.8784654624505860,
      -0.2302051549830810,-0.9409736043275990,0.2481416179884810,
      -0.0791847103092564,-0.9956294659870780,0.0495151301272986,
      0.3432876609136320,-0.1606450217698800,-0.9253900576756810,
      0.6910132782044240,0.7150303514681560,0.1059822901455680,
      -0.3743211784491260,-0.1177211051182680,-0.9197963887590500,
      -0.0638117734968980,-0.6201179359188230,0.7819090759896310,
      -0.6081942748353220,0.4670488067500740,-0.6418451029421870,
      -0.9237110100984830,0.0253860825460826,0.3822479779355320,
      -0.9551685142832210,0.2944878574922000,-0.0304960835458829,
      0.2047274908347290,-0.2027142615186940,0.9575978188537430,
      0.4664286040672840,0.4413500991724570,-0.7665862295060520,
      0.2436221916378790,0.7468678862924580,-0.6187378994102360,
      0.7739427622698460,-0.5762620863418250,0.2625540107769950,
      -0.8824775950813450,-0.1623522085699230,-0.4414465477857010,
      -0.5674249283656180,-0.3593537766554100,-0.7408736828048120,
      0.6114933730446530,0.6602715260117740,0.4360245023626030,
      0.6580967800369840,-0.1033246068935280,0.7458100654424360,
      -0.0651725243592964,-0.5903861242505850,-0.8044854046911040,
      0.6288524174560490,0.2026776272049530,0.7506440011684980,
      -0.5594322362469380,0.7479956050065880,0.3571248352307300,
      0.5065956414163710,-0.4948790783785480,-0.7060138482219270,
      -0.7796248885518470,0.2823581769849720,-0.5589802259832120,
      0.1069521912958730,-0.0945188164394620,-0.9897612955232660,
      -0.1677987609066020,-0.9702091453514290,-0.1747506512567480,
      0.6497726089471500,0.7581321700271420,-0.0550560571776502,
      -0.7436099555683260,-0.5675313008756160,-0.3534861475448520,
      0.8705569853960000,-0.4718161643221160,0.1397141448193180,
      -0.7367938401263080,0.6760868762135980,-0.0064321818744716,
      -0.8709771462534060,-0.3550310615284370,-0.3396347391746950,
      -0.5568793244548770,0.8296552833726470,-0.0394655389742952,
      0.8255863288934740,-0.1743226461918030,-0.5366738567965490,
      0.5123909885019620,-0.8475608044158260,-0.1381888480304350,
      0.3892383734256480,0.6628221159035160,-0.6396564166192190,
      -0.3850734793456500,-0.5125887952201850,-0.7674445533843810,
      -0.0303214011502939,0.1349348899639910,-0.9903904220572250,
      -0.1951010042782680,-0.3408158110301630,-0.9196630801991900,
      -0.7299481699543050,0.1633700135838910,0.6636911238233930,
      -0.3764204298254850,-0.8272654006443230,0.4170606873187450,
      -0.0207934145254701,-0.9354136997804840,-0.3529431174783800,
      0.0530508631426718,0.0286917551417508,0.9981795375115150,
      -0.9893103145653900,-0.1443535615447680,-0.0206676259852998,
      0.3520785503091250,-0.7515903016454040,0.5578106424986840,
      0.4646561319602880,-0.5645009731359150,0.6822267440963500,
      -0.3638080873049480,-0.7397671945737120,-0.5660284210567990,
      0.8993857269815550,-0.3263912150460360,0.2908162458368350,
      -0.1700781261443220,0.9625883335514640,-0.2109434310848530,
      -0.3098321331013300,-0.2959672665364290,0.9035526694315240,
      0.3772438334691890,0.2858348804818840,0.8809003980072750,
      0.4170539029542560,-0.8070208930488820,-0.4180709511712160,
      0.3535382771055950,0.9267047409094850,-0.1273931308865980,
      0.0162686253194627,0.8954558957620720,-0.4448528639619620,
      -0.4409570938666000,0.8843580437434560,-0.1531916833087520,
      0.1659388260557630,0.8826574008153690,0.4397501777067190,
      -0.6722877318211110,0.5534153014066750,-0.4916916816581130,
      -0.5022690191279730,-0.7127108731518800,0.4896621730492420,
      -0.2994694405516180,0.9013048000637180,-0.3129979417788570,
      -0.3172838107693270,-0.5624673796012650,0.7635191093274480,
      0.7315837873258040,0.0963804929381182,-0.6749044100485910,
      -0.2068469360516820,0.7944607239234560,0.5710048188842610,
      0.5378470897916880,-0.7741067897070790,0.3338849893780950,
      0.1257284809109600,-0.8835082232405290,-0.4512267374105670,
      0.3130995324769140,-0.4640943924714670,-0.8286042949679160,
      0.9894576229982930,0.0690358063959267,-0.1273093465768750,
      -0.0845800025595276,-0.7421395656655050,0.6648872748375060,
      0.5451534912742490,-0.6927392692238080,0.4721440202170950,
      -0.5063313485711320,0.1671271448910190,0.8459888196038490,
      0.9140701463180970,0.3632958280951350,0.1802551216988890,
      0.8112330574743470,0.5738372376167550,-0.1123020533436160,
      0.8691548124260960,0.2553703834185790,-0.4235042848770290,
      0.2587208365075750,-0.8528680460626400,0.4535191558932370,
      -0.7810165842314370,-0.4032429479626940,0.4768733794979690,
      0.6386218421663880,-0.3771596505315350,-0.6707553508686590,
      0.6209847697030420,-0.6181456007191380,-0.4819480595545900,
      0.1495129959273720,0.1671684875068860,0.9745258133233210,
      0.8500517134843810,0.5236094820204760,0.0569666107515227,
      0.2831566624639810,-0.3165192184973970,-0.9053385492864280,
      -0.9842250831135330,0.1106121936986040,-0.1380794277774220,
      0.5302801835192860,0.0861689235342689,0.8434321807849740,
      0.8470084161413430,-0.0276085553364133,0.5308620448458990,
      -0.2172067203511410,-0.5051072452535100,-0.8352771464769700,
      0.3204407112933310,0.0455278779726987,-0.9461738544649850,
      -0.4935635772769070,0.6256485875038120,0.6041182335769260,
      0.4980559918394660,0.2263222843527210,-0.8370892739715230,
      0.2021779827202400,-0.7110486153559470,0.6734492778996520,
      0.6602958653226490,0.7044162570524200,-0.2603979781758510,
      0.4825507527225460,-0.3146110298685950,0.8174134027112130,
      -0.8074813189441350,0.4434122850213180,0.3890494377947510,
      0.7706009639641810,-0.5764170340352260,-0.2718778350868380,
      0.9877213210351460,-0.1170482811531720,-0.1034712126713200,
      0.9471546743930560,-0.1633313871864270,-0.2760812937073770,
      0.9072494495203930,-0.1579003412593720,0.3898280628368310,
      -0.3139240581363140,0.0802169066474812,-0.9460533460704720,
      0.2464642940724080,0.9686222614596370,0.0320353921812441,
      -0.3117655950297000,-0.9062547169400450,-0.2854901080242450,
      0.0608632767228376,-0.7075111154985250,-0.7040764752444090,
      -0.4607335862628370,-0.6188008149745030,-0.6362468969482480,
      0.2190580158109430,0.7137403609578260,0.6652730889257150,
      -0.0823847846664380,-0.8044875371365980,-0.5882283143876660,
      -0.8295547427688960,-0.1162092353369240,0.5461999106298320,
      -0.0852410700430401,0.9961766497246490,0.0191322377491922,
      0.3131894593955590,-0.5541595443573950,0.7712454615238390,
      -0.2053809775103630,-0.8866791538596230,0.4142689128907330,
      -0.2803794157477670,0.8920364641234400,0.3544831870471280,
      0.3607197436800900,-0.9212300461377630,-0.1456587402540090,
      0.5099563266151200,0.8025819147395540,-0.3095267598741700,
      0.3434662076553160,-0.3991341139236100,0.8501311212402990,
      -0.8525024407450160,0.5081350695533460,0.1226309080689130,
      -0.7808884875619300,-0.2500741633167730,-0.5724299807266000,
      -0.5053813675827790,-0.8611316666645480,0.0551536577907309,
      -0.9210719657807660,0.3672285791438610,0.1294975077471940,
      0.6652607437466560,0.6063117536090580,-0.4356767153121150,
      0.3394557649489950,-0.6705912128459400,-0.6596038272301810
    };
    auto      pointshell
        = new vnl_matrix_fixed<double, 3, 150>();

    // List sorted for each row [x,y,z]
    // pointshell need [x...x
    //                  y...y
    //                  z...z]
    int i=0;
    for(int c =0; c < 150; c++)
    {
      for(int r=0; r<3; r++){
        pointshell->put(r,c,coords[i++]);
      }
    }

    return pointshell;
  }
};

}

#endif //__itkPointShell_txx__
