/*===================================================================

The Medical Imaging Interaction Toolkit (MITK)

Copyright (c) German Cancer Research Center,
Division of Medical and Biological Informatics.
All rights reserved.

This software is distributed WITHOUT ANY WARRANTY; without
even the implied warranty of MERCHANTABILITY or FITNESS FOR
A PARTICULAR PURPOSE.

See LICENSE.txt or http://www.mitk.org for details.

===================================================================*/

#include <mitkLookupTableProperty.h>
#include <mitkRenderingModeProperty.h>
#include <mitkTransferFunctionProperty.h>
#include "mitkFreeSurferParcellationTranslator.h"
using namespace mitk;
using namespace std;


/*###########   Static Members   ###########*/
LookupTable::Pointer                              FreeSurferParcellationTranslator::m_LookupTable      = FreeSurferParcellationTranslator::CreateLookupTable();
const FreeSurferParcellationTranslator::NameTable FreeSurferParcellationTranslator::m_NameTable        = FreeSurferParcellationTranslator::CreateNameTable();
TransferFunction::Pointer                         FreeSurferParcellationTranslator::m_TransferFunction = FreeSurferParcellationTranslator::CreateTransferFunction();


/*###########      Constructors      ###########*/
FreeSurferParcellationTranslator::FreeSurferParcellationTranslator()
{
}
FreeSurferParcellationTranslator::~FreeSurferParcellationTranslator()
{
}


/*###########      Methods      ###########*/
// Assign the lookup table to the given node
void FreeSurferParcellationTranslator::AssignLookupTable(DataNode::Pointer node) const
{
  LookupTableProperty::Pointer lookupTableProp = LookupTableProperty::New( this->GetLookupTable() );
  node->SetProperty( "LookupTable", lookupTableProp );
}
// Assign the color transfer function to the given node
void FreeSurferParcellationTranslator::AssignTransferFunction(DataNode::Pointer node) const
{
  TransferFunctionProperty::Pointer transferFunctionProp = TransferFunctionProperty::New( this->GetTransferFunction() );
  node->SetProperty( "Image Rendering.Transfer Function", transferFunctionProp );
}
// Get the label assigned to the given name
const string FreeSurferParcellationTranslator::GetLabel(const string& name) const
{
  NameTable::const_iterator iter;
  for( iter = m_NameTable.begin(); iter != m_NameTable.end(); ++iter )
  {
    if( iter->second == name )
      return iter->first;
  }
  return "0";
}
// Get the label assigned to the given name as integer
int FreeSurferParcellationTranslator::GetLabelAsNumber(const string &name) const
{
  return std::atoi(this->GetLabel( name ).c_str()) ;
}
// Return the lookup table
LookupTable::Pointer FreeSurferParcellationTranslator::GetLookupTable() const
{
  return m_LookupTable;
}
// Get the name assigned to the given label
const string FreeSurferParcellationTranslator::GetName(const string & label) const
{
  auto endIter = this->m_NameTable.end();
  auto iter = this->m_NameTable.find( label );
  if( iter != endIter )
  {
    return iter->second;
  }
  else
  {
    return "Unknown area or outside the cerebrum.";
  }
}
// Get the name assigned to the given label
const string FreeSurferParcellationTranslator::GetName(int label) const
{
  stringstream ss;//create a stringstream
  ss << label;//add number to the stream
  return  this->GetName( ss.str() );
}
// Get the transfer function
TransferFunction::Pointer FreeSurferParcellationTranslator::GetTransferFunction() const
{
  return m_TransferFunction;
}

LookupTable::Pointer FreeSurferParcellationTranslator::CreateLookupTable()
{
  vtkSmartPointer<vtkLookupTable> table = vtkSmartPointer<vtkLookupTable>::New();
    table->SetRange(0.0, 14175.0);
    table->SetNumberOfTableValues( 14176.0 );  // Count of the overall range (0 - 14175)
    table->Build();
    table->SetTableValue( 0.0, 0, 0, 0, 0);
    table->SetTableValue( 1.0, 0.27451, 0.509804, 0.705882 );
    table->SetTableValue( 2.0, 0.960784, 0.960784, 0.960784 );
    table->SetTableValue( 3.0, 0.803922, 0.243137, 0.305882 );
    table->SetTableValue( 4.0, 0.470588, 0.0705882, 0.52549 );
    table->SetTableValue( 5.0, 0.768627, 0.227451, 0.980392 );
    table->SetTableValue( 6.0, 0, 0.580392, 0 );
    table->SetTableValue( 7.0, 0.862745, 0.972549, 0.643137 );
    table->SetTableValue( 8.0, 0.901961, 0.580392, 0.133333 );
    table->SetTableValue( 9.0, 0, 0.462745, 0.054902 );
    table->SetTableValue( 10.0, 0, 0.462745, 0.054902 );
    table->SetTableValue( 11.0, 0.478431, 0.729412, 0.862745 );
    table->SetTableValue( 12.0, 0.92549, 0.0509804, 0.690196 );
    table->SetTableValue( 13.0, 0.0470588, 0.188235, 1 );
    table->SetTableValue( 14.0, 0.8, 0.713725, 0.556863 );
    table->SetTableValue( 15.0, 0.164706, 0.8, 0.643137 );
    table->SetTableValue( 16.0, 0.466667, 0.623529, 0.690196 );
    table->SetTableValue( 17.0, 0.862745, 0.847059, 0.0784314 );
    table->SetTableValue( 18.0, 0.403922, 1, 1 );
    table->SetTableValue( 19.0, 0.313725, 0.768627, 0.384314 );
    table->SetTableValue( 20.0, 0.235294, 0.227451, 0.823529 );
    table->SetTableValue( 21.0, 0.235294, 0.227451, 0.823529 );
    table->SetTableValue( 22.0, 0.235294, 0.227451, 0.823529 );
    table->SetTableValue( 23.0, 0.235294, 0.227451, 0.823529 );
    table->SetTableValue( 24.0, 0.235294, 0.235294, 0.235294 );
    table->SetTableValue( 25.0, 1, 0.647059, 0 );
    table->SetTableValue( 26.0, 1, 0.647059, 0 );
    table->SetTableValue( 27.0, 0, 1, 0.498039 );
    table->SetTableValue( 28.0, 0.647059, 0.164706, 0.164706 );
    table->SetTableValue( 29.0, 0.529412, 0.807843, 0.921569 );
    table->SetTableValue( 30.0, 0.627451, 0.12549, 0.941176 );
    table->SetTableValue( 31.0, 0, 0.784314, 0.784314 );
    table->SetTableValue( 32.0, 0.392157, 0.196078, 0.392157 );
    table->SetTableValue( 33.0, 0.529412, 0.196078, 0.290196 );
    table->SetTableValue( 34.0, 0.478431, 0.529412, 0.196078 );
    table->SetTableValue( 35.0, 0.2, 0.196078, 0.529412 );
    table->SetTableValue( 36.0, 0.290196, 0.607843, 0.235294 );
    table->SetTableValue( 37.0, 0.470588, 0.243137, 0.168627 );
    table->SetTableValue( 38.0, 0.290196, 0.607843, 0.235294 );
    table->SetTableValue( 39.0, 0.478431, 0.529412, 0.196078 );
    table->SetTableValue( 40.0, 0.27451, 0.509804, 0.705882 );
    table->SetTableValue( 41.0, 0, 0.882353, 0 );
    table->SetTableValue( 42.0, 0.803922, 0.243137, 0.305882 );
    table->SetTableValue( 43.0, 0.470588, 0.0705882, 0.52549 );
    table->SetTableValue( 44.0, 0.768627, 0.227451, 0.980392 );
    table->SetTableValue( 45.0, 0, 0.580392, 0 );
    table->SetTableValue( 46.0, 0.862745, 0.972549, 0.643137 );
    table->SetTableValue( 47.0, 0.901961, 0.580392, 0.133333 );
    table->SetTableValue( 48.0, 0, 0.462745, 0.054902 );
    table->SetTableValue( 49.0, 0, 0.462745, 0.054902 );
    table->SetTableValue( 50.0, 0.478431, 0.729412, 0.862745 );
    table->SetTableValue( 51.0, 0.92549, 0.0509804, 0.690196 );
    table->SetTableValue( 52.0, 0.0509804, 0.188235, 1 );
    table->SetTableValue( 53.0, 0.862745, 0.847059, 0.0784314 );
    table->SetTableValue( 54.0, 0.403922, 1, 1 );
    table->SetTableValue( 55.0, 0.313725, 0.768627, 0.384314 );
    table->SetTableValue( 56.0, 0.235294, 0.227451, 0.823529 );
    table->SetTableValue( 57.0, 1, 0.647059, 0 );
    table->SetTableValue( 58.0, 1, 0.647059, 0 );
    table->SetTableValue( 59.0, 0, 1, 0.498039 );
    table->SetTableValue( 60.0, 0.647059, 0.164706, 0.164706 );
    table->SetTableValue( 61.0, 0.529412, 0.807843, 0.921569 );
    table->SetTableValue( 62.0, 0.627451, 0.12549, 0.941176 );
    table->SetTableValue( 63.0, 0, 0.784314, 0.866667 );
    table->SetTableValue( 64.0, 0.392157, 0.196078, 0.392157 );
    table->SetTableValue( 65.0, 0.529412, 0.196078, 0.290196 );
    table->SetTableValue( 66.0, 0.478431, 0.529412, 0.196078 );
    table->SetTableValue( 67.0, 0.2, 0.196078, 0.529412 );
    table->SetTableValue( 68.0, 0.290196, 0.607843, 0.235294 );
    table->SetTableValue( 69.0, 0.470588, 0.243137, 0.168627 );
    table->SetTableValue( 70.0, 0.290196, 0.607843, 0.235294 );
    table->SetTableValue( 71.0, 0.478431, 0.529412, 0.196078 );
    table->SetTableValue( 72.0, 0.470588, 0.745098, 0.588235 );
    table->SetTableValue( 73.0, 0.478431, 0.529412, 0.196078 );
    table->SetTableValue( 74.0, 0.478431, 0.529412, 0.196078 );
    table->SetTableValue( 77.0, 0.784314, 0.27451, 1 );
    table->SetTableValue( 78.0, 1, 0.580392, 0.0392157 );
    table->SetTableValue( 79.0, 1, 0.580392, 0.0392157 );
    table->SetTableValue( 80.0, 0.643137, 0.423529, 0.886275 );
    table->SetTableValue( 81.0, 0.643137, 0.423529, 0.886275 );
    table->SetTableValue( 82.0, 0.643137, 0.423529, 0.886275 );
    table->SetTableValue( 83.0, 1, 0.854902, 0.72549 );
    table->SetTableValue( 84.0, 1, 0.854902, 0.72549 );
    table->SetTableValue( 85.0, 0.917647, 0.662745, 0.117647 );
    table->SetTableValue( 192.0, 0.980392, 1, 0.196078 );
    table->SetTableValue( 86.0, 0.784314, 0.470588, 1 );
    table->SetTableValue( 87.0, 0.784314, 0.47451, 1 );
    table->SetTableValue( 88.0, 0.784314, 0.478431, 1 );
    table->SetTableValue( 96.0, 0.803922, 0.0392157, 0.490196 );
    table->SetTableValue( 97.0, 0.803922, 0.0392157, 0.490196 );
    table->SetTableValue( 98.0, 0.627451, 0.12549, 0.941176 );
    table->SetTableValue( 100.0, 0.486275, 0.54902, 0.698039 );
    table->SetTableValue( 101.0, 0.490196, 0.54902, 0.698039 );
    table->SetTableValue( 102.0, 0.494118, 0.54902, 0.698039 );
    table->SetTableValue( 103.0, 0.498039, 0.54902, 0.698039 );
    table->SetTableValue( 104.0, 0.486275, 0.552941, 0.698039 );
    table->SetTableValue( 105.0, 0.486275, 0.556863, 0.698039 );
    table->SetTableValue( 106.0, 0.486275, 0.560784, 0.698039 );
    table->SetTableValue( 107.0, 0.486275, 0.564706, 0.698039 );
    table->SetTableValue( 108.0, 0.486275, 0.54902, 0.701961 );
    table->SetTableValue( 109.0, 0.486275, 0.54902, 0.698039 );
    table->SetTableValue( 110.0, 0.490196, 0.54902, 0.698039 );
    table->SetTableValue( 111.0, 0.494118, 0.54902, 0.698039 );
    table->SetTableValue( 112.0, 0.498039, 0.54902, 0.698039 );
    table->SetTableValue( 113.0, 0.486275, 0.552941, 0.698039 );
    table->SetTableValue( 114.0, 0.486275, 0.556863, 0.698039 );
    table->SetTableValue( 115.0, 0.486275, 0.560784, 0.698039 );
    table->SetTableValue( 116.0, 0.486275, 0.564706, 0.698039 );
    table->SetTableValue( 117.0, 0.486275, 0.54902, 0.701961 );
    table->SetTableValue( 118.0, 1, 0.0784314, 0.576471 );
    table->SetTableValue( 119.0, 0.803922, 0.701961, 0.545098 );
    table->SetTableValue( 120.0, 0.933333, 0.933333, 0.819608 );
    table->SetTableValue( 121.0, 0.784314, 0.784314, 0.784314 );
    table->SetTableValue( 122.0, 0.290196, 1, 0.290196 );
    table->SetTableValue( 123.0, 0.933333, 0, 0 );
    table->SetTableValue( 124.0, 0, 0, 0.545098 );
    table->SetTableValue( 125.0, 0.678431, 1, 0.184314 );
    table->SetTableValue( 126.0, 0.521569, 0.796078, 0.898039 );
    table->SetTableValue( 127.0, 0.101961, 0.929412, 0.223529 );
    table->SetTableValue( 128.0, 0.133333, 0.545098, 0.133333 );
    table->SetTableValue( 129.0, 0.117647, 0.564706, 1 );
    table->SetTableValue( 130.0, 0.576471, 0.0745098, 0.678431 );
    table->SetTableValue( 131.0, 0.933333, 0.231373, 0.231373 );
    table->SetTableValue( 132.0, 0.866667, 0.152941, 0.784314 );
    table->SetTableValue( 133.0, 0.933333, 0.682353, 0.933333 );
    table->SetTableValue( 134.0, 1, 0, 0 );
    table->SetTableValue( 135.0, 0.282353, 0.239216, 0.545098 );
    table->SetTableValue( 136.0, 0.0823529, 0.152941, 0.517647 );
    table->SetTableValue( 137.0, 0.0823529, 0.152941, 0.517647 );
    table->SetTableValue( 138.0, 0.254902, 0.529412, 0.0784314 );
    table->SetTableValue( 139.0, 0.254902, 0.529412, 0.0784314 );
    table->SetTableValue( 140.0, 0.52549, 0.0156863, 0.627451 );
    table->SetTableValue( 142.0, 0.866667, 0.886275, 0.266667 );
    table->SetTableValue( 143.0, 1, 1, 0.996078 );
    table->SetTableValue( 144.0, 0.203922, 0.819608, 0.886275 );
    table->SetTableValue( 145.0, 0.937255, 0.627451, 0.87451 );
    table->SetTableValue( 146.0, 0.27451, 0.509804, 0.705882 );
    table->SetTableValue( 147.0, 0.27451, 0.509804, 0.709804 );
    table->SetTableValue( 148.0, 0.545098, 0.47451, 0.368627 );
    table->SetTableValue( 149.0, 0.878431, 0.878431, 0.878431 );
    table->SetTableValue( 150.0, 1, 0, 0 );
    table->SetTableValue( 151.0, 0.803922, 0.803922, 0 );
    table->SetTableValue( 152.0, 0.933333, 0.933333, 0.819608 );
    table->SetTableValue( 153.0, 0.545098, 0.47451, 0.368627 );
    table->SetTableValue( 154.0, 0.933333, 0.231373, 0.231373 );
    table->SetTableValue( 155.0, 0.933333, 0.231373, 0.231373 );
    table->SetTableValue( 156.0, 0.933333, 0.231373, 0.231373 );
    table->SetTableValue( 157.0, 0.243137, 0.0392157, 0.803922 );
    table->SetTableValue( 158.0, 0.243137, 0.0392157, 0.803922 );
    table->SetTableValue( 159.0, 0, 0.462745, 0.054902 );
    table->SetTableValue( 160.0, 0, 0.462745, 0.054902 );
    table->SetTableValue( 161.0, 0.862745, 0.847059, 0.0823529 );
    table->SetTableValue( 162.0, 0.862745, 0.847059, 0.0823529 );
    table->SetTableValue( 163.0, 0.478431, 0.729412, 0.862745 );
    table->SetTableValue( 164.0, 0.478431, 0.729412, 0.862745 );
    table->SetTableValue( 165.0, 1, 0.647059, 0 );
    table->SetTableValue( 166.0, 0.054902, 0.188235, 1 );
    table->SetTableValue( 167.0, 0.65098, 0.164706, 0.164706 );
    table->SetTableValue( 168.0, 0.47451, 0.0705882, 0.52549 );
    table->SetTableValue( 169.0, 0.92549, 0.0509804, 0.498039 );
    table->SetTableValue( 176.0, 0.92549, 0.0509804, 0.494118 );
    table->SetTableValue( 170.0, 0.466667, 0.623529, 0.690196 );
    table->SetTableValue( 171.0, 0.466667, 0, 0.690196 );
    table->SetTableValue( 172.0, 0.466667, 0.392157, 0.690196 );
    table->SetTableValue( 173.0, 0.466667, 0.784314, 0.690196 );
    table->SetTableValue( 174.0, 0.466667, 0.623529, 0.392157 );
    table->SetTableValue( 175.0, 0.466667, 0.623529, 0.784314 );
    table->SetTableValue( 180.0, 0.286275, 0.239216, 0.545098 );
    table->SetTableValue( 181.0, 0.286275, 0.243137, 0.545098 );
    table->SetTableValue( 193.0, 0, 0.768627, 1 );
    table->SetTableValue( 194.0, 1, 0.643137, 0.643137 );
    table->SetTableValue( 195.0, 0.768627, 0.768627, 0 );
    table->SetTableValue( 196.0, 0, 0.392157, 1 );
    table->SetTableValue( 197.0, 0.501961, 0.768627, 0.643137 );
    table->SetTableValue( 198.0, 0, 0.494118, 0.294118 );
    table->SetTableValue( 199.0, 0.501961, 0.376471, 0.25098 );
    table->SetTableValue( 200.0, 0, 0.196078, 0.501961 );
    table->SetTableValue( 201.0, 1, 0.8, 0.6 );
    table->SetTableValue( 202.0, 1, 0.501961, 0.501961 );
    table->SetTableValue( 203.0, 1, 1, 0 );
    table->SetTableValue( 204.0, 0.25098, 0, 0.25098 );
    table->SetTableValue( 205.0, 0, 0, 1 );
    table->SetTableValue( 206.0, 1, 0, 0 );
    table->SetTableValue( 207.0, 0.501961, 0.501961, 1 );
    table->SetTableValue( 208.0, 0, 0.501961, 0 );
    table->SetTableValue( 209.0, 0.768627, 0.627451, 0.501961 );
    table->SetTableValue( 210.0, 0.12549, 0.784314, 1 );
    table->SetTableValue( 211.0, 0.501961, 1, 0.501961 );
    table->SetTableValue( 212.0, 0.8, 0.6, 0.8 );
    table->SetTableValue( 213.0, 0.47451, 0.0666667, 0.533333 );
    table->SetTableValue( 214.0, 0.501961, 0, 0 );
    table->SetTableValue( 215.0, 0.501961, 0.12549, 1 );
    table->SetTableValue( 216.0, 1, 0.8, 0.4 );
    table->SetTableValue( 217.0, 0.501961, 0.501961, 0.501961 );
    table->SetTableValue( 218.0, 0.407843, 1, 1 );
    table->SetTableValue( 219.0, 0, 0.886275, 0 );
    table->SetTableValue( 220.0, 0.803922, 0.247059, 0.305882 );
    table->SetTableValue( 221.0, 0.772549, 0.227451, 0.980392 );
    table->SetTableValue( 222.0, 0.129412, 0.588235, 0.980392 );
    table->SetTableValue( 223.0, 0.886275, 0, 0 );
    table->SetTableValue( 224.0, 0.392157, 0.392157, 0.392157 );
    table->SetTableValue( 225.0, 0.772549, 0.588235, 0.980392 );
    table->SetTableValue( 226.0, 0.666667, 0.666667, 1 );
    table->SetTableValue( 250.0, 1, 0, 0 );
    table->SetTableValue( 251.0, 0, 0, 0.25098 );
    table->SetTableValue( 252.0, 0, 0, 0.439216 );
    table->SetTableValue( 253.0, 0, 0, 0.627451 );
    table->SetTableValue( 254.0, 0, 0, 0.815686 );
    table->SetTableValue( 255.0, 0, 0, 1 );
    table->SetTableValue( 256.0, 0, 0, 0, 0);
    table->SetTableValue( 331.0, 1, 0, 0 );
    table->SetTableValue( 332.0, 1, 0.313725, 0 );
    table->SetTableValue( 333.0, 1, 0.627451, 0 );
    table->SetTableValue( 334.0, 1, 1, 0 );
    table->SetTableValue( 335.0, 0, 1, 0 );
    table->SetTableValue( 336.0, 1, 0, 0.627451 );
    table->SetTableValue( 337.0, 1, 0, 1 );
    table->SetTableValue( 338.0, 1, 0.196078, 0.313725 );
    table->SetTableValue( 339.0, 0.313725, 1, 0.196078 );
    table->SetTableValue( 340.0, 0.627451, 1, 0.196078 );
    table->SetTableValue( 341.0, 0.627451, 0.784314, 1 );
    table->SetTableValue( 342.0, 0, 1, 0.627451 );
    table->SetTableValue( 343.0, 0, 0, 1 );
    table->SetTableValue( 344.0, 0.313725, 0.196078, 1 );
    table->SetTableValue( 345.0, 0.627451, 0, 1 );
    table->SetTableValue( 346.0, 1, 0.823529, 0 );
    table->SetTableValue( 347.0, 0, 0.627451, 1 );
    table->SetTableValue( 348.0, 1, 0.784314, 0.313725 );
    table->SetTableValue( 349.0, 1, 0.784314, 0.627451 );
    table->SetTableValue( 350.0, 1, 0.313725, 0.784314 );
    table->SetTableValue( 351.0, 1, 0.627451, 0.784314 );
    table->SetTableValue( 352.0, 0.117647, 1, 0.313725 );
    table->SetTableValue( 353.0, 0.313725, 0.784314, 1 );
    table->SetTableValue( 354.0, 0.313725, 1, 0.784314 );
    table->SetTableValue( 355.0, 0.764706, 1, 0.784314 );
    table->SetTableValue( 356.0, 0.470588, 0.784314, 0.0784314 );
    table->SetTableValue( 357.0, 0.666667, 0.0392157, 0.784314 );
    table->SetTableValue( 358.0, 0.0784314, 0.509804, 0.705882 );
    table->SetTableValue( 359.0, 0.0784314, 0.705882, 0.509804 );
    table->SetTableValue( 400.0, 0.807843, 0.243137, 0.305882 );
    table->SetTableValue( 401.0, 0.47451, 0.0705882, 0.52549 );
    table->SetTableValue( 402.0, 0.780392, 0.227451, 0.980392 );
    table->SetTableValue( 403.0, 0.00392157, 0.580392, 0 );
    table->SetTableValue( 404.0, 0.866667, 0.972549, 0.643137 );
    table->SetTableValue( 405.0, 0.905882, 0.580392, 0.133333 );
    table->SetTableValue( 406.0, 0.00392157, 0.462745, 0.054902 );
    table->SetTableValue( 407.0, 0.470588, 0.462745, 0.054902 );
    table->SetTableValue( 408.0, 0.482353, 0.729412, 0.866667 );
    table->SetTableValue( 409.0, 0.933333, 0.0509804, 0.694118 );
    table->SetTableValue( 410.0, 0.482353, 0.729412, 0.862745 );
    table->SetTableValue( 411.0, 0.541176, 0.0509804, 0.807843 );
    table->SetTableValue( 412.0, 0.933333, 0.509804, 0.690196 );
    table->SetTableValue( 413.0, 0.854902, 0.901961, 0.298039 );
    table->SetTableValue( 414.0, 0.14902, 0.835294, 0.690196 );
    table->SetTableValue( 415.0, 0.00392157, 0.882353, 0.690196 );
    table->SetTableValue( 416.0, 0.00392157, 0.882353, 0.690196 );
    table->SetTableValue( 417.0, 0.784314, 0.00784314, 0.392157 );
    table->SetTableValue( 418.0, 0.784314, 0.00784314, 0.392157 );
    table->SetTableValue( 419.0, 0.0196078, 0.784314, 0.352941 );
    table->SetTableValue( 420.0, 0.0196078, 0.784314, 0.352941 );
    table->SetTableValue( 421.0, 0.392157, 0.0196078, 0.784314 );
    table->SetTableValue( 422.0, 0.0980392, 1, 0.392157 );
    table->SetTableValue( 423.0, 0.0980392, 1, 0.392157 );
    table->SetTableValue( 424.0, 0.901961, 0.027451, 0.392157 );
    table->SetTableValue( 425.0, 0.901961, 0.027451, 0.392157 );
    table->SetTableValue( 426.0, 0.392157, 0.0196078, 0.784314 );
    table->SetTableValue( 427.0, 0.588235, 0.0392157, 0.784314 );
    table->SetTableValue( 428.0, 0.588235, 0.0392157, 0.784314 );
    table->SetTableValue( 429.0, 0.686275, 0.0392157, 0.690196 );
    table->SetTableValue( 430.0, 0.686275, 0.0392157, 0.690196 );
    table->SetTableValue( 431.0, 0.0392157, 0.392157, 1 );
    table->SetTableValue( 432.0, 0.0392157, 0.392157, 1 );
    table->SetTableValue( 433.0, 0.588235, 0.176471, 0.27451 );
    table->SetTableValue( 434.0, 0.588235, 0.176471, 0.27451 );
    table->SetTableValue( 435.0, 0.176471, 0.784314, 0.0588235 );
    table->SetTableValue( 436.0, 0.176471, 0.784314, 0.0588235 );
    table->SetTableValue( 437.0, 0.890196, 0.176471, 0.392157 );
    table->SetTableValue( 438.0, 0.890196, 0.176471, 0.392157 );
    table->SetTableValue( 439.0, 0.890196, 0.176471, 0.392157 );
    table->SetTableValue( 498.0, 0.560784, 0.737255, 0.560784 );
    table->SetTableValue( 499.0, 1, 0.972549, 0.862745 );
    table->SetTableValue( 500.0, 0.0666667, 0.333333, 0.533333 );
    table->SetTableValue( 501.0, 0.466667, 0.733333, 0.4 );
    table->SetTableValue( 502.0, 0.8, 0.266667, 0.133333 );
    table->SetTableValue( 503.0, 0.8, 0, 1 );
    table->SetTableValue( 504.0, 0.866667, 0.733333, 0.0666667 );
    table->SetTableValue( 505.0, 0.6, 0.866667, 0.933333 );
    table->SetTableValue( 506.0, 0.2, 0.0666667, 0.0666667 );
    table->SetTableValue( 507.0, 0, 0.466667, 0.333333 );
    table->SetTableValue( 508.0, 0.0784314, 0.392157, 0.784314 );
    table->SetTableValue( 550.0, 0.0666667, 0.333333, 0.537255 );
    table->SetTableValue( 551.0, 0.466667, 0.733333, 0.403922 );
    table->SetTableValue( 552.0, 0.8, 0.266667, 0.137255 );
    table->SetTableValue( 553.0, 0.8, 0, 0.996078 );
    table->SetTableValue( 554.0, 0.866667, 0.733333, 0.0627451 );
    table->SetTableValue( 555.0, 0.6, 0.866667, 0.937255 );
    table->SetTableValue( 556.0, 0.2, 0.0666667, 0.0705882 );
    table->SetTableValue( 557.0, 0, 0.466667, 0.337255 );
    table->SetTableValue( 558.0, 0.0784314, 0.392157, 0.788235 );
    table->SetTableValue( 600.0, 0.996078, 0.996078, 0.996078 );
    table->SetTableValue( 601.0, 0.27451, 0.509804, 0.705882 );
    table->SetTableValue( 602.0, 0.960784, 0.960784, 0.960784 );
    table->SetTableValue( 603.0, 0.803922, 0.243137, 0.305882 );
    table->SetTableValue( 604.0, 0.470588, 0.0705882, 0.52549 );
    table->SetTableValue( 605.0, 0.768627, 0.227451, 0.980392 );
    table->SetTableValue( 606.0, 0, 0.580392, 0 );
    table->SetTableValue( 607.0, 0.862745, 0.972549, 0.643137 );
    table->SetTableValue( 608.0, 0.901961, 0.580392, 0.133333 );
    table->SetTableValue( 609.0, 0, 0.462745, 0.054902 );
    table->SetTableValue( 610.0, 0, 0.462745, 0.054902 );
    table->SetTableValue( 611.0, 0.478431, 0.729412, 0.862745 );
    table->SetTableValue( 612.0, 0.92549, 0.0509804, 0.690196 );
    table->SetTableValue( 613.0, 0.0470588, 0.188235, 1 );
    table->SetTableValue( 614.0, 0.8, 0.713725, 0.556863 );
    table->SetTableValue( 615.0, 0.164706, 0.8, 0.643137 );
    table->SetTableValue( 616.0, 0.466667, 0.623529, 0.690196 );
    table->SetTableValue( 617.0, 0.862745, 0.847059, 0.0784314 );
    table->SetTableValue( 618.0, 0.403922, 1, 1 );
    table->SetTableValue( 619.0, 0.313725, 0.768627, 0.384314 );
    table->SetTableValue( 620.0, 0.235294, 0.227451, 0.823529 );
    table->SetTableValue( 621.0, 0.235294, 0.227451, 0.823529 );
    table->SetTableValue( 622.0, 0.235294, 0.227451, 0.823529 );
    table->SetTableValue( 623.0, 0.235294, 0.227451, 0.823529 );
    table->SetTableValue( 624.0, 0.235294, 0.235294, 0.235294 );
    table->SetTableValue( 625.0, 1, 0.647059, 0 );
    table->SetTableValue( 626.0, 1, 0.647059, 0 );
    table->SetTableValue( 627.0, 0, 1, 0.498039 );
    table->SetTableValue( 628.0, 0.647059, 0.164706, 0.164706 );
    table->SetTableValue( 640.0, 0.8, 0, 0 );
    table->SetTableValue( 641.0, 1, 0, 0 );
    table->SetTableValue( 642.0, 0, 0, 1 );
    table->SetTableValue( 643.0, 0.117647, 0.564706, 1 );
    table->SetTableValue( 644.0, 0.392157, 0.831373, 0.929412 );
    table->SetTableValue( 645.0, 0.854902, 0.647059, 0.12549 );
    table->SetTableValue( 646.0, 1, 0.843137, 0 );
    table->SetTableValue( 647.0, 1, 1, 0.65098 );
    table->SetTableValue( 648.0, 0.6, 0, 0.8 );
    table->SetTableValue( 649.0, 0.6, 0.552941, 0.819608 );
    table->SetTableValue( 650.0, 0.8, 0.8, 1 );
    table->SetTableValue( 651.0, 0.121569, 0.831373, 0.760784 );
    table->SetTableValue( 652.0, 0.0117647, 1, 0.929412 );
    table->SetTableValue( 653.0, 0.8, 1, 1 );
    table->SetTableValue( 654.0, 0.337255, 0.290196, 0.576471 );
    table->SetTableValue( 655.0, 0.447059, 0.447059, 0.745098 );
    table->SetTableValue( 656.0, 0.721569, 0.698039, 1 );
    table->SetTableValue( 657.0, 0.494118, 0.541176, 0.145098 );
    table->SetTableValue( 658.0, 0.741176, 0.772549, 0.458824 );
    table->SetTableValue( 659.0, 0.941176, 0.901961, 0.54902 );
    table->SetTableValue( 660.0, 0.8, 0, 0 );
    table->SetTableValue( 661.0, 1, 0, 0 );
    table->SetTableValue( 662.0, 0, 0, 1 );
    table->SetTableValue( 663.0, 0.117647, 0.564706, 1 );
    table->SetTableValue( 664.0, 0.392157, 0.831373, 0.929412 );
    table->SetTableValue( 665.0, 0.854902, 0.647059, 0.12549 );
    table->SetTableValue( 666.0, 1, 0.843137, 0 );
    table->SetTableValue( 667.0, 1, 1, 0.65098 );
    table->SetTableValue( 668.0, 0.6, 0, 0.8 );
    table->SetTableValue( 669.0, 0.6, 0.552941, 0.819608 );
    table->SetTableValue( 670.0, 0.8, 0.8, 1 );
    table->SetTableValue( 671.0, 0.121569, 0.831373, 0.760784 );
    table->SetTableValue( 672.0, 0.0117647, 1, 0.929412 );
    table->SetTableValue( 673.0, 0.8, 1, 1 );
    table->SetTableValue( 674.0, 0.337255, 0.290196, 0.576471 );
    table->SetTableValue( 675.0, 0.447059, 0.447059, 0.745098 );
    table->SetTableValue( 676.0, 0.721569, 0.698039, 1 );
    table->SetTableValue( 677.0, 0.494118, 0.541176, 0.145098 );
    table->SetTableValue( 678.0, 0.741176, 0.772549, 0.458824 );
    table->SetTableValue( 679.0, 0.941176, 0.901961, 0.54902 );
    table->SetTableValue( 701.0, 0.470588, 0.0705882, 0.52549 );
    table->SetTableValue( 702.0, 0.803922, 0.243137, 0.305882 );
    table->SetTableValue( 703.0, 0, 0.882353, 0 );
    table->SetTableValue( 999.0, 1, 0.392157, 0.392157 );
    table->SetTableValue( 1000.0, 0.0980392, 0.0196078, 0.0980392 );
    table->SetTableValue( 1001.0, 0.0980392, 0.392157, 0.156863 );
    table->SetTableValue( 1002.0, 0.490196, 0.392157, 0.627451 );
    table->SetTableValue( 1003.0, 0.392157, 0.0980392, 0 );
    table->SetTableValue( 1004.0, 0.470588, 0.27451, 0.196078 );
    table->SetTableValue( 1005.0, 0.862745, 0.0784314, 0.392157 );
    table->SetTableValue( 1006.0, 0.862745, 0.0784314, 0.0392157 );
    table->SetTableValue( 1007.0, 0.705882, 0.862745, 0.54902 );
    table->SetTableValue( 1008.0, 0.862745, 0.235294, 0.862745 );
    table->SetTableValue( 1009.0, 0.705882, 0.156863, 0.470588 );
    table->SetTableValue( 1010.0, 0.54902, 0.0784314, 0.54902 );
    table->SetTableValue( 1011.0, 0.0784314, 0.117647, 0.54902 );
    table->SetTableValue( 1012.0, 0.137255, 0.294118, 0.196078 );
    table->SetTableValue( 1013.0, 0.882353, 0.54902, 0.54902 );
    table->SetTableValue( 1014.0, 0.784314, 0.137255, 0.294118 );
    table->SetTableValue( 1015.0, 0.627451, 0.392157, 0.196078 );
    table->SetTableValue( 1016.0, 0.0784314, 0.862745, 0.235294 );
    table->SetTableValue( 1017.0, 0.235294, 0.862745, 0.235294 );
    table->SetTableValue( 1018.0, 0.862745, 0.705882, 0.54902 );
    table->SetTableValue( 1019.0, 0.0784314, 0.392157, 0.196078 );
    table->SetTableValue( 1020.0, 0.862745, 0.235294, 0.0784314 );
    table->SetTableValue( 1021.0, 0.470588, 0.392157, 0.235294 );
    table->SetTableValue( 1022.0, 0.862745, 0.0784314, 0.0784314 );
    table->SetTableValue( 1023.0, 0.862745, 0.705882, 0.862745 );
    table->SetTableValue( 1024.0, 0.235294, 0.0784314, 0.862745 );
    table->SetTableValue( 1025.0, 0.627451, 0.54902, 0.705882 );
    table->SetTableValue( 1026.0, 0.313725, 0.0784314, 0.54902 );
    table->SetTableValue( 1027.0, 0.294118, 0.196078, 0.490196 );
    table->SetTableValue( 1028.0, 0.0784314, 0.862745, 0.627451 );
    table->SetTableValue( 1029.0, 0.0784314, 0.705882, 0.54902 );
    table->SetTableValue( 1030.0, 0.54902, 0.862745, 0.862745 );
    table->SetTableValue( 1031.0, 0.313725, 0.627451, 0.0784314 );
    table->SetTableValue( 1032.0, 0.392157, 0, 0.392157 );
    table->SetTableValue( 1033.0, 0.27451, 0.27451, 0.27451 );
    table->SetTableValue( 1034.0, 0.588235, 0.588235, 0.784314 );
    table->SetTableValue( 1035.0, 1, 0.752941, 0.12549 );
    table->SetTableValue( 2000.0, 0.0980392, 0.0196078, 0.0980392 );
    table->SetTableValue( 2001.0, 0.0980392, 0.392157, 0.156863 );
    table->SetTableValue( 2002.0, 0.490196, 0.392157, 0.627451 );
    table->SetTableValue( 2003.0, 0.392157, 0.0980392, 0 );
    table->SetTableValue( 2004.0, 0.470588, 0.27451, 0.196078 );
    table->SetTableValue( 2005.0, 0.862745, 0.0784314, 0.392157 );
    table->SetTableValue( 2006.0, 0.862745, 0.0784314, 0.0392157 );
    table->SetTableValue( 2007.0, 0.705882, 0.862745, 0.54902 );
    table->SetTableValue( 2008.0, 0.862745, 0.235294, 0.862745 );
    table->SetTableValue( 2009.0, 0.705882, 0.156863, 0.470588 );
    table->SetTableValue( 2010.0, 0.54902, 0.0784314, 0.54902 );
    table->SetTableValue( 2011.0, 0.0784314, 0.117647, 0.54902 );
    table->SetTableValue( 2012.0, 0.137255, 0.294118, 0.196078 );
    table->SetTableValue( 2013.0, 0.882353, 0.54902, 0.54902 );
    table->SetTableValue( 2014.0, 0.784314, 0.137255, 0.294118 );
    table->SetTableValue( 2015.0, 0.627451, 0.392157, 0.196078 );
    table->SetTableValue( 2016.0, 0.0784314, 0.862745, 0.235294 );
    table->SetTableValue( 2017.0, 0.235294, 0.862745, 0.235294 );
    table->SetTableValue( 2018.0, 0.862745, 0.705882, 0.54902 );
    table->SetTableValue( 2019.0, 0.0784314, 0.392157, 0.196078 );
    table->SetTableValue( 2020.0, 0.862745, 0.235294, 0.0784314 );
    table->SetTableValue( 2021.0, 0.470588, 0.392157, 0.235294 );
    table->SetTableValue( 2022.0, 0.862745, 0.0784314, 0.0784314 );
    table->SetTableValue( 2023.0, 0.862745, 0.705882, 0.862745 );
    table->SetTableValue( 2024.0, 0.235294, 0.0784314, 0.862745 );
    table->SetTableValue( 2025.0, 0.627451, 0.54902, 0.705882 );
    table->SetTableValue( 2026.0, 0.313725, 0.0784314, 0.54902 );
    table->SetTableValue( 2027.0, 0.294118, 0.196078, 0.490196 );
    table->SetTableValue( 2028.0, 0.0784314, 0.862745, 0.627451 );
    table->SetTableValue( 2029.0, 0.0784314, 0.705882, 0.54902 );
    table->SetTableValue( 2030.0, 0.54902, 0.862745, 0.862745 );
    table->SetTableValue( 2031.0, 0.313725, 0.627451, 0.0784314 );
    table->SetTableValue( 2032.0, 0.392157, 0, 0.392157 );
    table->SetTableValue( 2033.0, 0.27451, 0.27451, 0.27451 );
    table->SetTableValue( 2034.0, 0.588235, 0.588235, 0.784314 );
    table->SetTableValue( 2035.0, 1, 0.752941, 0.12549 );
    table->SetTableValue( 3000.0, 0.901961, 0.980392, 0.901961 );
    table->SetTableValue( 3001.0, 0.901961, 0.607843, 0.843137 );
    table->SetTableValue( 3002.0, 0.509804, 0.607843, 0.372549 );
    table->SetTableValue( 3003.0, 0.607843, 0.901961, 1 );
    table->SetTableValue( 3004.0, 0.529412, 0.72549, 0.803922 );
    table->SetTableValue( 3005.0, 0.137255, 0.921569, 0.607843 );
    table->SetTableValue( 3006.0, 0.137255, 0.921569, 0.960784 );
    table->SetTableValue( 3007.0, 0.294118, 0.137255, 0.45098 );
    table->SetTableValue( 3008.0, 0.137255, 0.764706, 0.137255 );
    table->SetTableValue( 3009.0, 0.294118, 0.843137, 0.529412 );
    table->SetTableValue( 3010.0, 0.45098, 0.921569, 0.45098 );
    table->SetTableValue( 3011.0, 0.921569, 0.882353, 0.45098 );
    table->SetTableValue( 3012.0, 0.862745, 0.705882, 0.803922 );
    table->SetTableValue( 3013.0, 0.117647, 0.45098, 0.45098 );
    table->SetTableValue( 3014.0, 0.215686, 0.862745, 0.705882 );
    table->SetTableValue( 3015.0, 0.372549, 0.607843, 0.803922 );
    table->SetTableValue( 3016.0, 0.921569, 0.137255, 0.764706 );
    table->SetTableValue( 3017.0, 0.764706, 0.137255, 0.764706 );
    table->SetTableValue( 3018.0, 0.137255, 0.294118, 0.45098 );
    table->SetTableValue( 3019.0, 0.921569, 0.607843, 0.803922 );
    table->SetTableValue( 3020.0, 0.137255, 0.764706, 0.921569 );
    table->SetTableValue( 3021.0, 0.529412, 0.607843, 0.764706 );
    table->SetTableValue( 3022.0, 0.137255, 0.921569, 0.921569 );
    table->SetTableValue( 3023.0, 0.137255, 0.294118, 0.137255 );
    table->SetTableValue( 3024.0, 0.764706, 0.921569, 0.137255 );
    table->SetTableValue( 3025.0, 0.372549, 0.45098, 0.294118 );
    table->SetTableValue( 3026.0, 0.686275, 0.921569, 0.45098 );
    table->SetTableValue( 3027.0, 0.705882, 0.803922, 0.509804 );
    table->SetTableValue( 3028.0, 0.921569, 0.137255, 0.372549 );
    table->SetTableValue( 3029.0, 0.921569, 0.294118, 0.45098 );
    table->SetTableValue( 3030.0, 0.45098, 0.137255, 0.137255 );
    table->SetTableValue( 3031.0, 0.686275, 0.372549, 0.921569 );
    table->SetTableValue( 3032.0, 0.607843, 1, 0.607843 );
    table->SetTableValue( 3033.0, 0.72549, 0.72549, 0.72549 );
    table->SetTableValue( 3034.0, 0.411765, 0.411765, 0.215686 );
    table->SetTableValue( 3035.0, 0.996078, 0.74902, 0.121569 );
    table->SetTableValue( 4000.0, 0.901961, 0.980392, 0.901961 );
    table->SetTableValue( 4001.0, 0.901961, 0.607843, 0.843137 );
    table->SetTableValue( 4002.0, 0.509804, 0.607843, 0.372549 );
    table->SetTableValue( 4003.0, 0.607843, 0.901961, 1 );
    table->SetTableValue( 4004.0, 0.529412, 0.72549, 0.803922 );
    table->SetTableValue( 4005.0, 0.137255, 0.921569, 0.607843 );
    table->SetTableValue( 4006.0, 0.137255, 0.921569, 0.960784 );
    table->SetTableValue( 4007.0, 0.294118, 0.137255, 0.45098 );
    table->SetTableValue( 4008.0, 0.137255, 0.764706, 0.137255 );
    table->SetTableValue( 4009.0, 0.294118, 0.843137, 0.529412 );
    table->SetTableValue( 4010.0, 0.45098, 0.921569, 0.45098 );
    table->SetTableValue( 4011.0, 0.921569, 0.882353, 0.45098 );
    table->SetTableValue( 4012.0, 0.862745, 0.705882, 0.803922 );
    table->SetTableValue( 4013.0, 0.117647, 0.45098, 0.45098 );
    table->SetTableValue( 4014.0, 0.215686, 0.862745, 0.705882 );
    table->SetTableValue( 4015.0, 0.372549, 0.607843, 0.803922 );
    table->SetTableValue( 4016.0, 0.921569, 0.137255, 0.764706 );
    table->SetTableValue( 4017.0, 0.764706, 0.137255, 0.764706 );
    table->SetTableValue( 4018.0, 0.137255, 0.294118, 0.45098 );
    table->SetTableValue( 4019.0, 0.921569, 0.607843, 0.803922 );
    table->SetTableValue( 4020.0, 0.137255, 0.764706, 0.921569 );
    table->SetTableValue( 4021.0, 0.529412, 0.607843, 0.764706 );
    table->SetTableValue( 4022.0, 0.137255, 0.921569, 0.921569 );
    table->SetTableValue( 4023.0, 0.137255, 0.294118, 0.137255 );
    table->SetTableValue( 4024.0, 0.764706, 0.921569, 0.137255 );
    table->SetTableValue( 4025.0, 0.372549, 0.45098, 0.294118 );
    table->SetTableValue( 4026.0, 0.686275, 0.921569, 0.45098 );
    table->SetTableValue( 4027.0, 0.705882, 0.803922, 0.509804 );
    table->SetTableValue( 4028.0, 0.921569, 0.137255, 0.372549 );
    table->SetTableValue( 4029.0, 0.921569, 0.294118, 0.45098 );
    table->SetTableValue( 4030.0, 0.45098, 0.137255, 0.137255 );
    table->SetTableValue( 4031.0, 0.686275, 0.372549, 0.921569 );
    table->SetTableValue( 4032.0, 0.607843, 1, 0.607843 );
    table->SetTableValue( 4033.0, 0.72549, 0.72549, 0.72549 );
    table->SetTableValue( 4034.0, 0.411765, 0.411765, 0.215686 );
    table->SetTableValue( 4035.0, 0.996078, 0.74902, 0.121569 );
    table->SetTableValue( 1100.0, 0, 0, 0, 0);
    table->SetTableValue( 1101.0, 0.196078, 0.196078, 0.196078 );
    table->SetTableValue( 1102.0, 0.705882, 0.0784314, 0.117647 );
    table->SetTableValue( 1103.0, 0.235294, 0.0980392, 0.0980392 );
    table->SetTableValue( 1104.0, 0.0980392, 0.235294, 0.235294 );
    table->SetTableValue( 1200.0, 0.0980392, 0.235294, 0.239216 );
    table->SetTableValue( 1201.0, 0.0980392, 0.352941, 0.235294 );
    table->SetTableValue( 1202.0, 0.0980392, 0.470588, 0.235294 );
    table->SetTableValue( 1205.0, 0.0980392, 0.588235, 0.235294 );
    table->SetTableValue( 1206.0, 0.0980392, 0.705882, 0.235294 );
    table->SetTableValue( 1207.0, 0.0980392, 0.823529, 0.235294 );
    table->SetTableValue( 1210.0, 0.0980392, 0.588235, 0.352941 );
    table->SetTableValue( 1211.0, 0.0980392, 0.705882, 0.352941 );
    table->SetTableValue( 1212.0, 0.0980392, 0.823529, 0.352941 );
    table->SetTableValue( 1105.0, 0.705882, 0.0784314, 0.0784314 );
    table->SetTableValue( 1106.0, 0.862745, 0.0784314, 0.392157 );
    table->SetTableValue( 1107.0, 0.54902, 0.235294, 0.235294 );
    table->SetTableValue( 1108.0, 0.705882, 0.862745, 0.54902 );
    table->SetTableValue( 1109.0, 0.54902, 0.392157, 0.705882 );
    table->SetTableValue( 1110.0, 0.705882, 0.0784314, 0.54902 );
    table->SetTableValue( 1111.0, 0.54902, 0.0784314, 0.54902 );
    table->SetTableValue( 1112.0, 0.0823529, 0.0392157, 0.0392157 );
    table->SetTableValue( 1113.0, 0.882353, 0.54902, 0.54902 );
    table->SetTableValue( 1114.0, 0.0901961, 0.235294, 0.705882 );
    table->SetTableValue( 1115.0, 0.705882, 0.235294, 0.705882 );
    table->SetTableValue( 1116.0, 0.0784314, 0.862745, 0.235294 );
    table->SetTableValue( 1117.0, 0.235294, 0.0784314, 0.54902 );
    table->SetTableValue( 1118.0, 0.862745, 0.705882, 0.54902 );
    table->SetTableValue( 1119.0, 0.254902, 0.392157, 0.0784314 );
    table->SetTableValue( 1120.0, 0.862745, 0.235294, 0.0784314 );
    table->SetTableValue( 1121.0, 0.235294, 0.392157, 0.235294 );
    table->SetTableValue( 1122.0, 0.0784314, 0.235294, 0.862745 );
    table->SetTableValue( 1123.0, 0.392157, 0.392157, 0.235294 );
    table->SetTableValue( 1124.0, 0.862745, 0.705882, 0.862745 );
    table->SetTableValue( 1125.0, 0.0784314, 0.705882, 0.54902 );
    table->SetTableValue( 1126.0, 0.235294, 0.54902, 0.705882 );
    table->SetTableValue( 1127.0, 0.0980392, 0.0784314, 0.54902 );
    table->SetTableValue( 1128.0, 0.0784314, 0.235294, 0.392157 );
    table->SetTableValue( 1129.0, 0.235294, 0.862745, 0.0784314 );
    table->SetTableValue( 1130.0, 0.235294, 0.0784314, 0.862745 );
    table->SetTableValue( 1131.0, 0.862745, 0.862745, 0.392157 );
    table->SetTableValue( 1132.0, 0.705882, 0.235294, 0.235294 );
    table->SetTableValue( 1133.0, 0.235294, 0.235294, 0.862745 );
    table->SetTableValue( 1134.0, 0.862745, 0.235294, 0.862745 );
    table->SetTableValue( 1135.0, 0.254902, 0.862745, 0.235294 );
    table->SetTableValue( 1136.0, 0.0980392, 0.54902, 0.0784314 );
    table->SetTableValue( 1137.0, 0.0509804, 0, 0.980392 );
    table->SetTableValue( 1138.0, 0.239216, 0.0784314, 0.862745 );
    table->SetTableValue( 1139.0, 0.239216, 0.0784314, 0.235294 );
    table->SetTableValue( 1140.0, 0.239216, 0.235294, 0.392157 );
    table->SetTableValue( 1141.0, 0.0980392, 0.0980392, 0.0980392 );
    table->SetTableValue( 1142.0, 0.54902, 0.0784314, 0.235294 );
    table->SetTableValue( 1143.0, 0.862745, 0.705882, 0.0784314 );
    table->SetTableValue( 1144.0, 0.247059, 0.705882, 0.705882 );
    table->SetTableValue( 1145.0, 0.866667, 0.0784314, 0.0392157 );
    table->SetTableValue( 1146.0, 0.0823529, 0.862745, 0.0784314 );
    table->SetTableValue( 1147.0, 0.717647, 0.392157, 0.0784314 );
    table->SetTableValue( 1148.0, 0.866667, 0.0784314, 0.392157 );
    table->SetTableValue( 1149.0, 0.866667, 0.235294, 0.54902 );
    table->SetTableValue( 1150.0, 0.866667, 0.0784314, 0.862745 );
    table->SetTableValue( 1151.0, 0.239216, 0.862745, 0.862745 );
    table->SetTableValue( 1152.0, 0.392157, 0.784314, 0.784314 );
    table->SetTableValue( 1153.0, 0.0392157, 0.784314, 0.784314 );
    table->SetTableValue( 1154.0, 0.866667, 0.862745, 0.0784314 );
    table->SetTableValue( 1155.0, 0.552941, 0.0784314, 0.392157 );
    table->SetTableValue( 1156.0, 0.239216, 0.862745, 0.392157 );
    table->SetTableValue( 1157.0, 0.0823529, 0.862745, 0.235294 );
    table->SetTableValue( 1158.0, 0.552941, 0.235294, 0.0784314 );
    table->SetTableValue( 1159.0, 0.560784, 0.0784314, 0.862745 );
    table->SetTableValue( 1160.0, 0.239216, 0.0784314, 0.705882 );
    table->SetTableValue( 1161.0, 0.396078, 0.235294, 0.862745 );
    table->SetTableValue( 1162.0, 0.0823529, 0.0784314, 0.54902 );
    table->SetTableValue( 1163.0, 0.866667, 0.54902, 0.0784314 );
    table->SetTableValue( 1164.0, 0.552941, 0.392157, 0.862745 );
    table->SetTableValue( 1165.0, 0.396078, 0.0784314, 0.0784314 );
    table->SetTableValue( 1166.0, 0.866667, 0.392157, 0.0784314 );
    table->SetTableValue( 1167.0, 0.709804, 0.784314, 0.0784314 );
    table->SetTableValue( 1168.0, 0.0823529, 0.705882, 0.54902 );
    table->SetTableValue( 1169.0, 0.396078, 0.392157, 0.705882 );
    table->SetTableValue( 1170.0, 0.709804, 0.862745, 0.0784314 );
    table->SetTableValue( 1171.0, 0.0823529, 0.54902, 0.784314 );
    table->SetTableValue( 1172.0, 0.0823529, 0.0784314, 0.941176 );
    table->SetTableValue( 1173.0, 0.0823529, 0.0784314, 0.784314 );
    table->SetTableValue( 1174.0, 0.239216, 0.705882, 0.235294 );
    table->SetTableValue( 1175.0, 0.239216, 0.705882, 0.980392 );
    table->SetTableValue( 1176.0, 0.0823529, 0.0784314, 0.235294 );
    table->SetTableValue( 1177.0, 0.396078, 0.235294, 0.235294 );
    table->SetTableValue( 1178.0, 0.0823529, 0.862745, 0.862745 );
    table->SetTableValue( 1179.0, 0.0823529, 0.705882, 0.705882 );
    table->SetTableValue( 1180.0, 0.87451, 0.862745, 0.235294 );
    table->SetTableValue( 1181.0, 0.866667, 0.235294, 0.235294 );
    table->SetTableValue( 2100.0, 0, 0, 0, 0);
    table->SetTableValue( 2101.0, 0.196078, 0.196078, 0.196078 );
    table->SetTableValue( 2102.0, 0.705882, 0.0784314, 0.117647 );
    table->SetTableValue( 2103.0, 0.235294, 0.0980392, 0.0980392 );
    table->SetTableValue( 2104.0, 0.0980392, 0.235294, 0.235294 );
    table->SetTableValue( 2105.0, 0.705882, 0.0784314, 0.0784314 );
    table->SetTableValue( 2106.0, 0.862745, 0.0784314, 0.392157 );
    table->SetTableValue( 2107.0, 0.54902, 0.235294, 0.235294 );
    table->SetTableValue( 2108.0, 0.705882, 0.862745, 0.54902 );
    table->SetTableValue( 2109.0, 0.54902, 0.392157, 0.705882 );
    table->SetTableValue( 2110.0, 0.705882, 0.0784314, 0.54902 );
    table->SetTableValue( 2111.0, 0.54902, 0.0784314, 0.54902 );
    table->SetTableValue( 2112.0, 0.0823529, 0.0392157, 0.0392157 );
    table->SetTableValue( 2113.0, 0.882353, 0.54902, 0.54902 );
    table->SetTableValue( 2114.0, 0.0901961, 0.235294, 0.705882 );
    table->SetTableValue( 2115.0, 0.705882, 0.235294, 0.705882 );
    table->SetTableValue( 2116.0, 0.0784314, 0.862745, 0.235294 );
    table->SetTableValue( 2117.0, 0.235294, 0.0784314, 0.54902 );
    table->SetTableValue( 2118.0, 0.862745, 0.705882, 0.54902 );
    table->SetTableValue( 2119.0, 0.254902, 0.392157, 0.0784314 );
    table->SetTableValue( 2120.0, 0.862745, 0.235294, 0.0784314 );
    table->SetTableValue( 2121.0, 0.235294, 0.392157, 0.235294 );
    table->SetTableValue( 2122.0, 0.0784314, 0.235294, 0.862745 );
    table->SetTableValue( 2123.0, 0.392157, 0.392157, 0.235294 );
    table->SetTableValue( 2124.0, 0.862745, 0.705882, 0.862745 );
    table->SetTableValue( 2125.0, 0.0784314, 0.705882, 0.54902 );
    table->SetTableValue( 2126.0, 0.235294, 0.54902, 0.705882 );
    table->SetTableValue( 2127.0, 0.0980392, 0.0784314, 0.54902 );
    table->SetTableValue( 2128.0, 0.0784314, 0.235294, 0.392157 );
    table->SetTableValue( 2129.0, 0.235294, 0.862745, 0.0784314 );
    table->SetTableValue( 2130.0, 0.235294, 0.0784314, 0.862745 );
    table->SetTableValue( 2131.0, 0.862745, 0.862745, 0.392157 );
    table->SetTableValue( 2132.0, 0.705882, 0.235294, 0.235294 );
    table->SetTableValue( 2133.0, 0.235294, 0.235294, 0.862745 );
    table->SetTableValue( 2134.0, 0.862745, 0.235294, 0.862745 );
    table->SetTableValue( 2135.0, 0.254902, 0.862745, 0.235294 );
    table->SetTableValue( 2136.0, 0.0980392, 0.54902, 0.0784314 );
    table->SetTableValue( 2137.0, 0.0509804, 0, 0.980392 );
    table->SetTableValue( 2138.0, 0.239216, 0.0784314, 0.862745 );
    table->SetTableValue( 2139.0, 0.239216, 0.0784314, 0.235294 );
    table->SetTableValue( 2140.0, 0.239216, 0.235294, 0.392157 );
    table->SetTableValue( 2141.0, 0.0980392, 0.0980392, 0.0980392 );
    table->SetTableValue( 2142.0, 0.54902, 0.0784314, 0.235294 );
    table->SetTableValue( 2143.0, 0.862745, 0.705882, 0.0784314 );
    table->SetTableValue( 2144.0, 0.247059, 0.705882, 0.705882 );
    table->SetTableValue( 2145.0, 0.866667, 0.0784314, 0.0392157 );
    table->SetTableValue( 2146.0, 0.0823529, 0.862745, 0.0784314 );
    table->SetTableValue( 2147.0, 0.717647, 0.392157, 0.0784314 );
    table->SetTableValue( 2148.0, 0.866667, 0.0784314, 0.392157 );
    table->SetTableValue( 2149.0, 0.866667, 0.235294, 0.54902 );
    table->SetTableValue( 2150.0, 0.866667, 0.0784314, 0.862745 );
    table->SetTableValue( 2151.0, 0.239216, 0.862745, 0.862745 );
    table->SetTableValue( 2152.0, 0.392157, 0.784314, 0.784314 );
    table->SetTableValue( 2153.0, 0.0392157, 0.784314, 0.784314 );
    table->SetTableValue( 2154.0, 0.866667, 0.862745, 0.0784314 );
    table->SetTableValue( 2155.0, 0.552941, 0.0784314, 0.392157 );
    table->SetTableValue( 2156.0, 0.239216, 0.862745, 0.392157 );
    table->SetTableValue( 2157.0, 0.0823529, 0.862745, 0.235294 );
    table->SetTableValue( 2158.0, 0.552941, 0.235294, 0.0784314 );
    table->SetTableValue( 2159.0, 0.560784, 0.0784314, 0.862745 );
    table->SetTableValue( 2160.0, 0.239216, 0.0784314, 0.705882 );
    table->SetTableValue( 2161.0, 0.396078, 0.235294, 0.862745 );
    table->SetTableValue( 2162.0, 0.0823529, 0.0784314, 0.54902 );
    table->SetTableValue( 2163.0, 0.866667, 0.54902, 0.0784314 );
    table->SetTableValue( 2164.0, 0.552941, 0.392157, 0.862745 );
    table->SetTableValue( 2165.0, 0.396078, 0.0784314, 0.0784314 );
    table->SetTableValue( 2166.0, 0.866667, 0.392157, 0.0784314 );
    table->SetTableValue( 2167.0, 0.709804, 0.784314, 0.0784314 );
    table->SetTableValue( 2168.0, 0.0823529, 0.705882, 0.54902 );
    table->SetTableValue( 2169.0, 0.396078, 0.392157, 0.705882 );
    table->SetTableValue( 2170.0, 0.709804, 0.862745, 0.0784314 );
    table->SetTableValue( 2171.0, 0.0823529, 0.54902, 0.784314 );
    table->SetTableValue( 2172.0, 0.0823529, 0.0784314, 0.941176 );
    table->SetTableValue( 2173.0, 0.0823529, 0.0784314, 0.784314 );
    table->SetTableValue( 2174.0, 0.239216, 0.705882, 0.235294 );
    table->SetTableValue( 2175.0, 0.239216, 0.705882, 0.980392 );
    table->SetTableValue( 2176.0, 0.0823529, 0.0784314, 0.235294 );
    table->SetTableValue( 2177.0, 0.396078, 0.235294, 0.235294 );
    table->SetTableValue( 2178.0, 0.0823529, 0.862745, 0.862745 );
    table->SetTableValue( 2179.0, 0.0823529, 0.705882, 0.705882 );
    table->SetTableValue( 2180.0, 0.87451, 0.862745, 0.235294 );
    table->SetTableValue( 2181.0, 0.866667, 0.235294, 0.235294 );
    table->SetTableValue( 2200.0, 0.0980392, 0.235294, 0.239216 );
    table->SetTableValue( 2201.0, 0.0980392, 0.352941, 0.235294 );
    table->SetTableValue( 2202.0, 0.0980392, 0.470588, 0.235294 );
    table->SetTableValue( 2205.0, 0.0980392, 0.588235, 0.235294 );
    table->SetTableValue( 2206.0, 0.0980392, 0.705882, 0.235294 );
    table->SetTableValue( 2207.0, 0.0980392, 0.823529, 0.235294 );
    table->SetTableValue( 2210.0, 0.0980392, 0.588235, 0.352941 );
    table->SetTableValue( 2211.0, 0.0980392, 0.705882, 0.352941 );
    table->SetTableValue( 2212.0, 0.0980392, 0.823529, 0.352941 );
    table->SetTableValue( 3100.0, 0, 0, 0, 0);
    table->SetTableValue( 3101.0, 0.196078, 0.196078, 0.196078 );
    table->SetTableValue( 3102.0, 0.705882, 0.0784314, 0.117647 );
    table->SetTableValue( 3103.0, 0.235294, 0.0980392, 0.0980392 );
    table->SetTableValue( 3104.0, 0.0980392, 0.235294, 0.235294 );
    table->SetTableValue( 3105.0, 0.705882, 0.0784314, 0.0784314 );
    table->SetTableValue( 3106.0, 0.862745, 0.0784314, 0.392157 );
    table->SetTableValue( 3107.0, 0.54902, 0.235294, 0.235294 );
    table->SetTableValue( 3108.0, 0.705882, 0.862745, 0.54902 );
    table->SetTableValue( 3109.0, 0.54902, 0.392157, 0.705882 );
    table->SetTableValue( 3110.0, 0.705882, 0.0784314, 0.54902 );
    table->SetTableValue( 3111.0, 0.54902, 0.0784314, 0.54902 );
    table->SetTableValue( 3112.0, 0.0823529, 0.0392157, 0.0392157 );
    table->SetTableValue( 3113.0, 0.882353, 0.54902, 0.54902 );
    table->SetTableValue( 3114.0, 0.0901961, 0.235294, 0.705882 );
    table->SetTableValue( 3115.0, 0.705882, 0.235294, 0.705882 );
    table->SetTableValue( 3116.0, 0.0784314, 0.862745, 0.235294 );
    table->SetTableValue( 3117.0, 0.235294, 0.0784314, 0.54902 );
    table->SetTableValue( 3118.0, 0.862745, 0.705882, 0.54902 );
    table->SetTableValue( 3119.0, 0.254902, 0.392157, 0.0784314 );
    table->SetTableValue( 3120.0, 0.862745, 0.235294, 0.0784314 );
    table->SetTableValue( 3121.0, 0.235294, 0.392157, 0.235294 );
    table->SetTableValue( 3122.0, 0.0784314, 0.235294, 0.862745 );
    table->SetTableValue( 3123.0, 0.392157, 0.392157, 0.235294 );
    table->SetTableValue( 3124.0, 0.862745, 0.705882, 0.862745 );
    table->SetTableValue( 3125.0, 0.0784314, 0.705882, 0.54902 );
    table->SetTableValue( 3126.0, 0.235294, 0.54902, 0.705882 );
    table->SetTableValue( 3127.0, 0.0980392, 0.0784314, 0.54902 );
    table->SetTableValue( 3128.0, 0.0784314, 0.235294, 0.392157 );
    table->SetTableValue( 3129.0, 0.235294, 0.862745, 0.0784314 );
    table->SetTableValue( 3130.0, 0.235294, 0.0784314, 0.862745 );
    table->SetTableValue( 3131.0, 0.862745, 0.862745, 0.392157 );
    table->SetTableValue( 3132.0, 0.705882, 0.235294, 0.235294 );
    table->SetTableValue( 3133.0, 0.235294, 0.235294, 0.862745 );
    table->SetTableValue( 3134.0, 0.862745, 0.235294, 0.862745 );
    table->SetTableValue( 3135.0, 0.254902, 0.862745, 0.235294 );
    table->SetTableValue( 3136.0, 0.0980392, 0.54902, 0.0784314 );
    table->SetTableValue( 3137.0, 0.0509804, 0, 0.980392 );
    table->SetTableValue( 3138.0, 0.239216, 0.0784314, 0.862745 );
    table->SetTableValue( 3139.0, 0.239216, 0.0784314, 0.235294 );
    table->SetTableValue( 3140.0, 0.239216, 0.235294, 0.392157 );
    table->SetTableValue( 3141.0, 0.0980392, 0.0980392, 0.0980392 );
    table->SetTableValue( 3142.0, 0.54902, 0.0784314, 0.235294 );
    table->SetTableValue( 3143.0, 0.862745, 0.705882, 0.0784314 );
    table->SetTableValue( 3144.0, 0.247059, 0.705882, 0.705882 );
    table->SetTableValue( 3145.0, 0.866667, 0.0784314, 0.0392157 );
    table->SetTableValue( 3146.0, 0.0823529, 0.862745, 0.0784314 );
    table->SetTableValue( 3147.0, 0.717647, 0.392157, 0.0784314 );
    table->SetTableValue( 3148.0, 0.866667, 0.0784314, 0.392157 );
    table->SetTableValue( 3149.0, 0.866667, 0.235294, 0.54902 );
    table->SetTableValue( 3150.0, 0.866667, 0.0784314, 0.862745 );
    table->SetTableValue( 3151.0, 0.239216, 0.862745, 0.862745 );
    table->SetTableValue( 3152.0, 0.392157, 0.784314, 0.784314 );
    table->SetTableValue( 3153.0, 0.0392157, 0.784314, 0.784314 );
    table->SetTableValue( 3154.0, 0.866667, 0.862745, 0.0784314 );
    table->SetTableValue( 3155.0, 0.552941, 0.0784314, 0.392157 );
    table->SetTableValue( 3156.0, 0.239216, 0.862745, 0.392157 );
    table->SetTableValue( 3157.0, 0.0823529, 0.862745, 0.235294 );
    table->SetTableValue( 3158.0, 0.552941, 0.235294, 0.0784314 );
    table->SetTableValue( 3159.0, 0.560784, 0.0784314, 0.862745 );
    table->SetTableValue( 3160.0, 0.239216, 0.0784314, 0.705882 );
    table->SetTableValue( 3161.0, 0.396078, 0.235294, 0.862745 );
    table->SetTableValue( 3162.0, 0.0823529, 0.0784314, 0.54902 );
    table->SetTableValue( 3163.0, 0.866667, 0.54902, 0.0784314 );
    table->SetTableValue( 3164.0, 0.552941, 0.392157, 0.862745 );
    table->SetTableValue( 3165.0, 0.396078, 0.0784314, 0.0784314 );
    table->SetTableValue( 3166.0, 0.866667, 0.392157, 0.0784314 );
    table->SetTableValue( 3167.0, 0.709804, 0.784314, 0.0784314 );
    table->SetTableValue( 3168.0, 0.0823529, 0.705882, 0.54902 );
    table->SetTableValue( 3169.0, 0.396078, 0.392157, 0.705882 );
    table->SetTableValue( 3170.0, 0.709804, 0.862745, 0.0784314 );
    table->SetTableValue( 3171.0, 0.0823529, 0.54902, 0.784314 );
    table->SetTableValue( 3172.0, 0.0823529, 0.0784314, 0.941176 );
    table->SetTableValue( 3173.0, 0.0823529, 0.0784314, 0.784314 );
    table->SetTableValue( 3174.0, 0.239216, 0.705882, 0.235294 );
    table->SetTableValue( 3175.0, 0.239216, 0.705882, 0.980392 );
    table->SetTableValue( 3176.0, 0.0823529, 0.0784314, 0.235294 );
    table->SetTableValue( 3177.0, 0.396078, 0.235294, 0.235294 );
    table->SetTableValue( 3178.0, 0.0823529, 0.862745, 0.862745 );
    table->SetTableValue( 3179.0, 0.0823529, 0.705882, 0.705882 );
    table->SetTableValue( 3180.0, 0.87451, 0.862745, 0.235294 );
    table->SetTableValue( 3181.0, 0.866667, 0.235294, 0.235294 );
    table->SetTableValue( 4100.0, 0, 0, 0, 0);
    table->SetTableValue( 4101.0, 0.196078, 0.196078, 0.196078 );
    table->SetTableValue( 4102.0, 0.705882, 0.0784314, 0.117647 );
    table->SetTableValue( 4103.0, 0.235294, 0.0980392, 0.0980392 );
    table->SetTableValue( 4104.0, 0.0980392, 0.235294, 0.235294 );
    table->SetTableValue( 4105.0, 0.705882, 0.0784314, 0.0784314 );
    table->SetTableValue( 4106.0, 0.862745, 0.0784314, 0.392157 );
    table->SetTableValue( 4107.0, 0.54902, 0.235294, 0.235294 );
    table->SetTableValue( 4108.0, 0.705882, 0.862745, 0.54902 );
    table->SetTableValue( 4109.0, 0.54902, 0.392157, 0.705882 );
    table->SetTableValue( 4110.0, 0.705882, 0.0784314, 0.54902 );
    table->SetTableValue( 4111.0, 0.54902, 0.0784314, 0.54902 );
    table->SetTableValue( 4112.0, 0.0823529, 0.0392157, 0.0392157 );
    table->SetTableValue( 4113.0, 0.882353, 0.54902, 0.54902 );
    table->SetTableValue( 4114.0, 0.0901961, 0.235294, 0.705882 );
    table->SetTableValue( 4115.0, 0.705882, 0.235294, 0.705882 );
    table->SetTableValue( 4116.0, 0.0784314, 0.862745, 0.235294 );
    table->SetTableValue( 4117.0, 0.235294, 0.0784314, 0.54902 );
    table->SetTableValue( 4118.0, 0.862745, 0.705882, 0.54902 );
    table->SetTableValue( 4119.0, 0.254902, 0.392157, 0.0784314 );
    table->SetTableValue( 4120.0, 0.862745, 0.235294, 0.0784314 );
    table->SetTableValue( 4121.0, 0.235294, 0.392157, 0.235294 );
    table->SetTableValue( 4122.0, 0.0784314, 0.235294, 0.862745 );
    table->SetTableValue( 4123.0, 0.392157, 0.392157, 0.235294 );
    table->SetTableValue( 4124.0, 0.862745, 0.705882, 0.862745 );
    table->SetTableValue( 4125.0, 0.0784314, 0.705882, 0.54902 );
    table->SetTableValue( 4126.0, 0.235294, 0.54902, 0.705882 );
    table->SetTableValue( 4127.0, 0.0980392, 0.0784314, 0.54902 );
    table->SetTableValue( 4128.0, 0.0784314, 0.235294, 0.392157 );
    table->SetTableValue( 4129.0, 0.235294, 0.862745, 0.0784314 );
    table->SetTableValue( 4130.0, 0.235294, 0.0784314, 0.862745 );
    table->SetTableValue( 4131.0, 0.862745, 0.862745, 0.392157 );
    table->SetTableValue( 4132.0, 0.705882, 0.235294, 0.235294 );
    table->SetTableValue( 4133.0, 0.235294, 0.235294, 0.862745 );
    table->SetTableValue( 4134.0, 0.862745, 0.235294, 0.862745 );
    table->SetTableValue( 4135.0, 0.254902, 0.862745, 0.235294 );
    table->SetTableValue( 4136.0, 0.0980392, 0.54902, 0.0784314 );
    table->SetTableValue( 4137.0, 0.0509804, 0, 0.980392 );
    table->SetTableValue( 4138.0, 0.239216, 0.0784314, 0.862745 );
    table->SetTableValue( 4139.0, 0.239216, 0.0784314, 0.235294 );
    table->SetTableValue( 4140.0, 0.239216, 0.235294, 0.392157 );
    table->SetTableValue( 4141.0, 0.0980392, 0.0980392, 0.0980392 );
    table->SetTableValue( 4142.0, 0.54902, 0.0784314, 0.235294 );
    table->SetTableValue( 4143.0, 0.862745, 0.705882, 0.0784314 );
    table->SetTableValue( 4144.0, 0.247059, 0.705882, 0.705882 );
    table->SetTableValue( 4145.0, 0.866667, 0.0784314, 0.0392157 );
    table->SetTableValue( 4146.0, 0.0823529, 0.862745, 0.0784314 );
    table->SetTableValue( 4147.0, 0.717647, 0.392157, 0.0784314 );
    table->SetTableValue( 4148.0, 0.866667, 0.0784314, 0.392157 );
    table->SetTableValue( 4149.0, 0.866667, 0.235294, 0.54902 );
    table->SetTableValue( 4150.0, 0.866667, 0.0784314, 0.862745 );
    table->SetTableValue( 4151.0, 0.239216, 0.862745, 0.862745 );
    table->SetTableValue( 4152.0, 0.392157, 0.784314, 0.784314 );
    table->SetTableValue( 4153.0, 0.0392157, 0.784314, 0.784314 );
    table->SetTableValue( 4154.0, 0.866667, 0.862745, 0.0784314 );
    table->SetTableValue( 4155.0, 0.552941, 0.0784314, 0.392157 );
    table->SetTableValue( 4156.0, 0.239216, 0.862745, 0.392157 );
    table->SetTableValue( 4157.0, 0.0823529, 0.862745, 0.235294 );
    table->SetTableValue( 4158.0, 0.552941, 0.235294, 0.0784314 );
    table->SetTableValue( 4159.0, 0.560784, 0.0784314, 0.862745 );
    table->SetTableValue( 4160.0, 0.239216, 0.0784314, 0.705882 );
    table->SetTableValue( 4161.0, 0.396078, 0.235294, 0.862745 );
    table->SetTableValue( 4162.0, 0.0823529, 0.0784314, 0.54902 );
    table->SetTableValue( 4163.0, 0.866667, 0.54902, 0.0784314 );
    table->SetTableValue( 4164.0, 0.552941, 0.392157, 0.862745 );
    table->SetTableValue( 4165.0, 0.396078, 0.0784314, 0.0784314 );
    table->SetTableValue( 4166.0, 0.866667, 0.392157, 0.0784314 );
    table->SetTableValue( 4167.0, 0.709804, 0.784314, 0.0784314 );
    table->SetTableValue( 4168.0, 0.0823529, 0.705882, 0.54902 );
    table->SetTableValue( 4169.0, 0.396078, 0.392157, 0.705882 );
    table->SetTableValue( 4170.0, 0.709804, 0.862745, 0.0784314 );
    table->SetTableValue( 4171.0, 0.0823529, 0.54902, 0.784314 );
    table->SetTableValue( 4172.0, 0.0823529, 0.0784314, 0.941176 );
    table->SetTableValue( 4173.0, 0.0823529, 0.0784314, 0.784314 );
    table->SetTableValue( 4174.0, 0.239216, 0.705882, 0.235294 );
    table->SetTableValue( 4175.0, 0.239216, 0.705882, 0.980392 );
    table->SetTableValue( 4176.0, 0.0823529, 0.0784314, 0.235294 );
    table->SetTableValue( 4177.0, 0.396078, 0.235294, 0.235294 );
    table->SetTableValue( 4178.0, 0.0823529, 0.862745, 0.862745 );
    table->SetTableValue( 4179.0, 0.0823529, 0.705882, 0.705882 );
    table->SetTableValue( 4180.0, 0.87451, 0.862745, 0.235294 );
    table->SetTableValue( 4181.0, 0.866667, 0.235294, 0.235294 );
    table->SetTableValue( 5001.0, 0.0784314, 0.117647, 0.156863 );
    table->SetTableValue( 5002.0, 0.0784314, 0.117647, 0.156863 );
    table->SetTableValue( 5100.0, 0.8, 0.4, 0.4 );
    table->SetTableValue( 5101.0, 0.8, 0.4, 0.4 );
    table->SetTableValue( 5102.0, 1, 1, 0.4 );
    table->SetTableValue( 5103.0, 0.6, 0.8, 0 );
    table->SetTableValue( 5104.0, 0, 0.6, 0.6 );
    table->SetTableValue( 5105.0, 0.8, 0.6, 1 );
    table->SetTableValue( 5106.0, 1, 0.6, 0.2 );
    table->SetTableValue( 5107.0, 0.8, 0.8, 0.8 );
    table->SetTableValue( 5108.0, 0.6, 1, 1 );
    table->SetTableValue( 5109.0, 0.4, 0.6, 1 );
    table->SetTableValue( 5110.0, 1, 1, 0.4 );
    table->SetTableValue( 5111.0, 0.6, 0.8, 0 );
    table->SetTableValue( 5112.0, 0, 0.6, 0.6 );
    table->SetTableValue( 5113.0, 0.8, 0.6, 1 );
    table->SetTableValue( 5114.0, 1, 0.6, 0.2 );
    table->SetTableValue( 5115.0, 0.8, 0.8, 0.8 );
    table->SetTableValue( 5116.0, 0.6, 1, 1 );
    table->SetTableValue( 5117.0, 0.4, 0.6, 1 );
    table->SetTableValue( 5200.0, 0.8, 0.4, 0.4 );
    table->SetTableValue( 5201.0, 0.8, 0.4, 0.4 );
    table->SetTableValue( 5202.0, 1, 1, 0.4 );
    table->SetTableValue( 5203.0, 0.6, 0.8, 0 );
    table->SetTableValue( 5204.0, 0, 0.6, 0.6 );
    table->SetTableValue( 5205.0, 0.8, 0.6, 1 );
    table->SetTableValue( 5206.0, 1, 0.6, 0.2 );
    table->SetTableValue( 5207.0, 0.8, 0.8, 0.8 );
    table->SetTableValue( 5208.0, 0.6, 1, 1 );
    table->SetTableValue( 5209.0, 0.4, 0.6, 1 );
    table->SetTableValue( 5210.0, 1, 1, 0.4 );
    table->SetTableValue( 5211.0, 0.6, 0.8, 0 );
    table->SetTableValue( 5212.0, 0, 0.6, 0.6 );
    table->SetTableValue( 5213.0, 0.8, 0.6, 1 );
    table->SetTableValue( 5214.0, 1, 0.6, 0.2 );
    table->SetTableValue( 5215.0, 0.8, 0.8, 0.8 );
    table->SetTableValue( 5216.0, 0.6, 1, 1 );
    table->SetTableValue( 5217.0, 0.4, 0.6, 1 );
    table->SetTableValue( 6000.0, 0, 1, 0 );
    table->SetTableValue( 6001.0, 1, 1, 0 );
    table->SetTableValue( 6002.0, 0, 1, 1 );
    table->SetTableValue( 6003.0, 0, 0, 1 );
    table->SetTableValue( 6010.0, 0.92549, 0.0627451, 0.905882 );
    table->SetTableValue( 6020.0, 0.929412, 0.0705882, 0.909804 );
    table->SetTableValue( 6030.0, 0.92549, 0.0509804, 0.890196 );
    table->SetTableValue( 6040.0, 0.92549, 0.0666667, 0.894118 );
    table->SetTableValue( 6050.0, 0.00392157, 1, 0.00392157 );
    table->SetTableValue( 6060.0, 0.00784314, 1, 0.00392157 );
    table->SetTableValue( 6070.0, 0.92549, 0.054902, 0.901961 );
    table->SetTableValue( 6080.0, 0.929412, 0.054902, 0.901961 );
    table->SetTableValue( 7001.0, 0.282353, 0.517647, 0.709804 );
    table->SetTableValue( 7002.0, 0.952941, 0.952941, 0.952941 );
    table->SetTableValue( 7003.0, 0.811765, 0.247059, 0.309804 );
    table->SetTableValue( 7004.0, 0.47451, 0.0784314, 0.529412 );
    table->SetTableValue( 7005.0, 0.772549, 0.235294, 0.972549 );
    table->SetTableValue( 7006.0, 0.00784314, 0.584314, 0.00784314 );
    table->SetTableValue( 7007.0, 0.866667, 0.976471, 0.65098 );
    table->SetTableValue( 7008.0, 0.909804, 0.572549, 0.137255 );
    table->SetTableValue( 7009.0, 0.0784314, 0.235294, 0.470588 );
    table->SetTableValue( 7010.0, 0.980392, 0.980392, 0 );
    table->SetTableValue( 7011.0, 0.478431, 0.733333, 0.870588 );
    table->SetTableValue( 7012.0, 0.929412, 0.0470588, 0.694118 );
    table->SetTableValue( 7013.0, 0.0392157, 0.192157, 1 );
    table->SetTableValue( 7014.0, 0.803922, 0.721569, 0.564706 );
    table->SetTableValue( 7015.0, 0.176471, 0.803922, 0.647059 );
    table->SetTableValue( 7016.0, 0.458824, 0.627451, 0.686275 );
    table->SetTableValue( 7017.0, 0.866667, 0.85098, 0.0823529 );
    table->SetTableValue( 7018.0, 0.0784314, 0.235294, 0.470588 );
    table->SetTableValue( 7019.0, 0.552941, 0.0823529, 0.392157 );
    table->SetTableValue( 7020.0, 0.882353, 0.54902, 0.552941 );
    table->SetTableValue( 7100.0, 0.164706, 0.788235, 0.658824 );
    table->SetTableValue( 7101.0, 0.658824, 0.407843, 0.635294 );
    table->SetTableValue( 8001.0, 0.290196, 0.509804, 0.709804 );
    table->SetTableValue( 8002.0, 0.94902, 0.945098, 0.941176 );
    table->SetTableValue( 8003.0, 0.807843, 0.254902, 0.305882 );
    table->SetTableValue( 8004.0, 0.470588, 0.0823529, 0.521569 );
    table->SetTableValue( 8005.0, 0.764706, 0.239216, 0.964706 );
    table->SetTableValue( 8006.0, 0.0117647, 0.576471, 0.0235294 );
    table->SetTableValue( 8007.0, 0.862745, 0.984314, 0.639216 );
    table->SetTableValue( 8008.0, 0.909804, 0.572549, 0.129412 );
    table->SetTableValue( 8009.0, 0.0156863, 0.447059, 0.054902 );
    table->SetTableValue( 8010.0, 0.47451, 0.721569, 0.862745 );
    table->SetTableValue( 8011.0, 0.921569, 0.0431373, 0.686275 );
    table->SetTableValue( 8012.0, 0.0470588, 0.180392, 0.980392 );
    table->SetTableValue( 8013.0, 0.796078, 0.713725, 0.560784 );
    table->SetTableValue( 8014.0, 0.164706, 0.8, 0.654902 );
    table->SetTableValue( 9000.0, 0.117647, 0.0196078, 0.117647 );
    table->SetTableValue( 9001.0, 0.117647, 0.392157, 0.176471 );
    table->SetTableValue( 9002.0, 0.509804, 0.392157, 0.647059 );
    table->SetTableValue( 9003.0, 0.411765, 0.0980392, 0.0196078 );
    table->SetTableValue( 9004.0, 0.490196, 0.27451, 0.215686 );
    table->SetTableValue( 9005.0, 0.882353, 0.0784314, 0.411765 );
    table->SetTableValue( 9006.0, 0.882353, 0.0784314, 0.0588235 );
    table->SetTableValue( 9500.0, 0.117647, 0.215686, 0.117647 );
    table->SetTableValue( 9501.0, 0.117647, 0.588235, 0.176471 );
    table->SetTableValue( 9502.0, 0.509804, 0.588235, 0.647059 );
    table->SetTableValue( 9503.0, 0.411765, 0.294118, 0.0196078 );
    table->SetTableValue( 9504.0, 0.490196, 0.470588, 0.215686 );
    table->SetTableValue( 9505.0, 0.882353, 0.27451, 0.411765 );
    table->SetTableValue( 9506.0, 0.882353, 0.27451, 0.0588235 );
    table->SetTableValue( 11100.0, 0, 0, 0, 0);
    table->SetTableValue( 11101.0, 0.0901961, 0.862745, 0.235294 );
    table->SetTableValue( 11102.0, 0.0901961, 0.235294, 0.705882 );
    table->SetTableValue( 11103.0, 0.247059, 0.392157, 0.235294 );
    table->SetTableValue( 11104.0, 0.247059, 0.0784314, 0.862745 );
    table->SetTableValue( 11105.0, 0.0509804, 0, 0.980392 );
    table->SetTableValue( 11106.0, 0.101961, 0.235294, 0 );
    table->SetTableValue( 11107.0, 0.101961, 0.235294, 0.294118 );
    table->SetTableValue( 11108.0, 0.101961, 0.235294, 0.588235 );
    table->SetTableValue( 11109.0, 0.0980392, 0.235294, 0.980392 );
    table->SetTableValue( 11110.0, 0.235294, 0.0980392, 0.0980392 );
    table->SetTableValue( 11111.0, 0.705882, 0.0784314, 0.0784314 );
    table->SetTableValue( 11112.0, 0.862745, 0.0784314, 0.392157 );
    table->SetTableValue( 11113.0, 0.54902, 0.235294, 0.235294 );
    table->SetTableValue( 11114.0, 0.705882, 0.862745, 0.54902 );
    table->SetTableValue( 11115.0, 0.54902, 0.392157, 0.705882 );
    table->SetTableValue( 11116.0, 0.705882, 0.0784314, 0.54902 );
    table->SetTableValue( 11117.0, 0.0901961, 0.0392157, 0.0392157 );
    table->SetTableValue( 11118.0, 0.882353, 0.54902, 0.54902 );
    table->SetTableValue( 11119.0, 0.705882, 0.235294, 0.705882 );
    table->SetTableValue( 11120.0, 0.0784314, 0.862745, 0.235294 );
    table->SetTableValue( 11121.0, 0.235294, 0.0784314, 0.54902 );
    table->SetTableValue( 11122.0, 0.862745, 0.705882, 0.54902 );
    table->SetTableValue( 11123.0, 0.254902, 0.392157, 0.0784314 );
    table->SetTableValue( 11124.0, 0.862745, 0.235294, 0.0784314 );
    table->SetTableValue( 11125.0, 0.0784314, 0.235294, 0.862745 );
    table->SetTableValue( 11126.0, 0.392157, 0.392157, 0.235294 );
    table->SetTableValue( 11127.0, 0.862745, 0.705882, 0.862745 );
    table->SetTableValue( 11128.0, 0.0784314, 0.705882, 0.54902 );
    table->SetTableValue( 11129.0, 0.235294, 0.54902, 0.705882 );
    table->SetTableValue( 11130.0, 0.0980392, 0.0784314, 0.54902 );
    table->SetTableValue( 11131.0, 0.0784314, 0.235294, 0.392157 );
    table->SetTableValue( 11132.0, 0.235294, 0.862745, 0.0784314 );
    table->SetTableValue( 11133.0, 0.235294, 0.235294, 0.862745 );
    table->SetTableValue( 11134.0, 0.862745, 0.235294, 0.862745 );
    table->SetTableValue( 11135.0, 0.254902, 0.862745, 0.235294 );
    table->SetTableValue( 11136.0, 0.0980392, 0.54902, 0.0784314 );
    table->SetTableValue( 11137.0, 0.862745, 0.862745, 0.392157 );
    table->SetTableValue( 11138.0, 0.705882, 0.235294, 0.235294 );
    table->SetTableValue( 11139.0, 0.239216, 0.0784314, 0.862745 );
    table->SetTableValue( 11140.0, 0.239216, 0.0784314, 0.235294 );
    table->SetTableValue( 11141.0, 0.239216, 0.235294, 0.392157 );
    table->SetTableValue( 11142.0, 0.0980392, 0.0980392, 0.0980392 );
    table->SetTableValue( 11143.0, 0.54902, 0.0784314, 0.235294 );
    table->SetTableValue( 11144.0, 0.862745, 0.705882, 0.0784314 );
    table->SetTableValue( 11145.0, 0.247059, 0.705882, 0.705882 );
    table->SetTableValue( 11146.0, 0.866667, 0.0784314, 0.0392157 );
    table->SetTableValue( 11147.0, 0.866667, 0.0784314, 0.392157 );
    table->SetTableValue( 11148.0, 0.866667, 0.235294, 0.54902 );
    table->SetTableValue( 11149.0, 0.866667, 0.0784314, 0.862745 );
    table->SetTableValue( 11150.0, 0.239216, 0.862745, 0.862745 );
    table->SetTableValue( 11151.0, 0.392157, 0.784314, 0.784314 );
    table->SetTableValue( 11152.0, 0.0392157, 0.784314, 0.784314 );
    table->SetTableValue( 11153.0, 0.866667, 0.862745, 0.0784314 );
    table->SetTableValue( 11154.0, 0.552941, 0.0784314, 0.392157 );
    table->SetTableValue( 11155.0, 0.239216, 0.862745, 0.392157 );
    table->SetTableValue( 11156.0, 0.552941, 0.235294, 0.0784314 );
    table->SetTableValue( 11157.0, 0.560784, 0.0784314, 0.862745 );
    table->SetTableValue( 11158.0, 0.396078, 0.235294, 0.862745 );
    table->SetTableValue( 11159.0, 0.0823529, 0.0784314, 0.54902 );
    table->SetTableValue( 11160.0, 0.239216, 0.0784314, 0.705882 );
    table->SetTableValue( 11161.0, 0.866667, 0.54902, 0.0784314 );
    table->SetTableValue( 11162.0, 0.552941, 0.392157, 0.862745 );
    table->SetTableValue( 11163.0, 0.866667, 0.392157, 0.0784314 );
    table->SetTableValue( 11164.0, 0.709804, 0.784314, 0.0784314 );
    table->SetTableValue( 11165.0, 0.396078, 0.0784314, 0.0784314 );
    table->SetTableValue( 11166.0, 0.396078, 0.392157, 0.705882 );
    table->SetTableValue( 11167.0, 0.709804, 0.862745, 0.0784314 );
    table->SetTableValue( 11168.0, 0.0823529, 0.54902, 0.784314 );
    table->SetTableValue( 11169.0, 0.0823529, 0.0784314, 0.941176 );
    table->SetTableValue( 11170.0, 0.0823529, 0.0784314, 0.784314 );
    table->SetTableValue( 11171.0, 0.0823529, 0.0784314, 0.235294 );
    table->SetTableValue( 11172.0, 0.396078, 0.235294, 0.235294 );
    table->SetTableValue( 11173.0, 0.0823529, 0.705882, 0.705882 );
    table->SetTableValue( 11174.0, 0.87451, 0.862745, 0.235294 );
    table->SetTableValue( 11175.0, 0.866667, 0.235294, 0.235294 );
    table->SetTableValue( 12100.0, 0, 0, 0, 0);
    table->SetTableValue( 12101.0, 0.0901961, 0.862745, 0.235294 );
    table->SetTableValue( 12102.0, 0.0901961, 0.235294, 0.705882 );
    table->SetTableValue( 12103.0, 0.247059, 0.392157, 0.235294 );
    table->SetTableValue( 12104.0, 0.247059, 0.0784314, 0.862745 );
    table->SetTableValue( 12105.0, 0.0509804, 0, 0.980392 );
    table->SetTableValue( 12106.0, 0.101961, 0.235294, 0 );
    table->SetTableValue( 12107.0, 0.101961, 0.235294, 0.294118 );
    table->SetTableValue( 12108.0, 0.101961, 0.235294, 0.588235 );
    table->SetTableValue( 12109.0, 0.0980392, 0.235294, 0.980392 );
    table->SetTableValue( 12110.0, 0.235294, 0.0980392, 0.0980392 );
    table->SetTableValue( 12111.0, 0.705882, 0.0784314, 0.0784314 );
    table->SetTableValue( 12112.0, 0.862745, 0.0784314, 0.392157 );
    table->SetTableValue( 12113.0, 0.54902, 0.235294, 0.235294 );
    table->SetTableValue( 12114.0, 0.705882, 0.862745, 0.54902 );
    table->SetTableValue( 12115.0, 0.54902, 0.392157, 0.705882 );
    table->SetTableValue( 12116.0, 0.705882, 0.0784314, 0.54902 );
    table->SetTableValue( 12117.0, 0.0901961, 0.0392157, 0.0392157 );
    table->SetTableValue( 12118.0, 0.882353, 0.54902, 0.54902 );
    table->SetTableValue( 12119.0, 0.705882, 0.235294, 0.705882 );
    table->SetTableValue( 12120.0, 0.0784314, 0.862745, 0.235294 );
    table->SetTableValue( 12121.0, 0.235294, 0.0784314, 0.54902 );
    table->SetTableValue( 12122.0, 0.862745, 0.705882, 0.54902 );
    table->SetTableValue( 12123.0, 0.254902, 0.392157, 0.0784314 );
    table->SetTableValue( 12124.0, 0.862745, 0.235294, 0.0784314 );
    table->SetTableValue( 12125.0, 0.0784314, 0.235294, 0.862745 );
    table->SetTableValue( 12126.0, 0.392157, 0.392157, 0.235294 );
    table->SetTableValue( 12127.0, 0.862745, 0.705882, 0.862745 );
    table->SetTableValue( 12128.0, 0.0784314, 0.705882, 0.54902 );
    table->SetTableValue( 12129.0, 0.235294, 0.54902, 0.705882 );
    table->SetTableValue( 12130.0, 0.0980392, 0.0784314, 0.54902 );
    table->SetTableValue( 12131.0, 0.0784314, 0.235294, 0.392157 );
    table->SetTableValue( 12132.0, 0.235294, 0.862745, 0.0784314 );
    table->SetTableValue( 12133.0, 0.235294, 0.235294, 0.862745 );
    table->SetTableValue( 12134.0, 0.862745, 0.235294, 0.862745 );
    table->SetTableValue( 12135.0, 0.254902, 0.862745, 0.235294 );
    table->SetTableValue( 12136.0, 0.0980392, 0.54902, 0.0784314 );
    table->SetTableValue( 12137.0, 0.862745, 0.862745, 0.392157 );
    table->SetTableValue( 12138.0, 0.705882, 0.235294, 0.235294 );
    table->SetTableValue( 12139.0, 0.239216, 0.0784314, 0.862745 );
    table->SetTableValue( 12140.0, 0.239216, 0.0784314, 0.235294 );
    table->SetTableValue( 12141.0, 0.239216, 0.235294, 0.392157 );
    table->SetTableValue( 12142.0, 0.0980392, 0.0980392, 0.0980392 );
    table->SetTableValue( 12143.0, 0.54902, 0.0784314, 0.235294 );
    table->SetTableValue( 12144.0, 0.862745, 0.705882, 0.0784314 );
    table->SetTableValue( 12145.0, 0.247059, 0.705882, 0.705882 );
    table->SetTableValue( 12146.0, 0.866667, 0.0784314, 0.0392157 );
    table->SetTableValue( 12147.0, 0.866667, 0.0784314, 0.392157 );
    table->SetTableValue( 12148.0, 0.866667, 0.235294, 0.54902 );
    table->SetTableValue( 12149.0, 0.866667, 0.0784314, 0.862745 );
    table->SetTableValue( 12150.0, 0.239216, 0.862745, 0.862745 );
    table->SetTableValue( 12151.0, 0.392157, 0.784314, 0.784314 );
    table->SetTableValue( 12152.0, 0.0392157, 0.784314, 0.784314 );
    table->SetTableValue( 12153.0, 0.866667, 0.862745, 0.0784314 );
    table->SetTableValue( 12154.0, 0.552941, 0.0784314, 0.392157 );
    table->SetTableValue( 12155.0, 0.239216, 0.862745, 0.392157 );
    table->SetTableValue( 12156.0, 0.552941, 0.235294, 0.0784314 );
    table->SetTableValue( 12157.0, 0.560784, 0.0784314, 0.862745 );
    table->SetTableValue( 12158.0, 0.396078, 0.235294, 0.862745 );
    table->SetTableValue( 12159.0, 0.0823529, 0.0784314, 0.54902 );
    table->SetTableValue( 12160.0, 0.239216, 0.0784314, 0.705882 );
    table->SetTableValue( 12161.0, 0.866667, 0.54902, 0.0784314 );
    table->SetTableValue( 12162.0, 0.552941, 0.392157, 0.862745 );
    table->SetTableValue( 12163.0, 0.866667, 0.392157, 0.0784314 );
    table->SetTableValue( 12164.0, 0.709804, 0.784314, 0.0784314 );
    table->SetTableValue( 12165.0, 0.396078, 0.0784314, 0.0784314 );
    table->SetTableValue( 12166.0, 0.396078, 0.392157, 0.705882 );
    table->SetTableValue( 12167.0, 0.709804, 0.862745, 0.0784314 );
    table->SetTableValue( 12168.0, 0.0823529, 0.54902, 0.784314 );
    table->SetTableValue( 12169.0, 0.0823529, 0.0784314, 0.941176 );
    table->SetTableValue( 12170.0, 0.0823529, 0.0784314, 0.784314 );
    table->SetTableValue( 12171.0, 0.0823529, 0.0784314, 0.235294 );
    table->SetTableValue( 12172.0, 0.396078, 0.235294, 0.235294 );
    table->SetTableValue( 12173.0, 0.0823529, 0.705882, 0.705882 );
    table->SetTableValue( 12174.0, 0.87451, 0.862745, 0.235294 );
    table->SetTableValue( 12175.0, 0.866667, 0.235294, 0.235294 );
    table->SetTableValue( 13100.0, 0, 0, 0, 0);
    table->SetTableValue( 13101.0, 0.0901961, 0.862745, 0.235294 );
    table->SetTableValue( 13102.0, 0.0901961, 0.235294, 0.705882 );
    table->SetTableValue( 13103.0, 0.247059, 0.392157, 0.235294 );
    table->SetTableValue( 13104.0, 0.247059, 0.0784314, 0.862745 );
    table->SetTableValue( 13105.0, 0.0509804, 0, 0.980392 );
    table->SetTableValue( 13106.0, 0.101961, 0.235294, 0 );
    table->SetTableValue( 13107.0, 0.101961, 0.235294, 0.294118 );
    table->SetTableValue( 13108.0, 0.101961, 0.235294, 0.588235 );
    table->SetTableValue( 13109.0, 0.0980392, 0.235294, 0.980392 );
    table->SetTableValue( 13110.0, 0.235294, 0.0980392, 0.0980392 );
    table->SetTableValue( 13111.0, 0.705882, 0.0784314, 0.0784314 );
    table->SetTableValue( 13112.0, 0.862745, 0.0784314, 0.392157 );
    table->SetTableValue( 13113.0, 0.54902, 0.235294, 0.235294 );
    table->SetTableValue( 13114.0, 0.705882, 0.862745, 0.54902 );
    table->SetTableValue( 13115.0, 0.54902, 0.392157, 0.705882 );
    table->SetTableValue( 13116.0, 0.705882, 0.0784314, 0.54902 );
    table->SetTableValue( 13117.0, 0.0901961, 0.0392157, 0.0392157 );
    table->SetTableValue( 13118.0, 0.882353, 0.54902, 0.54902 );
    table->SetTableValue( 13119.0, 0.705882, 0.235294, 0.705882 );
    table->SetTableValue( 13120.0, 0.0784314, 0.862745, 0.235294 );
    table->SetTableValue( 13121.0, 0.235294, 0.0784314, 0.54902 );
    table->SetTableValue( 13122.0, 0.862745, 0.705882, 0.54902 );
    table->SetTableValue( 13123.0, 0.254902, 0.392157, 0.0784314 );
    table->SetTableValue( 13124.0, 0.862745, 0.235294, 0.0784314 );
    table->SetTableValue( 13125.0, 0.0784314, 0.235294, 0.862745 );
    table->SetTableValue( 13126.0, 0.392157, 0.392157, 0.235294 );
    table->SetTableValue( 13127.0, 0.862745, 0.705882, 0.862745 );
    table->SetTableValue( 13128.0, 0.0784314, 0.705882, 0.54902 );
    table->SetTableValue( 13129.0, 0.235294, 0.54902, 0.705882 );
    table->SetTableValue( 13130.0, 0.0980392, 0.0784314, 0.54902 );
    table->SetTableValue( 13131.0, 0.0784314, 0.235294, 0.392157 );
    table->SetTableValue( 13132.0, 0.235294, 0.862745, 0.0784314 );
    table->SetTableValue( 13133.0, 0.235294, 0.235294, 0.862745 );
    table->SetTableValue( 13134.0, 0.862745, 0.235294, 0.862745 );
    table->SetTableValue( 13135.0, 0.254902, 0.862745, 0.235294 );
    table->SetTableValue( 13136.0, 0.0980392, 0.54902, 0.0784314 );
    table->SetTableValue( 13137.0, 0.862745, 0.862745, 0.392157 );
    table->SetTableValue( 13138.0, 0.705882, 0.235294, 0.235294 );
    table->SetTableValue( 13139.0, 0.239216, 0.0784314, 0.862745 );
    table->SetTableValue( 13140.0, 0.239216, 0.0784314, 0.235294 );
    table->SetTableValue( 13141.0, 0.239216, 0.235294, 0.392157 );
    table->SetTableValue( 13142.0, 0.0980392, 0.0980392, 0.0980392 );
    table->SetTableValue( 13143.0, 0.54902, 0.0784314, 0.235294 );
    table->SetTableValue( 13144.0, 0.862745, 0.705882, 0.0784314 );
    table->SetTableValue( 13145.0, 0.247059, 0.705882, 0.705882 );
    table->SetTableValue( 13146.0, 0.866667, 0.0784314, 0.0392157 );
    table->SetTableValue( 13147.0, 0.866667, 0.0784314, 0.392157 );
    table->SetTableValue( 13148.0, 0.866667, 0.235294, 0.54902 );
    table->SetTableValue( 13149.0, 0.866667, 0.0784314, 0.862745 );
    table->SetTableValue( 13150.0, 0.239216, 0.862745, 0.862745 );
    table->SetTableValue( 13151.0, 0.392157, 0.784314, 0.784314 );
    table->SetTableValue( 13152.0, 0.0392157, 0.784314, 0.784314 );
    table->SetTableValue( 13153.0, 0.866667, 0.862745, 0.0784314 );
    table->SetTableValue( 13154.0, 0.552941, 0.0784314, 0.392157 );
    table->SetTableValue( 13155.0, 0.239216, 0.862745, 0.392157 );
    table->SetTableValue( 13156.0, 0.552941, 0.235294, 0.0784314 );
    table->SetTableValue( 13157.0, 0.560784, 0.0784314, 0.862745 );
    table->SetTableValue( 13158.0, 0.396078, 0.235294, 0.862745 );
    table->SetTableValue( 13159.0, 0.0823529, 0.0784314, 0.54902 );
    table->SetTableValue( 13160.0, 0.239216, 0.0784314, 0.705882 );
    table->SetTableValue( 13161.0, 0.866667, 0.54902, 0.0784314 );
    table->SetTableValue( 13162.0, 0.552941, 0.392157, 0.862745 );
    table->SetTableValue( 13163.0, 0.866667, 0.392157, 0.0784314 );
    table->SetTableValue( 13164.0, 0.709804, 0.784314, 0.0784314 );
    table->SetTableValue( 13165.0, 0.396078, 0.0784314, 0.0784314 );
    table->SetTableValue( 13166.0, 0.396078, 0.392157, 0.705882 );
    table->SetTableValue( 13167.0, 0.709804, 0.862745, 0.0784314 );
    table->SetTableValue( 13168.0, 0.0823529, 0.54902, 0.784314 );
    table->SetTableValue( 13169.0, 0.0823529, 0.0784314, 0.941176 );
    table->SetTableValue( 13170.0, 0.0823529, 0.0784314, 0.784314 );
    table->SetTableValue( 13171.0, 0.0823529, 0.0784314, 0.235294 );
    table->SetTableValue( 13172.0, 0.396078, 0.235294, 0.235294 );
    table->SetTableValue( 13173.0, 0.0823529, 0.705882, 0.705882 );
    table->SetTableValue( 13174.0, 0.87451, 0.862745, 0.235294 );
    table->SetTableValue( 13175.0, 0.866667, 0.235294, 0.235294 );
    table->SetTableValue( 14100.0, 0, 0, 0, 0);
    table->SetTableValue( 14101.0, 0.0901961, 0.862745, 0.235294 );
    table->SetTableValue( 14102.0, 0.0901961, 0.235294, 0.705882 );
    table->SetTableValue( 14103.0, 0.247059, 0.392157, 0.235294 );
    table->SetTableValue( 14104.0, 0.247059, 0.0784314, 0.862745 );
    table->SetTableValue( 14105.0, 0.0509804, 0, 0.980392 );
    table->SetTableValue( 14106.0, 0.101961, 0.235294, 0 );
    table->SetTableValue( 14107.0, 0.101961, 0.235294, 0.294118 );
    table->SetTableValue( 14108.0, 0.101961, 0.235294, 0.588235 );
    table->SetTableValue( 14109.0, 0.0980392, 0.235294, 0.980392 );
    table->SetTableValue( 14110.0, 0.235294, 0.0980392, 0.0980392 );
    table->SetTableValue( 14111.0, 0.705882, 0.0784314, 0.0784314 );
    table->SetTableValue( 14112.0, 0.862745, 0.0784314, 0.392157 );
    table->SetTableValue( 14113.0, 0.54902, 0.235294, 0.235294 );
    table->SetTableValue( 14114.0, 0.705882, 0.862745, 0.54902 );
    table->SetTableValue( 14115.0, 0.54902, 0.392157, 0.705882 );
    table->SetTableValue( 14116.0, 0.705882, 0.0784314, 0.54902 );
    table->SetTableValue( 14117.0, 0.0901961, 0.0392157, 0.0392157 );
    table->SetTableValue( 14118.0, 0.882353, 0.54902, 0.54902 );
    table->SetTableValue( 14119.0, 0.705882, 0.235294, 0.705882 );
    table->SetTableValue( 14120.0, 0.0784314, 0.862745, 0.235294 );
    table->SetTableValue( 14121.0, 0.235294, 0.0784314, 0.54902 );
    table->SetTableValue( 14122.0, 0.862745, 0.705882, 0.54902 );
    table->SetTableValue( 14123.0, 0.254902, 0.392157, 0.0784314 );
    table->SetTableValue( 14124.0, 0.862745, 0.235294, 0.0784314 );
    table->SetTableValue( 14125.0, 0.0784314, 0.235294, 0.862745 );
    table->SetTableValue( 14126.0, 0.392157, 0.392157, 0.235294 );
    table->SetTableValue( 14127.0, 0.862745, 0.705882, 0.862745 );
    table->SetTableValue( 14128.0, 0.0784314, 0.705882, 0.54902 );
    table->SetTableValue( 14129.0, 0.235294, 0.54902, 0.705882 );
    table->SetTableValue( 14130.0, 0.0980392, 0.0784314, 0.54902 );
    table->SetTableValue( 14131.0, 0.0784314, 0.235294, 0.392157 );
    table->SetTableValue( 14132.0, 0.235294, 0.862745, 0.0784314 );
    table->SetTableValue( 14133.0, 0.235294, 0.235294, 0.862745 );
    table->SetTableValue( 14134.0, 0.862745, 0.235294, 0.862745 );
    table->SetTableValue( 14135.0, 0.254902, 0.862745, 0.235294 );
    table->SetTableValue( 14136.0, 0.0980392, 0.54902, 0.0784314 );
    table->SetTableValue( 14137.0, 0.862745, 0.862745, 0.392157 );
    table->SetTableValue( 14138.0, 0.705882, 0.235294, 0.235294 );
    table->SetTableValue( 14139.0, 0.239216, 0.0784314, 0.862745 );
    table->SetTableValue( 14140.0, 0.239216, 0.0784314, 0.235294 );
    table->SetTableValue( 14141.0, 0.239216, 0.235294, 0.392157 );
    table->SetTableValue( 14142.0, 0.0980392, 0.0980392, 0.0980392 );
    table->SetTableValue( 14143.0, 0.54902, 0.0784314, 0.235294 );
    table->SetTableValue( 14144.0, 0.862745, 0.705882, 0.0784314 );
    table->SetTableValue( 14145.0, 0.247059, 0.705882, 0.705882 );
    table->SetTableValue( 14146.0, 0.866667, 0.0784314, 0.0392157 );
    table->SetTableValue( 14147.0, 0.866667, 0.0784314, 0.392157 );
    table->SetTableValue( 14148.0, 0.866667, 0.235294, 0.54902 );
    table->SetTableValue( 14149.0, 0.866667, 0.0784314, 0.862745 );
    table->SetTableValue( 14150.0, 0.239216, 0.862745, 0.862745 );
    table->SetTableValue( 14151.0, 0.392157, 0.784314, 0.784314 );
    table->SetTableValue( 14152.0, 0.0392157, 0.784314, 0.784314 );
    table->SetTableValue( 14153.0, 0.866667, 0.862745, 0.0784314 );
    table->SetTableValue( 14154.0, 0.552941, 0.0784314, 0.392157 );
    table->SetTableValue( 14155.0, 0.239216, 0.862745, 0.392157 );
    table->SetTableValue( 14156.0, 0.552941, 0.235294, 0.0784314 );
    table->SetTableValue( 14157.0, 0.560784, 0.0784314, 0.862745 );
    table->SetTableValue( 14158.0, 0.396078, 0.235294, 0.862745 );
    table->SetTableValue( 14159.0, 0.0823529, 0.0784314, 0.54902 );
    table->SetTableValue( 14160.0, 0.239216, 0.0784314, 0.705882 );
    table->SetTableValue( 14161.0, 0.866667, 0.54902, 0.0784314 );
    table->SetTableValue( 14162.0, 0.552941, 0.392157, 0.862745 );
    table->SetTableValue( 14163.0, 0.866667, 0.392157, 0.0784314 );
    table->SetTableValue( 14164.0, 0.709804, 0.784314, 0.0784314 );
    table->SetTableValue( 14165.0, 0.396078, 0.0784314, 0.0784314 );
    table->SetTableValue( 14166.0, 0.396078, 0.392157, 0.705882 );
    table->SetTableValue( 14167.0, 0.709804, 0.862745, 0.0784314 );
    table->SetTableValue( 14168.0, 0.0823529, 0.54902, 0.784314 );
    table->SetTableValue( 14169.0, 0.0823529, 0.0784314, 0.941176 );
    table->SetTableValue( 14170.0, 0.0823529, 0.0784314, 0.784314 );
    table->SetTableValue( 14171.0, 0.0823529, 0.0784314, 0.235294 );
    table->SetTableValue( 14172.0, 0.396078, 0.235294, 0.235294 );
    table->SetTableValue( 14173.0, 0.0823529, 0.705882, 0.705882 );
    table->SetTableValue( 14174.0, 0.87451, 0.862745, 0.235294 );
    table->SetTableValue( 14175.0, 0.866667, 0.235294, 0.235294 );
    mitk::LookupTable::Pointer lookupTable = mitk::LookupTable::New();
    lookupTable->SetVtkLookupTable( table );
    return lookupTable;
}

TransferFunction::Pointer FreeSurferParcellationTranslator::CreateTransferFunction()
{
  mitk::TransferFunction::Pointer transferFunction = mitk::TransferFunction::New();
    transferFunction->AddRGBPoint( 1.0, 0.27451, 0.509804, 0.705882 );
    transferFunction->AddRGBPoint( 2.0, 0.960784, 0.960784, 0.960784 );
    transferFunction->AddRGBPoint( 3.0, 0.803922, 0.243137, 0.305882 );
    transferFunction->AddRGBPoint( 4.0, 0.470588, 0.0705882, 0.52549 );
    transferFunction->AddRGBPoint( 5.0, 0.768627, 0.227451, 0.980392 );
    transferFunction->AddRGBPoint( 6.0, 0, 0.580392, 0 );
    transferFunction->AddRGBPoint( 7.0, 0.862745, 0.972549, 0.643137 );
    transferFunction->AddRGBPoint( 8.0, 0.901961, 0.580392, 0.133333 );
    transferFunction->AddRGBPoint( 9.0, 0, 0.462745, 0.054902 );
    transferFunction->AddRGBPoint( 10.0, 0, 0.462745, 0.054902 );
    transferFunction->AddRGBPoint( 11.0, 0.478431, 0.729412, 0.862745 );
    transferFunction->AddRGBPoint( 12.0, 0.92549, 0.0509804, 0.690196 );
    transferFunction->AddRGBPoint( 13.0, 0.0470588, 0.188235, 1 );
    transferFunction->AddRGBPoint( 14.0, 0.8, 0.713725, 0.556863 );
    transferFunction->AddRGBPoint( 15.0, 0.164706, 0.8, 0.643137 );
    transferFunction->AddRGBPoint( 16.0, 0.466667, 0.623529, 0.690196 );
    transferFunction->AddRGBPoint( 17.0, 0.862745, 0.847059, 0.0784314 );
    transferFunction->AddRGBPoint( 18.0, 0.403922, 1, 1 );
    transferFunction->AddRGBPoint( 19.0, 0.313725, 0.768627, 0.384314 );
    transferFunction->AddRGBPoint( 20.0, 0.235294, 0.227451, 0.823529 );
    transferFunction->AddRGBPoint( 21.0, 0.235294, 0.227451, 0.823529 );
    transferFunction->AddRGBPoint( 22.0, 0.235294, 0.227451, 0.823529 );
    transferFunction->AddRGBPoint( 23.0, 0.235294, 0.227451, 0.823529 );
    transferFunction->AddRGBPoint( 24.0, 0.235294, 0.235294, 0.235294 );
    transferFunction->AddRGBPoint( 25.0, 1, 0.647059, 0 );
    transferFunction->AddRGBPoint( 26.0, 1, 0.647059, 0 );
    transferFunction->AddRGBPoint( 27.0, 0, 1, 0.498039 );
    transferFunction->AddRGBPoint( 28.0, 0.647059, 0.164706, 0.164706 );
    transferFunction->AddRGBPoint( 29.0, 0.529412, 0.807843, 0.921569 );
    transferFunction->AddRGBPoint( 30.0, 0.627451, 0.12549, 0.941176 );
    transferFunction->AddRGBPoint( 31.0, 0, 0.784314, 0.784314 );
    transferFunction->AddRGBPoint( 32.0, 0.392157, 0.196078, 0.392157 );
    transferFunction->AddRGBPoint( 33.0, 0.529412, 0.196078, 0.290196 );
    transferFunction->AddRGBPoint( 34.0, 0.478431, 0.529412, 0.196078 );
    transferFunction->AddRGBPoint( 35.0, 0.2, 0.196078, 0.529412 );
    transferFunction->AddRGBPoint( 36.0, 0.290196, 0.607843, 0.235294 );
    transferFunction->AddRGBPoint( 37.0, 0.470588, 0.243137, 0.168627 );
    transferFunction->AddRGBPoint( 38.0, 0.290196, 0.607843, 0.235294 );
    transferFunction->AddRGBPoint( 39.0, 0.478431, 0.529412, 0.196078 );
    transferFunction->AddRGBPoint( 40.0, 0.27451, 0.509804, 0.705882 );
    transferFunction->AddRGBPoint( 41.0, 0, 0.882353, 0 );
    transferFunction->AddRGBPoint( 42.0, 0.803922, 0.243137, 0.305882 );
    transferFunction->AddRGBPoint( 43.0, 0.470588, 0.0705882, 0.52549 );
    transferFunction->AddRGBPoint( 44.0, 0.768627, 0.227451, 0.980392 );
    transferFunction->AddRGBPoint( 45.0, 0, 0.580392, 0 );
    transferFunction->AddRGBPoint( 46.0, 0.862745, 0.972549, 0.643137 );
    transferFunction->AddRGBPoint( 47.0, 0.901961, 0.580392, 0.133333 );
    transferFunction->AddRGBPoint( 48.0, 0, 0.462745, 0.054902 );
    transferFunction->AddRGBPoint( 49.0, 0, 0.462745, 0.054902 );
    transferFunction->AddRGBPoint( 50.0, 0.478431, 0.729412, 0.862745 );
    transferFunction->AddRGBPoint( 51.0, 0.92549, 0.0509804, 0.690196 );
    transferFunction->AddRGBPoint( 52.0, 0.0509804, 0.188235, 1 );
    transferFunction->AddRGBPoint( 53.0, 0.862745, 0.847059, 0.0784314 );
    transferFunction->AddRGBPoint( 54.0, 0.403922, 1, 1 );
    transferFunction->AddRGBPoint( 55.0, 0.313725, 0.768627, 0.384314 );
    transferFunction->AddRGBPoint( 56.0, 0.235294, 0.227451, 0.823529 );
    transferFunction->AddRGBPoint( 57.0, 1, 0.647059, 0 );
    transferFunction->AddRGBPoint( 58.0, 1, 0.647059, 0 );
    transferFunction->AddRGBPoint( 59.0, 0, 1, 0.498039 );
    transferFunction->AddRGBPoint( 60.0, 0.647059, 0.164706, 0.164706 );
    transferFunction->AddRGBPoint( 61.0, 0.529412, 0.807843, 0.921569 );
    transferFunction->AddRGBPoint( 62.0, 0.627451, 0.12549, 0.941176 );
    transferFunction->AddRGBPoint( 63.0, 0, 0.784314, 0.866667 );
    transferFunction->AddRGBPoint( 64.0, 0.392157, 0.196078, 0.392157 );
    transferFunction->AddRGBPoint( 65.0, 0.529412, 0.196078, 0.290196 );
    transferFunction->AddRGBPoint( 66.0, 0.478431, 0.529412, 0.196078 );
    transferFunction->AddRGBPoint( 67.0, 0.2, 0.196078, 0.529412 );
    transferFunction->AddRGBPoint( 68.0, 0.290196, 0.607843, 0.235294 );
    transferFunction->AddRGBPoint( 69.0, 0.470588, 0.243137, 0.168627 );
    transferFunction->AddRGBPoint( 70.0, 0.290196, 0.607843, 0.235294 );
    transferFunction->AddRGBPoint( 71.0, 0.478431, 0.529412, 0.196078 );
    transferFunction->AddRGBPoint( 72.0, 0.470588, 0.745098, 0.588235 );
    transferFunction->AddRGBPoint( 73.0, 0.478431, 0.529412, 0.196078 );
    transferFunction->AddRGBPoint( 74.0, 0.478431, 0.529412, 0.196078 );
    transferFunction->AddRGBPoint( 77.0, 0.784314, 0.27451, 1 );
    transferFunction->AddRGBPoint( 78.0, 1, 0.580392, 0.0392157 );
    transferFunction->AddRGBPoint( 79.0, 1, 0.580392, 0.0392157 );
    transferFunction->AddRGBPoint( 80.0, 0.643137, 0.423529, 0.886275 );
    transferFunction->AddRGBPoint( 81.0, 0.643137, 0.423529, 0.886275 );
    transferFunction->AddRGBPoint( 82.0, 0.643137, 0.423529, 0.886275 );
    transferFunction->AddRGBPoint( 83.0, 1, 0.854902, 0.72549 );
    transferFunction->AddRGBPoint( 84.0, 1, 0.854902, 0.72549 );
    transferFunction->AddRGBPoint( 85.0, 0.917647, 0.662745, 0.117647 );
    transferFunction->AddRGBPoint( 192.0, 0.980392, 1, 0.196078 );
    transferFunction->AddRGBPoint( 86.0, 0.784314, 0.470588, 1 );
    transferFunction->AddRGBPoint( 87.0, 0.784314, 0.47451, 1 );
    transferFunction->AddRGBPoint( 88.0, 0.784314, 0.478431, 1 );
    transferFunction->AddRGBPoint( 96.0, 0.803922, 0.0392157, 0.490196 );
    transferFunction->AddRGBPoint( 97.0, 0.803922, 0.0392157, 0.490196 );
    transferFunction->AddRGBPoint( 98.0, 0.627451, 0.12549, 0.941176 );
    transferFunction->AddRGBPoint( 100.0, 0.486275, 0.54902, 0.698039 );
    transferFunction->AddRGBPoint( 101.0, 0.490196, 0.54902, 0.698039 );
    transferFunction->AddRGBPoint( 102.0, 0.494118, 0.54902, 0.698039 );
    transferFunction->AddRGBPoint( 103.0, 0.498039, 0.54902, 0.698039 );
    transferFunction->AddRGBPoint( 104.0, 0.486275, 0.552941, 0.698039 );
    transferFunction->AddRGBPoint( 105.0, 0.486275, 0.556863, 0.698039 );
    transferFunction->AddRGBPoint( 106.0, 0.486275, 0.560784, 0.698039 );
    transferFunction->AddRGBPoint( 107.0, 0.486275, 0.564706, 0.698039 );
    transferFunction->AddRGBPoint( 108.0, 0.486275, 0.54902, 0.701961 );
    transferFunction->AddRGBPoint( 109.0, 0.486275, 0.54902, 0.698039 );
    transferFunction->AddRGBPoint( 110.0, 0.490196, 0.54902, 0.698039 );
    transferFunction->AddRGBPoint( 111.0, 0.494118, 0.54902, 0.698039 );
    transferFunction->AddRGBPoint( 112.0, 0.498039, 0.54902, 0.698039 );
    transferFunction->AddRGBPoint( 113.0, 0.486275, 0.552941, 0.698039 );
    transferFunction->AddRGBPoint( 114.0, 0.486275, 0.556863, 0.698039 );
    transferFunction->AddRGBPoint( 115.0, 0.486275, 0.560784, 0.698039 );
    transferFunction->AddRGBPoint( 116.0, 0.486275, 0.564706, 0.698039 );
    transferFunction->AddRGBPoint( 117.0, 0.486275, 0.54902, 0.701961 );
    transferFunction->AddRGBPoint( 118.0, 1, 0.0784314, 0.576471 );
    transferFunction->AddRGBPoint( 119.0, 0.803922, 0.701961, 0.545098 );
    transferFunction->AddRGBPoint( 120.0, 0.933333, 0.933333, 0.819608 );
    transferFunction->AddRGBPoint( 121.0, 0.784314, 0.784314, 0.784314 );
    transferFunction->AddRGBPoint( 122.0, 0.290196, 1, 0.290196 );
    transferFunction->AddRGBPoint( 123.0, 0.933333, 0, 0 );
    transferFunction->AddRGBPoint( 124.0, 0, 0, 0.545098 );
    transferFunction->AddRGBPoint( 125.0, 0.678431, 1, 0.184314 );
    transferFunction->AddRGBPoint( 126.0, 0.521569, 0.796078, 0.898039 );
    transferFunction->AddRGBPoint( 127.0, 0.101961, 0.929412, 0.223529 );
    transferFunction->AddRGBPoint( 128.0, 0.133333, 0.545098, 0.133333 );
    transferFunction->AddRGBPoint( 129.0, 0.117647, 0.564706, 1 );
    transferFunction->AddRGBPoint( 130.0, 0.576471, 0.0745098, 0.678431 );
    transferFunction->AddRGBPoint( 131.0, 0.933333, 0.231373, 0.231373 );
    transferFunction->AddRGBPoint( 132.0, 0.866667, 0.152941, 0.784314 );
    transferFunction->AddRGBPoint( 133.0, 0.933333, 0.682353, 0.933333 );
    transferFunction->AddRGBPoint( 134.0, 1, 0, 0 );
    transferFunction->AddRGBPoint( 135.0, 0.282353, 0.239216, 0.545098 );
    transferFunction->AddRGBPoint( 136.0, 0.0823529, 0.152941, 0.517647 );
    transferFunction->AddRGBPoint( 137.0, 0.0823529, 0.152941, 0.517647 );
    transferFunction->AddRGBPoint( 138.0, 0.254902, 0.529412, 0.0784314 );
    transferFunction->AddRGBPoint( 139.0, 0.254902, 0.529412, 0.0784314 );
    transferFunction->AddRGBPoint( 140.0, 0.52549, 0.0156863, 0.627451 );
    transferFunction->AddRGBPoint( 142.0, 0.866667, 0.886275, 0.266667 );
    transferFunction->AddRGBPoint( 143.0, 1, 1, 0.996078 );
    transferFunction->AddRGBPoint( 144.0, 0.203922, 0.819608, 0.886275 );
    transferFunction->AddRGBPoint( 145.0, 0.937255, 0.627451, 0.87451 );
    transferFunction->AddRGBPoint( 146.0, 0.27451, 0.509804, 0.705882 );
    transferFunction->AddRGBPoint( 147.0, 0.27451, 0.509804, 0.709804 );
    transferFunction->AddRGBPoint( 148.0, 0.545098, 0.47451, 0.368627 );
    transferFunction->AddRGBPoint( 149.0, 0.878431, 0.878431, 0.878431 );
    transferFunction->AddRGBPoint( 150.0, 1, 0, 0 );
    transferFunction->AddRGBPoint( 151.0, 0.803922, 0.803922, 0 );
    transferFunction->AddRGBPoint( 152.0, 0.933333, 0.933333, 0.819608 );
    transferFunction->AddRGBPoint( 153.0, 0.545098, 0.47451, 0.368627 );
    transferFunction->AddRGBPoint( 154.0, 0.933333, 0.231373, 0.231373 );
    transferFunction->AddRGBPoint( 155.0, 0.933333, 0.231373, 0.231373 );
    transferFunction->AddRGBPoint( 156.0, 0.933333, 0.231373, 0.231373 );
    transferFunction->AddRGBPoint( 157.0, 0.243137, 0.0392157, 0.803922 );
    transferFunction->AddRGBPoint( 158.0, 0.243137, 0.0392157, 0.803922 );
    transferFunction->AddRGBPoint( 159.0, 0, 0.462745, 0.054902 );
    transferFunction->AddRGBPoint( 160.0, 0, 0.462745, 0.054902 );
    transferFunction->AddRGBPoint( 161.0, 0.862745, 0.847059, 0.0823529 );
    transferFunction->AddRGBPoint( 162.0, 0.862745, 0.847059, 0.0823529 );
    transferFunction->AddRGBPoint( 163.0, 0.478431, 0.729412, 0.862745 );
    transferFunction->AddRGBPoint( 164.0, 0.478431, 0.729412, 0.862745 );
    transferFunction->AddRGBPoint( 165.0, 1, 0.647059, 0 );
    transferFunction->AddRGBPoint( 166.0, 0.054902, 0.188235, 1 );
    transferFunction->AddRGBPoint( 167.0, 0.65098, 0.164706, 0.164706 );
    transferFunction->AddRGBPoint( 168.0, 0.47451, 0.0705882, 0.52549 );
    transferFunction->AddRGBPoint( 169.0, 0.92549, 0.0509804, 0.498039 );
    transferFunction->AddRGBPoint( 176.0, 0.92549, 0.0509804, 0.494118 );
    transferFunction->AddRGBPoint( 170.0, 0.466667, 0.623529, 0.690196 );
    transferFunction->AddRGBPoint( 171.0, 0.466667, 0, 0.690196 );
    transferFunction->AddRGBPoint( 172.0, 0.466667, 0.392157, 0.690196 );
    transferFunction->AddRGBPoint( 173.0, 0.466667, 0.784314, 0.690196 );
    transferFunction->AddRGBPoint( 174.0, 0.466667, 0.623529, 0.392157 );
    transferFunction->AddRGBPoint( 175.0, 0.466667, 0.623529, 0.784314 );
    transferFunction->AddRGBPoint( 180.0, 0.286275, 0.239216, 0.545098 );
    transferFunction->AddRGBPoint( 181.0, 0.286275, 0.243137, 0.545098 );
    transferFunction->AddRGBPoint( 193.0, 0, 0.768627, 1 );
    transferFunction->AddRGBPoint( 194.0, 1, 0.643137, 0.643137 );
    transferFunction->AddRGBPoint( 195.0, 0.768627, 0.768627, 0 );
    transferFunction->AddRGBPoint( 196.0, 0, 0.392157, 1 );
    transferFunction->AddRGBPoint( 197.0, 0.501961, 0.768627, 0.643137 );
    transferFunction->AddRGBPoint( 198.0, 0, 0.494118, 0.294118 );
    transferFunction->AddRGBPoint( 199.0, 0.501961, 0.376471, 0.25098 );
    transferFunction->AddRGBPoint( 200.0, 0, 0.196078, 0.501961 );
    transferFunction->AddRGBPoint( 201.0, 1, 0.8, 0.6 );
    transferFunction->AddRGBPoint( 202.0, 1, 0.501961, 0.501961 );
    transferFunction->AddRGBPoint( 203.0, 1, 1, 0 );
    transferFunction->AddRGBPoint( 204.0, 0.25098, 0, 0.25098 );
    transferFunction->AddRGBPoint( 205.0, 0, 0, 1 );
    transferFunction->AddRGBPoint( 206.0, 1, 0, 0 );
    transferFunction->AddRGBPoint( 207.0, 0.501961, 0.501961, 1 );
    transferFunction->AddRGBPoint( 208.0, 0, 0.501961, 0 );
    transferFunction->AddRGBPoint( 209.0, 0.768627, 0.627451, 0.501961 );
    transferFunction->AddRGBPoint( 210.0, 0.12549, 0.784314, 1 );
    transferFunction->AddRGBPoint( 211.0, 0.501961, 1, 0.501961 );
    transferFunction->AddRGBPoint( 212.0, 0.8, 0.6, 0.8 );
    transferFunction->AddRGBPoint( 213.0, 0.47451, 0.0666667, 0.533333 );
    transferFunction->AddRGBPoint( 214.0, 0.501961, 0, 0 );
    transferFunction->AddRGBPoint( 215.0, 0.501961, 0.12549, 1 );
    transferFunction->AddRGBPoint( 216.0, 1, 0.8, 0.4 );
    transferFunction->AddRGBPoint( 217.0, 0.501961, 0.501961, 0.501961 );
    transferFunction->AddRGBPoint( 218.0, 0.407843, 1, 1 );
    transferFunction->AddRGBPoint( 219.0, 0, 0.886275, 0 );
    transferFunction->AddRGBPoint( 220.0, 0.803922, 0.247059, 0.305882 );
    transferFunction->AddRGBPoint( 221.0, 0.772549, 0.227451, 0.980392 );
    transferFunction->AddRGBPoint( 222.0, 0.129412, 0.588235, 0.980392 );
    transferFunction->AddRGBPoint( 223.0, 0.886275, 0, 0 );
    transferFunction->AddRGBPoint( 224.0, 0.392157, 0.392157, 0.392157 );
    transferFunction->AddRGBPoint( 225.0, 0.772549, 0.588235, 0.980392 );
    transferFunction->AddRGBPoint( 226.0, 0.666667, 0.666667, 1 );
    transferFunction->AddRGBPoint( 250.0, 1, 0, 0 );
    transferFunction->AddRGBPoint( 251.0, 0, 0, 0.25098 );
    transferFunction->AddRGBPoint( 252.0, 0, 0, 0.439216 );
    transferFunction->AddRGBPoint( 253.0, 0, 0, 0.627451 );
    transferFunction->AddRGBPoint( 254.0, 0, 0, 0.815686 );
    transferFunction->AddRGBPoint( 255.0, 0, 0, 1 );
    transferFunction->AddRGBPoint( 331.0, 1, 0, 0 );
    transferFunction->AddRGBPoint( 332.0, 1, 0.313725, 0 );
    transferFunction->AddRGBPoint( 333.0, 1, 0.627451, 0 );
    transferFunction->AddRGBPoint( 334.0, 1, 1, 0 );
    transferFunction->AddRGBPoint( 335.0, 0, 1, 0 );
    transferFunction->AddRGBPoint( 336.0, 1, 0, 0.627451 );
    transferFunction->AddRGBPoint( 337.0, 1, 0, 1 );
    transferFunction->AddRGBPoint( 338.0, 1, 0.196078, 0.313725 );
    transferFunction->AddRGBPoint( 339.0, 0.313725, 1, 0.196078 );
    transferFunction->AddRGBPoint( 340.0, 0.627451, 1, 0.196078 );
    transferFunction->AddRGBPoint( 341.0, 0.627451, 0.784314, 1 );
    transferFunction->AddRGBPoint( 342.0, 0, 1, 0.627451 );
    transferFunction->AddRGBPoint( 343.0, 0, 0, 1 );
    transferFunction->AddRGBPoint( 344.0, 0.313725, 0.196078, 1 );
    transferFunction->AddRGBPoint( 345.0, 0.627451, 0, 1 );
    transferFunction->AddRGBPoint( 346.0, 1, 0.823529, 0 );
    transferFunction->AddRGBPoint( 347.0, 0, 0.627451, 1 );
    transferFunction->AddRGBPoint( 348.0, 1, 0.784314, 0.313725 );
    transferFunction->AddRGBPoint( 349.0, 1, 0.784314, 0.627451 );
    transferFunction->AddRGBPoint( 350.0, 1, 0.313725, 0.784314 );
    transferFunction->AddRGBPoint( 351.0, 1, 0.627451, 0.784314 );
    transferFunction->AddRGBPoint( 352.0, 0.117647, 1, 0.313725 );
    transferFunction->AddRGBPoint( 353.0, 0.313725, 0.784314, 1 );
    transferFunction->AddRGBPoint( 354.0, 0.313725, 1, 0.784314 );
    transferFunction->AddRGBPoint( 355.0, 0.764706, 1, 0.784314 );
    transferFunction->AddRGBPoint( 356.0, 0.470588, 0.784314, 0.0784314 );
    transferFunction->AddRGBPoint( 357.0, 0.666667, 0.0392157, 0.784314 );
    transferFunction->AddRGBPoint( 358.0, 0.0784314, 0.509804, 0.705882 );
    transferFunction->AddRGBPoint( 359.0, 0.0784314, 0.705882, 0.509804 );
    transferFunction->AddRGBPoint( 400.0, 0.807843, 0.243137, 0.305882 );
    transferFunction->AddRGBPoint( 401.0, 0.47451, 0.0705882, 0.52549 );
    transferFunction->AddRGBPoint( 402.0, 0.780392, 0.227451, 0.980392 );
    transferFunction->AddRGBPoint( 403.0, 0.00392157, 0.580392, 0 );
    transferFunction->AddRGBPoint( 404.0, 0.866667, 0.972549, 0.643137 );
    transferFunction->AddRGBPoint( 405.0, 0.905882, 0.580392, 0.133333 );
    transferFunction->AddRGBPoint( 406.0, 0.00392157, 0.462745, 0.054902 );
    transferFunction->AddRGBPoint( 407.0, 0.470588, 0.462745, 0.054902 );
    transferFunction->AddRGBPoint( 408.0, 0.482353, 0.729412, 0.866667 );
    transferFunction->AddRGBPoint( 409.0, 0.933333, 0.0509804, 0.694118 );
    transferFunction->AddRGBPoint( 410.0, 0.482353, 0.729412, 0.862745 );
    transferFunction->AddRGBPoint( 411.0, 0.541176, 0.0509804, 0.807843 );
    transferFunction->AddRGBPoint( 412.0, 0.933333, 0.509804, 0.690196 );
    transferFunction->AddRGBPoint( 413.0, 0.854902, 0.901961, 0.298039 );
    transferFunction->AddRGBPoint( 414.0, 0.14902, 0.835294, 0.690196 );
    transferFunction->AddRGBPoint( 415.0, 0.00392157, 0.882353, 0.690196 );
    transferFunction->AddRGBPoint( 416.0, 0.00392157, 0.882353, 0.690196 );
    transferFunction->AddRGBPoint( 417.0, 0.784314, 0.00784314, 0.392157 );
    transferFunction->AddRGBPoint( 418.0, 0.784314, 0.00784314, 0.392157 );
    transferFunction->AddRGBPoint( 419.0, 0.0196078, 0.784314, 0.352941 );
    transferFunction->AddRGBPoint( 420.0, 0.0196078, 0.784314, 0.352941 );
    transferFunction->AddRGBPoint( 421.0, 0.392157, 0.0196078, 0.784314 );
    transferFunction->AddRGBPoint( 422.0, 0.0980392, 1, 0.392157 );
    transferFunction->AddRGBPoint( 423.0, 0.0980392, 1, 0.392157 );
    transferFunction->AddRGBPoint( 424.0, 0.901961, 0.027451, 0.392157 );
    transferFunction->AddRGBPoint( 425.0, 0.901961, 0.027451, 0.392157 );
    transferFunction->AddRGBPoint( 426.0, 0.392157, 0.0196078, 0.784314 );
    transferFunction->AddRGBPoint( 427.0, 0.588235, 0.0392157, 0.784314 );
    transferFunction->AddRGBPoint( 428.0, 0.588235, 0.0392157, 0.784314 );
    transferFunction->AddRGBPoint( 429.0, 0.686275, 0.0392157, 0.690196 );
    transferFunction->AddRGBPoint( 430.0, 0.686275, 0.0392157, 0.690196 );
    transferFunction->AddRGBPoint( 431.0, 0.0392157, 0.392157, 1 );
    transferFunction->AddRGBPoint( 432.0, 0.0392157, 0.392157, 1 );
    transferFunction->AddRGBPoint( 433.0, 0.588235, 0.176471, 0.27451 );
    transferFunction->AddRGBPoint( 434.0, 0.588235, 0.176471, 0.27451 );
    transferFunction->AddRGBPoint( 435.0, 0.176471, 0.784314, 0.0588235 );
    transferFunction->AddRGBPoint( 436.0, 0.176471, 0.784314, 0.0588235 );
    transferFunction->AddRGBPoint( 437.0, 0.890196, 0.176471, 0.392157 );
    transferFunction->AddRGBPoint( 438.0, 0.890196, 0.176471, 0.392157 );
    transferFunction->AddRGBPoint( 439.0, 0.890196, 0.176471, 0.392157 );
    transferFunction->AddRGBPoint( 498.0, 0.560784, 0.737255, 0.560784 );
    transferFunction->AddRGBPoint( 499.0, 1, 0.972549, 0.862745 );
    transferFunction->AddRGBPoint( 500.0, 0.0666667, 0.333333, 0.533333 );
    transferFunction->AddRGBPoint( 501.0, 0.466667, 0.733333, 0.4 );
    transferFunction->AddRGBPoint( 502.0, 0.8, 0.266667, 0.133333 );
    transferFunction->AddRGBPoint( 503.0, 0.8, 0, 1 );
    transferFunction->AddRGBPoint( 504.0, 0.866667, 0.733333, 0.0666667 );
    transferFunction->AddRGBPoint( 505.0, 0.6, 0.866667, 0.933333 );
    transferFunction->AddRGBPoint( 506.0, 0.2, 0.0666667, 0.0666667 );
    transferFunction->AddRGBPoint( 507.0, 0, 0.466667, 0.333333 );
    transferFunction->AddRGBPoint( 508.0, 0.0784314, 0.392157, 0.784314 );
    transferFunction->AddRGBPoint( 550.0, 0.0666667, 0.333333, 0.537255 );
    transferFunction->AddRGBPoint( 551.0, 0.466667, 0.733333, 0.403922 );
    transferFunction->AddRGBPoint( 552.0, 0.8, 0.266667, 0.137255 );
    transferFunction->AddRGBPoint( 553.0, 0.8, 0, 0.996078 );
    transferFunction->AddRGBPoint( 554.0, 0.866667, 0.733333, 0.0627451 );
    transferFunction->AddRGBPoint( 555.0, 0.6, 0.866667, 0.937255 );
    transferFunction->AddRGBPoint( 556.0, 0.2, 0.0666667, 0.0705882 );
    transferFunction->AddRGBPoint( 557.0, 0, 0.466667, 0.337255 );
    transferFunction->AddRGBPoint( 558.0, 0.0784314, 0.392157, 0.788235 );
    transferFunction->AddRGBPoint( 600.0, 0.996078, 0.996078, 0.996078 );
    transferFunction->AddRGBPoint( 601.0, 0.27451, 0.509804, 0.705882 );
    transferFunction->AddRGBPoint( 602.0, 0.960784, 0.960784, 0.960784 );
    transferFunction->AddRGBPoint( 603.0, 0.803922, 0.243137, 0.305882 );
    transferFunction->AddRGBPoint( 604.0, 0.470588, 0.0705882, 0.52549 );
    transferFunction->AddRGBPoint( 605.0, 0.768627, 0.227451, 0.980392 );
    transferFunction->AddRGBPoint( 606.0, 0, 0.580392, 0 );
    transferFunction->AddRGBPoint( 607.0, 0.862745, 0.972549, 0.643137 );
    transferFunction->AddRGBPoint( 608.0, 0.901961, 0.580392, 0.133333 );
    transferFunction->AddRGBPoint( 609.0, 0, 0.462745, 0.054902 );
    transferFunction->AddRGBPoint( 610.0, 0, 0.462745, 0.054902 );
    transferFunction->AddRGBPoint( 611.0, 0.478431, 0.729412, 0.862745 );
    transferFunction->AddRGBPoint( 612.0, 0.92549, 0.0509804, 0.690196 );
    transferFunction->AddRGBPoint( 613.0, 0.0470588, 0.188235, 1 );
    transferFunction->AddRGBPoint( 614.0, 0.8, 0.713725, 0.556863 );
    transferFunction->AddRGBPoint( 615.0, 0.164706, 0.8, 0.643137 );
    transferFunction->AddRGBPoint( 616.0, 0.466667, 0.623529, 0.690196 );
    transferFunction->AddRGBPoint( 617.0, 0.862745, 0.847059, 0.0784314 );
    transferFunction->AddRGBPoint( 618.0, 0.403922, 1, 1 );
    transferFunction->AddRGBPoint( 619.0, 0.313725, 0.768627, 0.384314 );
    transferFunction->AddRGBPoint( 620.0, 0.235294, 0.227451, 0.823529 );
    transferFunction->AddRGBPoint( 621.0, 0.235294, 0.227451, 0.823529 );
    transferFunction->AddRGBPoint( 622.0, 0.235294, 0.227451, 0.823529 );
    transferFunction->AddRGBPoint( 623.0, 0.235294, 0.227451, 0.823529 );
    transferFunction->AddRGBPoint( 624.0, 0.235294, 0.235294, 0.235294 );
    transferFunction->AddRGBPoint( 625.0, 1, 0.647059, 0 );
    transferFunction->AddRGBPoint( 626.0, 1, 0.647059, 0 );
    transferFunction->AddRGBPoint( 627.0, 0, 1, 0.498039 );
    transferFunction->AddRGBPoint( 628.0, 0.647059, 0.164706, 0.164706 );
    transferFunction->AddRGBPoint( 640.0, 0.8, 0, 0 );
    transferFunction->AddRGBPoint( 641.0, 1, 0, 0 );
    transferFunction->AddRGBPoint( 642.0, 0, 0, 1 );
    transferFunction->AddRGBPoint( 643.0, 0.117647, 0.564706, 1 );
    transferFunction->AddRGBPoint( 644.0, 0.392157, 0.831373, 0.929412 );
    transferFunction->AddRGBPoint( 645.0, 0.854902, 0.647059, 0.12549 );
    transferFunction->AddRGBPoint( 646.0, 1, 0.843137, 0 );
    transferFunction->AddRGBPoint( 647.0, 1, 1, 0.65098 );
    transferFunction->AddRGBPoint( 648.0, 0.6, 0, 0.8 );
    transferFunction->AddRGBPoint( 649.0, 0.6, 0.552941, 0.819608 );
    transferFunction->AddRGBPoint( 650.0, 0.8, 0.8, 1 );
    transferFunction->AddRGBPoint( 651.0, 0.121569, 0.831373, 0.760784 );
    transferFunction->AddRGBPoint( 652.0, 0.0117647, 1, 0.929412 );
    transferFunction->AddRGBPoint( 653.0, 0.8, 1, 1 );
    transferFunction->AddRGBPoint( 654.0, 0.337255, 0.290196, 0.576471 );
    transferFunction->AddRGBPoint( 655.0, 0.447059, 0.447059, 0.745098 );
    transferFunction->AddRGBPoint( 656.0, 0.721569, 0.698039, 1 );
    transferFunction->AddRGBPoint( 657.0, 0.494118, 0.541176, 0.145098 );
    transferFunction->AddRGBPoint( 658.0, 0.741176, 0.772549, 0.458824 );
    transferFunction->AddRGBPoint( 659.0, 0.941176, 0.901961, 0.54902 );
    transferFunction->AddRGBPoint( 660.0, 0.8, 0, 0 );
    transferFunction->AddRGBPoint( 661.0, 1, 0, 0 );
    transferFunction->AddRGBPoint( 662.0, 0, 0, 1 );
    transferFunction->AddRGBPoint( 663.0, 0.117647, 0.564706, 1 );
    transferFunction->AddRGBPoint( 664.0, 0.392157, 0.831373, 0.929412 );
    transferFunction->AddRGBPoint( 665.0, 0.854902, 0.647059, 0.12549 );
    transferFunction->AddRGBPoint( 666.0, 1, 0.843137, 0 );
    transferFunction->AddRGBPoint( 667.0, 1, 1, 0.65098 );
    transferFunction->AddRGBPoint( 668.0, 0.6, 0, 0.8 );
    transferFunction->AddRGBPoint( 669.0, 0.6, 0.552941, 0.819608 );
    transferFunction->AddRGBPoint( 670.0, 0.8, 0.8, 1 );
    transferFunction->AddRGBPoint( 671.0, 0.121569, 0.831373, 0.760784 );
    transferFunction->AddRGBPoint( 672.0, 0.0117647, 1, 0.929412 );
    transferFunction->AddRGBPoint( 673.0, 0.8, 1, 1 );
    transferFunction->AddRGBPoint( 674.0, 0.337255, 0.290196, 0.576471 );
    transferFunction->AddRGBPoint( 675.0, 0.447059, 0.447059, 0.745098 );
    transferFunction->AddRGBPoint( 676.0, 0.721569, 0.698039, 1 );
    transferFunction->AddRGBPoint( 677.0, 0.494118, 0.541176, 0.145098 );
    transferFunction->AddRGBPoint( 678.0, 0.741176, 0.772549, 0.458824 );
    transferFunction->AddRGBPoint( 679.0, 0.941176, 0.901961, 0.54902 );
    transferFunction->AddRGBPoint( 701.0, 0.470588, 0.0705882, 0.52549 );
    transferFunction->AddRGBPoint( 702.0, 0.803922, 0.243137, 0.305882 );
    transferFunction->AddRGBPoint( 703.0, 0, 0.882353, 0 );
    transferFunction->AddRGBPoint( 999.0, 1, 0.392157, 0.392157 );
    transferFunction->AddRGBPoint( 1000.0, 0.0980392, 0.0196078, 0.0980392 );
    transferFunction->AddRGBPoint( 1001.0, 0.0980392, 0.392157, 0.156863 );
    transferFunction->AddRGBPoint( 1002.0, 0.490196, 0.392157, 0.627451 );
    transferFunction->AddRGBPoint( 1003.0, 0.392157, 0.0980392, 0 );
    transferFunction->AddRGBPoint( 1004.0, 0.470588, 0.27451, 0.196078 );
    transferFunction->AddRGBPoint( 1005.0, 0.862745, 0.0784314, 0.392157 );
    transferFunction->AddRGBPoint( 1006.0, 0.862745, 0.0784314, 0.0392157 );
    transferFunction->AddRGBPoint( 1007.0, 0.705882, 0.862745, 0.54902 );
    transferFunction->AddRGBPoint( 1008.0, 0.862745, 0.235294, 0.862745 );
    transferFunction->AddRGBPoint( 1009.0, 0.705882, 0.156863, 0.470588 );
    transferFunction->AddRGBPoint( 1010.0, 0.54902, 0.0784314, 0.54902 );
    transferFunction->AddRGBPoint( 1011.0, 0.0784314, 0.117647, 0.54902 );
    transferFunction->AddRGBPoint( 1012.0, 0.137255, 0.294118, 0.196078 );
    transferFunction->AddRGBPoint( 1013.0, 0.882353, 0.54902, 0.54902 );
    transferFunction->AddRGBPoint( 1014.0, 0.784314, 0.137255, 0.294118 );
    transferFunction->AddRGBPoint( 1015.0, 0.627451, 0.392157, 0.196078 );
    transferFunction->AddRGBPoint( 1016.0, 0.0784314, 0.862745, 0.235294 );
    transferFunction->AddRGBPoint( 1017.0, 0.235294, 0.862745, 0.235294 );
    transferFunction->AddRGBPoint( 1018.0, 0.862745, 0.705882, 0.54902 );
    transferFunction->AddRGBPoint( 1019.0, 0.0784314, 0.392157, 0.196078 );
    transferFunction->AddRGBPoint( 1020.0, 0.862745, 0.235294, 0.0784314 );
    transferFunction->AddRGBPoint( 1021.0, 0.470588, 0.392157, 0.235294 );
    transferFunction->AddRGBPoint( 1022.0, 0.862745, 0.0784314, 0.0784314 );
    transferFunction->AddRGBPoint( 1023.0, 0.862745, 0.705882, 0.862745 );
    transferFunction->AddRGBPoint( 1024.0, 0.235294, 0.0784314, 0.862745 );
    transferFunction->AddRGBPoint( 1025.0, 0.627451, 0.54902, 0.705882 );
    transferFunction->AddRGBPoint( 1026.0, 0.313725, 0.0784314, 0.54902 );
    transferFunction->AddRGBPoint( 1027.0, 0.294118, 0.196078, 0.490196 );
    transferFunction->AddRGBPoint( 1028.0, 0.0784314, 0.862745, 0.627451 );
    transferFunction->AddRGBPoint( 1029.0, 0.0784314, 0.705882, 0.54902 );
    transferFunction->AddRGBPoint( 1030.0, 0.54902, 0.862745, 0.862745 );
    transferFunction->AddRGBPoint( 1031.0, 0.313725, 0.627451, 0.0784314 );
    transferFunction->AddRGBPoint( 1032.0, 0.392157, 0, 0.392157 );
    transferFunction->AddRGBPoint( 1033.0, 0.27451, 0.27451, 0.27451 );
    transferFunction->AddRGBPoint( 1034.0, 0.588235, 0.588235, 0.784314 );
    transferFunction->AddRGBPoint( 1035.0, 1, 0.752941, 0.12549 );
    transferFunction->AddRGBPoint( 2000.0, 0.0980392, 0.0196078, 0.0980392 );
    transferFunction->AddRGBPoint( 2001.0, 0.0980392, 0.392157, 0.156863 );
    transferFunction->AddRGBPoint( 2002.0, 0.490196, 0.392157, 0.627451 );
    transferFunction->AddRGBPoint( 2003.0, 0.392157, 0.0980392, 0 );
    transferFunction->AddRGBPoint( 2004.0, 0.470588, 0.27451, 0.196078 );
    transferFunction->AddRGBPoint( 2005.0, 0.862745, 0.0784314, 0.392157 );
    transferFunction->AddRGBPoint( 2006.0, 0.862745, 0.0784314, 0.0392157 );
    transferFunction->AddRGBPoint( 2007.0, 0.705882, 0.862745, 0.54902 );
    transferFunction->AddRGBPoint( 2008.0, 0.862745, 0.235294, 0.862745 );
    transferFunction->AddRGBPoint( 2009.0, 0.705882, 0.156863, 0.470588 );
    transferFunction->AddRGBPoint( 2010.0, 0.54902, 0.0784314, 0.54902 );
    transferFunction->AddRGBPoint( 2011.0, 0.0784314, 0.117647, 0.54902 );
    transferFunction->AddRGBPoint( 2012.0, 0.137255, 0.294118, 0.196078 );
    transferFunction->AddRGBPoint( 2013.0, 0.882353, 0.54902, 0.54902 );
    transferFunction->AddRGBPoint( 2014.0, 0.784314, 0.137255, 0.294118 );
    transferFunction->AddRGBPoint( 2015.0, 0.627451, 0.392157, 0.196078 );
    transferFunction->AddRGBPoint( 2016.0, 0.0784314, 0.862745, 0.235294 );
    transferFunction->AddRGBPoint( 2017.0, 0.235294, 0.862745, 0.235294 );
    transferFunction->AddRGBPoint( 2018.0, 0.862745, 0.705882, 0.54902 );
    transferFunction->AddRGBPoint( 2019.0, 0.0784314, 0.392157, 0.196078 );
    transferFunction->AddRGBPoint( 2020.0, 0.862745, 0.235294, 0.0784314 );
    transferFunction->AddRGBPoint( 2021.0, 0.470588, 0.392157, 0.235294 );
    transferFunction->AddRGBPoint( 2022.0, 0.862745, 0.0784314, 0.0784314 );
    transferFunction->AddRGBPoint( 2023.0, 0.862745, 0.705882, 0.862745 );
    transferFunction->AddRGBPoint( 2024.0, 0.235294, 0.0784314, 0.862745 );
    transferFunction->AddRGBPoint( 2025.0, 0.627451, 0.54902, 0.705882 );
    transferFunction->AddRGBPoint( 2026.0, 0.313725, 0.0784314, 0.54902 );
    transferFunction->AddRGBPoint( 2027.0, 0.294118, 0.196078, 0.490196 );
    transferFunction->AddRGBPoint( 2028.0, 0.0784314, 0.862745, 0.627451 );
    transferFunction->AddRGBPoint( 2029.0, 0.0784314, 0.705882, 0.54902 );
    transferFunction->AddRGBPoint( 2030.0, 0.54902, 0.862745, 0.862745 );
    transferFunction->AddRGBPoint( 2031.0, 0.313725, 0.627451, 0.0784314 );
    transferFunction->AddRGBPoint( 2032.0, 0.392157, 0, 0.392157 );
    transferFunction->AddRGBPoint( 2033.0, 0.27451, 0.27451, 0.27451 );
    transferFunction->AddRGBPoint( 2034.0, 0.588235, 0.588235, 0.784314 );
    transferFunction->AddRGBPoint( 2035.0, 1, 0.752941, 0.12549 );
    transferFunction->AddRGBPoint( 3000.0, 0.901961, 0.980392, 0.901961 );
    transferFunction->AddRGBPoint( 3001.0, 0.901961, 0.607843, 0.843137 );
    transferFunction->AddRGBPoint( 3002.0, 0.509804, 0.607843, 0.372549 );
    transferFunction->AddRGBPoint( 3003.0, 0.607843, 0.901961, 1 );
    transferFunction->AddRGBPoint( 3004.0, 0.529412, 0.72549, 0.803922 );
    transferFunction->AddRGBPoint( 3005.0, 0.137255, 0.921569, 0.607843 );
    transferFunction->AddRGBPoint( 3006.0, 0.137255, 0.921569, 0.960784 );
    transferFunction->AddRGBPoint( 3007.0, 0.294118, 0.137255, 0.45098 );
    transferFunction->AddRGBPoint( 3008.0, 0.137255, 0.764706, 0.137255 );
    transferFunction->AddRGBPoint( 3009.0, 0.294118, 0.843137, 0.529412 );
    transferFunction->AddRGBPoint( 3010.0, 0.45098, 0.921569, 0.45098 );
    transferFunction->AddRGBPoint( 3011.0, 0.921569, 0.882353, 0.45098 );
    transferFunction->AddRGBPoint( 3012.0, 0.862745, 0.705882, 0.803922 );
    transferFunction->AddRGBPoint( 3013.0, 0.117647, 0.45098, 0.45098 );
    transferFunction->AddRGBPoint( 3014.0, 0.215686, 0.862745, 0.705882 );
    transferFunction->AddRGBPoint( 3015.0, 0.372549, 0.607843, 0.803922 );
    transferFunction->AddRGBPoint( 3016.0, 0.921569, 0.137255, 0.764706 );
    transferFunction->AddRGBPoint( 3017.0, 0.764706, 0.137255, 0.764706 );
    transferFunction->AddRGBPoint( 3018.0, 0.137255, 0.294118, 0.45098 );
    transferFunction->AddRGBPoint( 3019.0, 0.921569, 0.607843, 0.803922 );
    transferFunction->AddRGBPoint( 3020.0, 0.137255, 0.764706, 0.921569 );
    transferFunction->AddRGBPoint( 3021.0, 0.529412, 0.607843, 0.764706 );
    transferFunction->AddRGBPoint( 3022.0, 0.137255, 0.921569, 0.921569 );
    transferFunction->AddRGBPoint( 3023.0, 0.137255, 0.294118, 0.137255 );
    transferFunction->AddRGBPoint( 3024.0, 0.764706, 0.921569, 0.137255 );
    transferFunction->AddRGBPoint( 3025.0, 0.372549, 0.45098, 0.294118 );
    transferFunction->AddRGBPoint( 3026.0, 0.686275, 0.921569, 0.45098 );
    transferFunction->AddRGBPoint( 3027.0, 0.705882, 0.803922, 0.509804 );
    transferFunction->AddRGBPoint( 3028.0, 0.921569, 0.137255, 0.372549 );
    transferFunction->AddRGBPoint( 3029.0, 0.921569, 0.294118, 0.45098 );
    transferFunction->AddRGBPoint( 3030.0, 0.45098, 0.137255, 0.137255 );
    transferFunction->AddRGBPoint( 3031.0, 0.686275, 0.372549, 0.921569 );
    transferFunction->AddRGBPoint( 3032.0, 0.607843, 1, 0.607843 );
    transferFunction->AddRGBPoint( 3033.0, 0.72549, 0.72549, 0.72549 );
    transferFunction->AddRGBPoint( 3034.0, 0.411765, 0.411765, 0.215686 );
    transferFunction->AddRGBPoint( 3035.0, 0.996078, 0.74902, 0.121569 );
    transferFunction->AddRGBPoint( 4000.0, 0.901961, 0.980392, 0.901961 );
    transferFunction->AddRGBPoint( 4001.0, 0.901961, 0.607843, 0.843137 );
    transferFunction->AddRGBPoint( 4002.0, 0.509804, 0.607843, 0.372549 );
    transferFunction->AddRGBPoint( 4003.0, 0.607843, 0.901961, 1 );
    transferFunction->AddRGBPoint( 4004.0, 0.529412, 0.72549, 0.803922 );
    transferFunction->AddRGBPoint( 4005.0, 0.137255, 0.921569, 0.607843 );
    transferFunction->AddRGBPoint( 4006.0, 0.137255, 0.921569, 0.960784 );
    transferFunction->AddRGBPoint( 4007.0, 0.294118, 0.137255, 0.45098 );
    transferFunction->AddRGBPoint( 4008.0, 0.137255, 0.764706, 0.137255 );
    transferFunction->AddRGBPoint( 4009.0, 0.294118, 0.843137, 0.529412 );
    transferFunction->AddRGBPoint( 4010.0, 0.45098, 0.921569, 0.45098 );
    transferFunction->AddRGBPoint( 4011.0, 0.921569, 0.882353, 0.45098 );
    transferFunction->AddRGBPoint( 4012.0, 0.862745, 0.705882, 0.803922 );
    transferFunction->AddRGBPoint( 4013.0, 0.117647, 0.45098, 0.45098 );
    transferFunction->AddRGBPoint( 4014.0, 0.215686, 0.862745, 0.705882 );
    transferFunction->AddRGBPoint( 4015.0, 0.372549, 0.607843, 0.803922 );
    transferFunction->AddRGBPoint( 4016.0, 0.921569, 0.137255, 0.764706 );
    transferFunction->AddRGBPoint( 4017.0, 0.764706, 0.137255, 0.764706 );
    transferFunction->AddRGBPoint( 4018.0, 0.137255, 0.294118, 0.45098 );
    transferFunction->AddRGBPoint( 4019.0, 0.921569, 0.607843, 0.803922 );
    transferFunction->AddRGBPoint( 4020.0, 0.137255, 0.764706, 0.921569 );
    transferFunction->AddRGBPoint( 4021.0, 0.529412, 0.607843, 0.764706 );
    transferFunction->AddRGBPoint( 4022.0, 0.137255, 0.921569, 0.921569 );
    transferFunction->AddRGBPoint( 4023.0, 0.137255, 0.294118, 0.137255 );
    transferFunction->AddRGBPoint( 4024.0, 0.764706, 0.921569, 0.137255 );
    transferFunction->AddRGBPoint( 4025.0, 0.372549, 0.45098, 0.294118 );
    transferFunction->AddRGBPoint( 4026.0, 0.686275, 0.921569, 0.45098 );
    transferFunction->AddRGBPoint( 4027.0, 0.705882, 0.803922, 0.509804 );
    transferFunction->AddRGBPoint( 4028.0, 0.921569, 0.137255, 0.372549 );
    transferFunction->AddRGBPoint( 4029.0, 0.921569, 0.294118, 0.45098 );
    transferFunction->AddRGBPoint( 4030.0, 0.45098, 0.137255, 0.137255 );
    transferFunction->AddRGBPoint( 4031.0, 0.686275, 0.372549, 0.921569 );
    transferFunction->AddRGBPoint( 4032.0, 0.607843, 1, 0.607843 );
    transferFunction->AddRGBPoint( 4033.0, 0.72549, 0.72549, 0.72549 );
    transferFunction->AddRGBPoint( 4034.0, 0.411765, 0.411765, 0.215686 );
    transferFunction->AddRGBPoint( 4035.0, 0.996078, 0.74902, 0.121569 );
    transferFunction->AddRGBPoint( 1101.0, 0.196078, 0.196078, 0.196078 );
    transferFunction->AddRGBPoint( 1102.0, 0.705882, 0.0784314, 0.117647 );
    transferFunction->AddRGBPoint( 1103.0, 0.235294, 0.0980392, 0.0980392 );
    transferFunction->AddRGBPoint( 1104.0, 0.0980392, 0.235294, 0.235294 );
    transferFunction->AddRGBPoint( 1200.0, 0.0980392, 0.235294, 0.239216 );
    transferFunction->AddRGBPoint( 1201.0, 0.0980392, 0.352941, 0.235294 );
    transferFunction->AddRGBPoint( 1202.0, 0.0980392, 0.470588, 0.235294 );
    transferFunction->AddRGBPoint( 1205.0, 0.0980392, 0.588235, 0.235294 );
    transferFunction->AddRGBPoint( 1206.0, 0.0980392, 0.705882, 0.235294 );
    transferFunction->AddRGBPoint( 1207.0, 0.0980392, 0.823529, 0.235294 );
    transferFunction->AddRGBPoint( 1210.0, 0.0980392, 0.588235, 0.352941 );
    transferFunction->AddRGBPoint( 1211.0, 0.0980392, 0.705882, 0.352941 );
    transferFunction->AddRGBPoint( 1212.0, 0.0980392, 0.823529, 0.352941 );
    transferFunction->AddRGBPoint( 1105.0, 0.705882, 0.0784314, 0.0784314 );
    transferFunction->AddRGBPoint( 1106.0, 0.862745, 0.0784314, 0.392157 );
    transferFunction->AddRGBPoint( 1107.0, 0.54902, 0.235294, 0.235294 );
    transferFunction->AddRGBPoint( 1108.0, 0.705882, 0.862745, 0.54902 );
    transferFunction->AddRGBPoint( 1109.0, 0.54902, 0.392157, 0.705882 );
    transferFunction->AddRGBPoint( 1110.0, 0.705882, 0.0784314, 0.54902 );
    transferFunction->AddRGBPoint( 1111.0, 0.54902, 0.0784314, 0.54902 );
    transferFunction->AddRGBPoint( 1112.0, 0.0823529, 0.0392157, 0.0392157 );
    transferFunction->AddRGBPoint( 1113.0, 0.882353, 0.54902, 0.54902 );
    transferFunction->AddRGBPoint( 1114.0, 0.0901961, 0.235294, 0.705882 );
    transferFunction->AddRGBPoint( 1115.0, 0.705882, 0.235294, 0.705882 );
    transferFunction->AddRGBPoint( 1116.0, 0.0784314, 0.862745, 0.235294 );
    transferFunction->AddRGBPoint( 1117.0, 0.235294, 0.0784314, 0.54902 );
    transferFunction->AddRGBPoint( 1118.0, 0.862745, 0.705882, 0.54902 );
    transferFunction->AddRGBPoint( 1119.0, 0.254902, 0.392157, 0.0784314 );
    transferFunction->AddRGBPoint( 1120.0, 0.862745, 0.235294, 0.0784314 );
    transferFunction->AddRGBPoint( 1121.0, 0.235294, 0.392157, 0.235294 );
    transferFunction->AddRGBPoint( 1122.0, 0.0784314, 0.235294, 0.862745 );
    transferFunction->AddRGBPoint( 1123.0, 0.392157, 0.392157, 0.235294 );
    transferFunction->AddRGBPoint( 1124.0, 0.862745, 0.705882, 0.862745 );
    transferFunction->AddRGBPoint( 1125.0, 0.0784314, 0.705882, 0.54902 );
    transferFunction->AddRGBPoint( 1126.0, 0.235294, 0.54902, 0.705882 );
    transferFunction->AddRGBPoint( 1127.0, 0.0980392, 0.0784314, 0.54902 );
    transferFunction->AddRGBPoint( 1128.0, 0.0784314, 0.235294, 0.392157 );
    transferFunction->AddRGBPoint( 1129.0, 0.235294, 0.862745, 0.0784314 );
    transferFunction->AddRGBPoint( 1130.0, 0.235294, 0.0784314, 0.862745 );
    transferFunction->AddRGBPoint( 1131.0, 0.862745, 0.862745, 0.392157 );
    transferFunction->AddRGBPoint( 1132.0, 0.705882, 0.235294, 0.235294 );
    transferFunction->AddRGBPoint( 1133.0, 0.235294, 0.235294, 0.862745 );
    transferFunction->AddRGBPoint( 1134.0, 0.862745, 0.235294, 0.862745 );
    transferFunction->AddRGBPoint( 1135.0, 0.254902, 0.862745, 0.235294 );
    transferFunction->AddRGBPoint( 1136.0, 0.0980392, 0.54902, 0.0784314 );
    transferFunction->AddRGBPoint( 1137.0, 0.0509804, 0, 0.980392 );
    transferFunction->AddRGBPoint( 1138.0, 0.239216, 0.0784314, 0.862745 );
    transferFunction->AddRGBPoint( 1139.0, 0.239216, 0.0784314, 0.235294 );
    transferFunction->AddRGBPoint( 1140.0, 0.239216, 0.235294, 0.392157 );
    transferFunction->AddRGBPoint( 1141.0, 0.0980392, 0.0980392, 0.0980392 );
    transferFunction->AddRGBPoint( 1142.0, 0.54902, 0.0784314, 0.235294 );
    transferFunction->AddRGBPoint( 1143.0, 0.862745, 0.705882, 0.0784314 );
    transferFunction->AddRGBPoint( 1144.0, 0.247059, 0.705882, 0.705882 );
    transferFunction->AddRGBPoint( 1145.0, 0.866667, 0.0784314, 0.0392157 );
    transferFunction->AddRGBPoint( 1146.0, 0.0823529, 0.862745, 0.0784314 );
    transferFunction->AddRGBPoint( 1147.0, 0.717647, 0.392157, 0.0784314 );
    transferFunction->AddRGBPoint( 1148.0, 0.866667, 0.0784314, 0.392157 );
    transferFunction->AddRGBPoint( 1149.0, 0.866667, 0.235294, 0.54902 );
    transferFunction->AddRGBPoint( 1150.0, 0.866667, 0.0784314, 0.862745 );
    transferFunction->AddRGBPoint( 1151.0, 0.239216, 0.862745, 0.862745 );
    transferFunction->AddRGBPoint( 1152.0, 0.392157, 0.784314, 0.784314 );
    transferFunction->AddRGBPoint( 1153.0, 0.0392157, 0.784314, 0.784314 );
    transferFunction->AddRGBPoint( 1154.0, 0.866667, 0.862745, 0.0784314 );
    transferFunction->AddRGBPoint( 1155.0, 0.552941, 0.0784314, 0.392157 );
    transferFunction->AddRGBPoint( 1156.0, 0.239216, 0.862745, 0.392157 );
    transferFunction->AddRGBPoint( 1157.0, 0.0823529, 0.862745, 0.235294 );
    transferFunction->AddRGBPoint( 1158.0, 0.552941, 0.235294, 0.0784314 );
    transferFunction->AddRGBPoint( 1159.0, 0.560784, 0.0784314, 0.862745 );
    transferFunction->AddRGBPoint( 1160.0, 0.239216, 0.0784314, 0.705882 );
    transferFunction->AddRGBPoint( 1161.0, 0.396078, 0.235294, 0.862745 );
    transferFunction->AddRGBPoint( 1162.0, 0.0823529, 0.0784314, 0.54902 );
    transferFunction->AddRGBPoint( 1163.0, 0.866667, 0.54902, 0.0784314 );
    transferFunction->AddRGBPoint( 1164.0, 0.552941, 0.392157, 0.862745 );
    transferFunction->AddRGBPoint( 1165.0, 0.396078, 0.0784314, 0.0784314 );
    transferFunction->AddRGBPoint( 1166.0, 0.866667, 0.392157, 0.0784314 );
    transferFunction->AddRGBPoint( 1167.0, 0.709804, 0.784314, 0.0784314 );
    transferFunction->AddRGBPoint( 1168.0, 0.0823529, 0.705882, 0.54902 );
    transferFunction->AddRGBPoint( 1169.0, 0.396078, 0.392157, 0.705882 );
    transferFunction->AddRGBPoint( 1170.0, 0.709804, 0.862745, 0.0784314 );
    transferFunction->AddRGBPoint( 1171.0, 0.0823529, 0.54902, 0.784314 );
    transferFunction->AddRGBPoint( 1172.0, 0.0823529, 0.0784314, 0.941176 );
    transferFunction->AddRGBPoint( 1173.0, 0.0823529, 0.0784314, 0.784314 );
    transferFunction->AddRGBPoint( 1174.0, 0.239216, 0.705882, 0.235294 );
    transferFunction->AddRGBPoint( 1175.0, 0.239216, 0.705882, 0.980392 );
    transferFunction->AddRGBPoint( 1176.0, 0.0823529, 0.0784314, 0.235294 );
    transferFunction->AddRGBPoint( 1177.0, 0.396078, 0.235294, 0.235294 );
    transferFunction->AddRGBPoint( 1178.0, 0.0823529, 0.862745, 0.862745 );
    transferFunction->AddRGBPoint( 1179.0, 0.0823529, 0.705882, 0.705882 );
    transferFunction->AddRGBPoint( 1180.0, 0.87451, 0.862745, 0.235294 );
    transferFunction->AddRGBPoint( 1181.0, 0.866667, 0.235294, 0.235294 );
    transferFunction->AddRGBPoint( 2101.0, 0.196078, 0.196078, 0.196078 );
    transferFunction->AddRGBPoint( 2102.0, 0.705882, 0.0784314, 0.117647 );
    transferFunction->AddRGBPoint( 2103.0, 0.235294, 0.0980392, 0.0980392 );
    transferFunction->AddRGBPoint( 2104.0, 0.0980392, 0.235294, 0.235294 );
    transferFunction->AddRGBPoint( 2105.0, 0.705882, 0.0784314, 0.0784314 );
    transferFunction->AddRGBPoint( 2106.0, 0.862745, 0.0784314, 0.392157 );
    transferFunction->AddRGBPoint( 2107.0, 0.54902, 0.235294, 0.235294 );
    transferFunction->AddRGBPoint( 2108.0, 0.705882, 0.862745, 0.54902 );
    transferFunction->AddRGBPoint( 2109.0, 0.54902, 0.392157, 0.705882 );
    transferFunction->AddRGBPoint( 2110.0, 0.705882, 0.0784314, 0.54902 );
    transferFunction->AddRGBPoint( 2111.0, 0.54902, 0.0784314, 0.54902 );
    transferFunction->AddRGBPoint( 2112.0, 0.0823529, 0.0392157, 0.0392157 );
    transferFunction->AddRGBPoint( 2113.0, 0.882353, 0.54902, 0.54902 );
    transferFunction->AddRGBPoint( 2114.0, 0.0901961, 0.235294, 0.705882 );
    transferFunction->AddRGBPoint( 2115.0, 0.705882, 0.235294, 0.705882 );
    transferFunction->AddRGBPoint( 2116.0, 0.0784314, 0.862745, 0.235294 );
    transferFunction->AddRGBPoint( 2117.0, 0.235294, 0.0784314, 0.54902 );
    transferFunction->AddRGBPoint( 2118.0, 0.862745, 0.705882, 0.54902 );
    transferFunction->AddRGBPoint( 2119.0, 0.254902, 0.392157, 0.0784314 );
    transferFunction->AddRGBPoint( 2120.0, 0.862745, 0.235294, 0.0784314 );
    transferFunction->AddRGBPoint( 2121.0, 0.235294, 0.392157, 0.235294 );
    transferFunction->AddRGBPoint( 2122.0, 0.0784314, 0.235294, 0.862745 );
    transferFunction->AddRGBPoint( 2123.0, 0.392157, 0.392157, 0.235294 );
    transferFunction->AddRGBPoint( 2124.0, 0.862745, 0.705882, 0.862745 );
    transferFunction->AddRGBPoint( 2125.0, 0.0784314, 0.705882, 0.54902 );
    transferFunction->AddRGBPoint( 2126.0, 0.235294, 0.54902, 0.705882 );
    transferFunction->AddRGBPoint( 2127.0, 0.0980392, 0.0784314, 0.54902 );
    transferFunction->AddRGBPoint( 2128.0, 0.0784314, 0.235294, 0.392157 );
    transferFunction->AddRGBPoint( 2129.0, 0.235294, 0.862745, 0.0784314 );
    transferFunction->AddRGBPoint( 2130.0, 0.235294, 0.0784314, 0.862745 );
    transferFunction->AddRGBPoint( 2131.0, 0.862745, 0.862745, 0.392157 );
    transferFunction->AddRGBPoint( 2132.0, 0.705882, 0.235294, 0.235294 );
    transferFunction->AddRGBPoint( 2133.0, 0.235294, 0.235294, 0.862745 );
    transferFunction->AddRGBPoint( 2134.0, 0.862745, 0.235294, 0.862745 );
    transferFunction->AddRGBPoint( 2135.0, 0.254902, 0.862745, 0.235294 );
    transferFunction->AddRGBPoint( 2136.0, 0.0980392, 0.54902, 0.0784314 );
    transferFunction->AddRGBPoint( 2137.0, 0.0509804, 0, 0.980392 );
    transferFunction->AddRGBPoint( 2138.0, 0.239216, 0.0784314, 0.862745 );
    transferFunction->AddRGBPoint( 2139.0, 0.239216, 0.0784314, 0.235294 );
    transferFunction->AddRGBPoint( 2140.0, 0.239216, 0.235294, 0.392157 );
    transferFunction->AddRGBPoint( 2141.0, 0.0980392, 0.0980392, 0.0980392 );
    transferFunction->AddRGBPoint( 2142.0, 0.54902, 0.0784314, 0.235294 );
    transferFunction->AddRGBPoint( 2143.0, 0.862745, 0.705882, 0.0784314 );
    transferFunction->AddRGBPoint( 2144.0, 0.247059, 0.705882, 0.705882 );
    transferFunction->AddRGBPoint( 2145.0, 0.866667, 0.0784314, 0.0392157 );
    transferFunction->AddRGBPoint( 2146.0, 0.0823529, 0.862745, 0.0784314 );
    transferFunction->AddRGBPoint( 2147.0, 0.717647, 0.392157, 0.0784314 );
    transferFunction->AddRGBPoint( 2148.0, 0.866667, 0.0784314, 0.392157 );
    transferFunction->AddRGBPoint( 2149.0, 0.866667, 0.235294, 0.54902 );
    transferFunction->AddRGBPoint( 2150.0, 0.866667, 0.0784314, 0.862745 );
    transferFunction->AddRGBPoint( 2151.0, 0.239216, 0.862745, 0.862745 );
    transferFunction->AddRGBPoint( 2152.0, 0.392157, 0.784314, 0.784314 );
    transferFunction->AddRGBPoint( 2153.0, 0.0392157, 0.784314, 0.784314 );
    transferFunction->AddRGBPoint( 2154.0, 0.866667, 0.862745, 0.0784314 );
    transferFunction->AddRGBPoint( 2155.0, 0.552941, 0.0784314, 0.392157 );
    transferFunction->AddRGBPoint( 2156.0, 0.239216, 0.862745, 0.392157 );
    transferFunction->AddRGBPoint( 2157.0, 0.0823529, 0.862745, 0.235294 );
    transferFunction->AddRGBPoint( 2158.0, 0.552941, 0.235294, 0.0784314 );
    transferFunction->AddRGBPoint( 2159.0, 0.560784, 0.0784314, 0.862745 );
    transferFunction->AddRGBPoint( 2160.0, 0.239216, 0.0784314, 0.705882 );
    transferFunction->AddRGBPoint( 2161.0, 0.396078, 0.235294, 0.862745 );
    transferFunction->AddRGBPoint( 2162.0, 0.0823529, 0.0784314, 0.54902 );
    transferFunction->AddRGBPoint( 2163.0, 0.866667, 0.54902, 0.0784314 );
    transferFunction->AddRGBPoint( 2164.0, 0.552941, 0.392157, 0.862745 );
    transferFunction->AddRGBPoint( 2165.0, 0.396078, 0.0784314, 0.0784314 );
    transferFunction->AddRGBPoint( 2166.0, 0.866667, 0.392157, 0.0784314 );
    transferFunction->AddRGBPoint( 2167.0, 0.709804, 0.784314, 0.0784314 );
    transferFunction->AddRGBPoint( 2168.0, 0.0823529, 0.705882, 0.54902 );
    transferFunction->AddRGBPoint( 2169.0, 0.396078, 0.392157, 0.705882 );
    transferFunction->AddRGBPoint( 2170.0, 0.709804, 0.862745, 0.0784314 );
    transferFunction->AddRGBPoint( 2171.0, 0.0823529, 0.54902, 0.784314 );
    transferFunction->AddRGBPoint( 2172.0, 0.0823529, 0.0784314, 0.941176 );
    transferFunction->AddRGBPoint( 2173.0, 0.0823529, 0.0784314, 0.784314 );
    transferFunction->AddRGBPoint( 2174.0, 0.239216, 0.705882, 0.235294 );
    transferFunction->AddRGBPoint( 2175.0, 0.239216, 0.705882, 0.980392 );
    transferFunction->AddRGBPoint( 2176.0, 0.0823529, 0.0784314, 0.235294 );
    transferFunction->AddRGBPoint( 2177.0, 0.396078, 0.235294, 0.235294 );
    transferFunction->AddRGBPoint( 2178.0, 0.0823529, 0.862745, 0.862745 );
    transferFunction->AddRGBPoint( 2179.0, 0.0823529, 0.705882, 0.705882 );
    transferFunction->AddRGBPoint( 2180.0, 0.87451, 0.862745, 0.235294 );
    transferFunction->AddRGBPoint( 2181.0, 0.866667, 0.235294, 0.235294 );
    transferFunction->AddRGBPoint( 2200.0, 0.0980392, 0.235294, 0.239216 );
    transferFunction->AddRGBPoint( 2201.0, 0.0980392, 0.352941, 0.235294 );
    transferFunction->AddRGBPoint( 2202.0, 0.0980392, 0.470588, 0.235294 );
    transferFunction->AddRGBPoint( 2205.0, 0.0980392, 0.588235, 0.235294 );
    transferFunction->AddRGBPoint( 2206.0, 0.0980392, 0.705882, 0.235294 );
    transferFunction->AddRGBPoint( 2207.0, 0.0980392, 0.823529, 0.235294 );
    transferFunction->AddRGBPoint( 2210.0, 0.0980392, 0.588235, 0.352941 );
    transferFunction->AddRGBPoint( 2211.0, 0.0980392, 0.705882, 0.352941 );
    transferFunction->AddRGBPoint( 2212.0, 0.0980392, 0.823529, 0.352941 );
    transferFunction->AddRGBPoint( 3101.0, 0.196078, 0.196078, 0.196078 );
    transferFunction->AddRGBPoint( 3102.0, 0.705882, 0.0784314, 0.117647 );
    transferFunction->AddRGBPoint( 3103.0, 0.235294, 0.0980392, 0.0980392 );
    transferFunction->AddRGBPoint( 3104.0, 0.0980392, 0.235294, 0.235294 );
    transferFunction->AddRGBPoint( 3105.0, 0.705882, 0.0784314, 0.0784314 );
    transferFunction->AddRGBPoint( 3106.0, 0.862745, 0.0784314, 0.392157 );
    transferFunction->AddRGBPoint( 3107.0, 0.54902, 0.235294, 0.235294 );
    transferFunction->AddRGBPoint( 3108.0, 0.705882, 0.862745, 0.54902 );
    transferFunction->AddRGBPoint( 3109.0, 0.54902, 0.392157, 0.705882 );
    transferFunction->AddRGBPoint( 3110.0, 0.705882, 0.0784314, 0.54902 );
    transferFunction->AddRGBPoint( 3111.0, 0.54902, 0.0784314, 0.54902 );
    transferFunction->AddRGBPoint( 3112.0, 0.0823529, 0.0392157, 0.0392157 );
    transferFunction->AddRGBPoint( 3113.0, 0.882353, 0.54902, 0.54902 );
    transferFunction->AddRGBPoint( 3114.0, 0.0901961, 0.235294, 0.705882 );
    transferFunction->AddRGBPoint( 3115.0, 0.705882, 0.235294, 0.705882 );
    transferFunction->AddRGBPoint( 3116.0, 0.0784314, 0.862745, 0.235294 );
    transferFunction->AddRGBPoint( 3117.0, 0.235294, 0.0784314, 0.54902 );
    transferFunction->AddRGBPoint( 3118.0, 0.862745, 0.705882, 0.54902 );
    transferFunction->AddRGBPoint( 3119.0, 0.254902, 0.392157, 0.0784314 );
    transferFunction->AddRGBPoint( 3120.0, 0.862745, 0.235294, 0.0784314 );
    transferFunction->AddRGBPoint( 3121.0, 0.235294, 0.392157, 0.235294 );
    transferFunction->AddRGBPoint( 3122.0, 0.0784314, 0.235294, 0.862745 );
    transferFunction->AddRGBPoint( 3123.0, 0.392157, 0.392157, 0.235294 );
    transferFunction->AddRGBPoint( 3124.0, 0.862745, 0.705882, 0.862745 );
    transferFunction->AddRGBPoint( 3125.0, 0.0784314, 0.705882, 0.54902 );
    transferFunction->AddRGBPoint( 3126.0, 0.235294, 0.54902, 0.705882 );
    transferFunction->AddRGBPoint( 3127.0, 0.0980392, 0.0784314, 0.54902 );
    transferFunction->AddRGBPoint( 3128.0, 0.0784314, 0.235294, 0.392157 );
    transferFunction->AddRGBPoint( 3129.0, 0.235294, 0.862745, 0.0784314 );
    transferFunction->AddRGBPoint( 3130.0, 0.235294, 0.0784314, 0.862745 );
    transferFunction->AddRGBPoint( 3131.0, 0.862745, 0.862745, 0.392157 );
    transferFunction->AddRGBPoint( 3132.0, 0.705882, 0.235294, 0.235294 );
    transferFunction->AddRGBPoint( 3133.0, 0.235294, 0.235294, 0.862745 );
    transferFunction->AddRGBPoint( 3134.0, 0.862745, 0.235294, 0.862745 );
    transferFunction->AddRGBPoint( 3135.0, 0.254902, 0.862745, 0.235294 );
    transferFunction->AddRGBPoint( 3136.0, 0.0980392, 0.54902, 0.0784314 );
    transferFunction->AddRGBPoint( 3137.0, 0.0509804, 0, 0.980392 );
    transferFunction->AddRGBPoint( 3138.0, 0.239216, 0.0784314, 0.862745 );
    transferFunction->AddRGBPoint( 3139.0, 0.239216, 0.0784314, 0.235294 );
    transferFunction->AddRGBPoint( 3140.0, 0.239216, 0.235294, 0.392157 );
    transferFunction->AddRGBPoint( 3141.0, 0.0980392, 0.0980392, 0.0980392 );
    transferFunction->AddRGBPoint( 3142.0, 0.54902, 0.0784314, 0.235294 );
    transferFunction->AddRGBPoint( 3143.0, 0.862745, 0.705882, 0.0784314 );
    transferFunction->AddRGBPoint( 3144.0, 0.247059, 0.705882, 0.705882 );
    transferFunction->AddRGBPoint( 3145.0, 0.866667, 0.0784314, 0.0392157 );
    transferFunction->AddRGBPoint( 3146.0, 0.0823529, 0.862745, 0.0784314 );
    transferFunction->AddRGBPoint( 3147.0, 0.717647, 0.392157, 0.0784314 );
    transferFunction->AddRGBPoint( 3148.0, 0.866667, 0.0784314, 0.392157 );
    transferFunction->AddRGBPoint( 3149.0, 0.866667, 0.235294, 0.54902 );
    transferFunction->AddRGBPoint( 3150.0, 0.866667, 0.0784314, 0.862745 );
    transferFunction->AddRGBPoint( 3151.0, 0.239216, 0.862745, 0.862745 );
    transferFunction->AddRGBPoint( 3152.0, 0.392157, 0.784314, 0.784314 );
    transferFunction->AddRGBPoint( 3153.0, 0.0392157, 0.784314, 0.784314 );
    transferFunction->AddRGBPoint( 3154.0, 0.866667, 0.862745, 0.0784314 );
    transferFunction->AddRGBPoint( 3155.0, 0.552941, 0.0784314, 0.392157 );
    transferFunction->AddRGBPoint( 3156.0, 0.239216, 0.862745, 0.392157 );
    transferFunction->AddRGBPoint( 3157.0, 0.0823529, 0.862745, 0.235294 );
    transferFunction->AddRGBPoint( 3158.0, 0.552941, 0.235294, 0.0784314 );
    transferFunction->AddRGBPoint( 3159.0, 0.560784, 0.0784314, 0.862745 );
    transferFunction->AddRGBPoint( 3160.0, 0.239216, 0.0784314, 0.705882 );
    transferFunction->AddRGBPoint( 3161.0, 0.396078, 0.235294, 0.862745 );
    transferFunction->AddRGBPoint( 3162.0, 0.0823529, 0.0784314, 0.54902 );
    transferFunction->AddRGBPoint( 3163.0, 0.866667, 0.54902, 0.0784314 );
    transferFunction->AddRGBPoint( 3164.0, 0.552941, 0.392157, 0.862745 );
    transferFunction->AddRGBPoint( 3165.0, 0.396078, 0.0784314, 0.0784314 );
    transferFunction->AddRGBPoint( 3166.0, 0.866667, 0.392157, 0.0784314 );
    transferFunction->AddRGBPoint( 3167.0, 0.709804, 0.784314, 0.0784314 );
    transferFunction->AddRGBPoint( 3168.0, 0.0823529, 0.705882, 0.54902 );
    transferFunction->AddRGBPoint( 3169.0, 0.396078, 0.392157, 0.705882 );
    transferFunction->AddRGBPoint( 3170.0, 0.709804, 0.862745, 0.0784314 );
    transferFunction->AddRGBPoint( 3171.0, 0.0823529, 0.54902, 0.784314 );
    transferFunction->AddRGBPoint( 3172.0, 0.0823529, 0.0784314, 0.941176 );
    transferFunction->AddRGBPoint( 3173.0, 0.0823529, 0.0784314, 0.784314 );
    transferFunction->AddRGBPoint( 3174.0, 0.239216, 0.705882, 0.235294 );
    transferFunction->AddRGBPoint( 3175.0, 0.239216, 0.705882, 0.980392 );
    transferFunction->AddRGBPoint( 3176.0, 0.0823529, 0.0784314, 0.235294 );
    transferFunction->AddRGBPoint( 3177.0, 0.396078, 0.235294, 0.235294 );
    transferFunction->AddRGBPoint( 3178.0, 0.0823529, 0.862745, 0.862745 );
    transferFunction->AddRGBPoint( 3179.0, 0.0823529, 0.705882, 0.705882 );
    transferFunction->AddRGBPoint( 3180.0, 0.87451, 0.862745, 0.235294 );
    transferFunction->AddRGBPoint( 3181.0, 0.866667, 0.235294, 0.235294 );
    transferFunction->AddRGBPoint( 4101.0, 0.196078, 0.196078, 0.196078 );
    transferFunction->AddRGBPoint( 4102.0, 0.705882, 0.0784314, 0.117647 );
    transferFunction->AddRGBPoint( 4103.0, 0.235294, 0.0980392, 0.0980392 );
    transferFunction->AddRGBPoint( 4104.0, 0.0980392, 0.235294, 0.235294 );
    transferFunction->AddRGBPoint( 4105.0, 0.705882, 0.0784314, 0.0784314 );
    transferFunction->AddRGBPoint( 4106.0, 0.862745, 0.0784314, 0.392157 );
    transferFunction->AddRGBPoint( 4107.0, 0.54902, 0.235294, 0.235294 );
    transferFunction->AddRGBPoint( 4108.0, 0.705882, 0.862745, 0.54902 );
    transferFunction->AddRGBPoint( 4109.0, 0.54902, 0.392157, 0.705882 );
    transferFunction->AddRGBPoint( 4110.0, 0.705882, 0.0784314, 0.54902 );
    transferFunction->AddRGBPoint( 4111.0, 0.54902, 0.0784314, 0.54902 );
    transferFunction->AddRGBPoint( 4112.0, 0.0823529, 0.0392157, 0.0392157 );
    transferFunction->AddRGBPoint( 4113.0, 0.882353, 0.54902, 0.54902 );
    transferFunction->AddRGBPoint( 4114.0, 0.0901961, 0.235294, 0.705882 );
    transferFunction->AddRGBPoint( 4115.0, 0.705882, 0.235294, 0.705882 );
    transferFunction->AddRGBPoint( 4116.0, 0.0784314, 0.862745, 0.235294 );
    transferFunction->AddRGBPoint( 4117.0, 0.235294, 0.0784314, 0.54902 );
    transferFunction->AddRGBPoint( 4118.0, 0.862745, 0.705882, 0.54902 );
    transferFunction->AddRGBPoint( 4119.0, 0.254902, 0.392157, 0.0784314 );
    transferFunction->AddRGBPoint( 4120.0, 0.862745, 0.235294, 0.0784314 );
    transferFunction->AddRGBPoint( 4121.0, 0.235294, 0.392157, 0.235294 );
    transferFunction->AddRGBPoint( 4122.0, 0.0784314, 0.235294, 0.862745 );
    transferFunction->AddRGBPoint( 4123.0, 0.392157, 0.392157, 0.235294 );
    transferFunction->AddRGBPoint( 4124.0, 0.862745, 0.705882, 0.862745 );
    transferFunction->AddRGBPoint( 4125.0, 0.0784314, 0.705882, 0.54902 );
    transferFunction->AddRGBPoint( 4126.0, 0.235294, 0.54902, 0.705882 );
    transferFunction->AddRGBPoint( 4127.0, 0.0980392, 0.0784314, 0.54902 );
    transferFunction->AddRGBPoint( 4128.0, 0.0784314, 0.235294, 0.392157 );
    transferFunction->AddRGBPoint( 4129.0, 0.235294, 0.862745, 0.0784314 );
    transferFunction->AddRGBPoint( 4130.0, 0.235294, 0.0784314, 0.862745 );
    transferFunction->AddRGBPoint( 4131.0, 0.862745, 0.862745, 0.392157 );
    transferFunction->AddRGBPoint( 4132.0, 0.705882, 0.235294, 0.235294 );
    transferFunction->AddRGBPoint( 4133.0, 0.235294, 0.235294, 0.862745 );
    transferFunction->AddRGBPoint( 4134.0, 0.862745, 0.235294, 0.862745 );
    transferFunction->AddRGBPoint( 4135.0, 0.254902, 0.862745, 0.235294 );
    transferFunction->AddRGBPoint( 4136.0, 0.0980392, 0.54902, 0.0784314 );
    transferFunction->AddRGBPoint( 4137.0, 0.0509804, 0, 0.980392 );
    transferFunction->AddRGBPoint( 4138.0, 0.239216, 0.0784314, 0.862745 );
    transferFunction->AddRGBPoint( 4139.0, 0.239216, 0.0784314, 0.235294 );
    transferFunction->AddRGBPoint( 4140.0, 0.239216, 0.235294, 0.392157 );
    transferFunction->AddRGBPoint( 4141.0, 0.0980392, 0.0980392, 0.0980392 );
    transferFunction->AddRGBPoint( 4142.0, 0.54902, 0.0784314, 0.235294 );
    transferFunction->AddRGBPoint( 4143.0, 0.862745, 0.705882, 0.0784314 );
    transferFunction->AddRGBPoint( 4144.0, 0.247059, 0.705882, 0.705882 );
    transferFunction->AddRGBPoint( 4145.0, 0.866667, 0.0784314, 0.0392157 );
    transferFunction->AddRGBPoint( 4146.0, 0.0823529, 0.862745, 0.0784314 );
    transferFunction->AddRGBPoint( 4147.0, 0.717647, 0.392157, 0.0784314 );
    transferFunction->AddRGBPoint( 4148.0, 0.866667, 0.0784314, 0.392157 );
    transferFunction->AddRGBPoint( 4149.0, 0.866667, 0.235294, 0.54902 );
    transferFunction->AddRGBPoint( 4150.0, 0.866667, 0.0784314, 0.862745 );
    transferFunction->AddRGBPoint( 4151.0, 0.239216, 0.862745, 0.862745 );
    transferFunction->AddRGBPoint( 4152.0, 0.392157, 0.784314, 0.784314 );
    transferFunction->AddRGBPoint( 4153.0, 0.0392157, 0.784314, 0.784314 );
    transferFunction->AddRGBPoint( 4154.0, 0.866667, 0.862745, 0.0784314 );
    transferFunction->AddRGBPoint( 4155.0, 0.552941, 0.0784314, 0.392157 );
    transferFunction->AddRGBPoint( 4156.0, 0.239216, 0.862745, 0.392157 );
    transferFunction->AddRGBPoint( 4157.0, 0.0823529, 0.862745, 0.235294 );
    transferFunction->AddRGBPoint( 4158.0, 0.552941, 0.235294, 0.0784314 );
    transferFunction->AddRGBPoint( 4159.0, 0.560784, 0.0784314, 0.862745 );
    transferFunction->AddRGBPoint( 4160.0, 0.239216, 0.0784314, 0.705882 );
    transferFunction->AddRGBPoint( 4161.0, 0.396078, 0.235294, 0.862745 );
    transferFunction->AddRGBPoint( 4162.0, 0.0823529, 0.0784314, 0.54902 );
    transferFunction->AddRGBPoint( 4163.0, 0.866667, 0.54902, 0.0784314 );
    transferFunction->AddRGBPoint( 4164.0, 0.552941, 0.392157, 0.862745 );
    transferFunction->AddRGBPoint( 4165.0, 0.396078, 0.0784314, 0.0784314 );
    transferFunction->AddRGBPoint( 4166.0, 0.866667, 0.392157, 0.0784314 );
    transferFunction->AddRGBPoint( 4167.0, 0.709804, 0.784314, 0.0784314 );
    transferFunction->AddRGBPoint( 4168.0, 0.0823529, 0.705882, 0.54902 );
    transferFunction->AddRGBPoint( 4169.0, 0.396078, 0.392157, 0.705882 );
    transferFunction->AddRGBPoint( 4170.0, 0.709804, 0.862745, 0.0784314 );
    transferFunction->AddRGBPoint( 4171.0, 0.0823529, 0.54902, 0.784314 );
    transferFunction->AddRGBPoint( 4172.0, 0.0823529, 0.0784314, 0.941176 );
    transferFunction->AddRGBPoint( 4173.0, 0.0823529, 0.0784314, 0.784314 );
    transferFunction->AddRGBPoint( 4174.0, 0.239216, 0.705882, 0.235294 );
    transferFunction->AddRGBPoint( 4175.0, 0.239216, 0.705882, 0.980392 );
    transferFunction->AddRGBPoint( 4176.0, 0.0823529, 0.0784314, 0.235294 );
    transferFunction->AddRGBPoint( 4177.0, 0.396078, 0.235294, 0.235294 );
    transferFunction->AddRGBPoint( 4178.0, 0.0823529, 0.862745, 0.862745 );
    transferFunction->AddRGBPoint( 4179.0, 0.0823529, 0.705882, 0.705882 );
    transferFunction->AddRGBPoint( 4180.0, 0.87451, 0.862745, 0.235294 );
    transferFunction->AddRGBPoint( 4181.0, 0.866667, 0.235294, 0.235294 );
    transferFunction->AddRGBPoint( 5001.0, 0.0784314, 0.117647, 0.156863 );
    transferFunction->AddRGBPoint( 5002.0, 0.0784314, 0.117647, 0.156863 );
    transferFunction->AddRGBPoint( 5100.0, 0.8, 0.4, 0.4 );
    transferFunction->AddRGBPoint( 5101.0, 0.8, 0.4, 0.4 );
    transferFunction->AddRGBPoint( 5102.0, 1, 1, 0.4 );
    transferFunction->AddRGBPoint( 5103.0, 0.6, 0.8, 0 );
    transferFunction->AddRGBPoint( 5104.0, 0, 0.6, 0.6 );
    transferFunction->AddRGBPoint( 5105.0, 0.8, 0.6, 1 );
    transferFunction->AddRGBPoint( 5106.0, 1, 0.6, 0.2 );
    transferFunction->AddRGBPoint( 5107.0, 0.8, 0.8, 0.8 );
    transferFunction->AddRGBPoint( 5108.0, 0.6, 1, 1 );
    transferFunction->AddRGBPoint( 5109.0, 0.4, 0.6, 1 );
    transferFunction->AddRGBPoint( 5110.0, 1, 1, 0.4 );
    transferFunction->AddRGBPoint( 5111.0, 0.6, 0.8, 0 );
    transferFunction->AddRGBPoint( 5112.0, 0, 0.6, 0.6 );
    transferFunction->AddRGBPoint( 5113.0, 0.8, 0.6, 1 );
    transferFunction->AddRGBPoint( 5114.0, 1, 0.6, 0.2 );
    transferFunction->AddRGBPoint( 5115.0, 0.8, 0.8, 0.8 );
    transferFunction->AddRGBPoint( 5116.0, 0.6, 1, 1 );
    transferFunction->AddRGBPoint( 5117.0, 0.4, 0.6, 1 );
    transferFunction->AddRGBPoint( 5200.0, 0.8, 0.4, 0.4 );
    transferFunction->AddRGBPoint( 5201.0, 0.8, 0.4, 0.4 );
    transferFunction->AddRGBPoint( 5202.0, 1, 1, 0.4 );
    transferFunction->AddRGBPoint( 5203.0, 0.6, 0.8, 0 );
    transferFunction->AddRGBPoint( 5204.0, 0, 0.6, 0.6 );
    transferFunction->AddRGBPoint( 5205.0, 0.8, 0.6, 1 );
    transferFunction->AddRGBPoint( 5206.0, 1, 0.6, 0.2 );
    transferFunction->AddRGBPoint( 5207.0, 0.8, 0.8, 0.8 );
    transferFunction->AddRGBPoint( 5208.0, 0.6, 1, 1 );
    transferFunction->AddRGBPoint( 5209.0, 0.4, 0.6, 1 );
    transferFunction->AddRGBPoint( 5210.0, 1, 1, 0.4 );
    transferFunction->AddRGBPoint( 5211.0, 0.6, 0.8, 0 );
    transferFunction->AddRGBPoint( 5212.0, 0, 0.6, 0.6 );
    transferFunction->AddRGBPoint( 5213.0, 0.8, 0.6, 1 );
    transferFunction->AddRGBPoint( 5214.0, 1, 0.6, 0.2 );
    transferFunction->AddRGBPoint( 5215.0, 0.8, 0.8, 0.8 );
    transferFunction->AddRGBPoint( 5216.0, 0.6, 1, 1 );
    transferFunction->AddRGBPoint( 5217.0, 0.4, 0.6, 1 );
    transferFunction->AddRGBPoint( 6000.0, 0, 1, 0 );
    transferFunction->AddRGBPoint( 6001.0, 1, 1, 0 );
    transferFunction->AddRGBPoint( 6002.0, 0, 1, 1 );
    transferFunction->AddRGBPoint( 6003.0, 0, 0, 1 );
    transferFunction->AddRGBPoint( 6010.0, 0.92549, 0.0627451, 0.905882 );
    transferFunction->AddRGBPoint( 6020.0, 0.929412, 0.0705882, 0.909804 );
    transferFunction->AddRGBPoint( 6030.0, 0.92549, 0.0509804, 0.890196 );
    transferFunction->AddRGBPoint( 6040.0, 0.92549, 0.0666667, 0.894118 );
    transferFunction->AddRGBPoint( 6050.0, 0.00392157, 1, 0.00392157 );
    transferFunction->AddRGBPoint( 6060.0, 0.00784314, 1, 0.00392157 );
    transferFunction->AddRGBPoint( 6070.0, 0.92549, 0.054902, 0.901961 );
    transferFunction->AddRGBPoint( 6080.0, 0.929412, 0.054902, 0.901961 );
    transferFunction->AddRGBPoint( 7001.0, 0.282353, 0.517647, 0.709804 );
    transferFunction->AddRGBPoint( 7002.0, 0.952941, 0.952941, 0.952941 );
    transferFunction->AddRGBPoint( 7003.0, 0.811765, 0.247059, 0.309804 );
    transferFunction->AddRGBPoint( 7004.0, 0.47451, 0.0784314, 0.529412 );
    transferFunction->AddRGBPoint( 7005.0, 0.772549, 0.235294, 0.972549 );
    transferFunction->AddRGBPoint( 7006.0, 0.00784314, 0.584314, 0.00784314 );
    transferFunction->AddRGBPoint( 7007.0, 0.866667, 0.976471, 0.65098 );
    transferFunction->AddRGBPoint( 7008.0, 0.909804, 0.572549, 0.137255 );
    transferFunction->AddRGBPoint( 7009.0, 0.0784314, 0.235294, 0.470588 );
    transferFunction->AddRGBPoint( 7010.0, 0.980392, 0.980392, 0 );
    transferFunction->AddRGBPoint( 7011.0, 0.478431, 0.733333, 0.870588 );
    transferFunction->AddRGBPoint( 7012.0, 0.929412, 0.0470588, 0.694118 );
    transferFunction->AddRGBPoint( 7013.0, 0.0392157, 0.192157, 1 );
    transferFunction->AddRGBPoint( 7014.0, 0.803922, 0.721569, 0.564706 );
    transferFunction->AddRGBPoint( 7015.0, 0.176471, 0.803922, 0.647059 );
    transferFunction->AddRGBPoint( 7016.0, 0.458824, 0.627451, 0.686275 );
    transferFunction->AddRGBPoint( 7017.0, 0.866667, 0.85098, 0.0823529 );
    transferFunction->AddRGBPoint( 7018.0, 0.0784314, 0.235294, 0.470588 );
    transferFunction->AddRGBPoint( 7019.0, 0.552941, 0.0823529, 0.392157 );
    transferFunction->AddRGBPoint( 7020.0, 0.882353, 0.54902, 0.552941 );
    transferFunction->AddRGBPoint( 7100.0, 0.164706, 0.788235, 0.658824 );
    transferFunction->AddRGBPoint( 7101.0, 0.658824, 0.407843, 0.635294 );
    transferFunction->AddRGBPoint( 8001.0, 0.290196, 0.509804, 0.709804 );
    transferFunction->AddRGBPoint( 8002.0, 0.94902, 0.945098, 0.941176 );
    transferFunction->AddRGBPoint( 8003.0, 0.807843, 0.254902, 0.305882 );
    transferFunction->AddRGBPoint( 8004.0, 0.470588, 0.0823529, 0.521569 );
    transferFunction->AddRGBPoint( 8005.0, 0.764706, 0.239216, 0.964706 );
    transferFunction->AddRGBPoint( 8006.0, 0.0117647, 0.576471, 0.0235294 );
    transferFunction->AddRGBPoint( 8007.0, 0.862745, 0.984314, 0.639216 );
    transferFunction->AddRGBPoint( 8008.0, 0.909804, 0.572549, 0.129412 );
    transferFunction->AddRGBPoint( 8009.0, 0.0156863, 0.447059, 0.054902 );
    transferFunction->AddRGBPoint( 8010.0, 0.47451, 0.721569, 0.862745 );
    transferFunction->AddRGBPoint( 8011.0, 0.921569, 0.0431373, 0.686275 );
    transferFunction->AddRGBPoint( 8012.0, 0.0470588, 0.180392, 0.980392 );
    transferFunction->AddRGBPoint( 8013.0, 0.796078, 0.713725, 0.560784 );
    transferFunction->AddRGBPoint( 8014.0, 0.164706, 0.8, 0.654902 );
    transferFunction->AddRGBPoint( 9000.0, 0.117647, 0.0196078, 0.117647 );
    transferFunction->AddRGBPoint( 9001.0, 0.117647, 0.392157, 0.176471 );
    transferFunction->AddRGBPoint( 9002.0, 0.509804, 0.392157, 0.647059 );
    transferFunction->AddRGBPoint( 9003.0, 0.411765, 0.0980392, 0.0196078 );
    transferFunction->AddRGBPoint( 9004.0, 0.490196, 0.27451, 0.215686 );
    transferFunction->AddRGBPoint( 9005.0, 0.882353, 0.0784314, 0.411765 );
    transferFunction->AddRGBPoint( 9006.0, 0.882353, 0.0784314, 0.0588235 );
    transferFunction->AddRGBPoint( 9500.0, 0.117647, 0.215686, 0.117647 );
    transferFunction->AddRGBPoint( 9501.0, 0.117647, 0.588235, 0.176471 );
    transferFunction->AddRGBPoint( 9502.0, 0.509804, 0.588235, 0.647059 );
    transferFunction->AddRGBPoint( 9503.0, 0.411765, 0.294118, 0.0196078 );
    transferFunction->AddRGBPoint( 9504.0, 0.490196, 0.470588, 0.215686 );
    transferFunction->AddRGBPoint( 9505.0, 0.882353, 0.27451, 0.411765 );
    transferFunction->AddRGBPoint( 9506.0, 0.882353, 0.27451, 0.0588235 );
    transferFunction->AddRGBPoint( 11101.0, 0.0901961, 0.862745, 0.235294 );
    transferFunction->AddRGBPoint( 11102.0, 0.0901961, 0.235294, 0.705882 );
    transferFunction->AddRGBPoint( 11103.0, 0.247059, 0.392157, 0.235294 );
    transferFunction->AddRGBPoint( 11104.0, 0.247059, 0.0784314, 0.862745 );
    transferFunction->AddRGBPoint( 11105.0, 0.0509804, 0, 0.980392 );
    transferFunction->AddRGBPoint( 11106.0, 0.101961, 0.235294, 0 );
    transferFunction->AddRGBPoint( 11107.0, 0.101961, 0.235294, 0.294118 );
    transferFunction->AddRGBPoint( 11108.0, 0.101961, 0.235294, 0.588235 );
    transferFunction->AddRGBPoint( 11109.0, 0.0980392, 0.235294, 0.980392 );
    transferFunction->AddRGBPoint( 11110.0, 0.235294, 0.0980392, 0.0980392 );
    transferFunction->AddRGBPoint( 11111.0, 0.705882, 0.0784314, 0.0784314 );
    transferFunction->AddRGBPoint( 11112.0, 0.862745, 0.0784314, 0.392157 );
    transferFunction->AddRGBPoint( 11113.0, 0.54902, 0.235294, 0.235294 );
    transferFunction->AddRGBPoint( 11114.0, 0.705882, 0.862745, 0.54902 );
    transferFunction->AddRGBPoint( 11115.0, 0.54902, 0.392157, 0.705882 );
    transferFunction->AddRGBPoint( 11116.0, 0.705882, 0.0784314, 0.54902 );
    transferFunction->AddRGBPoint( 11117.0, 0.0901961, 0.0392157, 0.0392157 );
    transferFunction->AddRGBPoint( 11118.0, 0.882353, 0.54902, 0.54902 );
    transferFunction->AddRGBPoint( 11119.0, 0.705882, 0.235294, 0.705882 );
    transferFunction->AddRGBPoint( 11120.0, 0.0784314, 0.862745, 0.235294 );
    transferFunction->AddRGBPoint( 11121.0, 0.235294, 0.0784314, 0.54902 );
    transferFunction->AddRGBPoint( 11122.0, 0.862745, 0.705882, 0.54902 );
    transferFunction->AddRGBPoint( 11123.0, 0.254902, 0.392157, 0.0784314 );
    transferFunction->AddRGBPoint( 11124.0, 0.862745, 0.235294, 0.0784314 );
    transferFunction->AddRGBPoint( 11125.0, 0.0784314, 0.235294, 0.862745 );
    transferFunction->AddRGBPoint( 11126.0, 0.392157, 0.392157, 0.235294 );
    transferFunction->AddRGBPoint( 11127.0, 0.862745, 0.705882, 0.862745 );
    transferFunction->AddRGBPoint( 11128.0, 0.0784314, 0.705882, 0.54902 );
    transferFunction->AddRGBPoint( 11129.0, 0.235294, 0.54902, 0.705882 );
    transferFunction->AddRGBPoint( 11130.0, 0.0980392, 0.0784314, 0.54902 );
    transferFunction->AddRGBPoint( 11131.0, 0.0784314, 0.235294, 0.392157 );
    transferFunction->AddRGBPoint( 11132.0, 0.235294, 0.862745, 0.0784314 );
    transferFunction->AddRGBPoint( 11133.0, 0.235294, 0.235294, 0.862745 );
    transferFunction->AddRGBPoint( 11134.0, 0.862745, 0.235294, 0.862745 );
    transferFunction->AddRGBPoint( 11135.0, 0.254902, 0.862745, 0.235294 );
    transferFunction->AddRGBPoint( 11136.0, 0.0980392, 0.54902, 0.0784314 );
    transferFunction->AddRGBPoint( 11137.0, 0.862745, 0.862745, 0.392157 );
    transferFunction->AddRGBPoint( 11138.0, 0.705882, 0.235294, 0.235294 );
    transferFunction->AddRGBPoint( 11139.0, 0.239216, 0.0784314, 0.862745 );
    transferFunction->AddRGBPoint( 11140.0, 0.239216, 0.0784314, 0.235294 );
    transferFunction->AddRGBPoint( 11141.0, 0.239216, 0.235294, 0.392157 );
    transferFunction->AddRGBPoint( 11142.0, 0.0980392, 0.0980392, 0.0980392 );
    transferFunction->AddRGBPoint( 11143.0, 0.54902, 0.0784314, 0.235294 );
    transferFunction->AddRGBPoint( 11144.0, 0.862745, 0.705882, 0.0784314 );
    transferFunction->AddRGBPoint( 11145.0, 0.247059, 0.705882, 0.705882 );
    transferFunction->AddRGBPoint( 11146.0, 0.866667, 0.0784314, 0.0392157 );
    transferFunction->AddRGBPoint( 11147.0, 0.866667, 0.0784314, 0.392157 );
    transferFunction->AddRGBPoint( 11148.0, 0.866667, 0.235294, 0.54902 );
    transferFunction->AddRGBPoint( 11149.0, 0.866667, 0.0784314, 0.862745 );
    transferFunction->AddRGBPoint( 11150.0, 0.239216, 0.862745, 0.862745 );
    transferFunction->AddRGBPoint( 11151.0, 0.392157, 0.784314, 0.784314 );
    transferFunction->AddRGBPoint( 11152.0, 0.0392157, 0.784314, 0.784314 );
    transferFunction->AddRGBPoint( 11153.0, 0.866667, 0.862745, 0.0784314 );
    transferFunction->AddRGBPoint( 11154.0, 0.552941, 0.0784314, 0.392157 );
    transferFunction->AddRGBPoint( 11155.0, 0.239216, 0.862745, 0.392157 );
    transferFunction->AddRGBPoint( 11156.0, 0.552941, 0.235294, 0.0784314 );
    transferFunction->AddRGBPoint( 11157.0, 0.560784, 0.0784314, 0.862745 );
    transferFunction->AddRGBPoint( 11158.0, 0.396078, 0.235294, 0.862745 );
    transferFunction->AddRGBPoint( 11159.0, 0.0823529, 0.0784314, 0.54902 );
    transferFunction->AddRGBPoint( 11160.0, 0.239216, 0.0784314, 0.705882 );
    transferFunction->AddRGBPoint( 11161.0, 0.866667, 0.54902, 0.0784314 );
    transferFunction->AddRGBPoint( 11162.0, 0.552941, 0.392157, 0.862745 );
    transferFunction->AddRGBPoint( 11163.0, 0.866667, 0.392157, 0.0784314 );
    transferFunction->AddRGBPoint( 11164.0, 0.709804, 0.784314, 0.0784314 );
    transferFunction->AddRGBPoint( 11165.0, 0.396078, 0.0784314, 0.0784314 );
    transferFunction->AddRGBPoint( 11166.0, 0.396078, 0.392157, 0.705882 );
    transferFunction->AddRGBPoint( 11167.0, 0.709804, 0.862745, 0.0784314 );
    transferFunction->AddRGBPoint( 11168.0, 0.0823529, 0.54902, 0.784314 );
    transferFunction->AddRGBPoint( 11169.0, 0.0823529, 0.0784314, 0.941176 );
    transferFunction->AddRGBPoint( 11170.0, 0.0823529, 0.0784314, 0.784314 );
    transferFunction->AddRGBPoint( 11171.0, 0.0823529, 0.0784314, 0.235294 );
    transferFunction->AddRGBPoint( 11172.0, 0.396078, 0.235294, 0.235294 );
    transferFunction->AddRGBPoint( 11173.0, 0.0823529, 0.705882, 0.705882 );
    transferFunction->AddRGBPoint( 11174.0, 0.87451, 0.862745, 0.235294 );
    transferFunction->AddRGBPoint( 11175.0, 0.866667, 0.235294, 0.235294 );
    transferFunction->AddRGBPoint( 12101.0, 0.0901961, 0.862745, 0.235294 );
    transferFunction->AddRGBPoint( 12102.0, 0.0901961, 0.235294, 0.705882 );
    transferFunction->AddRGBPoint( 12103.0, 0.247059, 0.392157, 0.235294 );
    transferFunction->AddRGBPoint( 12104.0, 0.247059, 0.0784314, 0.862745 );
    transferFunction->AddRGBPoint( 12105.0, 0.0509804, 0, 0.980392 );
    transferFunction->AddRGBPoint( 12106.0, 0.101961, 0.235294, 0 );
    transferFunction->AddRGBPoint( 12107.0, 0.101961, 0.235294, 0.294118 );
    transferFunction->AddRGBPoint( 12108.0, 0.101961, 0.235294, 0.588235 );
    transferFunction->AddRGBPoint( 12109.0, 0.0980392, 0.235294, 0.980392 );
    transferFunction->AddRGBPoint( 12110.0, 0.235294, 0.0980392, 0.0980392 );
    transferFunction->AddRGBPoint( 12111.0, 0.705882, 0.0784314, 0.0784314 );
    transferFunction->AddRGBPoint( 12112.0, 0.862745, 0.0784314, 0.392157 );
    transferFunction->AddRGBPoint( 12113.0, 0.54902, 0.235294, 0.235294 );
    transferFunction->AddRGBPoint( 12114.0, 0.705882, 0.862745, 0.54902 );
    transferFunction->AddRGBPoint( 12115.0, 0.54902, 0.392157, 0.705882 );
    transferFunction->AddRGBPoint( 12116.0, 0.705882, 0.0784314, 0.54902 );
    transferFunction->AddRGBPoint( 12117.0, 0.0901961, 0.0392157, 0.0392157 );
    transferFunction->AddRGBPoint( 12118.0, 0.882353, 0.54902, 0.54902 );
    transferFunction->AddRGBPoint( 12119.0, 0.705882, 0.235294, 0.705882 );
    transferFunction->AddRGBPoint( 12120.0, 0.0784314, 0.862745, 0.235294 );
    transferFunction->AddRGBPoint( 12121.0, 0.235294, 0.0784314, 0.54902 );
    transferFunction->AddRGBPoint( 12122.0, 0.862745, 0.705882, 0.54902 );
    transferFunction->AddRGBPoint( 12123.0, 0.254902, 0.392157, 0.0784314 );
    transferFunction->AddRGBPoint( 12124.0, 0.862745, 0.235294, 0.0784314 );
    transferFunction->AddRGBPoint( 12125.0, 0.0784314, 0.235294, 0.862745 );
    transferFunction->AddRGBPoint( 12126.0, 0.392157, 0.392157, 0.235294 );
    transferFunction->AddRGBPoint( 12127.0, 0.862745, 0.705882, 0.862745 );
    transferFunction->AddRGBPoint( 12128.0, 0.0784314, 0.705882, 0.54902 );
    transferFunction->AddRGBPoint( 12129.0, 0.235294, 0.54902, 0.705882 );
    transferFunction->AddRGBPoint( 12130.0, 0.0980392, 0.0784314, 0.54902 );
    transferFunction->AddRGBPoint( 12131.0, 0.0784314, 0.235294, 0.392157 );
    transferFunction->AddRGBPoint( 12132.0, 0.235294, 0.862745, 0.0784314 );
    transferFunction->AddRGBPoint( 12133.0, 0.235294, 0.235294, 0.862745 );
    transferFunction->AddRGBPoint( 12134.0, 0.862745, 0.235294, 0.862745 );
    transferFunction->AddRGBPoint( 12135.0, 0.254902, 0.862745, 0.235294 );
    transferFunction->AddRGBPoint( 12136.0, 0.0980392, 0.54902, 0.0784314 );
    transferFunction->AddRGBPoint( 12137.0, 0.862745, 0.862745, 0.392157 );
    transferFunction->AddRGBPoint( 12138.0, 0.705882, 0.235294, 0.235294 );
    transferFunction->AddRGBPoint( 12139.0, 0.239216, 0.0784314, 0.862745 );
    transferFunction->AddRGBPoint( 12140.0, 0.239216, 0.0784314, 0.235294 );
    transferFunction->AddRGBPoint( 12141.0, 0.239216, 0.235294, 0.392157 );
    transferFunction->AddRGBPoint( 12142.0, 0.0980392, 0.0980392, 0.0980392 );
    transferFunction->AddRGBPoint( 12143.0, 0.54902, 0.0784314, 0.235294 );
    transferFunction->AddRGBPoint( 12144.0, 0.862745, 0.705882, 0.0784314 );
    transferFunction->AddRGBPoint( 12145.0, 0.247059, 0.705882, 0.705882 );
    transferFunction->AddRGBPoint( 12146.0, 0.866667, 0.0784314, 0.0392157 );
    transferFunction->AddRGBPoint( 12147.0, 0.866667, 0.0784314, 0.392157 );
    transferFunction->AddRGBPoint( 12148.0, 0.866667, 0.235294, 0.54902 );
    transferFunction->AddRGBPoint( 12149.0, 0.866667, 0.0784314, 0.862745 );
    transferFunction->AddRGBPoint( 12150.0, 0.239216, 0.862745, 0.862745 );
    transferFunction->AddRGBPoint( 12151.0, 0.392157, 0.784314, 0.784314 );
    transferFunction->AddRGBPoint( 12152.0, 0.0392157, 0.784314, 0.784314 );
    transferFunction->AddRGBPoint( 12153.0, 0.866667, 0.862745, 0.0784314 );
    transferFunction->AddRGBPoint( 12154.0, 0.552941, 0.0784314, 0.392157 );
    transferFunction->AddRGBPoint( 12155.0, 0.239216, 0.862745, 0.392157 );
    transferFunction->AddRGBPoint( 12156.0, 0.552941, 0.235294, 0.0784314 );
    transferFunction->AddRGBPoint( 12157.0, 0.560784, 0.0784314, 0.862745 );
    transferFunction->AddRGBPoint( 12158.0, 0.396078, 0.235294, 0.862745 );
    transferFunction->AddRGBPoint( 12159.0, 0.0823529, 0.0784314, 0.54902 );
    transferFunction->AddRGBPoint( 12160.0, 0.239216, 0.0784314, 0.705882 );
    transferFunction->AddRGBPoint( 12161.0, 0.866667, 0.54902, 0.0784314 );
    transferFunction->AddRGBPoint( 12162.0, 0.552941, 0.392157, 0.862745 );
    transferFunction->AddRGBPoint( 12163.0, 0.866667, 0.392157, 0.0784314 );
    transferFunction->AddRGBPoint( 12164.0, 0.709804, 0.784314, 0.0784314 );
    transferFunction->AddRGBPoint( 12165.0, 0.396078, 0.0784314, 0.0784314 );
    transferFunction->AddRGBPoint( 12166.0, 0.396078, 0.392157, 0.705882 );
    transferFunction->AddRGBPoint( 12167.0, 0.709804, 0.862745, 0.0784314 );
    transferFunction->AddRGBPoint( 12168.0, 0.0823529, 0.54902, 0.784314 );
    transferFunction->AddRGBPoint( 12169.0, 0.0823529, 0.0784314, 0.941176 );
    transferFunction->AddRGBPoint( 12170.0, 0.0823529, 0.0784314, 0.784314 );
    transferFunction->AddRGBPoint( 12171.0, 0.0823529, 0.0784314, 0.235294 );
    transferFunction->AddRGBPoint( 12172.0, 0.396078, 0.235294, 0.235294 );
    transferFunction->AddRGBPoint( 12173.0, 0.0823529, 0.705882, 0.705882 );
    transferFunction->AddRGBPoint( 12174.0, 0.87451, 0.862745, 0.235294 );
    transferFunction->AddRGBPoint( 12175.0, 0.866667, 0.235294, 0.235294 );
    transferFunction->AddRGBPoint( 13101.0, 0.0901961, 0.862745, 0.235294 );
    transferFunction->AddRGBPoint( 13102.0, 0.0901961, 0.235294, 0.705882 );
    transferFunction->AddRGBPoint( 13103.0, 0.247059, 0.392157, 0.235294 );
    transferFunction->AddRGBPoint( 13104.0, 0.247059, 0.0784314, 0.862745 );
    transferFunction->AddRGBPoint( 13105.0, 0.0509804, 0, 0.980392 );
    transferFunction->AddRGBPoint( 13106.0, 0.101961, 0.235294, 0 );
    transferFunction->AddRGBPoint( 13107.0, 0.101961, 0.235294, 0.294118 );
    transferFunction->AddRGBPoint( 13108.0, 0.101961, 0.235294, 0.588235 );
    transferFunction->AddRGBPoint( 13109.0, 0.0980392, 0.235294, 0.980392 );
    transferFunction->AddRGBPoint( 13110.0, 0.235294, 0.0980392, 0.0980392 );
    transferFunction->AddRGBPoint( 13111.0, 0.705882, 0.0784314, 0.0784314 );
    transferFunction->AddRGBPoint( 13112.0, 0.862745, 0.0784314, 0.392157 );
    transferFunction->AddRGBPoint( 13113.0, 0.54902, 0.235294, 0.235294 );
    transferFunction->AddRGBPoint( 13114.0, 0.705882, 0.862745, 0.54902 );
    transferFunction->AddRGBPoint( 13115.0, 0.54902, 0.392157, 0.705882 );
    transferFunction->AddRGBPoint( 13116.0, 0.705882, 0.0784314, 0.54902 );
    transferFunction->AddRGBPoint( 13117.0, 0.0901961, 0.0392157, 0.0392157 );
    transferFunction->AddRGBPoint( 13118.0, 0.882353, 0.54902, 0.54902 );
    transferFunction->AddRGBPoint( 13119.0, 0.705882, 0.235294, 0.705882 );
    transferFunction->AddRGBPoint( 13120.0, 0.0784314, 0.862745, 0.235294 );
    transferFunction->AddRGBPoint( 13121.0, 0.235294, 0.0784314, 0.54902 );
    transferFunction->AddRGBPoint( 13122.0, 0.862745, 0.705882, 0.54902 );
    transferFunction->AddRGBPoint( 13123.0, 0.254902, 0.392157, 0.0784314 );
    transferFunction->AddRGBPoint( 13124.0, 0.862745, 0.235294, 0.0784314 );
    transferFunction->AddRGBPoint( 13125.0, 0.0784314, 0.235294, 0.862745 );
    transferFunction->AddRGBPoint( 13126.0, 0.392157, 0.392157, 0.235294 );
    transferFunction->AddRGBPoint( 13127.0, 0.862745, 0.705882, 0.862745 );
    transferFunction->AddRGBPoint( 13128.0, 0.0784314, 0.705882, 0.54902 );
    transferFunction->AddRGBPoint( 13129.0, 0.235294, 0.54902, 0.705882 );
    transferFunction->AddRGBPoint( 13130.0, 0.0980392, 0.0784314, 0.54902 );
    transferFunction->AddRGBPoint( 13131.0, 0.0784314, 0.235294, 0.392157 );
    transferFunction->AddRGBPoint( 13132.0, 0.235294, 0.862745, 0.0784314 );
    transferFunction->AddRGBPoint( 13133.0, 0.235294, 0.235294, 0.862745 );
    transferFunction->AddRGBPoint( 13134.0, 0.862745, 0.235294, 0.862745 );
    transferFunction->AddRGBPoint( 13135.0, 0.254902, 0.862745, 0.235294 );
    transferFunction->AddRGBPoint( 13136.0, 0.0980392, 0.54902, 0.0784314 );
    transferFunction->AddRGBPoint( 13137.0, 0.862745, 0.862745, 0.392157 );
    transferFunction->AddRGBPoint( 13138.0, 0.705882, 0.235294, 0.235294 );
    transferFunction->AddRGBPoint( 13139.0, 0.239216, 0.0784314, 0.862745 );
    transferFunction->AddRGBPoint( 13140.0, 0.239216, 0.0784314, 0.235294 );
    transferFunction->AddRGBPoint( 13141.0, 0.239216, 0.235294, 0.392157 );
    transferFunction->AddRGBPoint( 13142.0, 0.0980392, 0.0980392, 0.0980392 );
    transferFunction->AddRGBPoint( 13143.0, 0.54902, 0.0784314, 0.235294 );
    transferFunction->AddRGBPoint( 13144.0, 0.862745, 0.705882, 0.0784314 );
    transferFunction->AddRGBPoint( 13145.0, 0.247059, 0.705882, 0.705882 );
    transferFunction->AddRGBPoint( 13146.0, 0.866667, 0.0784314, 0.0392157 );
    transferFunction->AddRGBPoint( 13147.0, 0.866667, 0.0784314, 0.392157 );
    transferFunction->AddRGBPoint( 13148.0, 0.866667, 0.235294, 0.54902 );
    transferFunction->AddRGBPoint( 13149.0, 0.866667, 0.0784314, 0.862745 );
    transferFunction->AddRGBPoint( 13150.0, 0.239216, 0.862745, 0.862745 );
    transferFunction->AddRGBPoint( 13151.0, 0.392157, 0.784314, 0.784314 );
    transferFunction->AddRGBPoint( 13152.0, 0.0392157, 0.784314, 0.784314 );
    transferFunction->AddRGBPoint( 13153.0, 0.866667, 0.862745, 0.0784314 );
    transferFunction->AddRGBPoint( 13154.0, 0.552941, 0.0784314, 0.392157 );
    transferFunction->AddRGBPoint( 13155.0, 0.239216, 0.862745, 0.392157 );
    transferFunction->AddRGBPoint( 13156.0, 0.552941, 0.235294, 0.0784314 );
    transferFunction->AddRGBPoint( 13157.0, 0.560784, 0.0784314, 0.862745 );
    transferFunction->AddRGBPoint( 13158.0, 0.396078, 0.235294, 0.862745 );
    transferFunction->AddRGBPoint( 13159.0, 0.0823529, 0.0784314, 0.54902 );
    transferFunction->AddRGBPoint( 13160.0, 0.239216, 0.0784314, 0.705882 );
    transferFunction->AddRGBPoint( 13161.0, 0.866667, 0.54902, 0.0784314 );
    transferFunction->AddRGBPoint( 13162.0, 0.552941, 0.392157, 0.862745 );
    transferFunction->AddRGBPoint( 13163.0, 0.866667, 0.392157, 0.0784314 );
    transferFunction->AddRGBPoint( 13164.0, 0.709804, 0.784314, 0.0784314 );
    transferFunction->AddRGBPoint( 13165.0, 0.396078, 0.0784314, 0.0784314 );
    transferFunction->AddRGBPoint( 13166.0, 0.396078, 0.392157, 0.705882 );
    transferFunction->AddRGBPoint( 13167.0, 0.709804, 0.862745, 0.0784314 );
    transferFunction->AddRGBPoint( 13168.0, 0.0823529, 0.54902, 0.784314 );
    transferFunction->AddRGBPoint( 13169.0, 0.0823529, 0.0784314, 0.941176 );
    transferFunction->AddRGBPoint( 13170.0, 0.0823529, 0.0784314, 0.784314 );
    transferFunction->AddRGBPoint( 13171.0, 0.0823529, 0.0784314, 0.235294 );
    transferFunction->AddRGBPoint( 13172.0, 0.396078, 0.235294, 0.235294 );
    transferFunction->AddRGBPoint( 13173.0, 0.0823529, 0.705882, 0.705882 );
    transferFunction->AddRGBPoint( 13174.0, 0.87451, 0.862745, 0.235294 );
    transferFunction->AddRGBPoint( 13175.0, 0.866667, 0.235294, 0.235294 );
    transferFunction->AddRGBPoint( 14101.0, 0.0901961, 0.862745, 0.235294 );
    transferFunction->AddRGBPoint( 14102.0, 0.0901961, 0.235294, 0.705882 );
    transferFunction->AddRGBPoint( 14103.0, 0.247059, 0.392157, 0.235294 );
    transferFunction->AddRGBPoint( 14104.0, 0.247059, 0.0784314, 0.862745 );
    transferFunction->AddRGBPoint( 14105.0, 0.0509804, 0, 0.980392 );
    transferFunction->AddRGBPoint( 14106.0, 0.101961, 0.235294, 0 );
    transferFunction->AddRGBPoint( 14107.0, 0.101961, 0.235294, 0.294118 );
    transferFunction->AddRGBPoint( 14108.0, 0.101961, 0.235294, 0.588235 );
    transferFunction->AddRGBPoint( 14109.0, 0.0980392, 0.235294, 0.980392 );
    transferFunction->AddRGBPoint( 14110.0, 0.235294, 0.0980392, 0.0980392 );
    transferFunction->AddRGBPoint( 14111.0, 0.705882, 0.0784314, 0.0784314 );
    transferFunction->AddRGBPoint( 14112.0, 0.862745, 0.0784314, 0.392157 );
    transferFunction->AddRGBPoint( 14113.0, 0.54902, 0.235294, 0.235294 );
    transferFunction->AddRGBPoint( 14114.0, 0.705882, 0.862745, 0.54902 );
    transferFunction->AddRGBPoint( 14115.0, 0.54902, 0.392157, 0.705882 );
    transferFunction->AddRGBPoint( 14116.0, 0.705882, 0.0784314, 0.54902 );
    transferFunction->AddRGBPoint( 14117.0, 0.0901961, 0.0392157, 0.0392157 );
    transferFunction->AddRGBPoint( 14118.0, 0.882353, 0.54902, 0.54902 );
    transferFunction->AddRGBPoint( 14119.0, 0.705882, 0.235294, 0.705882 );
    transferFunction->AddRGBPoint( 14120.0, 0.0784314, 0.862745, 0.235294 );
    transferFunction->AddRGBPoint( 14121.0, 0.235294, 0.0784314, 0.54902 );
    transferFunction->AddRGBPoint( 14122.0, 0.862745, 0.705882, 0.54902 );
    transferFunction->AddRGBPoint( 14123.0, 0.254902, 0.392157, 0.0784314 );
    transferFunction->AddRGBPoint( 14124.0, 0.862745, 0.235294, 0.0784314 );
    transferFunction->AddRGBPoint( 14125.0, 0.0784314, 0.235294, 0.862745 );
    transferFunction->AddRGBPoint( 14126.0, 0.392157, 0.392157, 0.235294 );
    transferFunction->AddRGBPoint( 14127.0, 0.862745, 0.705882, 0.862745 );
    transferFunction->AddRGBPoint( 14128.0, 0.0784314, 0.705882, 0.54902 );
    transferFunction->AddRGBPoint( 14129.0, 0.235294, 0.54902, 0.705882 );
    transferFunction->AddRGBPoint( 14130.0, 0.0980392, 0.0784314, 0.54902 );
    transferFunction->AddRGBPoint( 14131.0, 0.0784314, 0.235294, 0.392157 );
    transferFunction->AddRGBPoint( 14132.0, 0.235294, 0.862745, 0.0784314 );
    transferFunction->AddRGBPoint( 14133.0, 0.235294, 0.235294, 0.862745 );
    transferFunction->AddRGBPoint( 14134.0, 0.862745, 0.235294, 0.862745 );
    transferFunction->AddRGBPoint( 14135.0, 0.254902, 0.862745, 0.235294 );
    transferFunction->AddRGBPoint( 14136.0, 0.0980392, 0.54902, 0.0784314 );
    transferFunction->AddRGBPoint( 14137.0, 0.862745, 0.862745, 0.392157 );
    transferFunction->AddRGBPoint( 14138.0, 0.705882, 0.235294, 0.235294 );
    transferFunction->AddRGBPoint( 14139.0, 0.239216, 0.0784314, 0.862745 );
    transferFunction->AddRGBPoint( 14140.0, 0.239216, 0.0784314, 0.235294 );
    transferFunction->AddRGBPoint( 14141.0, 0.239216, 0.235294, 0.392157 );
    transferFunction->AddRGBPoint( 14142.0, 0.0980392, 0.0980392, 0.0980392 );
    transferFunction->AddRGBPoint( 14143.0, 0.54902, 0.0784314, 0.235294 );
    transferFunction->AddRGBPoint( 14144.0, 0.862745, 0.705882, 0.0784314 );
    transferFunction->AddRGBPoint( 14145.0, 0.247059, 0.705882, 0.705882 );
    transferFunction->AddRGBPoint( 14146.0, 0.866667, 0.0784314, 0.0392157 );
    transferFunction->AddRGBPoint( 14147.0, 0.866667, 0.0784314, 0.392157 );
    transferFunction->AddRGBPoint( 14148.0, 0.866667, 0.235294, 0.54902 );
    transferFunction->AddRGBPoint( 14149.0, 0.866667, 0.0784314, 0.862745 );
    transferFunction->AddRGBPoint( 14150.0, 0.239216, 0.862745, 0.862745 );
    transferFunction->AddRGBPoint( 14151.0, 0.392157, 0.784314, 0.784314 );
    transferFunction->AddRGBPoint( 14152.0, 0.0392157, 0.784314, 0.784314 );
    transferFunction->AddRGBPoint( 14153.0, 0.866667, 0.862745, 0.0784314 );
    transferFunction->AddRGBPoint( 14154.0, 0.552941, 0.0784314, 0.392157 );
    transferFunction->AddRGBPoint( 14155.0, 0.239216, 0.862745, 0.392157 );
    transferFunction->AddRGBPoint( 14156.0, 0.552941, 0.235294, 0.0784314 );
    transferFunction->AddRGBPoint( 14157.0, 0.560784, 0.0784314, 0.862745 );
    transferFunction->AddRGBPoint( 14158.0, 0.396078, 0.235294, 0.862745 );
    transferFunction->AddRGBPoint( 14159.0, 0.0823529, 0.0784314, 0.54902 );
    transferFunction->AddRGBPoint( 14160.0, 0.239216, 0.0784314, 0.705882 );
    transferFunction->AddRGBPoint( 14161.0, 0.866667, 0.54902, 0.0784314 );
    transferFunction->AddRGBPoint( 14162.0, 0.552941, 0.392157, 0.862745 );
    transferFunction->AddRGBPoint( 14163.0, 0.866667, 0.392157, 0.0784314 );
    transferFunction->AddRGBPoint( 14164.0, 0.709804, 0.784314, 0.0784314 );
    transferFunction->AddRGBPoint( 14165.0, 0.396078, 0.0784314, 0.0784314 );
    transferFunction->AddRGBPoint( 14166.0, 0.396078, 0.392157, 0.705882 );
    transferFunction->AddRGBPoint( 14167.0, 0.709804, 0.862745, 0.0784314 );
    transferFunction->AddRGBPoint( 14168.0, 0.0823529, 0.54902, 0.784314 );
    transferFunction->AddRGBPoint( 14169.0, 0.0823529, 0.0784314, 0.941176 );
    transferFunction->AddRGBPoint( 14170.0, 0.0823529, 0.0784314, 0.784314 );
    transferFunction->AddRGBPoint( 14171.0, 0.0823529, 0.0784314, 0.235294 );
    transferFunction->AddRGBPoint( 14172.0, 0.396078, 0.235294, 0.235294 );
    transferFunction->AddRGBPoint( 14173.0, 0.0823529, 0.705882, 0.705882 );
    transferFunction->AddRGBPoint( 14174.0, 0.87451, 0.862745, 0.235294 );
    transferFunction->AddRGBPoint( 14175.0, 0.866667, 0.235294, 0.235294 );
    transferFunction->AddRGBPoint( 0.0, 0.0, 0.0, 0.0 );
    transferFunction->AddRGBPoint( 256.0, 0.0, 0.0, 0.0 );
    transferFunction->AddRGBPoint( 1100.0, 0.0, 0.0, 0.0 );
    transferFunction->AddRGBPoint( 2100.0, 0.0, 0.0, 0.0 );
    transferFunction->AddRGBPoint( 3100.0, 0.0, 0.0, 0.0 );
    transferFunction->AddRGBPoint( 4100.0, 0.0, 0.0, 0.0 );
    transferFunction->AddRGBPoint( 11100.0, 0.0, 0.0, 0.0 );
    transferFunction->AddRGBPoint( 12100.0, 0.0, 0.0, 0.0 );
    transferFunction->AddRGBPoint( 13100.0, 0.0, 0.0, 0.0 );
    transferFunction->AddRGBPoint( 14100.0, 0.0, 0.0, 0.0 );
    return transferFunction;
}

FreeSurferParcellationTranslator::NameTable FreeSurferParcellationTranslator::CreateNameTable()
{
  mitk::FreeSurferParcellationTranslator::NameTable nameTable;
    nameTable["0"] = "Unknown";
    nameTable["1"] = "Left-Cerebral-Exterior";
    nameTable["2"] = "Left-Cerebral-White-Matter";
    nameTable["3"] = "Left-Cerebral-Cortex";
    nameTable["4"] = "Left-Lateral-Ventricle";
    nameTable["5"] = "Left-Inf-Lat-Vent";
    nameTable["6"] = "Left-Cerebellum-Exterior";
    nameTable["7"] = "Left-Cerebellum-White-Matter";
    nameTable["8"] = "Left-Cerebellum-Cortex";
    nameTable["9"] = "Left-Thalamus";
    nameTable["10"] = "Left-Thalamus-Proper";
    nameTable["11"] = "Left-Caudate";
    nameTable["12"] = "Left-Putamen";
    nameTable["13"] = "Left-Pallidum";
    nameTable["14"] = "3rd-Ventricle";
    nameTable["15"] = "4th-Ventricle";
    nameTable["16"] = "Brain-Stem";
    nameTable["17"] = "Left-Hippocampus";
    nameTable["18"] = "Left-Amygdala";
    nameTable["19"] = "Left-Insula";
    nameTable["20"] = "Left-Operculum";
    nameTable["21"] = "Line-1";
    nameTable["22"] = "Line-2";
    nameTable["23"] = "Line-3";
    nameTable["24"] = "CSF";
    nameTable["25"] = "Left-Lesion";
    nameTable["26"] = "Left-Accumbens-area";
    nameTable["27"] = "Left-Substancia-Nigra";
    nameTable["28"] = "Left-VentralDC";
    nameTable["29"] = "Left-undetermined";
    nameTable["30"] = "Left-vessel";
    nameTable["31"] = "Left-choroid-plexus";
    nameTable["32"] = "Left-F3orb";
    nameTable["33"] = "Left-lOg";
    nameTable["34"] = "Left-aOg";
    nameTable["35"] = "Left-mOg";
    nameTable["36"] = "Left-pOg";
    nameTable["37"] = "Left-Stellate";
    nameTable["38"] = "Left-Porg";
    nameTable["39"] = "Left-Aorg";
    nameTable["40"] = "Right-Cerebral-Exterior";
    nameTable["41"] = "Right-Cerebral-White-Matter";
    nameTable["42"] = "Right-Cerebral-Cortex";
    nameTable["43"] = "Right-Lateral-Ventricle";
    nameTable["44"] = "Right-Inf-Lat-Vent";
    nameTable["45"] = "Right-Cerebellum-Exterior";
    nameTable["46"] = "Right-Cerebellum-White-Matter";
    nameTable["47"] = "Right-Cerebellum-Cortex";
    nameTable["48"] = "Right-Thalamus";
    nameTable["49"] = "Right-Thalamus-Proper";
    nameTable["50"] = "Right-Caudate";
    nameTable["51"] = "Right-Putamen";
    nameTable["52"] = "Right-Pallidum";
    nameTable["53"] = "Right-Hippocampus";
    nameTable["54"] = "Right-Amygdala";
    nameTable["55"] = "Right-Insula";
    nameTable["56"] = "Right-Operculum";
    nameTable["57"] = "Right-Lesion";
    nameTable["58"] = "Right-Accumbens-area";
    nameTable["59"] = "Right-Substancia-Nigra";
    nameTable["60"] = "Right-VentralDC";
    nameTable["61"] = "Right-undetermined";
    nameTable["62"] = "Right-vessel";
    nameTable["63"] = "Right-choroid-plexus";
    nameTable["64"] = "Right-F3orb";
    nameTable["65"] = "Right-lOg";
    nameTable["66"] = "Right-aOg";
    nameTable["67"] = "Right-mOg";
    nameTable["68"] = "Right-pOg";
    nameTable["69"] = "Right-Stellate";
    nameTable["70"] = "Right-Porg";
    nameTable["71"] = "Right-Aorg";
    nameTable["72"] = "5th-Ventricle";
    nameTable["73"] = "Left-Interior";
    nameTable["74"] = "Right-Interior";
    nameTable["77"] = "WM-hypointensities";
    nameTable["78"] = "Left-WM-hypointensities";
    nameTable["79"] = "Right-WM-hypointensities";
    nameTable["80"] = "non-WM-hypointensities";
    nameTable["81"] = "Left-non-WM-hypointensities";
    nameTable["82"] = "Right-non-WM-hypointensities";
    nameTable["83"] = "Left-F1";
    nameTable["84"] = "Right-F1";
    nameTable["85"] = "Optic-Chiasm";
    nameTable["192"] = "Corpus_Callosum";
    nameTable["86"] = "Left_future_WMSA";
    nameTable["87"] = "Right_future_WMSA";
    nameTable["88"] = "future_WMSA";
    nameTable["96"] = "Left-Amygdala-Anterior";
    nameTable["97"] = "Right-Amygdala-Anterior";
    nameTable["98"] = "Dura";
    nameTable["100"] = "Left-wm-intensity-abnormality";
    nameTable["101"] = "Left-caudate-intensity-abnormality";
    nameTable["102"] = "Left-putamen-intensity-abnormality";
    nameTable["103"] = "Left-accumbens-intensity-abnormality";
    nameTable["104"] = "Left-pallidum-intensity-abnormality";
    nameTable["105"] = "Left-amygdala-intensity-abnormality";
    nameTable["106"] = "Left-hippocampus-intensity-abnormality";
    nameTable["107"] = "Left-thalamus-intensity-abnormality";
    nameTable["108"] = "Left-VDC-intensity-abnormality";
    nameTable["109"] = "Right-wm-intensity-abnormality";
    nameTable["110"] = "Right-caudate-intensity-abnormality";
    nameTable["111"] = "Right-putamen-intensity-abnormality";
    nameTable["112"] = "Right-accumbens-intensity-abnormality";
    nameTable["113"] = "Right-pallidum-intensity-abnormality";
    nameTable["114"] = "Right-amygdala-intensity-abnormality";
    nameTable["115"] = "Right-hippocampus-intensity-abnormality";
    nameTable["116"] = "Right-thalamus-intensity-abnormality";
    nameTable["117"] = "Right-VDC-intensity-abnormality";
    nameTable["118"] = "Epidermis";
    nameTable["119"] = "Conn-Tissue";
    nameTable["120"] = "SC-Fat-Muscle";
    nameTable["121"] = "Cranium";
    nameTable["122"] = "CSF-SA";
    nameTable["123"] = "Muscle";
    nameTable["124"] = "Ear";
    nameTable["125"] = "Adipose";
    nameTable["126"] = "Spinal-Cord";
    nameTable["127"] = "Soft-Tissue";
    nameTable["128"] = "Nerve";
    nameTable["129"] = "Bone";
    nameTable["130"] = "Air";
    nameTable["131"] = "Orbital-Fat";
    nameTable["132"] = "Tongue";
    nameTable["133"] = "Nasal-Structures";
    nameTable["134"] = "Globe";
    nameTable["135"] = "Teeth";
    nameTable["136"] = "Left-Caudate-Putamen";
    nameTable["137"] = "Right-Caudate-Putamen";
    nameTable["138"] = "Left-Claustrum";
    nameTable["139"] = "Right-Claustrum";
    nameTable["140"] = "Cornea";
    nameTable["142"] = "Diploe";
    nameTable["143"] = "Vitreous-Humor";
    nameTable["144"] = "Lens";
    nameTable["145"] = "Aqueous-Humor";
    nameTable["146"] = "Outer-Table";
    nameTable["147"] = "Inner-Table";
    nameTable["148"] = "Periosteum";
    nameTable["149"] = "Endosteum";
    nameTable["150"] = "R-C-S";
    nameTable["151"] = "Iris";
    nameTable["152"] = "SC-Adipose-Muscle";
    nameTable["153"] = "SC-Tissue";
    nameTable["154"] = "Orbital-Adipose";
    nameTable["155"] = "Left-IntCapsule-Ant";
    nameTable["156"] = "Right-IntCapsule-Ant";
    nameTable["157"] = "Left-IntCapsule-Pos";
    nameTable["158"] = "Right-IntCapsule-Pos";
    nameTable["159"] = "Left-Cerebral-WM-unmyelinated";
    nameTable["160"] = "Right-Cerebral-WM-unmyelinated";
    nameTable["161"] = "Left-Cerebral-WM-myelinated";
    nameTable["162"] = "Right-Cerebral-WM-myelinated";
    nameTable["163"] = "Left-Subcortical-Gray-Matter";
    nameTable["164"] = "Right-Subcortical-Gray-Matter";
    nameTable["165"] = "Skull";
    nameTable["166"] = "Posterior-fossa";
    nameTable["167"] = "Scalp";
    nameTable["168"] = "Hematoma";
    nameTable["169"] = "Left-Basal-Ganglia";
    nameTable["176"] = "Right-Basal-Ganglia";
    nameTable["170"] = "brainstem";
    nameTable["171"] = "DCG";
    nameTable["172"] = "Vermis";
    nameTable["173"] = "Midbrain";
    nameTable["174"] = "Pons";
    nameTable["175"] = "Medulla";
    nameTable["180"] = "Left-Cortical-Dysplasia";
    nameTable["181"] = "Right-Cortical-Dysplasia";
    nameTable["193"] = "Left-hippocampal_fissure";
    nameTable["194"] = "Left-CADG-head";
    nameTable["195"] = "Left-subiculum";
    nameTable["196"] = "Left-fimbria";
    nameTable["197"] = "Right-hippocampal_fissure";
    nameTable["198"] = "Right-CADG-head";
    nameTable["199"] = "Right-subiculum";
    nameTable["200"] = "Right-fimbria";
    nameTable["201"] = "alveus";
    nameTable["202"] = "perforant_pathway";
    nameTable["203"] = "parasubiculum";
    nameTable["204"] = "presubiculum";
    nameTable["205"] = "subiculum";
    nameTable["206"] = "CA1";
    nameTable["207"] = "CA2";
    nameTable["208"] = "CA3";
    nameTable["209"] = "CA4";
    nameTable["210"] = "GC-ML-DG";
    nameTable["211"] = "HATA";
    nameTable["212"] = "fimbria";
    nameTable["213"] = "lateral_ventricle";
    nameTable["214"] = "molecular_layer_HP";
    nameTable["215"] = "hippocampal_fissure";
    nameTable["216"] = "entorhinal_cortex";
    nameTable["217"] = "molecular_layer_subiculum";
    nameTable["218"] = "Amygdala";
    nameTable["219"] = "Cerebral_White_Matter";
    nameTable["220"] = "Cerebral_Cortex";
    nameTable["221"] = "Inf_Lat_Vent";
    nameTable["222"] = "Perirhinal";
    nameTable["223"] = "Cerebral_White_Matter_Edge";
    nameTable["224"] = "Background";
    nameTable["225"] = "Ectorhinal";
    nameTable["226"] = "HP_tail";
    nameTable["250"] = "Fornix";
    nameTable["251"] = "CC_Posterior";
    nameTable["252"] = "CC_Mid_Posterior";
    nameTable["253"] = "CC_Central";
    nameTable["254"] = "CC_Mid_Anterior";
    nameTable["255"] = "CC_Anterior";
    nameTable["256"] = "Voxel-Unchanged";
    nameTable["331"] = "Aorta";
    nameTable["332"] = "Left-Common-IliacA";
    nameTable["333"] = "Right-Common-IliacA";
    nameTable["334"] = "Left-External-IliacA";
    nameTable["335"] = "Right-External-IliacA";
    nameTable["336"] = "Left-Internal-IliacA";
    nameTable["337"] = "Right-Internal-IliacA";
    nameTable["338"] = "Left-Lateral-SacralA";
    nameTable["339"] = "Right-Lateral-SacralA";
    nameTable["340"] = "Left-ObturatorA";
    nameTable["341"] = "Right-ObturatorA";
    nameTable["342"] = "Left-Internal-PudendalA";
    nameTable["343"] = "Right-Internal-PudendalA";
    nameTable["344"] = "Left-UmbilicalA";
    nameTable["345"] = "Right-UmbilicalA";
    nameTable["346"] = "Left-Inf-RectalA";
    nameTable["347"] = "Right-Inf-RectalA";
    nameTable["348"] = "Left-Common-IliacV";
    nameTable["349"] = "Right-Common-IliacV";
    nameTable["350"] = "Left-External-IliacV";
    nameTable["351"] = "Right-External-IliacV";
    nameTable["352"] = "Left-Internal-IliacV";
    nameTable["353"] = "Right-Internal-IliacV";
    nameTable["354"] = "Left-ObturatorV";
    nameTable["355"] = "Right-ObturatorV";
    nameTable["356"] = "Left-Internal-PudendalV";
    nameTable["357"] = "Right-Internal-PudendalV";
    nameTable["358"] = "Pos-Lymph";
    nameTable["359"] = "Neg-Lymph";
    nameTable["400"] = "V1";
    nameTable["401"] = "V2";
    nameTable["402"] = "BA44";
    nameTable["403"] = "BA45";
    nameTable["404"] = "BA4a";
    nameTable["405"] = "BA4p";
    nameTable["406"] = "BA6";
    nameTable["407"] = "BA2";
    nameTable["408"] = "BA1_old";
    nameTable["409"] = "BAun2";
    nameTable["410"] = "BA1";
    nameTable["411"] = "BA2b";
    nameTable["412"] = "BA3a";
    nameTable["413"] = "BA3b";
    nameTable["414"] = "MT";
    nameTable["415"] = "AIPS_AIP_l";
    nameTable["416"] = "AIPS_AIP_r";
    nameTable["417"] = "AIPS_VIP_l";
    nameTable["418"] = "AIPS_VIP_r";
    nameTable["419"] = "IPL_PFcm_l";
    nameTable["420"] = "IPL_PFcm_r";
    nameTable["421"] = "IPL_PF_l";
    nameTable["422"] = "IPL_PFm_l";
    nameTable["423"] = "IPL_PFm_r";
    nameTable["424"] = "IPL_PFop_l";
    nameTable["425"] = "IPL_PFop_r";
    nameTable["426"] = "IPL_PF_r";
    nameTable["427"] = "IPL_PFt_l";
    nameTable["428"] = "IPL_PFt_r";
    nameTable["429"] = "IPL_PGa_l";
    nameTable["430"] = "IPL_PGa_r";
    nameTable["431"] = "IPL_PGp_l";
    nameTable["432"] = "IPL_PGp_r";
    nameTable["433"] = "Visual_V3d_l";
    nameTable["434"] = "Visual_V3d_r";
    nameTable["435"] = "Visual_V4_l";
    nameTable["436"] = "Visual_V4_r";
    nameTable["437"] = "Visual_V5_b";
    nameTable["438"] = "Visual_VP_l";
    nameTable["439"] = "Visual_VP_r";
    nameTable["498"] = "wmsa";
    nameTable["499"] = "other_wmsa";
    nameTable["500"] = "right_CA2_3";
    nameTable["501"] = "right_alveus";
    nameTable["502"] = "right_CA1";
    nameTable["503"] = "right_fimbria";
    nameTable["504"] = "right_presubiculum";
    nameTable["505"] = "right_hippocampal_fissure";
    nameTable["506"] = "right_CA4_DG";
    nameTable["507"] = "right_subiculum";
    nameTable["508"] = "right_fornix";
    nameTable["550"] = "left_CA2_3";
    nameTable["551"] = "left_alveus";
    nameTable["552"] = "left_CA1";
    nameTable["553"] = "left_fimbria";
    nameTable["554"] = "left_presubiculum";
    nameTable["555"] = "left_hippocampal_fissure";
    nameTable["556"] = "left_CA4_DG";
    nameTable["557"] = "left_subiculum";
    nameTable["558"] = "left_fornix";
    nameTable["600"] = "Tumor";
    nameTable["601"] = "Cbm_Left_I_IV";
    nameTable["602"] = "Cbm_Right_I_IV";
    nameTable["603"] = "Cbm_Left_V";
    nameTable["604"] = "Cbm_Right_V";
    nameTable["605"] = "Cbm_Left_VI";
    nameTable["606"] = "Cbm_Vermis_VI";
    nameTable["607"] = "Cbm_Right_VI";
    nameTable["608"] = "Cbm_Left_CrusI";
    nameTable["609"] = "Cbm_Vermis_CrusI";
    nameTable["610"] = "Cbm_Right_CrusI";
    nameTable["611"] = "Cbm_Left_CrusII";
    nameTable["612"] = "Cbm_Vermis_CrusII";
    nameTable["613"] = "Cbm_Right_CrusII";
    nameTable["614"] = "Cbm_Left_VIIb";
    nameTable["615"] = "Cbm_Vermis_VIIb";
    nameTable["616"] = "Cbm_Right_VIIb";
    nameTable["617"] = "Cbm_Left_VIIIa";
    nameTable["618"] = "Cbm_Vermis_VIIIa";
    nameTable["619"] = "Cbm_Right_VIIIa";
    nameTable["620"] = "Cbm_Left_VIIIb";
    nameTable["621"] = "Cbm_Vermis_VIIIb";
    nameTable["622"] = "Cbm_Right_VIIIb";
    nameTable["623"] = "Cbm_Left_IX";
    nameTable["624"] = "Cbm_Vermis_IX";
    nameTable["625"] = "Cbm_Right_IX";
    nameTable["626"] = "Cbm_Left_X";
    nameTable["627"] = "Cbm_Vermis_X";
    nameTable["628"] = "Cbm_Right_X";
    nameTable["640"] = "Cbm_Right_I_V_med";
    nameTable["641"] = "Cbm_Right_I_V_mid";
    nameTable["642"] = "Cbm_Right_VI_med";
    nameTable["643"] = "Cbm_Right_VI_mid";
    nameTable["644"] = "Cbm_Right_VI_lat";
    nameTable["645"] = "Cbm_Right_CrusI_med";
    nameTable["646"] = "Cbm_Right_CrusI_mid";
    nameTable["647"] = "Cbm_Right_CrusI_lat";
    nameTable["648"] = "Cbm_Right_CrusII_med";
    nameTable["649"] = "Cbm_Right_CrusII_mid";
    nameTable["650"] = "Cbm_Right_CrusII_lat";
    nameTable["651"] = "Cbm_Right_7med";
    nameTable["652"] = "Cbm_Right_7mid";
    nameTable["653"] = "Cbm_Right_7lat";
    nameTable["654"] = "Cbm_Right_8med";
    nameTable["655"] = "Cbm_Right_8mid";
    nameTable["656"] = "Cbm_Right_8lat";
    nameTable["657"] = "Cbm_Right_PUNs";
    nameTable["658"] = "Cbm_Right_TONs";
    nameTable["659"] = "Cbm_Right_FLOs";
    nameTable["660"] = "Cbm_Left_I_V_med";
    nameTable["661"] = "Cbm_Left_I_V_mid";
    nameTable["662"] = "Cbm_Left_VI_med";
    nameTable["663"] = "Cbm_Left_VI_mid";
    nameTable["664"] = "Cbm_Left_VI_lat";
    nameTable["665"] = "Cbm_Left_CrusI_med";
    nameTable["666"] = "Cbm_Left_CrusI_mid";
    nameTable["667"] = "Cbm_Left_CrusI_lat";
    nameTable["668"] = "Cbm_Left_CrusII_med";
    nameTable["669"] = "Cbm_Left_CrusII_mid";
    nameTable["670"] = "Cbm_Left_CrusII_lat";
    nameTable["671"] = "Cbm_Left_7med";
    nameTable["672"] = "Cbm_Left_7mid";
    nameTable["673"] = "Cbm_Left_7lat";
    nameTable["674"] = "Cbm_Left_8med";
    nameTable["675"] = "Cbm_Left_8mid";
    nameTable["676"] = "Cbm_Left_8lat";
    nameTable["677"] = "Cbm_Left_PUNs";
    nameTable["678"] = "Cbm_Left_TONs";
    nameTable["679"] = "Cbm_Left_FLOs";
    nameTable["701"] = "CSF-FSL-FAST";
    nameTable["702"] = "GrayMatter-FSL-FAST";
    nameTable["703"] = "WhiteMatter-FSL-FAST";
    nameTable["999"] = "SUSPICIOUS";
    nameTable["1000"] = "ctx-lh-unknown";
    nameTable["1001"] = "ctx-lh-bankssts";
    nameTable["1002"] = "ctx-lh-caudalanteriorcingulate";
    nameTable["1003"] = "ctx-lh-caudalmiddlefrontal";
    nameTable["1004"] = "ctx-lh-corpuscallosum";
    nameTable["1005"] = "ctx-lh-cuneus";
    nameTable["1006"] = "ctx-lh-entorhinal";
    nameTable["1007"] = "ctx-lh-fusiform";
    nameTable["1008"] = "ctx-lh-inferiorparietal";
    nameTable["1009"] = "ctx-lh-inferiortemporal";
    nameTable["1010"] = "ctx-lh-isthmuscingulate";
    nameTable["1011"] = "ctx-lh-lateraloccipital";
    nameTable["1012"] = "ctx-lh-lateralorbitofrontal";
    nameTable["1013"] = "ctx-lh-lingual";
    nameTable["1014"] = "ctx-lh-medialorbitofrontal";
    nameTable["1015"] = "ctx-lh-middletemporal";
    nameTable["1016"] = "ctx-lh-parahippocampal";
    nameTable["1017"] = "ctx-lh-paracentral";
    nameTable["1018"] = "ctx-lh-parsopercularis";
    nameTable["1019"] = "ctx-lh-parsorbitalis";
    nameTable["1020"] = "ctx-lh-parstriangularis";
    nameTable["1021"] = "ctx-lh-pericalcarine";
    nameTable["1022"] = "ctx-lh-postcentral";
    nameTable["1023"] = "ctx-lh-posteriorcingulate";
    nameTable["1024"] = "ctx-lh-precentral";
    nameTable["1025"] = "ctx-lh-precuneus";
    nameTable["1026"] = "ctx-lh-rostralanteriorcingulate";
    nameTable["1027"] = "ctx-lh-rostralmiddlefrontal";
    nameTable["1028"] = "ctx-lh-superiorfrontal";
    nameTable["1029"] = "ctx-lh-superiorparietal";
    nameTable["1030"] = "ctx-lh-superiortemporal";
    nameTable["1031"] = "ctx-lh-supramarginal";
    nameTable["1032"] = "ctx-lh-frontalpole";
    nameTable["1033"] = "ctx-lh-temporalpole";
    nameTable["1034"] = "ctx-lh-transversetemporal";
    nameTable["1035"] = "ctx-lh-insula";
    nameTable["2000"] = "ctx-rh-unknown";
    nameTable["2001"] = "ctx-rh-bankssts";
    nameTable["2002"] = "ctx-rh-caudalanteriorcingulate";
    nameTable["2003"] = "ctx-rh-caudalmiddlefrontal";
    nameTable["2004"] = "ctx-rh-corpuscallosum";
    nameTable["2005"] = "ctx-rh-cuneus";
    nameTable["2006"] = "ctx-rh-entorhinal";
    nameTable["2007"] = "ctx-rh-fusiform";
    nameTable["2008"] = "ctx-rh-inferiorparietal";
    nameTable["2009"] = "ctx-rh-inferiortemporal";
    nameTable["2010"] = "ctx-rh-isthmuscingulate";
    nameTable["2011"] = "ctx-rh-lateraloccipital";
    nameTable["2012"] = "ctx-rh-lateralorbitofrontal";
    nameTable["2013"] = "ctx-rh-lingual";
    nameTable["2014"] = "ctx-rh-medialorbitofrontal";
    nameTable["2015"] = "ctx-rh-middletemporal";
    nameTable["2016"] = "ctx-rh-parahippocampal";
    nameTable["2017"] = "ctx-rh-paracentral";
    nameTable["2018"] = "ctx-rh-parsopercularis";
    nameTable["2019"] = "ctx-rh-parsorbitalis";
    nameTable["2020"] = "ctx-rh-parstriangularis";
    nameTable["2021"] = "ctx-rh-pericalcarine";
    nameTable["2022"] = "ctx-rh-postcentral";
    nameTable["2023"] = "ctx-rh-posteriorcingulate";
    nameTable["2024"] = "ctx-rh-precentral";
    nameTable["2025"] = "ctx-rh-precuneus";
    nameTable["2026"] = "ctx-rh-rostralanteriorcingulate";
    nameTable["2027"] = "ctx-rh-rostralmiddlefrontal";
    nameTable["2028"] = "ctx-rh-superiorfrontal";
    nameTable["2029"] = "ctx-rh-superiorparietal";
    nameTable["2030"] = "ctx-rh-superiortemporal";
    nameTable["2031"] = "ctx-rh-supramarginal";
    nameTable["2032"] = "ctx-rh-frontalpole";
    nameTable["2033"] = "ctx-rh-temporalpole";
    nameTable["2034"] = "ctx-rh-transversetemporal";
    nameTable["2035"] = "ctx-rh-insula";
    nameTable["3000"] = "wm-lh-unknown";
    nameTable["3001"] = "wm-lh-bankssts";
    nameTable["3002"] = "wm-lh-caudalanteriorcingulate";
    nameTable["3003"] = "wm-lh-caudalmiddlefrontal";
    nameTable["3004"] = "wm-lh-corpuscallosum";
    nameTable["3005"] = "wm-lh-cuneus";
    nameTable["3006"] = "wm-lh-entorhinal";
    nameTable["3007"] = "wm-lh-fusiform";
    nameTable["3008"] = "wm-lh-inferiorparietal";
    nameTable["3009"] = "wm-lh-inferiortemporal";
    nameTable["3010"] = "wm-lh-isthmuscingulate";
    nameTable["3011"] = "wm-lh-lateraloccipital";
    nameTable["3012"] = "wm-lh-lateralorbitofrontal";
    nameTable["3013"] = "wm-lh-lingual";
    nameTable["3014"] = "wm-lh-medialorbitofrontal";
    nameTable["3015"] = "wm-lh-middletemporal";
    nameTable["3016"] = "wm-lh-parahippocampal";
    nameTable["3017"] = "wm-lh-paracentral";
    nameTable["3018"] = "wm-lh-parsopercularis";
    nameTable["3019"] = "wm-lh-parsorbitalis";
    nameTable["3020"] = "wm-lh-parstriangularis";
    nameTable["3021"] = "wm-lh-pericalcarine";
    nameTable["3022"] = "wm-lh-postcentral";
    nameTable["3023"] = "wm-lh-posteriorcingulate";
    nameTable["3024"] = "wm-lh-precentral";
    nameTable["3025"] = "wm-lh-precuneus";
    nameTable["3026"] = "wm-lh-rostralanteriorcingulate";
    nameTable["3027"] = "wm-lh-rostralmiddlefrontal";
    nameTable["3028"] = "wm-lh-superiorfrontal";
    nameTable["3029"] = "wm-lh-superiorparietal";
    nameTable["3030"] = "wm-lh-superiortemporal";
    nameTable["3031"] = "wm-lh-supramarginal";
    nameTable["3032"] = "wm-lh-frontalpole";
    nameTable["3033"] = "wm-lh-temporalpole";
    nameTable["3034"] = "wm-lh-transversetemporal";
    nameTable["3035"] = "wm-lh-insula";
    nameTable["4000"] = "wm-rh-unknown";
    nameTable["4001"] = "wm-rh-bankssts";
    nameTable["4002"] = "wm-rh-caudalanteriorcingulate";
    nameTable["4003"] = "wm-rh-caudalmiddlefrontal";
    nameTable["4004"] = "wm-rh-corpuscallosum";
    nameTable["4005"] = "wm-rh-cuneus";
    nameTable["4006"] = "wm-rh-entorhinal";
    nameTable["4007"] = "wm-rh-fusiform";
    nameTable["4008"] = "wm-rh-inferiorparietal";
    nameTable["4009"] = "wm-rh-inferiortemporal";
    nameTable["4010"] = "wm-rh-isthmuscingulate";
    nameTable["4011"] = "wm-rh-lateraloccipital";
    nameTable["4012"] = "wm-rh-lateralorbitofrontal";
    nameTable["4013"] = "wm-rh-lingual";
    nameTable["4014"] = "wm-rh-medialorbitofrontal";
    nameTable["4015"] = "wm-rh-middletemporal";
    nameTable["4016"] = "wm-rh-parahippocampal";
    nameTable["4017"] = "wm-rh-paracentral";
    nameTable["4018"] = "wm-rh-parsopercularis";
    nameTable["4019"] = "wm-rh-parsorbitalis";
    nameTable["4020"] = "wm-rh-parstriangularis";
    nameTable["4021"] = "wm-rh-pericalcarine";
    nameTable["4022"] = "wm-rh-postcentral";
    nameTable["4023"] = "wm-rh-posteriorcingulate";
    nameTable["4024"] = "wm-rh-precentral";
    nameTable["4025"] = "wm-rh-precuneus";
    nameTable["4026"] = "wm-rh-rostralanteriorcingulate";
    nameTable["4027"] = "wm-rh-rostralmiddlefrontal";
    nameTable["4028"] = "wm-rh-superiorfrontal";
    nameTable["4029"] = "wm-rh-superiorparietal";
    nameTable["4030"] = "wm-rh-superiortemporal";
    nameTable["4031"] = "wm-rh-supramarginal";
    nameTable["4032"] = "wm-rh-frontalpole";
    nameTable["4033"] = "wm-rh-temporalpole";
    nameTable["4034"] = "wm-rh-transversetemporal";
    nameTable["4035"] = "wm-rh-insula";
    nameTable["1100"] = "ctx-lh-Unknown";
    nameTable["1101"] = "ctx-lh-Corpus_callosum";
    nameTable["1102"] = "ctx-lh-G_and_S_Insula_ONLY_AVERAGE";
    nameTable["1103"] = "ctx-lh-G_cingulate-Isthmus";
    nameTable["1104"] = "ctx-lh-G_cingulate-Main_part";
    nameTable["1200"] = "ctx-lh-G_cingulate-caudal_ACC";
    nameTable["1201"] = "ctx-lh-G_cingulate-rostral_ACC";
    nameTable["1202"] = "ctx-lh-G_cingulate-posterior";
    nameTable["1205"] = "ctx-lh-S_cingulate-caudal_ACC";
    nameTable["1206"] = "ctx-lh-S_cingulate-rostral_ACC";
    nameTable["1207"] = "ctx-lh-S_cingulate-posterior";
    nameTable["1210"] = "ctx-lh-S_pericallosal-caudal";
    nameTable["1211"] = "ctx-lh-S_pericallosal-rostral";
    nameTable["1212"] = "ctx-lh-S_pericallosal-posterior";
    nameTable["1105"] = "ctx-lh-G_cuneus";
    nameTable["1106"] = "ctx-lh-G_frontal_inf-Opercular_part";
    nameTable["1107"] = "ctx-lh-G_frontal_inf-Orbital_part";
    nameTable["1108"] = "ctx-lh-G_frontal_inf-Triangular_part";
    nameTable["1109"] = "ctx-lh-G_frontal_middle";
    nameTable["1110"] = "ctx-lh-G_frontal_superior";
    nameTable["1111"] = "ctx-lh-G_frontomarginal";
    nameTable["1112"] = "ctx-lh-G_insular_long";
    nameTable["1113"] = "ctx-lh-G_insular_short";
    nameTable["1114"] = "ctx-lh-G_and_S_occipital_inferior";
    nameTable["1115"] = "ctx-lh-G_occipital_middle";
    nameTable["1116"] = "ctx-lh-G_occipital_superior";
    nameTable["1117"] = "ctx-lh-G_occipit-temp_lat-Or_fusiform";
    nameTable["1118"] = "ctx-lh-G_occipit-temp_med-Lingual_part";
    nameTable["1119"] = "ctx-lh-G_occipit-temp_med-Parahippocampal_part";
    nameTable["1120"] = "ctx-lh-G_orbital";
    nameTable["1121"] = "ctx-lh-G_paracentral";
    nameTable["1122"] = "ctx-lh-G_parietal_inferior-Angular_part";
    nameTable["1123"] = "ctx-lh-G_parietal_inferior-Supramarginal_part";
    nameTable["1124"] = "ctx-lh-G_parietal_superior";
    nameTable["1125"] = "ctx-lh-G_postcentral";
    nameTable["1126"] = "ctx-lh-G_precentral";
    nameTable["1127"] = "ctx-lh-G_precuneus";
    nameTable["1128"] = "ctx-lh-G_rectus";
    nameTable["1129"] = "ctx-lh-G_subcallosal";
    nameTable["1130"] = "ctx-lh-G_subcentral";
    nameTable["1131"] = "ctx-lh-G_temporal_inferior";
    nameTable["1132"] = "ctx-lh-G_temporal_middle";
    nameTable["1133"] = "ctx-lh-G_temp_sup-G_temp_transv_and_interm_S";
    nameTable["1134"] = "ctx-lh-G_temp_sup-Lateral_aspect";
    nameTable["1135"] = "ctx-lh-G_temp_sup-Planum_polare";
    nameTable["1136"] = "ctx-lh-G_temp_sup-Planum_tempolare";
    nameTable["1137"] = "ctx-lh-G_and_S_transverse_frontopolar";
    nameTable["1138"] = "ctx-lh-Lat_Fissure-ant_sgt-ramus_horizontal";
    nameTable["1139"] = "ctx-lh-Lat_Fissure-ant_sgt-ramus_vertical";
    nameTable["1140"] = "ctx-lh-Lat_Fissure-post_sgt";
    nameTable["1141"] = "ctx-lh-Medial_wall";
    nameTable["1142"] = "ctx-lh-Pole_occipital";
    nameTable["1143"] = "ctx-lh-Pole_temporal";
    nameTable["1144"] = "ctx-lh-S_calcarine";
    nameTable["1145"] = "ctx-lh-S_central";
    nameTable["1146"] = "ctx-lh-S_central_insula";
    nameTable["1147"] = "ctx-lh-S_cingulate-Main_part_and_Intracingulate";
    nameTable["1148"] = "ctx-lh-S_cingulate-Marginalis_part";
    nameTable["1149"] = "ctx-lh-S_circular_insula_anterior";
    nameTable["1150"] = "ctx-lh-S_circular_insula_inferior";
    nameTable["1151"] = "ctx-lh-S_circular_insula_superior";
    nameTable["1152"] = "ctx-lh-S_collateral_transverse_ant";
    nameTable["1153"] = "ctx-lh-S_collateral_transverse_post";
    nameTable["1154"] = "ctx-lh-S_frontal_inferior";
    nameTable["1155"] = "ctx-lh-S_frontal_middle";
    nameTable["1156"] = "ctx-lh-S_frontal_superior";
    nameTable["1157"] = "ctx-lh-S_frontomarginal";
    nameTable["1158"] = "ctx-lh-S_intermedius_primus-Jensen";
    nameTable["1159"] = "ctx-lh-S_intraparietal-and_Parietal_transverse";
    nameTable["1160"] = "ctx-lh-S_occipital_anterior";
    nameTable["1161"] = "ctx-lh-S_occipital_middle_and_Lunatus";
    nameTable["1162"] = "ctx-lh-S_occipital_superior_and_transversalis";
    nameTable["1163"] = "ctx-lh-S_occipito-temporal_lateral";
    nameTable["1164"] = "ctx-lh-S_occipito-temporal_medial_and_S_Lingual";
    nameTable["1165"] = "ctx-lh-S_orbital-H_shapped";
    nameTable["1166"] = "ctx-lh-S_orbital_lateral";
    nameTable["1167"] = "ctx-lh-S_orbital_medial-Or_olfactory";
    nameTable["1168"] = "ctx-lh-S_paracentral";
    nameTable["1169"] = "ctx-lh-S_parieto_occipital";
    nameTable["1170"] = "ctx-lh-S_pericallosal";
    nameTable["1171"] = "ctx-lh-S_postcentral";
    nameTable["1172"] = "ctx-lh-S_precentral-Inferior-part";
    nameTable["1173"] = "ctx-lh-S_precentral-Superior-part";
    nameTable["1174"] = "ctx-lh-S_subcentral_ant";
    nameTable["1175"] = "ctx-lh-S_subcentral_post";
    nameTable["1176"] = "ctx-lh-S_suborbital";
    nameTable["1177"] = "ctx-lh-S_subparietal";
    nameTable["1178"] = "ctx-lh-S_supracingulate";
    nameTable["1179"] = "ctx-lh-S_temporal_inferior";
    nameTable["1180"] = "ctx-lh-S_temporal_superior";
    nameTable["1181"] = "ctx-lh-S_temporal_transverse";
    nameTable["2100"] = "ctx-rh-Unknown";
    nameTable["2101"] = "ctx-rh-Corpus_callosum";
    nameTable["2102"] = "ctx-rh-G_and_S_Insula_ONLY_AVERAGE";
    nameTable["2103"] = "ctx-rh-G_cingulate-Isthmus";
    nameTable["2104"] = "ctx-rh-G_cingulate-Main_part";
    nameTable["2105"] = "ctx-rh-G_cuneus";
    nameTable["2106"] = "ctx-rh-G_frontal_inf-Opercular_part";
    nameTable["2107"] = "ctx-rh-G_frontal_inf-Orbital_part";
    nameTable["2108"] = "ctx-rh-G_frontal_inf-Triangular_part";
    nameTable["2109"] = "ctx-rh-G_frontal_middle";
    nameTable["2110"] = "ctx-rh-G_frontal_superior";
    nameTable["2111"] = "ctx-rh-G_frontomarginal";
    nameTable["2112"] = "ctx-rh-G_insular_long";
    nameTable["2113"] = "ctx-rh-G_insular_short";
    nameTable["2114"] = "ctx-rh-G_and_S_occipital_inferior";
    nameTable["2115"] = "ctx-rh-G_occipital_middle";
    nameTable["2116"] = "ctx-rh-G_occipital_superior";
    nameTable["2117"] = "ctx-rh-G_occipit-temp_lat-Or_fusiform";
    nameTable["2118"] = "ctx-rh-G_occipit-temp_med-Lingual_part";
    nameTable["2119"] = "ctx-rh-G_occipit-temp_med-Parahippocampal_part";
    nameTable["2120"] = "ctx-rh-G_orbital";
    nameTable["2121"] = "ctx-rh-G_paracentral";
    nameTable["2122"] = "ctx-rh-G_parietal_inferior-Angular_part";
    nameTable["2123"] = "ctx-rh-G_parietal_inferior-Supramarginal_part";
    nameTable["2124"] = "ctx-rh-G_parietal_superior";
    nameTable["2125"] = "ctx-rh-G_postcentral";
    nameTable["2126"] = "ctx-rh-G_precentral";
    nameTable["2127"] = "ctx-rh-G_precuneus";
    nameTable["2128"] = "ctx-rh-G_rectus";
    nameTable["2129"] = "ctx-rh-G_subcallosal";
    nameTable["2130"] = "ctx-rh-G_subcentral";
    nameTable["2131"] = "ctx-rh-G_temporal_inferior";
    nameTable["2132"] = "ctx-rh-G_temporal_middle";
    nameTable["2133"] = "ctx-rh-G_temp_sup-G_temp_transv_and_interm_S";
    nameTable["2134"] = "ctx-rh-G_temp_sup-Lateral_aspect";
    nameTable["2135"] = "ctx-rh-G_temp_sup-Planum_polare";
    nameTable["2136"] = "ctx-rh-G_temp_sup-Planum_tempolare";
    nameTable["2137"] = "ctx-rh-G_and_S_transverse_frontopolar";
    nameTable["2138"] = "ctx-rh-Lat_Fissure-ant_sgt-ramus_horizontal";
    nameTable["2139"] = "ctx-rh-Lat_Fissure-ant_sgt-ramus_vertical";
    nameTable["2140"] = "ctx-rh-Lat_Fissure-post_sgt";
    nameTable["2141"] = "ctx-rh-Medial_wall";
    nameTable["2142"] = "ctx-rh-Pole_occipital";
    nameTable["2143"] = "ctx-rh-Pole_temporal";
    nameTable["2144"] = "ctx-rh-S_calcarine";
    nameTable["2145"] = "ctx-rh-S_central";
    nameTable["2146"] = "ctx-rh-S_central_insula";
    nameTable["2147"] = "ctx-rh-S_cingulate-Main_part_and_Intracingulate";
    nameTable["2148"] = "ctx-rh-S_cingulate-Marginalis_part";
    nameTable["2149"] = "ctx-rh-S_circular_insula_anterior";
    nameTable["2150"] = "ctx-rh-S_circular_insula_inferior";
    nameTable["2151"] = "ctx-rh-S_circular_insula_superior";
    nameTable["2152"] = "ctx-rh-S_collateral_transverse_ant";
    nameTable["2153"] = "ctx-rh-S_collateral_transverse_post";
    nameTable["2154"] = "ctx-rh-S_frontal_inferior";
    nameTable["2155"] = "ctx-rh-S_frontal_middle";
    nameTable["2156"] = "ctx-rh-S_frontal_superior";
    nameTable["2157"] = "ctx-rh-S_frontomarginal";
    nameTable["2158"] = "ctx-rh-S_intermedius_primus-Jensen";
    nameTable["2159"] = "ctx-rh-S_intraparietal-and_Parietal_transverse";
    nameTable["2160"] = "ctx-rh-S_occipital_anterior";
    nameTable["2161"] = "ctx-rh-S_occipital_middle_and_Lunatus";
    nameTable["2162"] = "ctx-rh-S_occipital_superior_and_transversalis";
    nameTable["2163"] = "ctx-rh-S_occipito-temporal_lateral";
    nameTable["2164"] = "ctx-rh-S_occipito-temporal_medial_and_S_Lingual";
    nameTable["2165"] = "ctx-rh-S_orbital-H_shapped";
    nameTable["2166"] = "ctx-rh-S_orbital_lateral";
    nameTable["2167"] = "ctx-rh-S_orbital_medial-Or_olfactory";
    nameTable["2168"] = "ctx-rh-S_paracentral";
    nameTable["2169"] = "ctx-rh-S_parieto_occipital";
    nameTable["2170"] = "ctx-rh-S_pericallosal";
    nameTable["2171"] = "ctx-rh-S_postcentral";
    nameTable["2172"] = "ctx-rh-S_precentral-Inferior-part";
    nameTable["2173"] = "ctx-rh-S_precentral-Superior-part";
    nameTable["2174"] = "ctx-rh-S_subcentral_ant";
    nameTable["2175"] = "ctx-rh-S_subcentral_post";
    nameTable["2176"] = "ctx-rh-S_suborbital";
    nameTable["2177"] = "ctx-rh-S_subparietal";
    nameTable["2178"] = "ctx-rh-S_supracingulate";
    nameTable["2179"] = "ctx-rh-S_temporal_inferior";
    nameTable["2180"] = "ctx-rh-S_temporal_superior";
    nameTable["2181"] = "ctx-rh-S_temporal_transverse";
    nameTable["2200"] = "ctx-rh-G_cingulate-caudal_ACC";
    nameTable["2201"] = "ctx-rh-G_cingulate-rostral_ACC";
    nameTable["2202"] = "ctx-rh-G_cingulate-posterior";
    nameTable["2205"] = "ctx-rh-S_cingulate-caudal_ACC";
    nameTable["2206"] = "ctx-rh-S_cingulate-rostral_ACC";
    nameTable["2207"] = "ctx-rh-S_cingulate-posterior";
    nameTable["2210"] = "ctx-rh-S_pericallosal-caudal";
    nameTable["2211"] = "ctx-rh-S_pericallosal-rostral";
    nameTable["2212"] = "ctx-rh-S_pericallosal-posterior";
    nameTable["3100"] = "wm-lh-Unknown";
    nameTable["3101"] = "wm-lh-Corpus_callosum";
    nameTable["3102"] = "wm-lh-G_and_S_Insula_ONLY_AVERAGE";
    nameTable["3103"] = "wm-lh-G_cingulate-Isthmus";
    nameTable["3104"] = "wm-lh-G_cingulate-Main_part";
    nameTable["3105"] = "wm-lh-G_cuneus";
    nameTable["3106"] = "wm-lh-G_frontal_inf-Opercular_part";
    nameTable["3107"] = "wm-lh-G_frontal_inf-Orbital_part";
    nameTable["3108"] = "wm-lh-G_frontal_inf-Triangular_part";
    nameTable["3109"] = "wm-lh-G_frontal_middle";
    nameTable["3110"] = "wm-lh-G_frontal_superior";
    nameTable["3111"] = "wm-lh-G_frontomarginal";
    nameTable["3112"] = "wm-lh-G_insular_long";
    nameTable["3113"] = "wm-lh-G_insular_short";
    nameTable["3114"] = "wm-lh-G_and_S_occipital_inferior";
    nameTable["3115"] = "wm-lh-G_occipital_middle";
    nameTable["3116"] = "wm-lh-G_occipital_superior";
    nameTable["3117"] = "wm-lh-G_occipit-temp_lat-Or_fusiform";
    nameTable["3118"] = "wm-lh-G_occipit-temp_med-Lingual_part";
    nameTable["3119"] = "wm-lh-G_occipit-temp_med-Parahippocampal_part";
    nameTable["3120"] = "wm-lh-G_orbital";
    nameTable["3121"] = "wm-lh-G_paracentral";
    nameTable["3122"] = "wm-lh-G_parietal_inferior-Angular_part";
    nameTable["3123"] = "wm-lh-G_parietal_inferior-Supramarginal_part";
    nameTable["3124"] = "wm-lh-G_parietal_superior";
    nameTable["3125"] = "wm-lh-G_postcentral";
    nameTable["3126"] = "wm-lh-G_precentral";
    nameTable["3127"] = "wm-lh-G_precuneus";
    nameTable["3128"] = "wm-lh-G_rectus";
    nameTable["3129"] = "wm-lh-G_subcallosal";
    nameTable["3130"] = "wm-lh-G_subcentral";
    nameTable["3131"] = "wm-lh-G_temporal_inferior";
    nameTable["3132"] = "wm-lh-G_temporal_middle";
    nameTable["3133"] = "wm-lh-G_temp_sup-G_temp_transv_and_interm_S";
    nameTable["3134"] = "wm-lh-G_temp_sup-Lateral_aspect";
    nameTable["3135"] = "wm-lh-G_temp_sup-Planum_polare";
    nameTable["3136"] = "wm-lh-G_temp_sup-Planum_tempolare";
    nameTable["3137"] = "wm-lh-G_and_S_transverse_frontopolar";
    nameTable["3138"] = "wm-lh-Lat_Fissure-ant_sgt-ramus_horizontal";
    nameTable["3139"] = "wm-lh-Lat_Fissure-ant_sgt-ramus_vertical";
    nameTable["3140"] = "wm-lh-Lat_Fissure-post_sgt";
    nameTable["3141"] = "wm-lh-Medial_wall";
    nameTable["3142"] = "wm-lh-Pole_occipital";
    nameTable["3143"] = "wm-lh-Pole_temporal";
    nameTable["3144"] = "wm-lh-S_calcarine";
    nameTable["3145"] = "wm-lh-S_central";
    nameTable["3146"] = "wm-lh-S_central_insula";
    nameTable["3147"] = "wm-lh-S_cingulate-Main_part_and_Intracingulate";
    nameTable["3148"] = "wm-lh-S_cingulate-Marginalis_part";
    nameTable["3149"] = "wm-lh-S_circular_insula_anterior";
    nameTable["3150"] = "wm-lh-S_circular_insula_inferior";
    nameTable["3151"] = "wm-lh-S_circular_insula_superior";
    nameTable["3152"] = "wm-lh-S_collateral_transverse_ant";
    nameTable["3153"] = "wm-lh-S_collateral_transverse_post";
    nameTable["3154"] = "wm-lh-S_frontal_inferior";
    nameTable["3155"] = "wm-lh-S_frontal_middle";
    nameTable["3156"] = "wm-lh-S_frontal_superior";
    nameTable["3157"] = "wm-lh-S_frontomarginal";
    nameTable["3158"] = "wm-lh-S_intermedius_primus-Jensen";
    nameTable["3159"] = "wm-lh-S_intraparietal-and_Parietal_transverse";
    nameTable["3160"] = "wm-lh-S_occipital_anterior";
    nameTable["3161"] = "wm-lh-S_occipital_middle_and_Lunatus";
    nameTable["3162"] = "wm-lh-S_occipital_superior_and_transversalis";
    nameTable["3163"] = "wm-lh-S_occipito-temporal_lateral";
    nameTable["3164"] = "wm-lh-S_occipito-temporal_medial_and_S_Lingual";
    nameTable["3165"] = "wm-lh-S_orbital-H_shapped";
    nameTable["3166"] = "wm-lh-S_orbital_lateral";
    nameTable["3167"] = "wm-lh-S_orbital_medial-Or_olfactory";
    nameTable["3168"] = "wm-lh-S_paracentral";
    nameTable["3169"] = "wm-lh-S_parieto_occipital";
    nameTable["3170"] = "wm-lh-S_pericallosal";
    nameTable["3171"] = "wm-lh-S_postcentral";
    nameTable["3172"] = "wm-lh-S_precentral-Inferior-part";
    nameTable["3173"] = "wm-lh-S_precentral-Superior-part";
    nameTable["3174"] = "wm-lh-S_subcentral_ant";
    nameTable["3175"] = "wm-lh-S_subcentral_post";
    nameTable["3176"] = "wm-lh-S_suborbital";
    nameTable["3177"] = "wm-lh-S_subparietal";
    nameTable["3178"] = "wm-lh-S_supracingulate";
    nameTable["3179"] = "wm-lh-S_temporal_inferior";
    nameTable["3180"] = "wm-lh-S_temporal_superior";
    nameTable["3181"] = "wm-lh-S_temporal_transverse";
    nameTable["4100"] = "wm-rh-Unknown";
    nameTable["4101"] = "wm-rh-Corpus_callosum";
    nameTable["4102"] = "wm-rh-G_and_S_Insula_ONLY_AVERAGE";
    nameTable["4103"] = "wm-rh-G_cingulate-Isthmus";
    nameTable["4104"] = "wm-rh-G_cingulate-Main_part";
    nameTable["4105"] = "wm-rh-G_cuneus";
    nameTable["4106"] = "wm-rh-G_frontal_inf-Opercular_part";
    nameTable["4107"] = "wm-rh-G_frontal_inf-Orbital_part";
    nameTable["4108"] = "wm-rh-G_frontal_inf-Triangular_part";
    nameTable["4109"] = "wm-rh-G_frontal_middle";
    nameTable["4110"] = "wm-rh-G_frontal_superior";
    nameTable["4111"] = "wm-rh-G_frontomarginal";
    nameTable["4112"] = "wm-rh-G_insular_long";
    nameTable["4113"] = "wm-rh-G_insular_short";
    nameTable["4114"] = "wm-rh-G_and_S_occipital_inferior";
    nameTable["4115"] = "wm-rh-G_occipital_middle";
    nameTable["4116"] = "wm-rh-G_occipital_superior";
    nameTable["4117"] = "wm-rh-G_occipit-temp_lat-Or_fusiform";
    nameTable["4118"] = "wm-rh-G_occipit-temp_med-Lingual_part";
    nameTable["4119"] = "wm-rh-G_occipit-temp_med-Parahippocampal_part";
    nameTable["4120"] = "wm-rh-G_orbital";
    nameTable["4121"] = "wm-rh-G_paracentral";
    nameTable["4122"] = "wm-rh-G_parietal_inferior-Angular_part";
    nameTable["4123"] = "wm-rh-G_parietal_inferior-Supramarginal_part";
    nameTable["4124"] = "wm-rh-G_parietal_superior";
    nameTable["4125"] = "wm-rh-G_postcentral";
    nameTable["4126"] = "wm-rh-G_precentral";
    nameTable["4127"] = "wm-rh-G_precuneus";
    nameTable["4128"] = "wm-rh-G_rectus";
    nameTable["4129"] = "wm-rh-G_subcallosal";
    nameTable["4130"] = "wm-rh-G_subcentral";
    nameTable["4131"] = "wm-rh-G_temporal_inferior";
    nameTable["4132"] = "wm-rh-G_temporal_middle";
    nameTable["4133"] = "wm-rh-G_temp_sup-G_temp_transv_and_interm_S";
    nameTable["4134"] = "wm-rh-G_temp_sup-Lateral_aspect";
    nameTable["4135"] = "wm-rh-G_temp_sup-Planum_polare";
    nameTable["4136"] = "wm-rh-G_temp_sup-Planum_tempolare";
    nameTable["4137"] = "wm-rh-G_and_S_transverse_frontopolar";
    nameTable["4138"] = "wm-rh-Lat_Fissure-ant_sgt-ramus_horizontal";
    nameTable["4139"] = "wm-rh-Lat_Fissure-ant_sgt-ramus_vertical";
    nameTable["4140"] = "wm-rh-Lat_Fissure-post_sgt";
    nameTable["4141"] = "wm-rh-Medial_wall";
    nameTable["4142"] = "wm-rh-Pole_occipital";
    nameTable["4143"] = "wm-rh-Pole_temporal";
    nameTable["4144"] = "wm-rh-S_calcarine";
    nameTable["4145"] = "wm-rh-S_central";
    nameTable["4146"] = "wm-rh-S_central_insula";
    nameTable["4147"] = "wm-rh-S_cingulate-Main_part_and_Intracingulate";
    nameTable["4148"] = "wm-rh-S_cingulate-Marginalis_part";
    nameTable["4149"] = "wm-rh-S_circular_insula_anterior";
    nameTable["4150"] = "wm-rh-S_circular_insula_inferior";
    nameTable["4151"] = "wm-rh-S_circular_insula_superior";
    nameTable["4152"] = "wm-rh-S_collateral_transverse_ant";
    nameTable["4153"] = "wm-rh-S_collateral_transverse_post";
    nameTable["4154"] = "wm-rh-S_frontal_inferior";
    nameTable["4155"] = "wm-rh-S_frontal_middle";
    nameTable["4156"] = "wm-rh-S_frontal_superior";
    nameTable["4157"] = "wm-rh-S_frontomarginal";
    nameTable["4158"] = "wm-rh-S_intermedius_primus-Jensen";
    nameTable["4159"] = "wm-rh-S_intraparietal-and_Parietal_transverse";
    nameTable["4160"] = "wm-rh-S_occipital_anterior";
    nameTable["4161"] = "wm-rh-S_occipital_middle_and_Lunatus";
    nameTable["4162"] = "wm-rh-S_occipital_superior_and_transversalis";
    nameTable["4163"] = "wm-rh-S_occipito-temporal_lateral";
    nameTable["4164"] = "wm-rh-S_occipito-temporal_medial_and_S_Lingual";
    nameTable["4165"] = "wm-rh-S_orbital-H_shapped";
    nameTable["4166"] = "wm-rh-S_orbital_lateral";
    nameTable["4167"] = "wm-rh-S_orbital_medial-Or_olfactory";
    nameTable["4168"] = "wm-rh-S_paracentral";
    nameTable["4169"] = "wm-rh-S_parieto_occipital";
    nameTable["4170"] = "wm-rh-S_pericallosal";
    nameTable["4171"] = "wm-rh-S_postcentral";
    nameTable["4172"] = "wm-rh-S_precentral-Inferior-part";
    nameTable["4173"] = "wm-rh-S_precentral-Superior-part";
    nameTable["4174"] = "wm-rh-S_subcentral_ant";
    nameTable["4175"] = "wm-rh-S_subcentral_post";
    nameTable["4176"] = "wm-rh-S_suborbital";
    nameTable["4177"] = "wm-rh-S_subparietal";
    nameTable["4178"] = "wm-rh-S_supracingulate";
    nameTable["4179"] = "wm-rh-S_temporal_inferior";
    nameTable["4180"] = "wm-rh-S_temporal_superior";
    nameTable["4181"] = "wm-rh-S_temporal_transverse";
    nameTable["5001"] = "Left-UnsegmentedWhiteMatter";
    nameTable["5002"] = "Right-UnsegmentedWhiteMatter";
    nameTable["5100"] = "fmajor";
    nameTable["5101"] = "fminor";
    nameTable["5102"] = "lh.atr";
    nameTable["5103"] = "lh.cab";
    nameTable["5104"] = "lh.ccg";
    nameTable["5105"] = "lh.cst";
    nameTable["5106"] = "lh.ilf";
    nameTable["5107"] = "lh.slfp";
    nameTable["5108"] = "lh.slft";
    nameTable["5109"] = "lh.unc";
    nameTable["5110"] = "rh.atr";
    nameTable["5111"] = "rh.cab";
    nameTable["5112"] = "rh.ccg";
    nameTable["5113"] = "rh.cst";
    nameTable["5114"] = "rh.ilf";
    nameTable["5115"] = "rh.slfp";
    nameTable["5116"] = "rh.slft";
    nameTable["5117"] = "rh.unc";
    nameTable["5200"] = "CC-ForcepsMajor";
    nameTable["5201"] = "CC-ForcepsMinor";
    nameTable["5202"] = "LAntThalRadiation";
    nameTable["5203"] = "LCingulumAngBundle";
    nameTable["5204"] = "LCingulumCingGyrus";
    nameTable["5205"] = "LCorticospinalTract";
    nameTable["5206"] = "LInfLongFas";
    nameTable["5207"] = "LSupLongFasParietal";
    nameTable["5208"] = "LSupLongFasTemporal";
    nameTable["5209"] = "LUncinateFas";
    nameTable["5210"] = "RAntThalRadiation";
    nameTable["5211"] = "RCingulumAngBundle";
    nameTable["5212"] = "RCingulumCingGyrus";
    nameTable["5213"] = "RCorticospinalTract";
    nameTable["5214"] = "RInfLongFas";
    nameTable["5215"] = "RSupLongFasParietal";
    nameTable["5216"] = "RSupLongFasTemporal";
    nameTable["5217"] = "RUncinateFas";
    nameTable["6000"] = "CST-orig";
    nameTable["6001"] = "CST-hammer";
    nameTable["6002"] = "CST-CVS";
    nameTable["6003"] = "CST-flirt";
    nameTable["6010"] = "Left-SLF1";
    nameTable["6020"] = "Right-SLF1";
    nameTable["6030"] = "Left-SLF3";
    nameTable["6040"] = "Right-SLF3";
    nameTable["6050"] = "Left-CST";
    nameTable["6060"] = "Right-CST";
    nameTable["6070"] = "Left-SLF2";
    nameTable["6080"] = "Right-SLF2";
    nameTable["7001"] = "Lateral-nucleus";
    nameTable["7002"] = "Basolateral-nucleus";
    nameTable["7003"] = "Basal-nucleus";
    nameTable["7004"] = "Centromedial-nucleus";
    nameTable["7005"] = "Central-nucleus";
    nameTable["7006"] = "Medial-nucleus";
    nameTable["7007"] = "Cortical-nucleus";
    nameTable["7008"] = "Accessory-Basal-nucleus";
    nameTable["7009"] = "Corticoamygdaloid-transitio";
    nameTable["7010"] = "Anterior-amygdaloid-area-AAA";
    nameTable["7011"] = "Fusion-amygdala-HP-FAH";
    nameTable["7012"] = "Hippocampal-amygdala-transition-HATA";
    nameTable["7013"] = "Endopiriform-nucleus";
    nameTable["7014"] = "Lateral-nucleus-olfactory-tract";
    nameTable["7015"] = "Paralaminar-nucleus";
    nameTable["7016"] = "Intercalated-nucleus";
    nameTable["7017"] = "Prepiriform-cortex";
    nameTable["7018"] = "Periamygdaloid-cortex";
    nameTable["7019"] = "Envelope-Amygdala";
    nameTable["7020"] = "Extranuclear-Amydala";
    nameTable["7100"] = "Brainstem-inferior-colliculus";
    nameTable["7101"] = "Brainstem-cochlear-nucleus";
    nameTable["8001"] = "Thalamus-Anterior";
    nameTable["8002"] = "Thalamus-Ventral-anterior";
    nameTable["8003"] = "Thalamus-Lateral-dorsal";
    nameTable["8004"] = "Thalamus-Lateral-posterior";
    nameTable["8005"] = "Thalamus-Ventral-lateral";
    nameTable["8006"] = "Thalamus-Ventral-posterior-medial";
    nameTable["8007"] = "Thalamus-Ventral-posterior-lateral";
    nameTable["8008"] = "Thalamus-intralaminar";
    nameTable["8009"] = "Thalamus-centromedian";
    nameTable["8010"] = "Thalamus-mediodorsal";
    nameTable["8011"] = "Thalamus-medial";
    nameTable["8012"] = "Thalamus-pulvinar";
    nameTable["8013"] = "Thalamus-lateral-geniculate";
    nameTable["8014"] = "Thalamus-medial-geniculate";
    nameTable["9000"] = "ctx-lh-prefrontal";
    nameTable["9001"] = "ctx-lh-primary-motor";
    nameTable["9002"] = "ctx-lh-premotor";
    nameTable["9003"] = "ctx-lh-temporal";
    nameTable["9004"] = "ctx-lh-posterior-parietal";
    nameTable["9005"] = "ctx-lh-prim-sec-somatosensory";
    nameTable["9006"] = "ctx-lh-occipital";
    nameTable["9500"] = "ctx-rh-prefrontal";
    nameTable["9501"] = "ctx-rh-primary-motor";
    nameTable["9502"] = "ctx-rh-premotor";
    nameTable["9503"] = "ctx-rh-temporal";
    nameTable["9504"] = "ctx-rh-posterior-parietal";
    nameTable["9505"] = "ctx-rh-prim-sec-somatosensory";
    nameTable["9506"] = "ctx-rh-occipital";
    nameTable["11100"] = "ctx_lh_Unknown";
    nameTable["11101"] = "ctx_lh_G_and_S_frontomargin";
    nameTable["11102"] = "ctx_lh_G_and_S_occipital_inf";
    nameTable["11103"] = "ctx_lh_G_and_S_paracentral";
    nameTable["11104"] = "ctx_lh_G_and_S_subcentral";
    nameTable["11105"] = "ctx_lh_G_and_S_transv_frontopol";
    nameTable["11106"] = "ctx_lh_G_and_S_cingul-Ant";
    nameTable["11107"] = "ctx_lh_G_and_S_cingul-Mid-Ant";
    nameTable["11108"] = "ctx_lh_G_and_S_cingul-Mid-Post";
    nameTable["11109"] = "ctx_lh_G_cingul-Post-dorsal";
    nameTable["11110"] = "ctx_lh_G_cingul-Post-ventral";
    nameTable["11111"] = "ctx_lh_G_cuneus";
    nameTable["11112"] = "ctx_lh_G_front_inf-Opercular";
    nameTable["11113"] = "ctx_lh_G_front_inf-Orbital";
    nameTable["11114"] = "ctx_lh_G_front_inf-Triangul";
    nameTable["11115"] = "ctx_lh_G_front_middle";
    nameTable["11116"] = "ctx_lh_G_front_sup";
    nameTable["11117"] = "ctx_lh_G_Ins_lg_and_S_cent_ins";
    nameTable["11118"] = "ctx_lh_G_insular_short";
    nameTable["11119"] = "ctx_lh_G_occipital_middle";
    nameTable["11120"] = "ctx_lh_G_occipital_sup";
    nameTable["11121"] = "ctx_lh_G_oc-temp_lat-fusifor";
    nameTable["11122"] = "ctx_lh_G_oc-temp_med-Lingual";
    nameTable["11123"] = "ctx_lh_G_oc-temp_med-Parahip";
    nameTable["11124"] = "ctx_lh_G_orbital";
    nameTable["11125"] = "ctx_lh_G_pariet_inf-Angular";
    nameTable["11126"] = "ctx_lh_G_pariet_inf-Supramar";
    nameTable["11127"] = "ctx_lh_G_parietal_sup";
    nameTable["11128"] = "ctx_lh_G_postcentral";
    nameTable["11129"] = "ctx_lh_G_precentral";
    nameTable["11130"] = "ctx_lh_G_precuneus";
    nameTable["11131"] = "ctx_lh_G_rectus";
    nameTable["11132"] = "ctx_lh_G_subcallosal";
    nameTable["11133"] = "ctx_lh_G_temp_sup-G_T_transv";
    nameTable["11134"] = "ctx_lh_G_temp_sup-Lateral";
    nameTable["11135"] = "ctx_lh_G_temp_sup-Plan_polar";
    nameTable["11136"] = "ctx_lh_G_temp_sup-Plan_tempo";
    nameTable["11137"] = "ctx_lh_G_temporal_inf";
    nameTable["11138"] = "ctx_lh_G_temporal_middle";
    nameTable["11139"] = "ctx_lh_Lat_Fis-ant-Horizont";
    nameTable["11140"] = "ctx_lh_Lat_Fis-ant-Vertical";
    nameTable["11141"] = "ctx_lh_Lat_Fis-post";
    nameTable["11142"] = "ctx_lh_Medial_wall";
    nameTable["11143"] = "ctx_lh_Pole_occipital";
    nameTable["11144"] = "ctx_lh_Pole_temporal";
    nameTable["11145"] = "ctx_lh_S_calcarine";
    nameTable["11146"] = "ctx_lh_S_central";
    nameTable["11147"] = "ctx_lh_S_cingul-Marginalis";
    nameTable["11148"] = "ctx_lh_S_circular_insula_ant";
    nameTable["11149"] = "ctx_lh_S_circular_insula_inf";
    nameTable["11150"] = "ctx_lh_S_circular_insula_sup";
    nameTable["11151"] = "ctx_lh_S_collat_transv_ant";
    nameTable["11152"] = "ctx_lh_S_collat_transv_post";
    nameTable["11153"] = "ctx_lh_S_front_inf";
    nameTable["11154"] = "ctx_lh_S_front_middle";
    nameTable["11155"] = "ctx_lh_S_front_sup";
    nameTable["11156"] = "ctx_lh_S_interm_prim-Jensen";
    nameTable["11157"] = "ctx_lh_S_intrapariet_and_P_trans";
    nameTable["11158"] = "ctx_lh_S_oc_middle_and_Lunatus";
    nameTable["11159"] = "ctx_lh_S_oc_sup_and_transversal";
    nameTable["11160"] = "ctx_lh_S_occipital_ant";
    nameTable["11161"] = "ctx_lh_S_oc-temp_lat";
    nameTable["11162"] = "ctx_lh_S_oc-temp_med_and_Lingual";
    nameTable["11163"] = "ctx_lh_S_orbital_lateral";
    nameTable["11164"] = "ctx_lh_S_orbital_med-olfact";
    nameTable["11165"] = "ctx_lh_S_orbital-H_Shaped";
    nameTable["11166"] = "ctx_lh_S_parieto_occipital";
    nameTable["11167"] = "ctx_lh_S_pericallosal";
    nameTable["11168"] = "ctx_lh_S_postcentral";
    nameTable["11169"] = "ctx_lh_S_precentral-inf-part";
    nameTable["11170"] = "ctx_lh_S_precentral-sup-part";
    nameTable["11171"] = "ctx_lh_S_suborbital";
    nameTable["11172"] = "ctx_lh_S_subparietal";
    nameTable["11173"] = "ctx_lh_S_temporal_inf";
    nameTable["11174"] = "ctx_lh_S_temporal_sup";
    nameTable["11175"] = "ctx_lh_S_temporal_transverse";
    nameTable["12100"] = "ctx_rh_Unknown";
    nameTable["12101"] = "ctx_rh_G_and_S_frontomargin";
    nameTable["12102"] = "ctx_rh_G_and_S_occipital_inf";
    nameTable["12103"] = "ctx_rh_G_and_S_paracentral";
    nameTable["12104"] = "ctx_rh_G_and_S_subcentral";
    nameTable["12105"] = "ctx_rh_G_and_S_transv_frontopol";
    nameTable["12106"] = "ctx_rh_G_and_S_cingul-Ant";
    nameTable["12107"] = "ctx_rh_G_and_S_cingul-Mid-Ant";
    nameTable["12108"] = "ctx_rh_G_and_S_cingul-Mid-Post";
    nameTable["12109"] = "ctx_rh_G_cingul-Post-dorsal";
    nameTable["12110"] = "ctx_rh_G_cingul-Post-ventral";
    nameTable["12111"] = "ctx_rh_G_cuneus";
    nameTable["12112"] = "ctx_rh_G_front_inf-Opercular";
    nameTable["12113"] = "ctx_rh_G_front_inf-Orbital";
    nameTable["12114"] = "ctx_rh_G_front_inf-Triangul";
    nameTable["12115"] = "ctx_rh_G_front_middle";
    nameTable["12116"] = "ctx_rh_G_front_sup";
    nameTable["12117"] = "ctx_rh_G_Ins_lg_and_S_cent_ins";
    nameTable["12118"] = "ctx_rh_G_insular_short";
    nameTable["12119"] = "ctx_rh_G_occipital_middle";
    nameTable["12120"] = "ctx_rh_G_occipital_sup";
    nameTable["12121"] = "ctx_rh_G_oc-temp_lat-fusifor";
    nameTable["12122"] = "ctx_rh_G_oc-temp_med-Lingual";
    nameTable["12123"] = "ctx_rh_G_oc-temp_med-Parahip";
    nameTable["12124"] = "ctx_rh_G_orbital";
    nameTable["12125"] = "ctx_rh_G_pariet_inf-Angular";
    nameTable["12126"] = "ctx_rh_G_pariet_inf-Supramar";
    nameTable["12127"] = "ctx_rh_G_parietal_sup";
    nameTable["12128"] = "ctx_rh_G_postcentral";
    nameTable["12129"] = "ctx_rh_G_precentral";
    nameTable["12130"] = "ctx_rh_G_precuneus";
    nameTable["12131"] = "ctx_rh_G_rectus";
    nameTable["12132"] = "ctx_rh_G_subcallosal";
    nameTable["12133"] = "ctx_rh_G_temp_sup-G_T_transv";
    nameTable["12134"] = "ctx_rh_G_temp_sup-Lateral";
    nameTable["12135"] = "ctx_rh_G_temp_sup-Plan_polar";
    nameTable["12136"] = "ctx_rh_G_temp_sup-Plan_tempo";
    nameTable["12137"] = "ctx_rh_G_temporal_inf";
    nameTable["12138"] = "ctx_rh_G_temporal_middle";
    nameTable["12139"] = "ctx_rh_Lat_Fis-ant-Horizont";
    nameTable["12140"] = "ctx_rh_Lat_Fis-ant-Vertical";
    nameTable["12141"] = "ctx_rh_Lat_Fis-post";
    nameTable["12142"] = "ctx_rh_Medial_wall";
    nameTable["12143"] = "ctx_rh_Pole_occipital";
    nameTable["12144"] = "ctx_rh_Pole_temporal";
    nameTable["12145"] = "ctx_rh_S_calcarine";
    nameTable["12146"] = "ctx_rh_S_central";
    nameTable["12147"] = "ctx_rh_S_cingul-Marginalis";
    nameTable["12148"] = "ctx_rh_S_circular_insula_ant";
    nameTable["12149"] = "ctx_rh_S_circular_insula_inf";
    nameTable["12150"] = "ctx_rh_S_circular_insula_sup";
    nameTable["12151"] = "ctx_rh_S_collat_transv_ant";
    nameTable["12152"] = "ctx_rh_S_collat_transv_post";
    nameTable["12153"] = "ctx_rh_S_front_inf";
    nameTable["12154"] = "ctx_rh_S_front_middle";
    nameTable["12155"] = "ctx_rh_S_front_sup";
    nameTable["12156"] = "ctx_rh_S_interm_prim-Jensen";
    nameTable["12157"] = "ctx_rh_S_intrapariet_and_P_trans";
    nameTable["12158"] = "ctx_rh_S_oc_middle_and_Lunatus";
    nameTable["12159"] = "ctx_rh_S_oc_sup_and_transversal";
    nameTable["12160"] = "ctx_rh_S_occipital_ant";
    nameTable["12161"] = "ctx_rh_S_oc-temp_lat";
    nameTable["12162"] = "ctx_rh_S_oc-temp_med_and_Lingual";
    nameTable["12163"] = "ctx_rh_S_orbital_lateral";
    nameTable["12164"] = "ctx_rh_S_orbital_med-olfact";
    nameTable["12165"] = "ctx_rh_S_orbital-H_Shaped";
    nameTable["12166"] = "ctx_rh_S_parieto_occipital";
    nameTable["12167"] = "ctx_rh_S_pericallosal";
    nameTable["12168"] = "ctx_rh_S_postcentral";
    nameTable["12169"] = "ctx_rh_S_precentral-inf-part";
    nameTable["12170"] = "ctx_rh_S_precentral-sup-part";
    nameTable["12171"] = "ctx_rh_S_suborbital";
    nameTable["12172"] = "ctx_rh_S_subparietal";
    nameTable["12173"] = "ctx_rh_S_temporal_inf";
    nameTable["12174"] = "ctx_rh_S_temporal_sup";
    nameTable["12175"] = "ctx_rh_S_temporal_transverse";
    nameTable["13100"] = "wm_lh_Unknown";
    nameTable["13101"] = "wm_lh_G_and_S_frontomargin";
    nameTable["13102"] = "wm_lh_G_and_S_occipital_inf";
    nameTable["13103"] = "wm_lh_G_and_S_paracentral";
    nameTable["13104"] = "wm_lh_G_and_S_subcentral";
    nameTable["13105"] = "wm_lh_G_and_S_transv_frontopol";
    nameTable["13106"] = "wm_lh_G_and_S_cingul-Ant";
    nameTable["13107"] = "wm_lh_G_and_S_cingul-Mid-Ant";
    nameTable["13108"] = "wm_lh_G_and_S_cingul-Mid-Post";
    nameTable["13109"] = "wm_lh_G_cingul-Post-dorsal";
    nameTable["13110"] = "wm_lh_G_cingul-Post-ventral";
    nameTable["13111"] = "wm_lh_G_cuneus";
    nameTable["13112"] = "wm_lh_G_front_inf-Opercular";
    nameTable["13113"] = "wm_lh_G_front_inf-Orbital";
    nameTable["13114"] = "wm_lh_G_front_inf-Triangul";
    nameTable["13115"] = "wm_lh_G_front_middle";
    nameTable["13116"] = "wm_lh_G_front_sup";
    nameTable["13117"] = "wm_lh_G_Ins_lg_and_S_cent_ins";
    nameTable["13118"] = "wm_lh_G_insular_short";
    nameTable["13119"] = "wm_lh_G_occipital_middle";
    nameTable["13120"] = "wm_lh_G_occipital_sup";
    nameTable["13121"] = "wm_lh_G_oc-temp_lat-fusifor";
    nameTable["13122"] = "wm_lh_G_oc-temp_med-Lingual";
    nameTable["13123"] = "wm_lh_G_oc-temp_med-Parahip";
    nameTable["13124"] = "wm_lh_G_orbital";
    nameTable["13125"] = "wm_lh_G_pariet_inf-Angular";
    nameTable["13126"] = "wm_lh_G_pariet_inf-Supramar";
    nameTable["13127"] = "wm_lh_G_parietal_sup";
    nameTable["13128"] = "wm_lh_G_postcentral";
    nameTable["13129"] = "wm_lh_G_precentral";
    nameTable["13130"] = "wm_lh_G_precuneus";
    nameTable["13131"] = "wm_lh_G_rectus";
    nameTable["13132"] = "wm_lh_G_subcallosal";
    nameTable["13133"] = "wm_lh_G_temp_sup-G_T_transv";
    nameTable["13134"] = "wm_lh_G_temp_sup-Lateral";
    nameTable["13135"] = "wm_lh_G_temp_sup-Plan_polar";
    nameTable["13136"] = "wm_lh_G_temp_sup-Plan_tempo";
    nameTable["13137"] = "wm_lh_G_temporal_inf";
    nameTable["13138"] = "wm_lh_G_temporal_middle";
    nameTable["13139"] = "wm_lh_Lat_Fis-ant-Horizont";
    nameTable["13140"] = "wm_lh_Lat_Fis-ant-Vertical";
    nameTable["13141"] = "wm_lh_Lat_Fis-post";
    nameTable["13142"] = "wm_lh_Medial_wall";
    nameTable["13143"] = "wm_lh_Pole_occipital";
    nameTable["13144"] = "wm_lh_Pole_temporal";
    nameTable["13145"] = "wm_lh_S_calcarine";
    nameTable["13146"] = "wm_lh_S_central";
    nameTable["13147"] = "wm_lh_S_cingul-Marginalis";
    nameTable["13148"] = "wm_lh_S_circular_insula_ant";
    nameTable["13149"] = "wm_lh_S_circular_insula_inf";
    nameTable["13150"] = "wm_lh_S_circular_insula_sup";
    nameTable["13151"] = "wm_lh_S_collat_transv_ant";
    nameTable["13152"] = "wm_lh_S_collat_transv_post";
    nameTable["13153"] = "wm_lh_S_front_inf";
    nameTable["13154"] = "wm_lh_S_front_middle";
    nameTable["13155"] = "wm_lh_S_front_sup";
    nameTable["13156"] = "wm_lh_S_interm_prim-Jensen";
    nameTable["13157"] = "wm_lh_S_intrapariet_and_P_trans";
    nameTable["13158"] = "wm_lh_S_oc_middle_and_Lunatus";
    nameTable["13159"] = "wm_lh_S_oc_sup_and_transversal";
    nameTable["13160"] = "wm_lh_S_occipital_ant";
    nameTable["13161"] = "wm_lh_S_oc-temp_lat";
    nameTable["13162"] = "wm_lh_S_oc-temp_med_and_Lingual";
    nameTable["13163"] = "wm_lh_S_orbital_lateral";
    nameTable["13164"] = "wm_lh_S_orbital_med-olfact";
    nameTable["13165"] = "wm_lh_S_orbital-H_Shaped";
    nameTable["13166"] = "wm_lh_S_parieto_occipital";
    nameTable["13167"] = "wm_lh_S_pericallosal";
    nameTable["13168"] = "wm_lh_S_postcentral";
    nameTable["13169"] = "wm_lh_S_precentral-inf-part";
    nameTable["13170"] = "wm_lh_S_precentral-sup-part";
    nameTable["13171"] = "wm_lh_S_suborbital";
    nameTable["13172"] = "wm_lh_S_subparietal";
    nameTable["13173"] = "wm_lh_S_temporal_inf";
    nameTable["13174"] = "wm_lh_S_temporal_sup";
    nameTable["13175"] = "wm_lh_S_temporal_transverse";
    nameTable["14100"] = "wm_rh_Unknown";
    nameTable["14101"] = "wm_rh_G_and_S_frontomargin";
    nameTable["14102"] = "wm_rh_G_and_S_occipital_inf";
    nameTable["14103"] = "wm_rh_G_and_S_paracentral";
    nameTable["14104"] = "wm_rh_G_and_S_subcentral";
    nameTable["14105"] = "wm_rh_G_and_S_transv_frontopol";
    nameTable["14106"] = "wm_rh_G_and_S_cingul-Ant";
    nameTable["14107"] = "wm_rh_G_and_S_cingul-Mid-Ant";
    nameTable["14108"] = "wm_rh_G_and_S_cingul-Mid-Post";
    nameTable["14109"] = "wm_rh_G_cingul-Post-dorsal";
    nameTable["14110"] = "wm_rh_G_cingul-Post-ventral";
    nameTable["14111"] = "wm_rh_G_cuneus";
    nameTable["14112"] = "wm_rh_G_front_inf-Opercular";
    nameTable["14113"] = "wm_rh_G_front_inf-Orbital";
    nameTable["14114"] = "wm_rh_G_front_inf-Triangul";
    nameTable["14115"] = "wm_rh_G_front_middle";
    nameTable["14116"] = "wm_rh_G_front_sup";
    nameTable["14117"] = "wm_rh_G_Ins_lg_and_S_cent_ins";
    nameTable["14118"] = "wm_rh_G_insular_short";
    nameTable["14119"] = "wm_rh_G_occipital_middle";
    nameTable["14120"] = "wm_rh_G_occipital_sup";
    nameTable["14121"] = "wm_rh_G_oc-temp_lat-fusifor";
    nameTable["14122"] = "wm_rh_G_oc-temp_med-Lingual";
    nameTable["14123"] = "wm_rh_G_oc-temp_med-Parahip";
    nameTable["14124"] = "wm_rh_G_orbital";
    nameTable["14125"] = "wm_rh_G_pariet_inf-Angular";
    nameTable["14126"] = "wm_rh_G_pariet_inf-Supramar";
    nameTable["14127"] = "wm_rh_G_parietal_sup";
    nameTable["14128"] = "wm_rh_G_postcentral";
    nameTable["14129"] = "wm_rh_G_precentral";
    nameTable["14130"] = "wm_rh_G_precuneus";
    nameTable["14131"] = "wm_rh_G_rectus";
    nameTable["14132"] = "wm_rh_G_subcallosal";
    nameTable["14133"] = "wm_rh_G_temp_sup-G_T_transv";
    nameTable["14134"] = "wm_rh_G_temp_sup-Lateral";
    nameTable["14135"] = "wm_rh_G_temp_sup-Plan_polar";
    nameTable["14136"] = "wm_rh_G_temp_sup-Plan_tempo";
    nameTable["14137"] = "wm_rh_G_temporal_inf";
    nameTable["14138"] = "wm_rh_G_temporal_middle";
    nameTable["14139"] = "wm_rh_Lat_Fis-ant-Horizont";
    nameTable["14140"] = "wm_rh_Lat_Fis-ant-Vertical";
    nameTable["14141"] = "wm_rh_Lat_Fis-post";
    nameTable["14142"] = "wm_rh_Medial_wall";
    nameTable["14143"] = "wm_rh_Pole_occipital";
    nameTable["14144"] = "wm_rh_Pole_temporal";
    nameTable["14145"] = "wm_rh_S_calcarine";
    nameTable["14146"] = "wm_rh_S_central";
    nameTable["14147"] = "wm_rh_S_cingul-Marginalis";
    nameTable["14148"] = "wm_rh_S_circular_insula_ant";
    nameTable["14149"] = "wm_rh_S_circular_insula_inf";
    nameTable["14150"] = "wm_rh_S_circular_insula_sup";
    nameTable["14151"] = "wm_rh_S_collat_transv_ant";
    nameTable["14152"] = "wm_rh_S_collat_transv_post";
    nameTable["14153"] = "wm_rh_S_front_inf";
    nameTable["14154"] = "wm_rh_S_front_middle";
    nameTable["14155"] = "wm_rh_S_front_sup";
    nameTable["14156"] = "wm_rh_S_interm_prim-Jensen";
    nameTable["14157"] = "wm_rh_S_intrapariet_and_P_trans";
    nameTable["14158"] = "wm_rh_S_oc_middle_and_Lunatus";
    nameTable["14159"] = "wm_rh_S_oc_sup_and_transversal";
    nameTable["14160"] = "wm_rh_S_occipital_ant";
    nameTable["14161"] = "wm_rh_S_oc-temp_lat";
    nameTable["14162"] = "wm_rh_S_oc-temp_med_and_Lingual";
    nameTable["14163"] = "wm_rh_S_orbital_lateral";
    nameTable["14164"] = "wm_rh_S_orbital_med-olfact";
    nameTable["14165"] = "wm_rh_S_orbital-H_Shaped";
    nameTable["14166"] = "wm_rh_S_parieto_occipital";
    nameTable["14167"] = "wm_rh_S_pericallosal";
    nameTable["14168"] = "wm_rh_S_postcentral";
    nameTable["14169"] = "wm_rh_S_precentral-inf-part";
    nameTable["14170"] = "wm_rh_S_precentral-sup-part";
    nameTable["14171"] = "wm_rh_S_suborbital";
    nameTable["14172"] = "wm_rh_S_subparietal";
    nameTable["14173"] = "wm_rh_S_temporal_inf";
    nameTable["14174"] = "wm_rh_S_temporal_sup";
    nameTable["14175"] = "wm_rh_S_temporal_transverse";
    return nameTable;
}
